// Copyright (C) 2002 Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "KNN.h"

namespace Torch {

KNN::KNN(DataSet* data_,int K_)
{
  data = data_;
  distances = NULL;
  indices = NULL;
  setK(K_);
  n_outputs = data->n_targets;
  addToList(&outputs, n_outputs, (real *)xalloc(sizeof(real)*n_outputs));
  n_real_examples = 0;
  real_examples = NULL;
}

void KNN::reset()
{
  data->setExample(0);
  n_real_examples = data->n_examples;
  real_examples = (int*)xrealloc(real_examples,n_real_examples*sizeof(int));
  for (int i=0;i<data->n_examples;i++) {
    real_examples[i] = data->selected_examples[i];
  }
}

void KNN::setK(int K_)
{
  K = K_;
  distances = (real *)xrealloc(distances,K*sizeof(real));
  indices = (int *)xrealloc(indices,K*sizeof(int));
}

void KNN::forward(List* inputs)
{
  // keep current example in order to restore at the end
  int current_example = data->current_example;
  real* inputs_ptr = (real*)inputs->ptr;

  // verify that n_examples > K
  int old_K = K;
  if (n_real_examples < K)
    K = n_real_examples;

  // initialization of distances to big values;
  for (int i=0;i<K;i++) {
    distances[i] = 1e35;
    indices[i] = -1;
  }

  // compute the K nearest neighbords

  // for each vector in data
  int* i_ptr = real_examples;
  for (int i=0;i<n_real_examples;i++) {
    data->setRealExample(*i_ptr++);
    // calculate euclidean distance between example and current vector
    real dist = 0;
    real *x = inputs_ptr;
    real *datax = (real *)data->inputs->ptr;
    for(int j = 0; j < data->n_inputs; j++) {
      real diff = *x++ - *datax++;
      dist += diff*diff;
    }
    // eventually add current vector to K nearest neighbors
    if (dist < distances[K-1]) {
      // find insertion point
      real* bptr = distances;
      real* eptr = distances + K - 1;
      real* mptr = bptr + (eptr - bptr) / 2;
      do {
        if (dist < *mptr)
          eptr = mptr;
        else
          bptr = mptr + 1;
        mptr = bptr + (eptr - bptr) / 2;
      } while (mptr < eptr);
      // insert the point by shifting all subsequent distances
      eptr = distances + K - 1;
      bptr = eptr - 1;

      int* eptr_idx = indices + K - 1;
      int* bptr_idx = eptr_idx - 1;

      while (eptr > mptr) {
        *eptr-- = *bptr--;      /*   distances   */
        *eptr_idx-- = *bptr_idx--;      /*   indices   */
      }
      *mptr = dist;
      indices[mptr - distances] = data->current_example;
    }
  }

  // give an answer as the mean of the answers of the KNNs
  // initialize outputs to null
  real* out = (real*)outputs->ptr;
  for (int j=0;j<n_outputs;j++)
    *out++ = 0;
  for (int i=0;i<K;i++) {
    out = (real*)outputs->ptr;
    data->setRealExample(indices[i]);
    real *targ = (real*)data->targets;
    for (int j=0;j<n_outputs;j++)
      *out++ += *targ++;
  }
  out = (real*)outputs->ptr;
  for (int j=0;j<n_outputs;j++)
    *out++ /= (real)K;

  // in case K was modified
  K = old_K;
  // restore current_example
  data->setRealExample(current_example);
}

KNN::~KNN()
{
  freeList(&outputs,true);
  free(distances);
  free(indices);
  free(real_examples);
}

}

