// Copyright (C) 2002 Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef EDIT_DISTANCE_INC
#define EDIT_DISTANCE_INC

#include "general.h"
#include "Object.h"

namespace Torch {

/** This class can be used to compute the "edit distance" between
    two sequences. It computes the number of insertions, deletions and
    substitutions. The overall distance is the sum of these numbers
    weighted by their cost (which are intergers equal to 1 by default).

    @author Samy Bengio (bengio@idiap.ch)
*/
class EditDistance : public Object
{
  public:
    /// the total edit distance between two sequences
    real accuracy;
    /// the number of insertions (weighted by their cost)
    int n_insert;
    /// the number of deletions (weighted by their cost)
    int n_delete;
    /// the number of substitutions (weighted by their cost)
    int n_subst;
    /// the number of sequences measured (used to normalize #dist# by #n_seq#)
    int n_seq;
    /// the cost of one insertion
    int insert_cost;
    /// the cost of one deletion
    int delete_cost;
    /// the cost of one substitution
    int subst_cost;

    /// the obtained sequence
    int* obtained;
    /// the size of the obtained sequence
    int obt_size;

    /// the desired sequence
    int* desired;
    /// the size of the desired sequence
    int des_size;

    ///
    EditDistance();

    /// sets the different costs
    virtual void setCosts(int i_cost, int d_cost, int s_cost);

    /// computes the edit distance between #obtained# and #desired#.
    virtual void distance(int* obtained, int obt_size, int* desired, int des_size);

    /// accumulates the distances of the current object and the given object
    virtual void add(EditDistance* d);

    /// prints the edit distance and optionally the sequences
    virtual void print(FILE *f);

    /** prints the edit distance ratio (divided by the number of sequnces) 
        and optionally the obtained and desired sequences
    */
    virtual void printRatio(FILE *f);

    virtual void reset();
    ~EditDistance();
};


}

#endif
