// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "WeightedMseCriterion.h"

namespace Torch {

WeightedMseCriterion::WeightedMseCriterion(DataSet *data_)
{
  data = data_;
  n_inputs = data->n_targets;
  weights = NULL;
  weights_is_allocated = false;
}

WeightedMseCriterion::WeightedMseCriterion(DataSet *data_, real *weights_)
{
  data = data_;
  n_inputs = data->n_targets;
  weights = weights_;
  weights_is_allocated = false;
}

void WeightedMseCriterion::setDataSet(DataSet *data_)
{
  if(weights_is_allocated)
  {
    if(data_->n_real_examples != data->n_real_examples)
      error("WeightedMseCriterion: trying to use a wrong DataSet");
  }

  data = data_;
}

void WeightedMseCriterion::allocateMemory()
{
  GradientMachine::allocateMemory();  
  if(!weights)
  {
    weights = (real *)xalloc(sizeof(real)*data->n_real_examples);
    for(int i = 0; i < data->n_real_examples; i++)
      weights[i] = 1;
    weights_is_allocated = true;  
  }
}

void WeightedMseCriterion::freeMemory()
{
  if(is_free)
    return;

  GradientMachine::freeMemory();
  if(weights_is_allocated)
    free(weights);

  is_free = true;
}

void WeightedMseCriterion::forward(List *inputs)
{
  real *desired = (real *)data->targets;
  real err = 0;
  while(inputs)
  {
    real *x = (real *)inputs->ptr;
    for(int j = 0; j < inputs->n; j++)
    {
      real z = *desired++ - *x++;
      err += z*z;
    }
    inputs = inputs->next;
  }

  real *ptr_out = (real *)outputs->ptr;
  ptr_out[0] = err*weights[data->current_example];
//  printf("%g %g\n", data->targets[0], weights[data->current_example]);
}

void WeightedMseCriterion::backward(List *inputs, real *alpha)
{
  real *desired = (real *)data->targets;
  real *ptr_beta = beta;
  real z = weights[data->current_example];

  while(inputs)
  {
    real *x = (real *)inputs->ptr;
    for(int j = 0; j < inputs->n; j++)
      *ptr_beta++ = 2.*z*(*x++ - *desired++);
    inputs = inputs->next;
  }
//  printf("%g %g\n", data->targets[0], weights[data->current_example]);
}

WeightedMseCriterion::~WeightedMseCriterion()
{
  freeMemory();
}

}

