// Copyright (C) 2002 Zbigniew Leyk (zbigniew.leyk@anu.edu.au)
//                and David E. Stewart (david.stewart@anu.edu.au)
//                and Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef MAT_OPERATIONS_INC
#define MAT_OPERATIONS_INC

#include "Mat.h"
#include "Vec.h"

namespace Torch {

/** Collection of matrix operation functions.
    Based on the "Meschach Library", available at the
    anonymous ftp site thrain.anu.edu.au in the directory
    pub/meschach.

    @author David E. Stewart (david.stewart@anu.edu.au)
    @author Zbigniew Leyk (zbigniew.leyk@anu.edu.au)
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
//@{
/// Matrix addition -- may be in-situ
void mxMatAddMat(Mat * mat1, Mat * mat2, Mat * out);
/// Matrix subtraction -- may be in-situ
void mxMatSubMat(Mat * mat1, Mat * mat2, Mat * out);
/// Matrix-matrix multiplication
void mxMatMulMat(Mat * mat1, Mat * mat2, Mat * out);
/** Matrix-matrix transposed multiplication.
    -- #A.B^T# is stored in out */
void mxMatMulTrMat(Mat * mat1, Mat * mat2, Mat * out);
/** Matrix transposed-matrix multiplication.
    -- #A^T.B# is stored in out */
void mxTrMatMulMat(Mat * mat1, Mat * mat2, Mat * out);
/** Matrix-vector multiplication.
    -- Note: #b# is treated as a column vector */
void mxMatMulVec(Mat * mat, Vec * b, Vec * out);
/// Scalar-matrix multiply -- may be in-situ
void mxRealMulMat(real scalar, Mat * matrix, Mat * out);
/** Vector-matrix multiplication. 
    -- Note: #b# is treated as a row vector */
void mxVecMulMat(Vec * b, Mat * mat, Vec * out);
/// Transpose matrix
void mxTrMat(Mat * in, Mat * out);
/** Swaps rows i and j of matrix A upto column lim.
    #lo# and #hi# to -1 if you want to swap all */
void mxSwapRowsMat(Mat * mat, int i, int j, int lo, int hi);
/** Swap columns i and j of matrix A upto row lim.
    #lo# and #hi# to -1 if you want to swap all */
void mxSwapColsMat(Mat * mat, int i, int j, int lo, int hi);
/** Matrix-scalar multiply and add.
    -- may be in situ.
    -- returns out == A1 + s*A2 */
void mxMatAddRealMulMat(Mat * mat1, Mat * mat2, real s, Mat * out);
/** Matrix-vector multiply and add.
    -- may not be in situ
    -- returns out == v1 + alpha*A*v2 */
void mxVecAddRealMulMatMulVec(Vec * v1, real alpha, Mat * mat, Vec * v2,
			      Vec * out);
/** Vector-matrix multiply and add
    -- may not be in situ
    -- returns out' == v1' + alpha * v2'*A */
void mxVecAddRealMulVecMulMat(Vec * v1, real alpha, Vec * v2, Mat * mat,
			      Vec * out);
//@}


}

#endif
