/* tuxtime-conf.c -- Configuration manager for TuxTime
 *
 * Copyright (c) 1996-2000  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: tuxtime-conf.c,v $
 * Revision 1.1  1999/12/04 13:47:47  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static const char rcsid[]="$Id: tuxtime-conf.c,v 1.1 1999/12/04 13:47:47 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#include<sys/stat.h>
#include<errno.h>
#include<signal.h>
#include<paths.h>
#include<pwd.h>
#include<features.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif
#include<gtk/gtk.h>

#include"sci.h"
#include"hci.h"
#include"machine.h"
#include"alarms.h"
#include"config.h"
#include"misc.h"

#include"tuxtime.xpm"


#define USAGE_STRING \
"Usage: tuxtime-conf [OPTION]\n\n\
  -a,--alarms     start with the battery alarm settings\n\
  -s,--settings   start with the machine settings\n\
  -h,--help       display this help message\n\
  -v,--version    display version\n\
Report bugs to jonathan@buzzard.org.uk\n"

#define VERSION_STRING "\
TuxTime Configuration Manager version "VERSION"\n\
Copyright (c) 1996-2000 Jonathan A. Buzzard <jonathan@buzzard.org.uk>\n"

enum { ALARM, SETTINGS };


int id;
char *config;


/*
 * Deal with window manager delete events
 */
void Deleted(GtkWidget *widget, GdkEvent *event, gpointer *data)
{
	unlink(PID_FILE);
	gtk_main_quit();
}


/*
 * Catch any signals and exit the program nicely
 */
void CatchSignal(int x)
{
	fprintf(stderr, "tuxtime-conf: caught signal %d, cleaning up...\n", x);

	SciCloseInterface();
	unlink(PID_FILE);
	exit(0);
}


/*
 * If we recieve a signal, exit nicely freeing up resources
 */
void HandleSignals(void)
{
	signal(SIGHUP, CatchSignal);
	signal(SIGINT, CatchSignal);
	signal(SIGQUIT, CatchSignal);
	signal(SIGILL, CatchSignal);
	signal(SIGTRAP, CatchSignal);
	signal(SIGABRT, CatchSignal);
	signal(SIGIOT, CatchSignal);
	signal(SIGFPE, CatchSignal);
	signal(SIGKILL, CatchSignal);
	signal(SIGSEGV, CatchSignal);
	signal(SIGPIPE, CatchSignal);
	signal(SIGTERM, CatchSignal);
	signal(SIGCHLD, CatchSignal);
	signal(SIGSTOP, CatchSignal);
	signal(SIGTTIN, CatchSignal);
	signal(SIGTTOU, CatchSignal);
	signal(SIGUSR1, SIG_IGN);

	return;
}


/*
 * Process the non-GTK command line arguments
 */
int ProcessComandLine(int *argc, char ***argv)
{
	int i,j;

	j = SETTINGS;
	for (i=1;i<*argc;i++) {
		if ((!strcmp((*argv)[i], "-h")) ||
			(!strcmp((*argv)[i], "--help"))) {
				printf(USAGE_STRING);
				unlink(PID_FILE);
				exit(0);
		}
		if ((!strcmp((*argv)[i], "-v")) ||
			(!strcmp((*argv)[i], "--version"))) {
				printf(VERSION_STRING);
				unlink(PID_FILE);
				exit(0);
		}
		if ((!strcmp((*argv)[i], "-a")) ||
			(!strcmp((*argv)[i], "--alarms"))) {
				j = ALARM;
		}
		if ((!strcmp((*argv)[i], "-s")) ||
			(!strcmp((*argv)[i],"--settings"))) {
				j = SETTINGS;
		}
	}

	return j;
}


/*
 * the entry point of TuxTime configuration manager
 */
int main(int argc, char *argv[])
{
	int pid,version,dialog;
	FILE *str;
	struct passwd *pw;
	struct stat info;
	GtkWidget *window;
	GtkStyle *style;
	GdkPixmap *icon;
	GdkBitmap *mask;


	/* do a quick check on the laptop */

	if (SciSupportCheck(&version)==SCI_FAILURE) {
		fprintf(stderr, "tuxtime-conf: this computer is not "
			"supported or the kernel module not installed.\n");
		return 1;
	}
	if (HciGetMachineID(&id)==HCI_FAILURE) {
		fprintf(stderr, "tuxtime-conf: unable to get machine "
			"identification\n");
		return 1;
	}

	/* check to see if a copy of config manager is already running */

	if (!access(PID_FILE, R_OK)) {
		if ((str = fopen(PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);

			/* test to see if the other TuxTime died unexpectedly */

			if (kill(pid, SIGUSR1)==0) {
				fprintf(stderr, "tuxtime-conf: already running "
					"as process %d.\n", pid);
				exit(1);
			}

			fprintf(stderr, "tuxtime-conf: process %d appears to "
				"have died, continuing\n", pid);
			unlink(PID_FILE);
		}
	}

	/* get the path of the tuxtimerc file */

	pw = getpwuid(getuid());
	config = (char *) malloc(strlen(pw->pw_dir)+12);
	if (config==NULL) {
		fprintf(stderr, "tuxtime-conf: unable to allocate sufficent "
			"memory, exiting\n");
		exit(1);
		}
	strcpy(config, pw->pw_dir);
	strcat(config, "/.tuxtimerc");

	/* test to see if tuxtimerc file exists and if no create one */

	if ((stat(config, &info)!=0) && (errno==ENOENT)) {
		CreateDefaultRC(config);
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}

	/* initialize the GTK toolkit */

	gtk_init(&argc, &argv);

	/* install signal handling after GTK toolkit is initialize otherwise
	   it will override our signal handling */

	HandleSignals();

	/* Gtk options now cleared, process remaining command line arguments */

	dialog = ProcessComandLine(&argc, &argv);

	/* create the top level window */

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_container_border_width(GTK_CONTAINER(window), 6);
	gtk_signal_connect(GTK_OBJECT(window), "delete_event",
		GTK_SIGNAL_FUNC(Deleted), NULL);
	gtk_widget_realize(window);

	/* add window tile and icon */

	gtk_window_set_title(GTK_WINDOW(window),
		"TuxTime Configuration Manager");
	style = gtk_widget_get_style(window);
	icon = gdk_pixmap_create_from_xpm_d(window->window, &mask,
		&style->bg[GTK_STATE_NORMAL], (gchar **)tuxtime_xpm);
	gdk_window_set_icon(window->window, NULL, icon, mask);

	/* display the requested dialog box */

	if (dialog==ALARM)
		AlarmDialog(window);
	else
		MachineDialog(window);

	/* start the program */

	gtk_widget_show_all(window);
	gtk_main();

	return 0;
}
