/***********************************************************************
 *
 * File: event_cond.c
 *
 * Author: Martin Fluch
 *
 *     Copyright (c) 1999 Martin Fluch, All rights reserved
 *     
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *     GNU General Public License for more details.
 * 
 *     To receive a copy of the GNU General Public License, please write
 *     to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *     Boston, MA 02111-1307 USA
 *
 ***********************************************************************/

#include <unistd.h>
#include <stdlib.h>
#include <linux/types.h>
#include <stdio.h>
#include <string.h>

#include <curses.h>
#include <menu.h>

#include "ntpctl.h"


#define BIT_STANDBY     3
#define BIT_SUSPEND	4
#define BIT_REDISAFE	5
#define BIT_HIBERNATION 6
#define BIT_POWEROFF	7

/***********************************************************************
 *
 * get_event_bits();
 *
 ***********************************************************************/

int get_event_bits(int func, int nr, word *cap, word *cond)
{
  smapi_outparm_t *result;
  word tmp;

  if((result=do_smapi(func,0x00,0,0,0,0,0,FALSE))==NULL) return 1;
  
  switch(nr)
    {
    case 1: tmp=result->wParm1;	          break;
    case 2: tmp=result->wParm2;		  break;
    case 3: tmp=result->wParm3;		  break;
    case 4: tmp=result->dwParm4 & 0xffff; break;
    case 5: tmp=result->dwParm5 & 0xffff; break;
    }

  *cap =(tmp>>8) & 0xff;
  *cond= tmp     & 0xff;

  return 0;
}

/***********************************************************************
 *
 * set_event_bits();
 *
 ***********************************************************************/

int set_event_bits(int func, int nr, word cond)
{
  smapi_outparm_t *result;

  if((result=do_smapi(func,0x00,0,0,0,0,0,FALSE))==NULL) return 1;
  
  switch(nr)
    {
    case 1: result->wParm1  = cond; break;
    case 2: result->wParm2  = cond; break;
    case 3: result->wParm3  = cond; break;
    case 4: result->dwParm4 = cond; break;
    case 5: result->dwParm5 = cond; break;
    }

  if(do_smapi(func,0x01,result->wParm2,result->wParm1,
	      0,result->dwParm4, result->dwParm5,TRUE)==NULL) 
    return 1;

  return 0;
}

/***********************************************************************
 *
 * cond2str();
 *
 ***********************************************************************/

void cond2str(word cond, char *str)
{
  static const char tab[]="SZRHO";
  int i;

  for(i=0;i<5;i++) str[i]=(cond & (0x0001<<(i+3))) ? tab[i] : '-';

  str[5]=0;
}

/***********************************************************************
 *
 * pm_sedate_standby_timer();
 *
 ***********************************************************************/

int handle_system_event_cond(struct menuinfo *mi, int action,
			     int func, int nr)
{
  word cond,cap;
  static char condstr[6],capstr[6];
  int chconfig=FALSE;

  switch(action)
    {
    case ACTION_KEY_0: chconfig=TRUE; cond=0; break;
    case ACTION_KEY_S: chconfig=TRUE; cond=1<<BIT_STANDBY; break;
    case ACTION_KEY_Z: chconfig=TRUE; cond=1<<BIT_SUSPEND; break;
    case ACTION_KEY_R: chconfig=TRUE; cond=1<<BIT_REDISAFE; break;
    case ACTION_KEY_H: chconfig=TRUE; cond=1<<BIT_HIBERNATION; break;
    case ACTION_KEY_O: chconfig=TRUE; cond=1<<BIT_POWEROFF; break;
    }

  if(chconfig) if(set_event_bits(func,nr,cond)) return 1;
  if(get_event_bits(func,nr,&cap,&cond)) return 1;

  cond2str(cond, condstr);
  cond2str(cap,  capstr);

  snprintf(mi->desc,MAXDESCSIZE,"%s     (capable: %s)",condstr,capstr);

  return 0;
}

/***********************************************************************
 *
 * pm_sedate_hw_and_sw_event();
 *
 ***********************************************************************/

int pm_sedate_hw_and_sw_event(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM sedate hw and sw event",MAXNAMESIZE);
      strncpy(mi->desc,".....    (capable: .....)",MAXDESCSIZE);
      mi->help=helptext[HELP_szrho];
    }

  return handle_system_event_cond(mi,action,0x31,2);
}

/***********************************************************************
 *
 * pm_sedate_power_switch();
 *
 ***********************************************************************/

int pm_sedate_power_switch(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM sedate power switch",MAXNAMESIZE);
      strncpy(mi->desc,".....    (capable: .....)",MAXDESCSIZE);
      mi->help=helptext[HELP_szrho];
    }

  return handle_system_event_cond(mi,action,0x31,4);
}

/***********************************************************************
 *
 * pm_sedate_lid();
 *
 ***********************************************************************/

int pm_sedate_lid(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM sedate lid closure",MAXNAMESIZE);
      strncpy(mi->desc,".....    (capable: .....)",MAXDESCSIZE);
      mi->help=helptext[HELP_szrho];
    }

  return handle_system_event_cond(mi,action,0x31,5);
}


/***********************************************************************
 *
 * pm_sedate_system_timer();
 *
 ***********************************************************************/

int pm_sedate_system_timer(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM sedate system timer",MAXNAMESIZE);
      strncpy(mi->desc,".....    (capable: .....)",MAXDESCSIZE);
      mi->help=helptext[HELP_szrho];
    }

  return handle_system_event_cond(mi,action,0x32,2);
}

/***********************************************************************
 *
 * pm_sedate_standby_timer();
 *
 ***********************************************************************/

int pm_sedate_standby_timer(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM sedate standby timer",MAXNAMESIZE);
      strncpy(mi->desc,".....    (capable: .....)",MAXDESCSIZE);
      mi->help=helptext[HELP_szrho];
    }

  return handle_system_event_cond(mi,action,0x32,4);
}

/***********************************************************************
 *
 * pm_sedate_hib_from_sup();
 *
 ***********************************************************************/

int pm_sedate_hib_from_sup(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM sedate hibernate from suspend",MAXNAMESIZE);
      strncpy(mi->desc,".....    (capable: .....)",MAXDESCSIZE);
      mi->help=helptext[HELP_szrho];
    }

  return handle_system_event_cond(mi,action,0x32,5);
}

/***********************************************************************
 *
 * pm_sedate_low_battery();
 *
 ***********************************************************************/

int pm_sedate_low_battery(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM sedate battery low",MAXNAMESIZE);
      strncpy(mi->desc,".....    (capable: .....)",MAXDESCSIZE);
      mi->help=helptext[HELP_szrho];
    }

  return handle_system_event_cond(mi,action,0x33,2);
}

/***********************************************************************
 *
 * pm_sedate_env_exhausted();
 *
 ***********************************************************************/

int pm_sedate_env_exhausted(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM sedate env'mt exhausted",MAXNAMESIZE);
      strncpy(mi->desc,".....    (capable: .....)",MAXDESCSIZE);
      mi->help=helptext[HELP_szrho];
    }

  return handle_system_event_cond(mi,action,0x32,5);
}
