// main.c - main routines
//
// Written by
//  Timothy Stark <sword7@speakeasy.org>
//
// This file is part of the TS10 Emulator.
// See ReadMe for copyright notice.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <stdio.h>
#include <signal.h>

#ifdef HAVE_SIGACTION
#include <unistd.h>
#include <stropts.h>
#endif /* HAVE_SIGACTION */

#include "emu/defs.h"
#include "emu/socket.h"

void (*emu_IOTrap)(void) = NULL;
int  emu_State;
int  emu_logFile = -1;

extern char   *ts10_Prompt;
extern SOCKET *ts10_Stdout;
extern MAP_DEVICE *ts10_Select;

#ifdef HAVE_SIGACTION
struct sigaction sigInt;   // ^E Exit
struct sigaction sigSegV;  // Bad memory
struct sigaction sigQuit;  // ^T Trace
struct sigaction sigIO;    // Keyboard activity
#endif /* HAVE_SIGACTION */

void emu_Interrupt(int s)
{
	emu_State = EMU_HALT;

#ifndef HAVE_SIGACTION
	signal(SIGINT, emu_Interrupt);
#endif /* HAVE_SIGACTION */
}

void emu_Trace(int s)
{
#ifdef DEBUG
	if (dbg_Check(DBG_TRACE)) {
		dbg_ClearMode(DBG_TRACE|DBG_DATA);
		printf("[Trace off]\r\n");
	} else {
		dbg_SetMode(DBG_TRACE|DBG_DATA);
		printf("[Trace on]\r\n");
	}
#endif /* DEBUG */

#ifndef HAVE_SIGACTION
	signal(SIGQUIT, emu_Trace);
#endif /* HAVE_SIGACTION */
}

void emu_BadMemory(int s)
{
	ts10_StopTimer();

#if defined(HAVE_SIGACTION) && !defined(linux)
	if (ioctl(STDIN_FILENO, I_SETSIG, 0) < 0) {
		perror("IOCTL I_SETSIG");
	}
#endif /* HAVE_SIGACTION && !linux */

	ts10_Exit("Segmentation Failure");
}

void emu_IO(int s)
{
	if (emu_IOTrap)
		emu_IOTrap();

#ifndef HAVE_SIGACTION
	signal(SIGIO,  emu_IO);
	signal(SIGURG, emu_IO);
#endif /* HAVE_SIGACTION */
}

void ts10_Exit(char *Reason)
{
	char outReason[1024];

	sprintf(outReason, "\r\nTS10 Emulator Exit (Reason: %s)\r\n", Reason);
	sock_Send(1, outReason, 0);
	sock_CloseAll(outReason);
	panel_Cleanup();

#ifdef DEBUG
	// Flush and Close debug file.
	CloseDebug();
#endif /* DEBUG */

	exit(0);
}

int main(int argc, char **argv)
{
	int idx;

#ifdef HAVE_SIGACTION
	sigInt.sa_handler = emu_Interrupt;
	sigInt.sa_flags   = 0;
	sigaction(SIGINT, &sigInt, NULL);

	sigQuit.sa_handler = emu_Trace;
	sigQuit.sa_flags   = 0;
	sigaction(SIGQUIT, &sigQuit, NULL);

	sigSegV.sa_handler = emu_BadMemory;
	sigSegV.sa_flags   = 0;
//	sigaction(SIGSEGV, &sigSegV, NULL);

	sigIO.sa_handler = emu_IO;
	sigIO.sa_flags   = 0;
	sigaction(SIGIO, &sigIO, NULL);
	sigaction(SIGURG, &sigIO, NULL);
#else
	signal(SIGINT,  emu_Interrupt);
	signal(SIGQUIT, emu_Trace);
//	signal(SIGSEGV, emu_BadMemory);
	signal(SIGIO,   emu_IO);
	signal(SIGURG,  emu_IO);
#endif /* HAVE_SIGACTION */

	InitSystem();       // Initialize Emulator System
	InitSockets();      // Initialize Socket Handler
	InitControlPanel(); // Initialize Panel Control Handler

#ifdef DEBUG
	OpenDebug(NULL);
#endif /* DEBUG */

	emu_State = EMU_CONSOLE;

	for (idx = 1; idx < argc; idx++) {
		if (argv[idx][0] == '-') {
			switch (argv[idx][1]) {
				case 'f': // Configuration file
					console_ExecuteFile(argv[++idx]);
					break;

				default:
					printf("Unknown option: %s\n", argv[idx]);
			}
		}
	}
	
	// Tell operator that it is ready now.
	sock_Send(ts10_Stdout->idSocket, ts10_Prompt, 0);

	while (emu_State) {
		if (emu_State == EMU_RUN) {
			if (ts10_Select) {
				DEVICE *devInfo = ts10_Select->devInfo;
				if (devInfo->Execute)
					devInfo->Execute(ts10_Select);
				else {
					printf("%s: Execute Not Supported.\n",
						ts10_Select->devName);
				}
			} else
				printf("Please type 'select <device>' first.\n");
			emu_State = EMU_CONSOLE;
		}
		pause();
	}

	ts10_Exit("Exit");
}
