#ifndef INCLUDED_READ_H
#define INCLUDED_READ_H
/* vim:ts=4:sw=4:noet
 * (tabspace=4)
 * 
 * Copyright (C) 2004, 2005 Walter Doekes, <walter@djcvt.net>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "types.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 * Use this for loading smallish files. Larger files would not fit into memory
 * and behaviour for files larger than INT_MAX is undefined.
 *  
 * @param[in] fd The file descriptor to read from.
 * @param[in] read_size The size of individual read()'s. On linux it's wise
 *         to match stat.st_blocksize.
 * @param[in] total_size A guess of the required size, it is only significant
 *         for efficient memory allocation.
 * @param[out] dest The output buffer, allocated with malloc. Use free() to
 *         deallocate the memory. The buffer has a terminating NULL appended
 *         to the end.
 * @return The total amount of characters read from the fd or -1 if an error
 *         occurred.
 */
int readall_fd(int fd, unsigned read_size, unsigned total_size, char** dest);

/**
 * Use this for loading smallish files. See readall_fd for notes.
 *
 * @param[in] filename The file name to read from. Specify NULL to read from
 *         standard input.
 * @param[out] dest The output buffer, allocated with malloc. Use free() to
 *         deallocate the memory. The buffer has a terminating NULL appended
 *         to the end.
 * @return The total amount of characters read from the fd or -1 if an error
 *         occurred.
 */
int readall_file(const char* filename, char** dest);

/**
 * The mmap read struct.
 */
struct readmmap_data;

/**
 * Open a randomly large file.
 * Supply a pointer to a pointer to readmmap_data.
 * You don't need to know what's in it.
 * Returns -1 on failure, check errno for details (or GetLastError()
 * on WIN32).
 */
int readmmap_open(const char* filename, struct readmmap_data** rsd);

/**
 * Get info from the readmmap_data struct.
 * The blocksize specifies the sizes you'll get with readmmap_get, except near
 * EOF obviously.
 */
int readmmap_getinfo(unsigned* blocksize, uint64_t* filesize,
		struct readmmap_data* rsd);

/**
 * Get the next block of data.
 * Returns 1 when there's data, 0 when there is no more data and -1 if an
 * error occurred.
 */
int readmmap_get(const char** next, unsigned* size, struct readmmap_data* rsd);

/**
 * Close the file opened with readmmap_open. Always do this, unless
 * readmmap_open failed.
 */
int readmmap_close(struct readmmap_data** rsd);

#ifdef __cplusplus
} /* extern "C" */
#endif /* __cplusplus */

#endif /* INCLUDED_READ_H */
