//-*-c++-*-
/*
 Author: Delorme Maxime
 Email : Maxime.Delorme@gmail.com

 Last modification : 07/07/2005 (fr) 

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/

#ifndef _GLAD_AXIS_POINT_H_
#define _GLAD_AXIS_POINT_H_

#include "GlADPoint.h"
#include "GlADLine.h"

#define N_POINT_LINES 6 /**< The number of lines for an axis point*/

namespace tlp {
/** \brief General class to render points with axis as augmented displays
 *
 * This class is a generic class to render points with axis as augmented displays.
 * It does uses the basic parameters of the GlAugmentedDisplay(position and color).
 */
class TLP_GL_SCOPE GlADAxisPoint : public GlADPoint
{
 protected:

  float range; /**< The range indicates the length of the axis */

  int negativeThickness; /**< The thickness of the axis in negative coordinates */
  int positiveThickness; /**< The thickness of the axis in positive coordinates */

  bool displayDim[3]; /**< Indicates which axis will be displayed */

  GlADLine* lines[N_POINT_LINES]; /**< Every axis line (3 for the positive side, 3 for the negative side) */
  
  /**
   * The default constructor
   *
   * \attention It's usage is forbidden, so that's why it has been put in protected members.
   *
   */
  GlADAxisPoint();

  /**
   * Function computing the lines give all the parameters
   */
  void computeLines();
 public:

  /**
   * Constructor
   *
   * \param position Indicates the position of the center point.
   * \param color Indicates the color of the point and the axis at the center.
   * \param range Indicates the length of the axis.
   */
  GlADAxisPoint(const Coord& position, const Color& color, float range);

  /**
   * Constructor
   *
   * \param position Indicates the position of the center point.
   * \param color Indicates the color of the point and the axis at the center.
   * \param range Indicates the length of the axis.
   * \param negativeThickness Indicates the thickness of the axis pointing towards a negative side.
   * \param positiveThickness Indicates the thickness of the axis pointing towards a positive side.
   */
  GlADAxisPoint(const Coord& position, const Color& color, float range, int negativeThickness, int positiveThickness);

  /**
   * Destructor
   */
  virtual ~GlADAxisPoint();

  /**
   * Virtual function used to draw the point and the axis
   */
  virtual void draw(GlGraph* graph);

  /**
   * Virtual function used to trigger when the rendering options change.
   */
  virtual void changeRenderOptions();

  /**
   * Accessor in reading to the range of the axis.
   */
  float getRange() const;

  /** 
   * Accessor in reading to the negative thickness of the axis.
   */
  int getNegativeThickness() const;

  /**
   * Accessor in reading to the positive thickness of the axis.
   */
  int getPositiveThickness() const;

  /**
   * Accessor in writing to the range of the axis.
   */
  void setRange(const float range);

  /**
   * Accessor in writing to the negative thickness of the axis.
   */
  void setNegativeThickness(const int negativeThickness);

  /**
   * Accessor in writing to the positive thickness of the axis.
   */
  void setPositiveThickness(const int positiveThickness);

  /**
   * Accessor in writing to the axis to display 
   *
   * \attention displayDim[0] = X axis 
   * \attention displayDim[1] = Y axis
   * \attention displayDim[2] = Z axis
   */
  void setDisplayDim(bool displayDim[3]);
};
}
#endif
