//---------------------------------------------------------------------------
// Copyright (c) 2000 Magnus Danielson. All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

//---------------------------------------------------------------------------

// $Id: Hierarchy.cc,v 1.1 2003/12/04 19:13:18 dmartin Exp $

// $Author: dmartin $
// $Revision: 1.1 $

#include <map>
#include <string>

// Project includes
#include "SignalBase.hh"

// Module include
#include "Hierarchy.hh"

struct Hierarchy_member
{
	char *		name;
	int		type;
	Hierarchy *	hier;
	SignalBase *	signal;
};

Hierarchy::Hierarchy()
{
	// We do not (yeat) know of a leader
	head = NULL;

	// No, we have no members here
}

Hierarchy::~Hierarchy()
{
	// Doomsday - or the amount of fun we can have this side of Waco
	typedef map<string, Hierarchy_member *>::const_iterator CI;

	// Kill all members of sect
	for (CI index = members.begin(); index != members.end(); index++)
	{
		// Hey, wait! This member migth be the head of a cell that we
		// need to kill of in the process of wipping him out.
		if (index->second->hier)
			index->second->hier->~Hierarchy();

		// Now, this fellow is lonely by now, so no more slow terror,
		// just wipe him of the face of the earth.
		delete index->second;
	}

	// Close the sect safely
	members.clear();
}

void	Hierarchy::add_member(char * name, int type, Hierarchy * hier, SignalBase * signal)
{
	Hierarchy_member * member;

	// Ensure we have space for a new member in the Hierarchy
	//ensure();

	// Create the actual space for the member
	member = new Hierarchy_member;
	if (member == NULL)
	{
		fprintf(stderr, "Hierarchy::add_member(%u): Unable to allocate memory.\n", __LINE__);
		return;
	}

	// Initiate the new member
	member->name = name;
	member->type = type;
	member->hier = hier;
	member->signal = signal;

	// Let the new member into the group
	members[name] = member;
}

void	Hierarchy::add_block(char * name, Hierarchy * hier)
{
	// Add member
	add_member(name, HIERARCHY_TYPE_BLOCK, hier, NULL);

	// Oh, your leader is *me*
	hier->head = this;
}

void	Hierarchy::add_signal(char * name, SignalBase * signal)
{
	// Add member
	add_member(name, HIERARCHY_TYPE_SIGNAL, NULL, signal);
}

unsigned long	Hierarchy::get_count()
{
	return members.size();
}

int	Hierarchy::get_type(char * name)
{
	// Ensure a decent index value
	if (name == NULL)
		return -1;

	// Find member and return type
	return members[name]->type;
}

Hierarchy *	Hierarchy::get_hierarchy(char * name)
{
	// Ensure a decent index value
	if (name == NULL)
		return NULL;

	// OK, we only do this for the type of members which have been given
	// the right to carry their own cell.
	switch (members[name]->type)
	{
		case HIERARCHY_TYPE_BLOCK:
			return members[name]->hier;
		default:
			break;
	}
	return NULL;
}

SignalBase *	Hierarchy::get_signal(char * name)
{
	// Ensure a decent index value
	if (name == NULL)
		return NULL;

	// OK, we only do this for the type of members which have been given
	// the right to carry their own cell.
	switch (members[name]->type)
	{
		case HIERARCHY_TYPE_SIGNAL:
			return members[name]->signal;
		default:
			break;
	}
	return NULL;
}

void		Hierarchy::fdump(FILE * fp, int depth)
{
	typedef map<string, Hierarchy_member *>::const_iterator CI;
	int		i;
	Hierarchy *	hier;

	for (CI index = members.begin(); index != members.end(); index++)
	{
		for (i = 0; i < depth; i++)
			fprintf(fp, "    ");
		switch (index->second->type)
		{
			case HIERARCHY_TYPE_BLOCK:
				fprintf(fp, "%s\n", index->second->name);
				hier = index->second->hier;
				hier->fdump(fp, depth + 1);
				break;
			case HIERARCHY_TYPE_SIGNAL:
				fprintf(fp, "%s\n", index->second->name);
				break;
			default:
				break;
		}
	}
}
