/* Page block renderer.

   Copyright (C) 2000 Daiki Ueno <ueno@unixuser.org>

   Author: Daiki Ueno <ueno@unixuser.org>
   Created: 2000-05-16

   This file is part of UltraPoint.

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)  
   any later version.                                                   

   This program is distributed in the hope that it will be useful,      
   but WITHOUT ANY WARRANTY; without even the implied warranty of       
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the        
   GNU General Public License for more details.                         

   You should have received a copy of the GNU General Public License    
   along with GNU Emacs; see the file COPYING.  If not, write to the    
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,         
   Boston, MA 02111-1307, USA.                                          

*/

#include "page.h"

static void page_get_em_metric (UptPage *page, gint *width, gint *height);
static void page_free (UptPage *page);
static GSList *page_get_linewrap (GSList *start, GSList *end);
static void vf_page_put_line (UptPage *page, GSList *start, GSList *end,
			      gpointer data);
static void vf_page_finish_line (UptPage *page, gpointer data);
static void vf_page_put_line_1 (UptPage *page, GSList *start, GSList *end,
				gpointer data);
static void vf_page_finish_line_1 (UptPage *page, gpointer data);
static void vf_bitmap_to_pbuf_aa (UptCanvas *canvas, VF_BITMAP bm,
				  glong *width, glong *height,
				  guint8 **pbuf);
static void vf_bitmap_to_pbuf (UptCanvas *canvas, VF_BITMAP bm,
			       glong *width, glong *height,
			       guint8 **pbuf);
static void page_fill_glues (UptPage *page, gint n_glues, glong off_x,
			     glong min_pixel_width, glong max_pixel_width);

typedef void (*PageBitmapInstantiator) (UptCanvas *canvas, VF_BITMAP bm,
					glong *width, glong *height,
					guint8 **pbuf);

typedef void (*PagePutLineFunc) (UptPage *page, GSList *start, GSList *end,
				 gpointer data);
typedef void (*PageFinishLineFunc) (UptPage *page, gpointer data);

static void
page_free (page)
     UptPage *page;
{
  GSList *p;

  for (p = page->glyphs; p; p = g_slist_next (p))
    upt_object_unref ((UptGlyph *)p->data);
  g_slist_free (page->glyphs);
  if (!page->pbuf)
    g_free (page->pbuf);
  g_free (page);
}

UptPage *upt_page_new (base_width, base_height, base_glue, base_lineskip,
		       base_font, antialias)
     glong base_width, base_height;
     UptGlue *base_glue;
     gdouble base_lineskip;
     UptFont *base_font;
     gboolean antialias;
{
  UptPage *page;

  page = g_new0 (UptPage, 1);
  g_assert (page != NULL);

  if (antialias)
    {
      base_width *= 2;
      base_height *= 2;
      base_lineskip *= 2;

      base_glue = g_memdup (base_glue, sizeof(UptGlue));
      base_glue->pixel_width *= 2;

      base_font = g_memdup (base_font, sizeof(UptFont));
      base_font->pixel_size *= 2;
    }

  page->base_width = base_width;
  page->base_height = base_height;
  page->base_lineskip = base_lineskip;
  page->base_glue = base_glue;
  page->base_font = base_font;
  page->antialias = antialias;

  page->free = (GFreeFunc) page_free;

  return page;
}

void
upt_page_add_glyph (page, glyph)
     UptPage *page;
     UptGlyph *glyph;
{
  page->glyphs = g_slist_append (page->glyphs, glyph);
  upt_object_ref (glyph);
}
     
void
upt_page_add_text (page, utf)
     UptPage *page;
     guchar *utf;
{
  UptGlyph *glyph;
  UptGlue *agglutinative_glue;
  UptFont *font;
  unicode_char_t cp;
  gint em_width, em_height;
  guchar *p;

  page_get_em_metric (page, &em_width, &em_height);

  agglutinative_glue = g_new0 (UptGlue, 1);
  agglutinative_glue->linewrap_cost = 1;
  agglutinative_glue->pixel_width = 0.1 * em_width;

  for (glyph = NULL, p = utf; *p; p = unicode_next_utf8 (p))
    {
      unicode_get_utf8 (p, &cp);
      if (!unicode_iscntrl (cp) && !unicode_isspace (cp))
	{
	  font = upt_find_font (cp, page->base_font->pixel_size,
				page->base_font->family_id,
				page->base_font->face_id);
	  if (!font)
	    continue;
	  if (font->charset->converter)
	    cp = (*font->charset->converter) (font->charset, cp);
	  glyph = upt_glyph_new (font, cp);
	  glyph->right_glue = agglutinative_glue;
	  upt_object_ref (glyph->right_glue);
	  upt_page_add_glyph (page, glyph);
	}
      else if (glyph && glyph->right_glue)
	{
	  upt_object_unref (glyph->right_glue);
	  glyph->right_glue = page->base_glue;
	  upt_object_ref (page->base_glue);
	}
    }
}

static void
page_get_em_metric (page, width, height)
     UptPage *page;
     gint *width, *height;
{
  unicode_char_t em=0x4d;
  UptFont *font;
  struct vf_s_metric2 metric;

  font = upt_find_font (em, page->base_font->pixel_size,
			page->base_font->family_id,
			page->base_font->face_id);
  if (!font)
    {
      *height = *width = page->base_font->pixel_size / 2;
      return;
    }
  VF_GetMetric2 (font->id, em, &metric, 1, 1);
  *width = metric.bbx_width;
  *height = metric.bbx_width;
}

static GSList *
page_get_linewrap (start, end)
     GSList *start, *end;
{
  GSList *p=end;

  while (start != end)
    {
      UptGlyph *glyph;

      glyph = start->data;
      start = g_slist_next (start);
      if (glyph->right_glue->linewrap_cost < 1)
	p = start;
    }

  return p;
}

static void
vf_page_put_line (page, start, end, data)
     UptPage *page;
     GSList *start, *end;
     gpointer data;
{
  VF_BITMAPLIST bml = data;

  do {
    UptGlyph *glyph;

    glyph = start->data;
    upt_glyph_instantiate (glyph);
    VF_BitmapListPut (bml, glyph->bm, bml->off_x, bml->off_y);
    bml->off_x += glyph->bm->mv_x
      + glyph->right_glue->pixel_width;
    start = g_slist_next (start);
  } while (start && start != end);
}    

static void
vf_page_finish_line (page, data)
     UptPage *page;
     gpointer data;
{
  VF_BITMAPLIST bml = data;
  gint aa_factor = page->antialias ? 2 : 1;

  bml->off_x = 0;
  bml->off_y -= page->base_font->pixel_size / aa_factor
    * (page->base_lineskip + aa_factor);
}

static void
vf_page_put_line_1 (page, start, end, data)
     UptPage *page;
     GSList *start, *end;
     gpointer data;
{
  VF_METRIC2 metric = data;
  gint off_x = 0;

  do {
    struct vf_s_metric2 glyph_metric;
    UptGlyph *glyph;

    glyph = start->data;
    VF_GetMetric2 (glyph->font->id, glyph->codepoint, &glyph_metric, 1, 1);
    off_x += glyph_metric.mv_x
      + glyph->right_glue->pixel_width;
    start = g_slist_next (start);
  } while (start && start != end);

  if (off_x > metric->bbx_width)
    metric->bbx_width = off_x;
}    

static void
vf_page_finish_line_1 (page, data)
     UptPage *page;
     gpointer data;
{
  VF_METRIC2 metric = data;
  gint aa_factor = page->antialias ? 2 : 1;

  metric->bbx_height += page->base_font->pixel_size / aa_factor
    * (page->base_lineskip + aa_factor);
}

static void
vf_bitmap_to_pbuf_aa (canvas, bm, width, height, pbuf)
     UptCanvas *canvas;
     VF_BITMAP bm;
     glong *width, *height;
     guint8 **pbuf;
{
  static const guint8 aa_bits[] = { 0, 1, 1, 2 };
  gint i, j, bbx_width, bbx_height, aabw, aabh, n_bits, s;

  guint8 *p, *q, *aa_row;
  ggi_pixel bg;

  bbx_width = bm->bbx_width - bm->bbx_width % 2;
  bbx_height = bm->bbx_height - bm->bbx_height % 2;

  *width = aabw = bbx_width / 2;
  *height = aabh = bbx_height / 2;

  aa_row = *pbuf = g_malloc0 (aabw * aabh * canvas->psize);
  
  for (p = bm->bitmap, i = 0; i < aabh; i++, p += bm->raster * 2)
    {
      q = p + bm->raster;
      for (j = 0; j < aabw; j++)
	{
	  s = (0x03 & ~(j % 4)) << 1;
	  n_bits = aa_bits[(p[j / 4] >> s) & 0x03]
	    + aa_bits[(q[j / 4] >> s) & 0x03];
	  if (upt_canvas_get_aa_pixel (canvas, j, i, n_bits, &bg) > -1)
	    memcpy (&aa_row[j * canvas->psize], &bg, canvas->psize);
	}
      aa_row += aabw * canvas->psize;
    }
}

static void
vf_bitmap_to_pbuf (canvas, bm, width, height, pbuf)
     UptCanvas *canvas;
     VF_BITMAP bm;
     glong *width, *height;
     guint8 **pbuf;
{
  gint i, j, n_bits;
  guint8 *p, *pbuf_row;
  ggi_pixel pixel;
  static unsigned char  bits[] = {
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01
  };

  *width = bm->bbx_width;
  *height = bm->bbx_height;

  pbuf_row = *pbuf =
    g_malloc0 (bm->bbx_width * bm->bbx_height * canvas->psize);

  for (p = bm->bitmap, i = 0; i < bm->bbx_height; i++, p += bm->raster)
    {
      for (j = 0; j < bm->bbx_width; j++)
	{
	  n_bits = p[j/8] & bits[j%8] ? 4 : 0;
	  if (upt_canvas_get_aa_pixel (canvas, j, i, n_bits, &pixel) > -1)
	    memcpy (&pbuf_row[j * canvas->psize], &pixel, canvas->psize);
	}
      pbuf_row += bm->bbx_width * canvas->psize;
    }
}

static void
page_fill_glues (page, n_glues, off_x, min_pixel_width, max_pixel_width)
     UptPage *page;
     gint n_glues;
     glong off_x, min_pixel_width, max_pixel_width;
{
  if (n_glues > 0)
    page->base_glue->pixel_width =
      MIN (max_pixel_width,
	   MAX ((page->base_width - off_x) / n_glues,
		min_pixel_width));
  else
    page->base_glue->pixel_width = 1;
}

void
upt_page_instantiate_1 (page, put_line, finish_line, data)
     UptPage *page;
     PagePutLineFunc put_line;
     PageFinishLineFunc finish_line;
     gpointer data;
{
  glong off_x;
  glong glue_min_pixel_width, glue_max_pixel_width;
  gint bbx_width, dummy, n_glues;
  GSList *p, *q;

  page_get_em_metric (page, &bbx_width, &dummy);
  glue_min_pixel_width =
    page->base_glue->min_width * bbx_width;
  glue_max_pixel_width =
    page->base_glue->max_width * bbx_width;

  off_x = n_glues = 0;
  for (q = p = page->glyphs; p; p = g_slist_next (p))
    {
      struct vf_s_metric2 metric;
      UptGlyph *glyph;

    line_start:
      glyph = p->data;
      VF_GetMetric2 (glyph->font->id, glyph->codepoint, &metric, 1, 1);
      if (off_x + metric.bbx_width + glue_min_pixel_width * n_glues
	  < page->base_width)
	{
	  off_x += metric.mv_x + glyph->right_glue->pixel_width;
	  if (glyph->right_glue == page->base_glue)
	    n_glues++;
	}
      else
	{
	  page_fill_glues (page, n_glues, off_x, glue_min_pixel_width,
			   glue_max_pixel_width);
	  p = page_get_linewrap (q, p);
	  (*put_line) (page, q, p, data);
	  (*finish_line) (page, data);
	  off_x = 0;
	  n_glues = 0;

	  /* reset to the default glue width */
	  page->base_glue->pixel_width = 0;

	  if (p != q && (q = p))
	    goto line_start;
	}
    }
  if (off_x)
    {
      page_fill_glues (page, n_glues, off_x, glue_min_pixel_width,
		       glue_max_pixel_width);
      (*put_line) (page, q, p, data);
    }
}

void
upt_page_instantiate (canvas, page)
     UptCanvas *canvas;
     UptPage *page;
{
  VF_BITMAPLIST bml;
  VF_BITMAP bm;
  PageBitmapInstantiator bitmap_to_pbuf =
    page->antialias ? vf_bitmap_to_pbuf_aa : vf_bitmap_to_pbuf;

  bml = g_malloc0 (sizeof(struct vf_s_bitmaplist));
  VF_BitmapListInit (bml);

  upt_page_instantiate_1 (page, vf_page_put_line, vf_page_finish_line, bml);
  bm = VF_BitmapListCompose (bml);

  (*bitmap_to_pbuf) (canvas, bm, &page->width, &page->height, &page->pbuf);

  VF_FreeBitmap (bm);
  VF_BitmapListFinish (bml);
  g_free (bml);
}

void
upt_page_get_metric (page, bbx_width, bbx_height)
     UptPage *page;
     glong *bbx_width, *bbx_height;
{
  struct vf_s_metric2 metric;
  gint aa_factor = page->antialias ? 2 : 1;

  memset (&metric, 0, sizeof(metric));
  upt_page_instantiate_1 (page, vf_page_put_line_1, vf_page_finish_line_1,
			  &metric);
  *bbx_width = metric.bbx_width / aa_factor;
  *bbx_height = metric.bbx_height / aa_factor;
}
