/* $Id: hw_misc.c */

/*
 * hw driver utility routines
 * Copyright (c) 2000 The Utah-GLX Project
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FINESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * UTAH-GLX OR ANY OF THE CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Andreas Ehliar (memory scaling)
 * Ralph Giles <giles@ashlu.bc.ca> (secure temp file routine)
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>
#include <fcntl.h>



#include "hwtypes.h"	/* HW_FILENAME_LENGTH */
#include "hwlog.h"

/* Compare offset to size of physical memory. */

#ifdef __linux__
int checkmemoffset(int offset)
{
  FILE *meminfo;
  int memtotal;
  char templine[160]; /* This should be enough for one line in
		       * /proc/meminfo unless you have LOTS of
		       * memory :) */

  meminfo=fopen("/proc/meminfo","r");
  if(!meminfo){
    return 0;
  }


  while(fgets(templine,160,meminfo)){
    if(sscanf(templine,"MemTotal: %d kB",&memtotal)==1){
      fclose(meminfo);
      if(memtotal*1024 < offset){
	return 1;
      }
      return 0;
    }
  }
  fclose(meminfo);
  return 0;
}
#else
/* Not good, but better than nothing */
int checkmemoffset(int offset)
{
  if(offset < 16*1024*1024){
    return 0;
  }
  return 1;
}

#endif	/* __linux__ */


/* create a temp file securely
   (used for shared memory in the dma drivers)

   pass a prefix for the filename (truncated to 5 chars)
   pass a (char*) to return the full path of the new file
	this string must be freed by the caller

   returns a filedescriptor to the open file
   blocks until success
*/
int hwOpenTempFile( const char *pre, char **newfilename )
{
	char	*name, prefix[6];
	int	fd;

	/* the main issue is to make sure no one can fool us into
	   opening a symlink they've created rather than a fresh file,
	   so we ask for it to be new, and keep trying with different
	   names until we get one.

	   based on an example in the Gnome coding guidelines
	*/

	/* truncate filename prefix */
	strncpy ( prefix, pre, (strlen(pre) > 5) ? 5 : strlen(pre) );
	prefix[5] = '\0';

	do {
		name = tempnam( "/tmp", prefix );
		if ( name == NULL || strlen(name) >= HW_FILENAME_LENGTH ) {
			hwMsg( 1, "WARNING: could not get unique memory file!"
				" relying on unlink()\n");
			name = strdup( "/tmp/glxmemory" );
			unlink( name );
		}
		hwMsg( 1, "Using memory file file: %s\n", name );

		fd = open( name,
			 O_RDWR | O_CREAT | O_EXCL | O_TRUNC, 0600 );
		if ( fd == -1 ) {
			hwMsg( 1, "Couldn't open '%s' trying another...\n",
				name );
			free( name );
		}
	} while ( fd == -1 );

	*newfilename = name;
	return ( fd );
}
