/*
 * GLX Hardware Device Driver for S3 Savage3D and probably Savage/MX and
 * Savage/IX
 * Copyright (C) 2000 Dominik Behr
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on S3 Virge driver by Jim Duchek <jimduchek@ou.edu>
 *
 * Dominik Behr <behr@promail.pl>
 * thanks to Raja Koduri and Tim Roberts   
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#ifdef __FREEBSD__
#include <machine/cpufunc.h>
#endif

#include "compiler.h"

#include "xsmesaP.h"

#include "types.h"
#include "pb.h"
#include "dd.h"

#include "glx_log.h"

#include "s3savglx.h"
#include "glx_symbols.h"


/* FIXME: GH */
//#define S3VIRGE_CONTEXT(ctx)	((s3virgeContextPtr)(((XSMesaContext)(ctx)->DriverCtx)->hw_ctx))

/******************************************************************************

******************************************************************************/
//S3SAVDRAWCTRL
hwUI32
s3savDRAWCTRL(const GLcontext *pCtx)
{
 S3SAVDRAWCTRL stRes;
 
 stRes.uED = pCtx->Color.DitherFlag && stS3Sav.pDrawBuffer->stBackBuffer.nBPP == 2;
 stRes.uUVO = 0;

 #ifdef HARDCULL
 // culling is done by software
 if (pCtx->Polygon.CullFlag)
    {
     if ((pCtx->Polygon.CullFaceMode == GL_BACK) 
         ^ (pCtx->Polygon.FrontFace == GL_CCW))
        /*stRes.uBCM = 2; // cull cw
     else
        stRes.uBCM = 3; // cull ccw	*/
    // have to invert this because we do -Y	
	stRes.uBCM = 3; // cull cw
     else
        stRes.uBCM = 2; // cull ccw	
    }
 else
    stRes.uBCM = 1; // no cull
 #else
    stRes.uBCM = 1; // no cull
 #endif    
 stRes.uTVC = 0;
 stRes.uSM = (pCtx->Light.ShadeModel == GL_FLAT);
 stRes.uESS = 0;
 if (pCtx->Color.BlendEnabled) 
    {
     switch (pCtx->Color.BlendSrcRGB) {
      case GL_ZERO:
       stRes.uSABM = S3SAVSABM_ZERO;
       break;
      case GL_ONE:
       stRes.uSABM = S3SAVSABM_ONE;
       break;
      case GL_DST_COLOR:
       stRes.uSABM = S3SAVSABM_DEST_COLOR;
       break;
      case GL_ONE_MINUS_DST_COLOR:
       stRes.uSABM = S3SAVSABM_ONE_MINUS_DEST_COLOR;
       break;
      case GL_SRC_ALPHA:
       stRes.uSABM = S3SAVSABM_SOURCE_ALPHA;
       break;
      case GL_ONE_MINUS_SRC_ALPHA:
       stRes.uSABM = S3SAVSABM_ONE_MINUS_SOURCE_ALPHA;
       break;
      case GL_DST_ALPHA:
       // UNSUPPORTED
       stRes.uSABM = 0;
       break;
      case GL_ONE_MINUS_DST_ALPHA:
       // UNSUPPORTED
       stRes.uSABM = 0;
       break;
      case GL_SRC_ALPHA_SATURATE:
       // UNSUPPORTED
       stRes.uSABM = 0;
       break;
      default: 
     } // end of switch BlendSrcRGB
     
     switch (pCtx->Color.BlendDstRGB) {
      case GL_ZERO:
       stRes.uDABM = S3SAVDABM_ZERO;
       break;
      case GL_ONE:
       stRes.uDABM = S3SAVDABM_ONE;
       break;
      case GL_SRC_COLOR:
       stRes.uDABM = S3SAVDABM_SOURCE_COLOR;
       break;
      case GL_ONE_MINUS_SRC_COLOR:
       stRes.uDABM = S3SAVDABM_ONE_MINUS_SOURCE_COLOR;
       break;
      case GL_SRC_ALPHA:
       stRes.uDABM = S3SAVDABM_SOURCE_ALPHA;
       break;
      case GL_ONE_MINUS_SRC_ALPHA:
       stRes.uDABM = S3SAVDABM_ONE_MINUS_SOURCE_ALPHA;
       break;
      case GL_DST_ALPHA:
       // UNSUPPORTED
       stRes.uDABM = 0;
       break;
      case GL_ONE_MINUS_DST_ALPHA:
       // UNSUPPORTED
       stRes.uDABM = 0;
       break;
      default: 
     } // end of switch BlendDstRGB
    }
 else // blend disabled
    {
     stRes.uDABM = S3SAVDABM_ZERO;
     stRes.uSABM = S3SAVSABM_ONE;
    }   
 // FIXIT ATC and alpharef is global, make a flag when atc changes
 // and wait for 3d idle when it changes
 if (pCtx->Color.AlphaEnabled) 
    {
     stRes.uEAT = 1;
     stRes.uAlphaRef = pCtx->Color.AlphaRef;
     switch (pCtx->Color.AlphaFunc) {
      case GL_NEVER:
       stRes.uATC = S3SAVATC_NEVER;
       break;
      case GL_LESS:
       stRes.uATC = S3SAVATC_LESS;
       break; 
      case GL_EQUAL:
       stRes.uATC = S3SAVATC_EQUAL;
       break; 
      case GL_LEQUAL:
       stRes.uATC = S3SAVATC_LEQUAL;
       break; 
      case GL_GREATER:
       stRes.uATC = S3SAVATC_GREATER;
       break; 
      case GL_NOTEQUAL:
       stRes.uATC = S3SAVATC_NOTEQUAL;
       break; 
      case GL_GEQUAL:
       stRes.uATC = S3SAVATC_GEQUAL;
       break; 
      case GL_ALWAYS:
       stRes.uATC = S3SAVATC_ALWAYS;
       break; 
      default: 
     } // end of switch AlphaFunc
    }
 else // alpha test disabled
    {
     stRes.uEAT = 0;
     stRes.uATC = stS3Sav.stDRAWCTRL.uATC;
     stRes.uAlphaRef = stS3Sav.stDRAWCTRL.uAlphaRef;
    }   
 if (pCtx->Texture.Enabled)
    {
     switch (pCtx->Texture.Unit[0].EnvMode) {
      case GL_DECAL:
       stRes.uTBC = S3SAVTBC_DECAL;
       break;
      case GL_REPLACE:
       stRes.uTBC = S3SAVTBC_COPY;
       break;
      case GL_BLEND: // FIXIT
      case GL_MODULATE:
       stRes.uTBC = S3SAVTBC_MODULATEALPHA;
       break;
      default: 
     }
    }
 else
    stRes.uTBC = 0;
 stRes.uFDW = 1;
 stRes.uFZW = 1;
 stRes.uIM = 0;

 if (stRes.uBCM != stS3Sav.stDRAWCTRL.uBCM 
     || stRes.uATC != stS3Sav.stDRAWCTRL.uATC
     || stRes.uEAT != stS3Sav.stDRAWCTRL.uEAT
     || stRes.uAlphaRef != stS3Sav.stDRAWCTRL.uAlphaRef)
    stS3Sav.uChangedFlags |= S3SAV_FLAG_GLOBAL; 

 if (DW(stRes) != DW(stS3Sav.stDRAWCTRL))
    {
     stS3Sav.stDRAWCTRL = stRes;
     stS3Sav.uChangedFlags |= S3SAV_FLAG_DRAWCTRL;
    }
 #ifdef DEBUGx
 fprintf(stderr, "[s3sav] DRAWCTRL:%08X\n", DW(stRes));
 #endif
 return DW(stRes);
}

/******************************************************************************

******************************************************************************/
//S3SAVZBCTRL
hwUI32
s3savZBCTRL(const GLcontext *pCtx)
{
 S3SAVZBCTRL stRes;
 
 switch (pCtx->Depth.Func) {
  case GL_NEVER:
   stRes.uZBComp = S3SAVZBC_NEVER;
   break;
  case GL_LESS:
   stRes.uZBComp = S3SAVZBC_LESS;
   break;
  case GL_LEQUAL:
   stRes.uZBComp = S3SAVZBC_LEQUAL;
   break;
  case GL_GREATER:
   stRes.uZBComp = S3SAVZBC_GREATER;
   break;
  case GL_NOTEQUAL:
   stRes.uZBComp = S3SAVZBC_NOTEQUAL;
   break;
  case GL_GEQUAL:
   stRes.uZBComp = S3SAVZBC_GEQUAL;
   break;
  case GL_ALWAYS:
   stRes.uZBComp = S3SAVZBC_ALWAYS;
   break;
  default: 
 }

 if (pCtx->Depth.Test)
    stRes.uEZ = 1;
 else
    stRes.uEZ = 0;
    
 if (pCtx->Depth.Mask)
    stRes.uEZU = 1;
 else
    stRes.uEZU = 0;
    
 stRes.uEDU = pCtx->Color.ColorMask[0]
            && pCtx->Color.ColorMask[1]
            && pCtx->Color.ColorMask[2]
            && pCtx->Color.ColorMask[3];
 stRes.uReserved1 = 0;
 stRes.uZExponentOffset = 0;
 stRes.uZW = pCtx->Color.AlphaEnabled;
 stRes.uReserved2 = 0;

 if (stRes.uZBComp != stS3Sav.stZBCTRL.uZBComp
     || stRes.uEZ != stS3Sav.stZBCTRL.uEZ)
    stS3Sav.uChangedFlags |= S3SAV_FLAG_GLOBAL; 
    
 if (DW(stRes) != DW(stS3Sav.stZBCTRL))
    {
     stS3Sav.stZBCTRL = stRes;
     stS3Sav.uChangedFlags |= S3SAV_FLAG_ZBCTRL;
    }
 #ifdef DEBUGx
 fprintf(stderr, "[s3sav] ZBCTRL:%08X\n", DW(stRes));
 #endif
 return DW(stRes);
}

/******************************************************************************

******************************************************************************/
//S3SAVZBADDR
hwUI32
s3savZBADDR(const GLcontext *pCtx)
{
 S3SAVZBADDR stRes;
 
 if (!stS3Sav.pDrawBuffer->stZBuffer.pMemBlock)
    {
     stRes.uZBufferOffset = 0;
     stRes.uReserved1 = 0;
     stRes.uZBufferWidth = 0;
     stRes.uZBufferDepth = 0;
     #ifdef DEBUGx
     fprintf(stderr, "[s3sav] ZBADDR: NONE!\n");
     #endif 
     if (DW(stRes) != DW(stS3Sav.stZBADDR))
        {
         stS3Sav.stZBADDR = stRes;
         stS3Sav.uChangedFlags |= S3SAV_FLAG_ZBADDR;
        }
     return DW(stRes);
    }
 #ifdef UNIFIED_BUFFER
 /* z buffer is the other half of buffer
    buffer width is 4096 bytes = 32 tiles
    hald is 16 tiles * 2048 bytes
 */
 stRes.uZBufferOffset = (stS3Sav.pDrawBuffer->stZBuffer.pMemBlock->ofs + HALF_BUFFER) >> S3SAV_BUFFERALIGN;
 #else
 stRes.uZBufferOffset = stS3Sav.pDrawBuffer->stZBuffer.pMemBlock->ofs >> S3SAV_BUFFERALIGN;
 stRes.uReserved1 = 0;
 #endif
 stRes.uZBufferWidth = stS3Sav.pDrawBuffer->stZBuffer.nStride >> 7;
 stRes.uZBufferDepth = 0;
 if (DW(stRes) != DW(stS3Sav.stZBADDR))
    {
     stS3Sav.stZBADDR = stRes;
     stS3Sav.uChangedFlags |= S3SAV_FLAG_ZBADDR | S3SAV_FLAG_GLOBAL;
    }
 #ifdef DEBUGx
 fprintf(stderr, "[s3sav] ZBADDR:%08X\n", DW(stRes));
 #endif
 return DW(stRes);
}


/******************************************************************************

******************************************************************************/
//S3SAVDESTCTRL
hwUI32
s3savDESTCTRL(const GLcontext *pCtx)
{
 S3SAVDESTCTRL stRes;
 
 stRes.uDWT = stS3Sav.pDrawBuffer->stBackBuffer.nStride >> 7;
 stRes.uReserved1 = 0;
 stRes.uDestOffset = stS3Sav.pDrawBuffer->stBackBuffer.pMemBlock->ofs >> S3SAV_BUFFERALIGN;
 stRes.uReserved2 = 0;
 if (stS3Sav.pDrawBuffer->stBackBuffer.nBPP == 2)
    stRes.uDPF = S3SAVDPF_RGB565;
 else
    stRes.uDPF = S3SAVDPF_XRGB888;
 stRes.uReserved3 = 0;      

 if (DW(stRes) != DW(stS3Sav.stDESTCTRL))
    {
     stS3Sav.stDESTCTRL = stRes;
     stS3Sav.uChangedFlags |= S3SAV_FLAG_DESTCTRL | S3SAV_FLAG_GLOBAL;
    }
 #ifdef DEBUGx
 fprintf(stderr, "[s3sav] DESTCTRL:%08X\n", DW(stRes));
 #endif
 return DW(stRes);
}

/******************************************************************************

******************************************************************************/
//S3SAVTEXADDR
hwUI32
s3savTEXADDR(const GLcontext *pCtx,
             const struct gl_texture_object *pTex)
{
 S3SAVTEXADDR stRes;
 PS3SAVTEXTURE p;
 if (!pTex)
    {
     DW(stRes) = 0;
     if (DW(stRes) != DW(stS3Sav.stTEXADDR))
        {
         stS3Sav.stTEXADDR = stRes;
         stS3Sav.uChangedFlags |= S3SAV_FLAG_TEXADDR;
        }
     return DW(stRes);
    }
 p = (PS3SAVTEXTURE)(pTex->DriverData);
 stRes.uTL = 0;
 stRes.uTMT = 0;
 stRes.uReserved = 0;
 stRes.uTextureDataAddress = (p->pMemBlock->ofs) >> 3;
 if (DW(stRes) != DW(stS3Sav.stTEXADDR))
    {
     stS3Sav.stTEXADDR = stRes;
     stS3Sav.uChangedFlags |= S3SAV_FLAG_TEXADDR;
    }
 #if 0
 fprintf(stderr, "[s3sav] TEXADDR %08X\n", DW(stRes));
 #endif
 return DW(stRes); 
}

static int 
Log2(unsigned int a) 
{
 #if 0
 int nRes;

 nRes = 0;
 while (a > 1)
       {
        nRes++;
        a >>= 1;
       }
 return nRes;
 #endif

 unsigned i;
	
 for (i = 0; i < 32; i++) 
     {
	  if ((1<<i) >= a) 
         {
		  return i;
		 } 	
	 }
 return 31;
}

/******************************************************************************

******************************************************************************/
//S3SAVTEXDESC
hwUI32
s3savTEXDESC(const GLcontext *pCtx,
             const struct gl_texture_object *pTex)
{
 S3SAVTEXDESC stRes;
 PS3SAVTEXTURE p;
 if (!pTex)
    {
     DW(stRes) = 0;
     if (DW(stRes) != DW(stS3Sav.stTEXDESC))
        {
         stS3Sav.stTEXDESC = stRes;
         stS3Sav.uChangedFlags |= S3SAV_FLAG_TEXDESC;
        }
     return DW(stRes);	
    }
 p = (PS3SAVTEXTURE)(pTex->DriverData);
 stRes.uTexWidth = Log2(pTex->Image[0]->Width);
 stRes.uReserved1 = 0;
 stRes.uTexHeight = Log2(pTex->Image[0]->Height);
 stRes.uReserved2 = 0;
 stRes.uTexelFmt = p->uFormat;
 stRes.uTPS = 0;
 stRes.uReserved3 = 0;
 stRes.uLTP = 0;
 if (DW(stRes) != DW(stS3Sav.stTEXDESC))
    {
     stS3Sav.stTEXDESC = stRes;
     stS3Sav.uChangedFlags |= S3SAV_FLAG_TEXDESC;
    }
 #if 0
 fprintf(stderr, "[s3sav] tex %dx%d form %d\n",
         stRes.uTexWidth,
	 stRes.uTexHeight,
	 stRes.uTexelFmt);
 #endif	 
 return DW(stRes);
}

/******************************************************************************

******************************************************************************/
//S3SAVTEXCTRL
hwUI32
s3savTEXCTRL(const GLcontext *pCtx,
             const struct gl_texture_object *pTex)
{
 S3SAVTEXCTRL stRes;
 if (!pTex)
    {
     DW(stRes) = 0;
     if (DW(stRes) != DW(stS3Sav.stTEXCTRL))
        {
         stS3Sav.stTEXCTRL = stRes;
         stS3Sav.uChangedFlags |= S3SAV_FLAG_TEXCTRL;
        }
     return DW(stRes);
    }
 switch (pTex->MinFilter) {
  case GL_LINEAR:
   stRes.uFiltMode = 1;
   stRes.uEM = 1;
   break;
  case GL_LINEAR_MIPMAP_NEAREST:
   stRes.uFiltMode = 1;
   stRes.uEM = 0;
   break;
  case GL_LINEAR_MIPMAP_LINEAR:
   stRes.uFiltMode = 3;
   stRes.uEM = 0;
   break;
  case GL_NEAREST:
   stRes.uFiltMode = 0;
   stRes.uEM = 1;
   break;
  case GL_NEAREST_MIPMAP_LINEAR:
   stRes.uFiltMode = 0;
   stRes.uEM = 0;
   break;
  case GL_NEAREST_MIPMAP_NEAREST:
   stRes.uFiltMode = 0;
   stRes.uEM = 0;
   break;
  default: 
 }
 stRes.uMipLevelBias = 0;
 stRes.uTUW = 0;
 stRes.uTVW = 0;
 if (pTex->WrapS == GL_CLAMP)
    stRes.uTAM = 1;
 else
    stRes.uTAM = 0;
 stRes.uETM = 1;
 stRes.uDFA = 0;
 stRes.uReserved1 = 0;
 stRes.uCCA = 0; // i dont really understand this bit function
 if (pCtx->Color.BlendEnabled) 
    {
     if ((pCtx->Color.BlendSrcRGB == GL_ONE
          && pCtx->Color.BlendDstRGB == GL_ZERO)
	 || ((PS3SAVTEXTURE)pTex->DriverData)->uFormat == S3SAVTEXFMT_RGB565) 
	stRes.uETT = 0; 
     else
	stRes.uETT = 1;	
    }
 else // blend disabled
    {
     stRes.uETT = 0;
    }   
 if (pCtx->Color.AlphaEnabled
     && ((PS3SAVTEXTURE)pTex->DriverData)->uFormat != S3SAVTEXFMT_RGB565)      
    stRes.uETT = 1;	
 stRes.uReserved2 = 0;        
 if (DW(stRes) != DW(stS3Sav.stTEXCTRL))
    {
     stS3Sav.stTEXCTRL = stRes;
     stS3Sav.uChangedFlags |= S3SAV_FLAG_TEXCTRL;
    }
 return DW(stRes);
}

/******************************************************************************

******************************************************************************/
//S3SAVFOGCTRL
hwUI32
s3savFOGCTRL(const GLcontext *pCtx)
{
 S3SAVFOGCTRL stRes;
 
 stRes = stS3Sav.stFOGCTRL;
 
 if (pCtx->Fog.Enabled)
    {
     /* 
      EGCS 1.1.2 (from slack 7.1) has problem the with following code
      result stRes contains only blue color !!!
      rest of fields are cleared   
     */
     stRes.uFE = 1;
     stRes.uFM = 1; // vertex fog
     stRes.uR = (hwUI8)(pCtx->Fog.Color[0] * 255.0f);
     stRes.uG = (hwUI8)(pCtx->Fog.Color[1] * 255.0f);
     stRes.uB = (hwUI8)(pCtx->Fog.Color[2] * 255.0f);
    }
 else
    { 
     stRes.uFE = 0;
     stRes.uFM = 0; // clearing FE is not enough !!! 
    }    
 
 if (DW(stRes) != DW(stS3Sav.stFOGCTRL))
    {
     stS3Sav.stFOGCTRL = stRes;
     stS3Sav.uChangedFlags |= S3SAV_FLAG_FOGCTRL;
    }
 return DW(stRes);
}

/******************************************************************************

******************************************************************************/
hwUI32
s3savSCSTART(const GLcontext *pCtx)
{
 S3SAVSCSTART stRes;
 stRes.uReserved1 = 0;
 stRes.uReserved2 = 0;
 if (pCtx->Scissor.Enabled)
    {
     stRes.uXStart = pCtx->Scissor.X;
     stRes.uYStart = pCtx->Scissor.Y;
    }
 else
    {
     stRes.uXStart = 0;
     stRes.uYStart = 0;
    }    
 if (DW(stRes) != DW(stS3Sav.stSCSTART))
    {
     stS3Sav.stSCSTART = stRes;
     stS3Sav.uChangedFlags |= S3SAV_FLAG_SCSTART;
    }    
 return DW(stRes);
}

/******************************************************************************

******************************************************************************/
hwUI32
s3savSCEND(const GLcontext *pCtx)
{
 S3SAVSCEND stRes;
 stRes.uReserved1 = 0;
 stRes.uReserved2 = 0;
 if (pCtx->Scissor.Enabled)
    {
     stRes.uXEnd = pCtx->Scissor.X + pCtx->Scissor.Width;
     stRes.uYEnd = pCtx->Scissor.Y + pCtx->Scissor.Height;
    }
 else
    {
     stRes.uXEnd = stS3Sav.pDrawBuffer->stBackBuffer.nWidth;
     stRes.uYEnd = stS3Sav.pDrawBuffer->stBackBuffer.nHeight;
    }    
 if (DW(stRes) != DW(stS3Sav.stSCEND))
    {
     stS3Sav.stSCEND = stRes;
     stS3Sav.uChangedFlags |= S3SAV_FLAG_SCEND;
    }    
 return DW(stRes);
}

/******************************************************************************

******************************************************************************/
hwUI32
s3savTILE(unsigned int uOffset,   
          unsigned int uStride, /* stride in bytes */
	  unsigned int uBPP) /* bytes per pixel */
{
 hwUI32 uRes;
 uRes = (uOffset | ((uStride >> 7) << 24));
 if (uBPP == 1)
    uRes |= 0x40000000;
 else    
 if (uBPP == 2)
    uRes |= 0x80000000;
 else    
 if (uBPP == 4)
    uRes |= 0xC0000000;
 else    
    {
     // crap !
    }
  
 return uRes;
}	  

/******************************************************************************

******************************************************************************/
void
s3savSetTile(unsigned int uTiledAperture,
             unsigned int uOffset,
	     unsigned int uStride,
	     unsigned int uBPP)
{
 S3SAVWR(S3SAV_TILEDAPERTURE0 + uTiledAperture * 4, s3savTILE(uOffset, uStride, uBPP)); 
 #if 0
 fprintf(stderr, "setting TILE%d %08X\n",
         uTiledAperture,
         S3SAVRD(S3SAV_TILEDAPERTURE0 + uTiledAperture * 4));
 #endif	 
}	     

/*******************************************************************************

*******************************************************************************/
void 
s3savDDUpdateState(GLcontext *pCtx) 
{
 struct gl_texture_object *pTex;
 #ifdef TRACE
 fprintf(stderr, "[s3sav] s3savDDUpdateState(%08X)\n",
         pCtx);
 #endif	 
 #ifdef EVENT_TRACE
 s3savFIFOWait(1);
 fprintf(stderr, "[s3sav] UPDSTATE event %04X\n", s3savInsertEvent());
 #endif
 // no accel :(
 //ctx->Driver.TriangleFunc = 0;
 //ctx->Driver.LineFunc = 0;
 //gl_set_triangle_function(pCtx);
 
 //s3savEnableMMIO();

 //stS3Sav.pRegs->stZBADDR = s3savZBADDR(pCtx);
 //stS3Sav.pRegs->stZBCTRL = s3savZBCTRL(pCtx);
 //stS3Sav.pRegs->stDESTCTRL = s3savDESTCTRL(pCtx);
 //stS3Sav.pRegs->stDRAWCTRL = s3savDRAWCTRL(pCtx);

 /*fprintf(stderr,
         "status0: %08X\n",
	 *stS3Sav.pStatus0);*/

 
 pTex = 0;
 if (pCtx->Texture.Enabled == TEXTURE0_2D)
    { 
     pTex = pCtx->Texture.Unit[0].Current;
     if (pTex != pCtx->Texture.Unit[0].CurrentD[2])
        pTex = 0;
    }

 if (pTex)
    if (!pTex->DriverData)
	s3savCreateTexObj(stS3Sav.pContext, pTex);
 // now move the texture to the beginning of LRU

 if (pTex)
    {  
     PS3SAVTEXTURE pT;
     pT = (PS3SAVTEXTURE)(pTex->DriverData);
     // FIXIT check here for pT = 0
     if (pT && 
         pT != stS3Sav.pTextures)
        {
         PS3SAVTEXTURE pPrev, p;
         pPrev = 0;
         for (p = stS3Sav.pTextures; p; p = p->pNext)
             {
    	      if (p == pT)
   	         break;
   	      pPrev = p;      
	     }
         if (pPrev)
            {
             pPrev->pNext = pT->pNext;
             pT->pNext = stS3Sav.pTextures;
             stS3Sav.pTextures = pT;
   	    } 
        }
    }

 s3savZBADDR(pCtx);
 s3savZBCTRL(pCtx);
 s3savDESTCTRL(pCtx);
 s3savDRAWCTRL(pCtx);
 s3savSCSTART(pCtx);
 s3savSCEND(pCtx);
 s3savFOGCTRL(pCtx);
 
 if (pTex)
    {
     s3savTEXADDR(pCtx, pTex);
     s3savTEXDESC(pCtx, pTex);
     s3savTEXCTRL(pCtx, pTex);
     if (pCtx->Fog.Enabled)
        {
	 pCtx->Driver.TriangleFunc = s3savTriangle_TX_FG;
         pCtx->Driver.LineFunc = s3savLine_TX_FG;
	}
     else
        {	
         pCtx->Driver.TriangleFunc = s3savTriangle_TX_NF;
         pCtx->Driver.LineFunc = s3savLine_TX_NF;
        }
    }
 else 
    {
     if (stS3Sav.stTEXCTRL.uETM)
        {
	 stS3Sav.stTEXCTRL.uETM = 0;
         stS3Sav.uChangedFlags |= S3SAV_FLAG_TEXCTRL;
	}
     if (pCtx->Fog.Enabled)
        {	
         pCtx->Driver.TriangleFunc = s3savTriangle_NT_FG;
         pCtx->Driver.LineFunc = s3savLine_NT_FG;
        } 
     else
        {	
         pCtx->Driver.TriangleFunc = s3savTriangle_NT_NF;
         pCtx->Driver.LineFunc = s3savLine_NT_NF;
        }	
    } 

 #if 0    
 fprintf(stderr, "[s3sav] state DO "); 
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_ZBADDR)
    fprintf(stderr, "ZBADDR ");   
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_ZBCTRL)
    fprintf(stderr, "ZBCTRL ");
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_DESTCTRL)    
    fprintf(stderr, "DESTCTRL ");
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_DRAWCTRL)    
    fprintf(stderr, "DRAWCTRL ");
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_SCSTART)    
    fprintf(stderr, "SCSTART ");
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_SCEND)    
    fprintf(stderr, "SCEND ");
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_TEXCTRL)        
    fprintf(stderr, "TEXCTRL ");
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_TEXADDR)
    fprintf(stderr, "TEXADDR ");
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_TEXDESC)
    fprintf(stderr, "TEXDESC ");
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_FOGCTRL)
    fprintf(stderr, "FOGCTRL ");    
 fprintf(stderr, "\n");    
 #endif 

 BCIRST;
 s3savFIFOWait(128); 
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_GLOBAL)
    {
     //s3savWait3DIdle();
     #ifdef TRACE
     fprintf(stderr, "[s3sav] global change\n");
     #endif
     BCIWR(BCI_CMD_WAIT | BCI_WAIT_3D_IDLE);
     BCIWR(BCI_CMD_WAIT | BCI_WAIT_3D_IDLE);
    }
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_ZBADDR)
    { BCISETREG(BCI_ZBADDR, DW(stS3Sav.stZBADDR)); }
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_ZBCTRL)
    { BCISETREG(BCI_ZBCTRL, DW(stS3Sav.stZBCTRL)); }
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_DESTCTRL)    
    { BCISETREG(BCI_DESTCTRL, DW(stS3Sav.stDESTCTRL)); }
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_DRAWCTRL)    
    { BCISETREG(BCI_DRAWCTRL, DW(stS3Sav.stDRAWCTRL)); }
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_SCSTART)    
    { BCISETREG(BCI_SCSTART, DW(stS3Sav.stSCSTART)); }
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_SCEND)    
    { BCISETREG(BCI_SCEND, DW(stS3Sav.stSCEND)); }
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_TEXCTRL)      
    { BCISETREG(BCI_TEXCTRL, DW(stS3Sav.stTEXCTRL)); }
 if (stS3Sav.uChangedFlags & S3SAV_FLAG_FOGCTRL)      
    { BCISETREG(BCI_FOGCTRL, DW(stS3Sav.stFOGCTRL)); }    
 // those two we always send togerther
 if (stS3Sav.uChangedFlags & (S3SAV_FLAG_TEXADDR | S3SAV_FLAG_TEXDESC))      
    {
     BCISETREG(BCI_TEXADDR, DW(stS3Sav.stTEXADDR));
     BCISETREG(BCI_TEXDESC, DW(stS3Sav.stTEXDESC));
     if (stS3Sav.pTextures)
        {
	 stS3Sav.nTextureID++;
         stS3Sav.nTextureID &= 0xFFFF;
	 stS3Sav.pTextures->nTextureID = stS3Sav.nTextureID;
	} 
    }
 #ifdef FORCE_SHADOW_UPDATE
 s3savUpdateShadow();
 #endif    
 stS3Sav.uChangedFlags = 0;            
    
 #ifdef DEBUGx 
 fprintf(stderr, "%08X, %08X, %08X, %08X, %08X, %08X\n",
         DW(stS3Sav.pRegs->stZBADDR),
         DW(stS3Sav.pRegs->stZBCTRL),
         DW(stS3Sav.pRegs->stDESTCTRL),
         DW(stS3Sav.pRegs->stDRAWCTRL),
	 DW(stS3Sav.pRegs->stSCSTART),
	 DW(stS3Sav.pRegs->stSCEND));
 #endif           

 stS3Sav.pContext->fViewportHeight = (float)stS3Sav.pDrawBuffer->stBackBuffer.nHeight;
 
 //gl_set_line_function(pCtx);
 //gl_set_triangle_function(pCtx);
 #ifdef TRACE
 fprintf(stderr, "[s3sav] s3savDDUpdateState(%08X) OK\n",
         pCtx);
 #endif	 
 return;
}	

/*******************************************************************************

*******************************************************************************/
void 
s3savDDInitStatePointers(GLcontext *ctx)
{
 ctx->Driver.Enable = 0;
 ctx->Driver.LightModelfv = 0;
 ctx->Driver.AlphaFunc = 0;
 ctx->Driver.BlendEquation = 0;
 ctx->Driver.BlendFunc = 0;
 ctx->Driver.BlendFuncSeparate = 0;
 ctx->Driver.DepthFunc = 0;
 ctx->Driver.DepthMask = 0;
 ctx->Driver.Fogfv = 0;
 ctx->Driver.Scissor = 0;
 ctx->Driver.CullFace = 0;
 ctx->Driver.FrontFace = 0;
 ctx->Driver.ShadeModel = 0;
 ctx->Driver.ColorMask = 0;
 ctx->Driver.ReducedPrimitiveChange = 0;
 ctx->Driver.RenderStart = s3savDDUpdateState;
 ctx->Driver.RenderFinish = 0;
 ctx->Driver.UpdateState = 0;
 ctx->Driver.UpdateTexturePalette = s3savUpdateTexturePalette;
 ctx->Driver.TexImage = s3savTexImage;
 ctx->Driver.TexSubImage = s3savTexSubImage;
 ctx->Driver.DeleteTexture = s3savDeleteTexture;
 ctx->Driver.IsTextureResident = s3savIsTextureResident;
}

