/*
 * GLX Server Extension
 * Copyright (C) 1996  Steven G. Parker  (sparker@cs.utah.edu)
 * Copyright (C) 1998, 1999 Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * STEVEN PARKER, TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR 
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/* TODO:
 - destroy context should not destroy until the context is made uncurrent
 - pixmaps
 */

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>


#include "extnsionst.h"
#include "dixstruct.h"
#include "scrnintstr.h"
#include "pixmapstr.h"
#include "windowstr.h"
#include "GL/GLXproto.h"
#include "xsmesaP.h"
#include "xsmesa.h"
#include "glxcommon.h"
#include "glx_varray.h"

#include "mesaglx/config.h"
#include "context.h"
#include "glx_log.h"

#include "glxrender.h"
#include "glxdecode.h"
#include "fontstruct.h"
#include "servermd.h"
#include "glxvisual.h"

#include "glx_macros.h"
#include "glx_clients.h"
#include "glx_symbols.h"

#include "xf86Version.h"

#ifdef XFree86LOADER
    #include "xf86_libc.h"
#endif

extern XSMesaVisual GLfind_vis(VisualID vid);



/* valid extension are those which are handled by the existing protocol */

char *valid_gl_ext[] = {
   "GL_EXT_vertex_array", 
#if 0
   "GL_EXT_texture_object", "GL_EXT_abgr", "GL_EXT_texture",
#endif   
// the hardware drivers don't deal well with paletted textures right now,
// so we shouldn't export them as supported. 
#if 0	// BUILD_COLOR_TABLE_EXT
   "GL_EXT_paletted_texture", "GL_EXT_shared_texture_palette",
#endif
   NULL 
};


int GLnoop(ClientPtr client)
{
   log_print(LOG_TRACE, "GLnoop called\n");
   return BadRequest;
}

int GLRender(ClientPtr client)
{
    XSMesaContext ctx;
    char* p;
    char* data;
    int nbytes;
    REQUEST(xGLXRenderReq);
    GLXRenderFunc renderfunc;
    int err = 0;

    REQUEST_AT_LEAST_SIZE(xGLXRenderReq);
    __GLX_MAKE_CURRENT;

    nbytes=(stuff->length<<2) - sizeof(xGLXRenderReq);
    p=(char*)(stuff+1);
    while (nbytes>0)
    {
        CARD16 render_length=*(CARD16*)p;
        CARD16 render_op=*(CARD16*)(p+2);
        if ((render_length <4) || (nbytes - render_length < 0 ))
        {
            ErrorF("bad length in render: %d %d\n", render_op, render_length);
            return (BadLength);
        }

        __GLX_VERIFY_OPCODE(render_op);
        log_print(LOG_TRACE, "ogl op: %#", render_op);

        /* get pointer to appropriate function */
        renderfunc=GLX_render_funcs[render_op];
        data=(char*)(p+4);

        /* call function, via function pointer and check for errors */
        err=(*renderfunc)(render_length-4, data);
        if (err != Success)
            return (err);
        log_print(LOG_TRACE, "*\n");

        nbytes-=render_length;
        p+=render_length;
    }

    return (Success);
}

int GLRender_swapped(ClientPtr client)
{
    XSMesaContext ctx;
    char* p;
    char* data;
    int nbytes;
    REQUEST(xGLXRenderReq);
    GLXRenderFunc renderfunc;
    int err = 0;
    register int n;

    swaps(&stuff->length, n);
    REQUEST_AT_LEAST_SIZE(xGLXRenderReq);
    swapl(&stuff->context_tag, n);
    __GLX_MAKE_CURRENT;

    nbytes=(stuff->length<<2) - sizeof(xGLXRenderReq);
    p=(char*)(stuff+1);
    while (nbytes>0)
    {
        CARD16 render_length=*(CARD16*)p;
        CARD16 render_op=*(CARD16*)(p+2);
        swaps(&render_length, n);
        swaps(&render_op, n);

        if ((render_length <4) || (nbytes - render_length < 0))
        {
            ErrorF("bad length in render: %d %d\n", render_op, render_length);
            return (BadLength);
        }

        __GLX_VERIFY_OPCODE(render_op);
        log_print(LOG_TRACE, "ogl op: %#", render_op);

        /* get pointer to appropriate function */
        renderfunc=GLX_render_funcs_swapped[render_op];
        data=(char*)(p+4);

        /* call function, via function pointer and check for errors */
        err=(*renderfunc)(render_length-4, data);
        if (err != Success)
            return (err);
        log_print(LOG_TRACE, "*\n");

        nbytes-=render_length;
        p+=render_length;
    }

    return (Success);
}

int GLRenderLarge(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXRenderLargeReq);
    GLXRenderFunc renderfunc;
    int err;
    char* data;

    log_print(LOG_TRACE, "GLRenderLarge\n");

    REQUEST_AT_LEAST_SIZE(xGLXRenderLargeReq);

    __GLX_MAKE_CURRENT;
    __GLX_PARSE_RENDER_LARGE(ctx);
    __GLX_VERIFY_OPCODE(ctx->large_op);

    log_print(LOG_TRACE, "Sending a large request\n");
    renderfunc=GLX_render_funcs[ctx->large_op];
    data=(char*)(ctx->large_buf);
    err=(*renderfunc)(ctx->large_len-8, data);
    xfree(ctx->large_buf);
    if (err != Success)
        return (err);

    return (Success);
}

int GLRenderLarge_swapped(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXRenderLargeReq);
    GLXRenderFunc renderfunc;
    int err;
    char* data;
    register int n;

    log_print(LOG_TRACE, "GLRenderLarge swapped\n");

    swaps(&stuff->length, n);
    REQUEST_AT_LEAST_SIZE(xGLXRenderLargeReq);
    swapl(&stuff->context_tag, n);
    swaps(&stuff->request_number, n);
    swaps(&stuff->request_total, n);
    swapl(&stuff->data_bytes, n);

    __GLX_MAKE_CURRENT;
    __GLX_PARSE_RENDER_LARGE(ctx);
    __GLX_VERIFY_OPCODE(ctx->large_op);

    log_print(LOG_TRACE, "Sending a large swapped request\n");
    log_print(LOG_TRACE, "ogl op: %#", ctx->large_op);

    renderfunc=GLX_render_funcs_swapped[ctx->large_op];
    data=(char*)(ctx->large_buf);
    err=(*renderfunc)(ctx->large_len-8, data);
    xfree(ctx->large_buf);
    if (err != Success)
        return (err);

    return (Success);
}

int GLCreateContext(ClientPtr client)
{
    ScreenPtr pScreen;
    GLScreen* glScreen;
    XSMesaContext ctx;
    XSMesaContext sharectx;
    XSMesaVisual mesavisual;
    __glxClient *tmp_client = NULL;
    int i;
    REQUEST(xGLXCreateContextReq);

    log_print(LOG_TRACE, "Entered CreateContext\n");
    REQUEST_SIZE_MATCH(xGLXCreateContextReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context, n);
        swapl(&stuff->visual, n);
        swapl(&stuff->screen, n);
        swapl(&stuff->sharelist, n);
    }
    if (stuff->screen >= screenInfo.numScreens)
    {
        client->errorValue=stuff->screen;
        return (BadValue);
    }
    pScreen = screenInfo.screens[stuff->screen];
    glScreen = &__glScreens[stuff->screen];
    for (i=0;i<pScreen->numVisuals;i++)
    {
        if (pScreen->visuals[i].vid == stuff->visual) 	  
            break;
    }
    if (i>=pScreen->numVisuals)
    {
        client->errorValue=i;
        return (BadValue);
    }
    mesavisual=glScreen->visuals[i];

    if (stuff->sharelist)
    {
        sharectx=LookupIDByType(stuff->sharelist, glContexts);
        if (!sharectx)
            return (GLXBadContextState);
    }
    else
    {
        sharectx=0;
    }

    tmp_client = __glXFindClient(client->index);
    if (tmp_client == NULL)
    {
        /** shouldn't happen! **/
        log_print(LOG_ALWAYS, "Client array corrupt: ClientInfo");
        return (BadMatch);
    }

    /* Get a client-local data structure */
    ctx=GLXProcs.CreateContext(/* client, stuff->context,*/
                                  mesavisual, sharectx);
    if (!ctx)
    {
        return (BadAlloc);
    }
    ctx->last_large_request=0;

	/* if the connection was started with the GLX_NOHW flag, each
	 * context should get a no_accel flag set.
	 */
	ctx->no_accel = tmp_client->no_accel;
	
    /* hook for Keith's Direct Rendering setup 
     */
    /* Not much to do really - going direct is really a client-side
     * activity.  The client will query the context with 'glxIsDirect'
     * to figure out synchronously whether the context was correctly
     * created, and whether it is direct.  If both are true, it will
     * initiate a direct connection.
     */
    ctx->try_direct = 0;

    if (!LocalClient(client)) {
       log_print(LOG_ALWAYS, "-- Non-local client (cannot be direct)");
    } else if (!tmp_client->try_direct) {
       log_print(LOG_ALWAYS, "-- Client has not provided try-direct string");
    } else if (!GLXProcs.AllowDirect(client)) {
       log_print(LOG_ALWAYS, "-- Driver refuses to allow direct client");
    } else 
       ctx->try_direct = 1;
    
    if (!AddResource(stuff->context, glContexts, (pointer)ctx))
    {
        GLXProcs.DestroyContext(ctx);
        return (BadAlloc);
    }
    return (Success);
}

int GLDestroyContext(ClientPtr client)
{
    XSMesaContext ctx;
    /* XSMesaContext sharectx; */
    REQUEST(xGLXDestroyContextReq);

    REQUEST_SIZE_MATCH(xGLXDestroyContextReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context, n);
    }
    ctx=LookupIDByType(stuff->context, glContexts);
    
    fprintf(stderr, "GLDestroyContext\n");
    
    GLXProcs.DestroyContext(ctx);
    if (!ctx)
    {
        return (GLXBadContext);
    }

#ifdef GLX_PROFILING
    /* Hack to make profiling work with the quake games...
     */
    if (getenv("LD_PRELOAD") != 0)
       glx_force_fini_prof();
#endif

    FreeResource(stuff->context, glContexts);
    return (Success);
}

int GLWaitGL(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXWaitGLReq);

    REQUEST_SIZE_MATCH(xGLXWaitGLReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
    }
    __GLX_MAKE_CURRENT;

    /** basically, we want all GL events to finish RIGHT NOW! **/
    glFinish();

    return (Success);
}

int GLSwapBuffers(ClientPtr client)
{
    /* DrawablePtr pDraw; */
    XSMesaContext ctx;
    XSMesaBuffer buf;
    REQUEST(xGLXSwapBuffersReq);

    REQUEST_SIZE_MATCH(xGLXSwapBuffersReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->drawable, n);
    }
    if (stuff->context_tag)
    {
       __GLX_MAKE_CURRENT;
    }

    buf=LookupIDByType(stuff->drawable, glWindows);
    if (!buf)
    {
        return (BadWindow);
    }

	/* make sure mesa flushes any open vertex buffers */
	if ( CC ) {
		FLUSH_VB( CC, "swapbuffers" );
	}

    GLXProcs.SwapBuffers(buf);

    return (Success);
}

int GLUseXFont(ClientPtr client)
{
    int i;
    int encoding;
    FontPtr pfont;
    XSMesaContext ctx;
    REQUEST(xGLXUseXFontReq);

    log_print(LOG_TRACE, "UseXFont");

    REQUEST_SIZE_MATCH(xGLXUseXFontReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->font, n);
        swapl(&stuff->first, n);
        swapl(&stuff->count, n);
        swapl(&stuff->listbase, n);
    }
    __GLX_MAKE_CURRENT;

    pfont = (FontPtr)LookupIDByType(stuff->font, RT_FONT);
    if (!pfont)
    {
        client->errorValue=stuff->font;
        return (BadFont);
    }

    encoding = TwoD16Bit;
    if (pfont->info.lastRow == 0)
        encoding = Linear16Bit;
    for (i=0;i<stuff->count;i++)
    {
        char* bits;
        unsigned long n;
        CharInfoPtr ch;
        int xorig, yorig;
        int width, height;
        int xmove, ymove;
        int nrow;
        int j;
        char* src;
        char* dst;
        CARD8 c[2];
        c[0]=(stuff->first+i)>>8;
        c[1]=(stuff->first+i);
        (*pfont->get_glyphs)(pfont, 1, c, encoding, &n, &ch);

        xorig=-ch->metrics.leftSideBearing;
        yorig=ch->metrics.descent-1;
        width=ch->metrics.rightSideBearing-ch->metrics.leftSideBearing;
        height=ch->metrics.ascent+ch->metrics.descent;
        xmove=ch->metrics.characterWidth;
        ymove=0;

        nrow=BYTES_PER_ROW(width, GLYPHPADBYTES);
        bits=(char*)xalloc(sizeof(char)*nrow*height);
        src=ch->bits;
        dst=bits+nrow*(height-1);
        for (j=0;j<height;j++)
        {
            int k;
            for (k=0;k<nrow;k++)
            {
                dst[k]=src[k];
            }
            dst-=nrow;
            src+=nrow;
        }
        glNewList(stuff->listbase+i, GL_COMPILE);
        if (IMAGE_BYTE_ORDER == LSBFirst)
        {
            glPixelStorei(GL_UNPACK_SWAP_BYTES, GL_TRUE);
        }
        else
        {
            glPixelStorei(GL_UNPACK_SWAP_BYTES, GL_FALSE);
        }
        if (BITMAP_BIT_ORDER == LSBFirst)
        {
            glPixelStorei(GL_UNPACK_LSB_FIRST, GL_TRUE);
        }
        else
        {
            glPixelStorei(GL_UNPACK_LSB_FIRST, GL_FALSE);
        }
        glPixelStorei(GL_UNPACK_ALIGNMENT, GLYPHPADBYTES);
        if (width > 0 || height > 0)
        {
            glBitmap(width, height, xorig, yorig, xmove, ymove,
                     (unsigned char*)bits);
        }
        glEndList();
        xfree(bits);
    }
    log_print(LOG_TRACE, "Done with XFont");
    return (Success);
}


int GLCreateGLXPixmap(ClientPtr client)
{
    XSMesaBuffer buf;
    XSMesaVisual vis;
    PixmapPtr p;
    REQUEST(xGLXCreateGLXPixmapReq);

    log_print(LOG_TRACE, "CreateGLXPixmap");
    REQUEST_SIZE_MATCH(xGLXCreateGLXPixmapReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->screen, n);
        swapl(&stuff->visual, n);
        swapl(&stuff->pixmap, n);
        swapl(&stuff->glx_pixmap, n);
    }

    vis = GLfind_vis(stuff->visual);
    if (vis == 0) return (BadValue);

    p = LookupIDByType(stuff->pixmap, RT_PIXMAP);
    if (p == 0) return (BadPixmap);

    buf = GLXProcs.CreatePixmapBuffer(vis, client, p, 0);
    if (buf == 0) return (BadAlloc);

    AddResource(stuff->glx_pixmap, glPixmaps, (pointer)buf);

    return (Success);
}

int GLDestroyGLXPixmap(ClientPtr client)
{
    XSMesaBuffer buf;
    REQUEST(xGLXDestroyGLXPixmapReq);

    log_print(LOG_TRACE, "DestroyGLXPixmap");
    REQUEST_SIZE_MATCH(xGLXDestroyGLXPixmapReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->glx_pixmap, n);
    }

    buf = LookupIDByType(stuff->glx_pixmap, glPixmaps);
    if (buf == NULL) return (GLXBadPixmap);

    log_print(LOG_TRACE, "freeing the buffer");
    FreeResource(stuff->glx_pixmap, glPixmaps);

    log_print(LOG_TRACE, "destroying the buffer");
    GLXProcs.DestroyBuffer(buf);

    return (Success);
}


int GLWaitX(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXWaitXReq);

    REQUEST_SIZE_MATCH(xGLXWaitXReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
    }
    __GLX_MAKE_CURRENT;

    /** we want all X events to occur NOW! **/

    return (Success);
}

int GLCopyContext(ClientPtr client)
{
/*
    ScreenPtr pScreen;
    GLScreen* glScreen;
*/
    XSMesaContext src_ctx;
    XSMesaContext dst_ctx;
/*
    XSMesaContext sharectx;
    VisualPtr visual;
*/
    REQUEST(xGLXCopyContextReq);

    REQUEST_SIZE_MATCH(xGLXCopyContextReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->source_context, n);
        swapl(&stuff->dest_context, n);
        swapl(&stuff->mask, n);
        swapl(&stuff->source_tag, n);
    }
    /* If there is a source tag, do a glFlush() */
    if (stuff->source_tag)
    {
        src_ctx=LookupIDByType(stuff->source_tag, glContexts);
        if (!src_ctx)
        {
            return (GLXBadContextTag);
        }
        GLXProcs.MakeCurrent(src_ctx);
        glFlush();
    }

    src_ctx=LookupIDByType(stuff->source_context, glContexts);
    if (!src_ctx)
    {
        return (GLXBadContext);
    }

    dst_ctx=LookupIDByType(stuff->dest_context, glContexts);
    if (!dst_ctx)
    {
        return (GLXBadContext);
    }

    GLXProcs.CopyContext(src_ctx, dst_ctx, stuff->mask);
    return (Success);
}

int GLIsDirect(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXIsDirectReq);
    xGLXIsDirectReply reply;
    extern ClientPtr direct_client;

    REQUEST_SIZE_MATCH(xGLXIsDirectReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.length=0;
    reply.sequenceNumber = client->sequence;
    reply.is_direct = 0;

    if (!ctx->try_direct) {
       log_print(LOG_ALWAYS, "Context is not direct");
    } else if (direct_client != 0 && direct_client != client) {
       log_print(LOG_ALWAYS, "There is already a direct client");
    } else
       reply.is_direct = 1;

    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
    }
    WriteToClient(client, sizeof(xGLXIsDirectReply), (char*)&reply);
    return(client->noClientException);
}

/* There are still some unreported error conditions here... */
int GLMakeCurrent(ClientPtr client)
{
    REQUEST(xGLXMakeCurrentReq);
    xGLXMakeCurrentReply reply;
    XSMesaContext ctx = NULL;
    XSMesaBuffer buf = NULL;
    DrawablePtr pDraw;

    REQUEST_SIZE_MATCH(xGLXMakeCurrentReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->drawable, n);
        swapl(&stuff->context, n);
        swapl(&stuff->old_context, n);
    }

    if (stuff->context != 0)
    {
        ctx=LookupIDByType(stuff->context, glContexts);
        if (!ctx)
            return (GLXBadContext);
    }

    if (stuff->drawable)
    {
        buf=LookupIDByType(stuff->drawable, glPixmaps);
        if (!buf)
        {
            buf=LookupIDByType(stuff->drawable, glWindows);
            if (!buf)
            {
                XSMesaVisual v;
                WindowPtr win;
                VERIFY_DRAWABLE(pDraw, stuff->drawable, client);
                if (pDraw->type != DRAWABLE_WINDOW)
                {
                    return (GLXBadDrawable);
                }
                win = (WindowPtr)pDraw;
                v   = GLfind_vis(wVisual(win));
                buf = GLXProcs.CreateWindowBuffer( v, client, win );
                if (!buf)
                {
                    log_print(LOG_ALWAYS, "Couldn't allocate MesaWindowBuffer");
                    return (BadAlloc);
                }
		fprintf(stderr, "Adding drawable %ld to glWindows list:\n",
			stuff->drawable);
                if (!AddResource(stuff->drawable, glWindows, (pointer)buf))
                {
                    return (BadAlloc);
                }
            }
        }
    }

    if (!GLXProcs.BindBuffer(ctx, buf))
    {
        log_print(LOG_ALWAYS, "buffer not bound!!");
    }
    reply.type=X_Reply;
    reply.length=0;
    reply.sequenceNumber=client->sequence;
    reply.context_tag=stuff->context;

    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.context_tag, n);
    }

    WriteToClient(client, sizeof(xGLXMakeCurrentReply), (char*)&reply);
    return(client->noClientException);
}

int GLQueryVersion(ClientPtr client)
{
    REQUEST(xGLXQueryVersionReq);
    xGLXQueryVersionReply reply;

    REQUEST_SIZE_MATCH(xGLXQueryVersionReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->major_version, n);
        swapl(&stuff->minor_version, n);
    }
    reply.type=X_Reply;
    reply.length=0;
    reply.sequenceNumber = client->sequence;
    reply.major_version = GLX_MAJOR_VERSION;
    reply.minor_version = GLX_MINOR_VERSION;
    if (reply.major_version != stuff->major_version)
        reply.major_version = 0;
    if (reply.minor_version > stuff->minor_version)
        reply.minor_version = stuff->minor_version;
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.major_version, n);
        swapl(&reply.minor_version, n);
    }
    WriteToClient(client, sizeof(xGLXQueryVersionReply), (char*)&reply);
    return(client->noClientException);
}

int GLAreTexturesResident(ClientPtr client)
{
    XSMesaContext ctx;
    void *data = NULL;
    GLboolean *residences = NULL;
    GLuint res_size = 0;
    xGLXAreTexturesResidentReply reply;

    REQUEST(xGLXAreTexturesResidentReq);

    log_print(LOG_TRACE, "entering AreTexturesResident");
    REQUEST_AT_LEAST_SIZE(xGLXAreTexturesResidentReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->length, n);
        swapl(&stuff->n, n);
    }
    __GLX_MAKE_CURRENT;

    data = (void *) ( ((char *) stuff) + sz_xGLXAreTexturesResidentReq);

    if (client->swapped)
    {
        register int n;
        int i;
        int *data_ptr = (int *) data;
        for (i = 0; i < stuff->n; i++)
        {
            swapl(&data_ptr[i], n);
        }
    }

#ifdef GLX_LOG
    {
        int i;
        GLuint *data_ptr = (GLuint *) data;

        log_print(LOG_DATA, "resident textures (%d): \n", stuff->n);
        for ( i = 0; i < stuff->n; i++)
        {
            log_print(LOG_DATA, "data (%d): %d\n", 
               ( ( (unsigned char *) data_ptr) + i), data_ptr[i]);
        }
    }
#endif

    res_size = (stuff->n + 3) >> 2;
    residences = (GLboolean *) xalloc(res_size);
    if (residences == NULL) 
       log_print(LOG_ALWAYS, "couldn't allocate memory!!!!\n");

    /* XSMesaMakeCurrent(ctx); */
    reply.value = glAreTexturesResident(stuff->n, (GLuint *) data, residences);

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    reply.length=0;
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
    }

    WriteToClient(client, sizeof(xGLXAreTexturesResidentReply), (char*)&reply);
    WriteToClient(client, res_size, (char *) residences);
    xfree(residences);
    return(client->noClientException);
}

int GLDeleteLists(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXDeleteListsReq);

    REQUEST_AT_LEAST_SIZE(xGLXDeleteListsReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->list, n);
        swapl(&stuff->range, n);
    }
    __GLX_MAKE_CURRENT;
    glDeleteLists(stuff->list, stuff->range);

    return (Success);
}

int GLDeleteTextures(ClientPtr client)
{
    XSMesaContext ctx;
    void *data = NULL;
    REQUEST(xGLXDeleteTexturesReq);

    log_print(LOG_TRACE, "entering DeleteTextures");
    REQUEST_AT_LEAST_SIZE(xGLXDeleteTexturesReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->length, n);
        swapl(&stuff->n, n);
    }
    __GLX_MAKE_CURRENT;
    data = (void *) ( ((char *) stuff) + sz_xGLXDeleteTexturesReq);

    if (client->swapped)
    {
        register int n;
        int i;
        int *data_ptr = (int *) data;
        for (i = 0; i < stuff->n; i++)
        {
            swapl(&data_ptr[i], n);
        }
    }

#if GLX_LOG
    {
        int i;
        int *data_ptr = (int *) data;

        log_print(LOG_DATA, "deleting textures (%d): ", stuff->n);
        for ( i = 0; i < stuff->n; i++)
        {
            log_print(LOG_DATA, "%d ", data_ptr[i] );
        }
        log_print(LOG_DATA, "\n");
    }
#endif

    GLXProcs.MakeCurrent(ctx);
    glDeleteTextures(stuff->n, (GLuint *) data);

    return (Success);
}

int GLClientInfo(ClientPtr client)
{
    __glxClient *tmp_client = NULL;
    char *client_exts = NULL;
    char *p = NULL;
    char *shared = NULL;
    int i;

    REQUEST(xGLXClientInfoReq);

    REQUEST_AT_LEAST_SIZE(xGLXClientInfoReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->major_version, n);
        swapl(&stuff->minor_version, n);
        swapl(&stuff->stringlength, n);
    }

    tmp_client = __glXFindClient(client->index);
    if (tmp_client == NULL)
    {
        /** shouldn't happen! **/
        log_print(LOG_ALWAYS, "Client array corrupt: ClientInfo");
        return (BadMatch);
    }

    if (stuff->major_version != GLX_MAJOR_VERSION)
    {
        log_print(LOG_ALWAYS, "GLX Client info: Major version mismatch!");
        return (BadMatch);
    }

    tmp_client->major = stuff->major_version;
    if (stuff->minor_version < GLX_MINOR_VERSION)
    {
        tmp_client->minor = stuff->minor_version; 
    }
    else
    {
        tmp_client->minor = GLX_MINOR_VERSION;
    }

    tmp_client->no_accel = GL_FALSE;

    /** still need to do more with client extensions **/
    client_exts = (char *) ( ((char *) stuff) + sz_xGLXClientInfoReq);
    log_print(LOG_TRACE, "client exts: %s\n", client_exts);
    shared = (char *) xalloc(strlen(client_exts) + 1);
    shared[0] = '\0';
    for (p = strtok(client_exts, " "); p ; p = strtok(NULL, " "))
    {
        for (i=0;valid_gl_ext[i];i++) 
        if (!strcmp(p, valid_gl_ext[i]))
        {
            strcat(shared, p);
            strcat(shared, " ");
        }
        if (!strcmp(p, GLX_NOHW_STRING)) {
           log_print(LOG_TRACE, "Got no hw accel string!\n");
           tmp_client->no_accel = GL_TRUE;
        }
        if (!strcmp(p, GLX_TRY_DIRECT_STRING)) {
           log_print(LOG_TRACE, "Got try direct string!\n");
           tmp_client->try_direct = GL_TRUE;
        }
    }
    tmp_client->exts = shared;
    log_print(LOG_TRACE, "shared exts: %s\n", shared);

    return (Success);
}


int GLVendorPrivate(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXVendorPrivateReq);
    REQUEST_AT_LEAST_SIZE(xGLXVendorPrivateReq);

    if (client->swapped)
    {
        register int n;
        swapl(&stuff->length, n);
        swapl(&stuff->opcode, n);
        swapl(&stuff->context_tag, n);
    }

    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (ctx)
       GLXProcs.MakeCurrent(ctx);

    switch (stuff->opcode) {
        default:
	   return GLXProcs.VendorPrivate( client, ctx, stuff );
    }

    return Success;
}

int GLVendorPrivateWithReply(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXVendorPrivateReq);
    REQUEST_AT_LEAST_SIZE(xGLXVendorPrivateReq);

    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (ctx)
       GLXProcs.MakeCurrent(ctx);

    if (client->swapped)
    {
        register int n;
        swapl(&stuff->length, n);
        swapl(&stuff->opcode, n);
        swapl(&stuff->context_tag, n);
    }

    return GLXProcs.VendorPrivate( client, ctx, stuff );
}

static int bit_count(int i)
{
    int c=0;
    while (i)
    {
        if (i&1)
            c++;
        i>>=1;
    }
    return (c);
}

static void send_visual(ClientPtr client, XSMesaVisual v, INT32* props)
{
    GLvisual* glv=v->gl_visual;
    INT32* p=props;
    *p++=v->pVisual->vid;       /* vid */
    *p++=v->pVisual->class;     /* class */
         *p++=glv->RGBAflag;         /* rgba */
    if (glv->RGBAflag)
    {
        *p++=bit_count(v->pVisual->redMask);        /* redSize */
        *p++=bit_count(v->pVisual->greenMask);  /* greenSize */
        *p++=bit_count(v->pVisual->blueMask);   /* blueSize */
#ifdef MESA31      
        if (glv->SoftwareAlpha)
#else      
        if (glv->FrontAlphaEnabled || glv->BackAlphaEnabled)
#endif	
	
            *p++=8;             /* alphaSize */
        else
            *p++=0;
        *p++=glv->AccumBits;        /* accumRedSize */
        *p++=glv->AccumBits;        /* accumGreenSize */
        *p++=glv->AccumBits;        /* accumBlueSize */
        *p++=glv->AccumBits;        /* accumAlphaSize */
    }
    else
    {
        *p++=0;             /* redSize */
        *p++=0;             /* greenSize */
        *p++=0;             /* blueSize */
        *p++=0;             /* alphaSize */
        *p++=0;             /* accumRedSize */
        *p++=0;             /* accumGreenSize */
        *p++=0;             /* accumBlueSize */
        *p++=0;             /* accumAlphaSize */
    }
    *p++=glv->DBflag;           /* doubleBuffer */
    *p++=GL_FALSE;              /* stereo */
    *p++=v->pVisual->nplanes;       /* bufferSize */
    *p++=glv->DepthBits;        /* depthSize */
    *p++=glv->StencilBits;      /* stencilSize */
    *p++=GL_FALSE;          /* auxBuffers */
    *p++=v->level;          /* level */
    if (client->swapped)
    {
        register int n;
        register int i;
        for (i=0;i<18;i++)
        {
            swapl(&props[i], n);
        }
    }
    WriteToClient(client, sizeof(INT32)*18, (char*)props);
}

int GLGetVisualConfigs(ClientPtr client)
{
    REQUEST(xGLXGetVisualConfigsReq);
    xGLXGetVisualConfigsReply reply;
    int numvisuals, numprops;
    INT32* props;
    /** INT32* p; **/
    GLScreen* glScreen;
    int i;

    REQUEST_SIZE_MATCH(xGLXGetVisualConfigsReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->screen, n);
    }

    if (stuff->screen > screenInfo.numScreens)
    {
        client->errorValue=stuff->screen;
        return (BadValue);
    }

    glScreen = &__glScreens[stuff->screen];
    numvisuals=0;
    for (i=0;i<glScreen->numVisuals;i++)
    {
        if (glScreen->visuals[i])
            numvisuals++;
    }
    numprops=18;
    reply.type=X_Reply;
    reply.length=numvisuals*numprops;
    reply.sequenceNumber = client->sequence;
    reply.num_visuals=numvisuals;
    reply.num_props=numprops;
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.num_visuals, n);
        swapl(&reply.num_props, n);
    }
    WriteToClient(client, sizeof(xGLXGetVisualConfigsReply), (char*)&reply);


    props=(INT32*)ALLOCATE_LOCAL(numprops*sizeof(INT32));
    /* Count the RGB, single buffered visuals */
    for (i=0;i<glScreen->numVisuals;i++)
    {
        XSMesaVisual v=glScreen->visuals[i];
        send_visual(client, v, props);
    }
    DEALLOCATE_LOCAL(props);
    return(client->noClientException);
}

int GLNewList(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXNewListReq);

    log_print(LOG_TRACE, "NewList");
    REQUEST_SIZE_MATCH(xGLXNewListReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->list, n);
        swapl(&stuff->mode, n);
    }
    __GLX_MAKE_CURRENT;

    glNewList(stuff->list, stuff->mode);
    return (Success);
}

int GLEndList(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXEndListReq);

    REQUEST_SIZE_MATCH(xGLXEndListReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
    }
    __GLX_MAKE_CURRENT;

    glEndList();
    return (Success);
}

int GLGenLists(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGenListsReq);
    xGLXGenListsReply reply;

    REQUEST_SIZE_MATCH(xGLXGenListsReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.length=0;
    reply.sequenceNumber = client->sequence;
    reply.data=glGenLists(stuff->data);
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.data, n);
    }
    WriteToClient(client, sizeof(xGLXGenListsReply), (char*)&reply);
    return(client->noClientException);
}

int GLGenTextures(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGenTexturesReq);
    xGLXGenTexturesReply reply;
    unsigned int *data = NULL;
    unsigned int data_size = 0;

    log_print(LOG_TRACE, "entering GenTextures");
    REQUEST_SIZE_MATCH(xGLXGenTexturesReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->n, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.length=0;
    reply.sequenceNumber = client->sequence;

    data_size = stuff->n * 4;
    data = (unsigned int *) xalloc(data_size);

    glGenTextures(stuff->n, data);

#ifdef GLX_LOG
    {
        int i;
        log_print(LOG_DATA, "new textures (%d): ", stuff->n);
        for (i = 0; i < stuff->n; i++)
        {
            log_print(LOG_DATA, "%d ", data[i]);
        }
        log_print(LOG_DATA, "\n");
    }
#endif

    if (client->swapped)
    {
        register int n, i;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        for (i = 0; i < stuff->n; i++)
        {
            swapl(&data[i], n);
        }
    }
    WriteToClient(client, sizeof(xGLXGenTexturesReply), (char *) &reply);
    WriteToClient(client, data_size, (char *) data);
    log_print(LOG_TRACE, "done with GenTextures");
    xfree(data);
    return(client->noClientException);
}

int GLIsTexture(ClientPtr client)
{
  XSMesaContext ctx;
  REQUEST(xGLXIsTextureReq);
  xGLXIsTextureReply reply;
  int result;

  log_print(LOG_TRACE,"entering IsTexture");
  REQUEST_SIZE_MATCH(xGLXIsTextureReq);
  if(client->swapped){
    register int n;
    swapl(&stuff->context_tag,n);
    swapl(&stuff->texture,n);
  }

  __GLX_MAKE_CURRENT;
  
  result=glIsTexture(stuff->texture);
  reply.type=X_Reply;
  reply.value=result;
  reply.length=0;
  reply.sequenceNumber = client->sequence;

  if(client->swapped){
    register int n;
    swapl(&reply.sequenceNumber,n);
    swapl(&reply.length,n);
    swapl(&reply.value,n);
  }
  WriteToClient(client,sizeof(xGLXIsTextureReply),(char *) &reply);
  return(client->noClientException);
}


int GLRenderMode(ClientPtr client)
{
    char* buffer;
    GLint oldrendermode;
    GLint nbuf;
    GLint ret;
    int l;
    XSMesaContext ctx;
    REQUEST(xGLXRenderModeReq);
    xGLXRenderModeReply reply;
    REQUEST_SIZE_MATCH(xGLXRenderModeReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->mode, n);
    }
    __GLX_MAKE_CURRENT;

    glGetIntegerv(GL_RENDER_MODE, &oldrendermode);
    ret=glRenderMode(stuff->mode);
    if (oldrendermode != GL_RENDER)
    {
        reply.type=X_Reply;
        if (oldrendermode == GL_FEEDBACK)
        {
            nbuf=ret;
        }
        else
        {
            int i;
            GLuint* sbuf = (GLuint*)GLXProcs.SelectionBuffer(ctx);
            GLuint* p    = sbuf;
            nbuf         = 0;
            for (i=0;i<ret;i++)
            {
                int reclen = 3 + *p;
                nbuf      += reclen;
                p         += reclen;
            }
        }
        reply.n              = nbuf;
        reply.length         = l = nbuf;
        reply.retval         = ret;
        reply.newmode        = stuff->mode;
        reply.sequenceNumber = client->sequence;
        if (client->swapped)
        {
            register int n;
            swaps(&reply.sequenceNumber, n);
            swapl(&reply.n, n);
            swapl(&reply.length, n);
            swapl(&reply.retval, n);
            swapl(&reply.newmode, n);
        }
        WriteToClient(client, sizeof(xGLXRenderModeReply), (char*)&reply);
        switch (oldrendermode)
        {
            case GL_FEEDBACK:
                buffer = GLXProcs.FeedbackBuffer(ctx);
                if (client->swapped)
                {
                    register int n;
                    register int i;
                    GLfloat* fb = (GLfloat*)buffer;
                    for (i=0;i<l;i++)
                    {
                        swapl(&fb[i], n);
                    }
                }
                WriteToClient(client, l*4, (char*)buffer);
                break;
            case GL_SELECT:
                buffer = GLXProcs.SelectionBuffer(ctx);
                if (client->swapped)
                {
                    register int n;
                    register int i;
                    GLuint* sb = (GLuint*)buffer;
                    for (i=0;i<l;i++)
                    {
                        swapl(&sb[i], n);
                    }
                }
                WriteToClient(client, l*4, (char*)buffer);
                break;
        }
    }
    return (Success);
}

int GLFeedbackBuffer(ClientPtr client)
{
    GLfloat* buffer;
    XSMesaContext ctx;
    REQUEST(xGLXFeedbackBufferReq);

    REQUEST_SIZE_MATCH(xGLXFeedbackBufferReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->size, n);
        swapl(&stuff->type, n);
    }
    __GLX_MAKE_CURRENT;

    /* If Mesa already has a feedback buffer, free it and allocate
     * a new one, and give the new one to Mesa. There should be no
     * xfree() to free buffer for this reason.
     */
    if (GLXProcs.FeedbackBuffer(ctx))
        xfree(GLXProcs.FeedbackBuffer(ctx));
    buffer=(GLfloat*)xalloc(4*stuff->size);
    if (!buffer)
        return (BadAlloc);
    glFeedbackBuffer(stuff->size, stuff->type, buffer);
    return (Success);
}

int GLSelectBuffer(ClientPtr client)
{
    GLuint* buffer;
    XSMesaContext ctx;
    REQUEST(xGLXSelectBufferReq);

    REQUEST_SIZE_MATCH(xGLXSelectBufferReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->size, n);
    }
    __GLX_MAKE_CURRENT;

    /* If Mesa already has a selection buffer, free it and allocate
     * a new one, and give the new one to Mesa. There should be no
     * xfree() to free buffer for this reason.
     */
    if (GLXProcs.SelectionBuffer(ctx))
        xfree(GLXProcs.SelectionBuffer(ctx));
    buffer=(GLuint*)xalloc(4*stuff->size);
    if (!buffer)
        return (BadAlloc);
    glSelectBuffer(stuff->size, buffer);
    return (Success);
}


int GLIsList(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXIsListReq);
    xGLXIsListReply reply;

    REQUEST_SIZE_MATCH(xGLXIsListReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.length=0;
    reply.sequenceNumber = client->sequence;
    reply.data=glIsList(stuff->data);
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.data, n);
    }
    WriteToClient(client, sizeof(xGLXIsListReply), (char*)&reply);
    return(client->noClientException);
}

int GLIsEnabled(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXIsEnabledReq);
    xGLXIsEnabledReply reply;

    REQUEST_SIZE_MATCH(xGLXIsEnabledReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.length=0;
    reply.sequenceNumber = client->sequence;
    reply.data=glIsEnabled(stuff->data);
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.data, n);
    }
    WriteToClient(client, sizeof(xGLXIsEnabledReply), (char*)&reply);
    return(client->noClientException);
}

int GLFlush(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXFlushReq);

    REQUEST_SIZE_MATCH(xGLXFlushReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
    }
    __GLX_MAKE_CURRENT;

    glFlush();
    return (Success);
}

int GLFinish(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXFinishReq);
    xGLXFinishReply(reply);

    REQUEST_SIZE_MATCH(xGLXFinishReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
    }
    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (!ctx)
    {
        return (GLXBadContextState);
    }
    __GLX_MAKE_CURRENT;

    glFinish();
    reply.type           = X_Reply;
    reply.length         = 0;
    reply.sequenceNumber = client->sequence;
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
    }
    WriteToClient(client, sizeof(xGLXFinishReply), (char*)&reply);
    return(client->noClientException);
}

int GLGetIntegerv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetIntegervReq);
    xGLXGetIntegervReply reply;
    int n;
    int ret[16];

    REQUEST_SIZE_MATCH(xGLXGetIntegervReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (!ctx)
    {
        return (GLXBadContextState);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetIntegerv(stuff->data, ret);
    n=GLX_get_size(stuff->data);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetIntegervReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}

int GLGetError(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetErrorReq);
    xGLXGetErrorReply reply;

    /* This is at_least_size, because SGI sends 3 words for this
       request, instead of 2 as the spec requires.  I am not sure
       what the third word does...  Steve Parker
       */
    REQUEST_AT_LEAST_SIZE(xGLXGetErrorReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
    }
    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (!ctx)
    {
        return (GLXBadContextState);
    }
    __GLX_MAKE_CURRENT;

    reply.type           = X_Reply;
    reply.sequenceNumber = client->sequence;
    reply.length         = 0;
    reply.data           = glGetError();
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.data, n);
    }
    WriteToClient(client, sizeof(xGLXGetErrorReply), (char*)&reply);

    return(client->noClientException);
}

int GLGetClipPlane(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetClipPlaneReq);
    xGLXGetClipPlaneReply reply;
    double equation[4];

    REQUEST_SIZE_MATCH(xGLXGetClipPlaneReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
    }

    if ( (stuff->data < GL_CLIP_PLANE0) || 
         (stuff->data > GL_CLIP_PLANE0+GL_MAX_CLIP_PLANES) )
    {
        reply.length=0;
        if (client->swapped)
        {
            register int n;
            swapl(&reply.length, n);
        }
        WriteToClient(client, sizeof(xGLXGetClipPlaneReply), (char*)&reply);
    }
    else
    {
        glGetClipPlane(stuff->data, equation);
        reply.length=8;
        if (client->swapped)
        {
            register int n;
            swaps(&reply.length, n);
            swapd(&equation[0], n);
            swapd(&equation[1], n);
            swapd(&equation[2], n);
            swapd(&equation[3], n);
        }
        WriteToClient(client, sizeof(xGLXGetClipPlaneReply), (char*)&reply);
        WriteToClient(client, sizeof(double)*4, (char*)equation);
    }
    return(client->noClientException);
}

int GLGetBooleanv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetBooleanvReq);
    xGLXGetBooleanvReply reply;
    int n;
    GLboolean ret[16];

    REQUEST_SIZE_MATCH(xGLXGetBooleanvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetBooleanv(stuff->data, ret);
    n=GLX_get_size(stuff->data);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singlebool=ret[0];
    }
    else
    {
        reply.length=(n+3)/4;
    }

    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.n, n);
    }
    WriteToClient(client, sizeof(xGLXGetIntegervReply), (char*)&reply);
    if (n!=1)
    {
        n=((n+3)/4)*4;
        /* does ret need to be swapped? */
        WriteToClient(client, n, (char*)ret);
    }
    return(client->noClientException);
}

int GLGetString(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetStringReq);
    xGLXGetStringReply reply;
    const char* string;
    const char* nullstring = "\0\0\0\0";
    int n;
    int l;

    REQUEST_SIZE_MATCH(xGLXGetIntegervReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    string = (char *) glGetString(stuff->data);

    /* make sure only valid extensions pass through: */
    if (stuff->data == GL_EXTENSIONS)
    {
       __glxClient *clientPtr = NULL;
       clientPtr = __glXFindClient(client->index);
       if (clientPtr) string = clientPtr->exts;
       log_print(LOG_ALWAYS, "Can't find client for %d!!\n", client->index);
    }

    log_print(LOG_TRACE, "%e : %s\n", stuff->data, string);

    /* FIXME: find the cause why getstring fails
      in assert_outside_begin_end_and_flush_with_return */
    if(!string) string = nullstring;
    n=strlen((char*)string)+1;
    reply.length=l=(n+3)/4;
    reply.n=n;
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.n, n);
    }

    WriteToClient(client, sizeof(xGLXGetStringReply), (char*)&reply);
    WriteToClient(client, sizeof(int)*l, (char*)string);
    return(client->noClientException);
}

int GLGetFloatv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetFloatvReq);
    xGLXGetFloatvReply reply;
    int n;
    float ret[16];

    log_print(LOG_TRACE, "In GetFloat");
    REQUEST_SIZE_MATCH(xGLXGetFloatvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    __GLX_MAKE_CURRENT;

    log_print(LOG_TRACE, "Calling MakeCurrent");
    GLXProcs.MakeCurrent(ctx);
    log_print(LOG_TRACE, "Done");

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetFloatv(stuff->data, ret);
    n=GLX_get_size(stuff->data);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetFloatvReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    log_print(LOG_TRACE, "Done with GetFloat");
    return(client->noClientException);
}

int GLReadPixels(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXReadPixelsReq);
    xGLXReadPixelsReply reply;
    int size;
    char* image;

    REQUEST_SIZE_MATCH(xGLXReadPixelsReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->x, n);
        swapl(&stuff->y, n);
        swapl(&stuff->width, n);
        swapl(&stuff->height, n);
        swapl(&stuff->format, n);
        swapl(&stuff->type, n);
    }

    size=GLX_image_size(stuff->width, stuff->height, stuff->format,
                        stuff->type, 4);
    /* 4?  See PACK_ALIGNMENT below!! wtf? grantham@alt.net */
    if (size < 0) {
       return (GLXBadRenderRequest);
    }
    __GLX_MAKE_CURRENT;

    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (!ctx)
    {
        return (GLXBadContextState);
    }
    GLXProcs.MakeCurrent(ctx);

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glPixelStorei(GL_PACK_SWAP_BYTES, stuff->swap_bytes);
    glPixelStorei(GL_PACK_LSB_FIRST, stuff->lsb_first);
    glPixelStorei(GL_PACK_ALIGNMENT, 4);
    image=(char*)xalloc(size);
    if (image == NULL) return BadAlloc;
    memset(image, 0, size);

    log_print(LOG_TRACE, "Calling glReadPixels");
    glReadPixels(stuff->x, stuff->y, stuff->width, stuff->height,
                 stuff->format, stuff->type, (void *)image);
    log_print(LOG_TRACE, "Done with glReadPixels");

    reply.length=size>>2;
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
    }
    WriteToClient(client, sizeof(xGLXReadPixelsReply), (char*)&reply);
    WriteToClient(client, size, image);
    FlushAllOutput();
    xfree(image);
    return(client->noClientException);
}

int GLGetTexImage(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetTexImageReq);
    xGLXGetTexImageReply reply;
    int size;
    char* image;
    int width, height;

    REQUEST_SIZE_MATCH(xGLXGetTexImageReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->target, n);
        swapl(&stuff->level, n);
        swapl(&stuff->format, n);
        swapl(&stuff->type, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glPixelStorei(GL_PACK_SWAP_BYTES, stuff->swap_bytes);
    glPixelStorei(GL_PACK_ALIGNMENT, 4);

    glGetTexLevelParameteriv(stuff->target, stuff->level,
                             GL_TEXTURE_WIDTH, &width);
    switch (stuff->target)
    {
        case GL_TEXTURE_1D:
            height=1;
            break;
        case GL_TEXTURE_2D:
            glGetTexLevelParameteriv(stuff->target, stuff->level,
                                     GL_TEXTURE_HEIGHT, &height);
            break;
            break;
    }
    size=GLX_image_size(width, height, stuff->format, stuff->type, 4);
    /* alignment 4???  See PACK_ALIGNMENT above!!! wtf? grantham@alt.net */
    image=(char*)xalloc(size);
    glGetTexImage(stuff->target, stuff->level, stuff->format, stuff->type,
                  image);
    reply.length=size>>2;
    reply.width=width;
    reply.height=height;
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.width, n);
        swapl(&reply.height, n);
    }
    WriteToClient(client, sizeof(xGLXGetTexImageReply), (char*)&reply);
    WriteToClient(client, size, image);
    xfree(image);
    return(client->noClientException);
}

int GLGetLightfv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetLightfvReq);
    xGLXGetLightfvReply reply;
    int n;
    float ret[4];

    REQUEST_SIZE_MATCH(xGLXGetLightfvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->light, n);
        swapl(&stuff->pname, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetLightfv(stuff->light, stuff->pname, ret);
    n=GLX_light_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetLightfvReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(float)*n, (char*)ret);
    }
    return(client->noClientException);
}

int GLGetLightiv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetLightivReq);
    xGLXGetLightivReply reply;
    int n;
    GLint ret[4];

    REQUEST_SIZE_MATCH(xGLXGetLightivReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->light, n);
        swapl(&stuff->pname, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetLightiv(stuff->light, stuff->pname, ret);
    n=GLX_light_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetLightivReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}

int GLGetPolygonStipple(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetPolygonStippleReq);
    xGLXGetPolygonStippleReply reply;
    GLubyte mask[128];

    REQUEST_SIZE_MATCH(xGLXGetPolygonStippleReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glPixelStorei(GL_PACK_LSB_FIRST, stuff->lsb_first);
    glPixelStorei(GL_PACK_ALIGNMENT, 1);
    glGetPolygonStipple(mask);
    reply.length=32;

    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
    }
    WriteToClient(client, sizeof(xGLXGetPolygonStippleReply), (char*)&reply);
    WriteToClient(client, 128, (char*)mask);
    return(client->noClientException);
}

int GLGetMaterialfv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetMaterialfvReq);
    xGLXGetMaterialfvReply reply;
    int n;
    float ret[4];

    REQUEST_SIZE_MATCH(xGLXGetMaterialfvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->face, n);
        swapl(&stuff->pname, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetMaterialfv(stuff->face, stuff->pname, ret);
    n=GLX_material_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetMaterialfvReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(float)*n, (char*)ret);
    }
    return(client->noClientException);
}

int GLGetPixelMapfv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetPixelMapfvReq);
    xGLXGetPixelMapfvReply reply;
    int n;
    float* ret;
    int sizetoken;

    REQUEST_SIZE_MATCH(xGLXGetPixelMapfvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    switch (stuff->data)
    {
        case GL_PIXEL_MAP_I_TO_I:
            sizetoken=GL_PIXEL_MAP_I_TO_I;
            break;
        case GL_PIXEL_MAP_S_TO_S:
            sizetoken=GL_PIXEL_MAP_S_TO_S;
            break;
        case GL_PIXEL_MAP_I_TO_R:
            sizetoken=GL_PIXEL_MAP_I_TO_R;
            break;
        case GL_PIXEL_MAP_I_TO_G:
            sizetoken=GL_PIXEL_MAP_I_TO_G;
            break;
        case GL_PIXEL_MAP_I_TO_B:
            sizetoken=GL_PIXEL_MAP_I_TO_B;
            break;
        case GL_PIXEL_MAP_I_TO_A:
            sizetoken=GL_PIXEL_MAP_I_TO_A;
            break;
        case GL_PIXEL_MAP_R_TO_R:
            sizetoken=GL_PIXEL_MAP_R_TO_R;
            break;
        case GL_PIXEL_MAP_G_TO_G:
            sizetoken=GL_PIXEL_MAP_G_TO_G;
            break;
        case GL_PIXEL_MAP_B_TO_B:
            sizetoken=GL_PIXEL_MAP_B_TO_B;
            break;
        case GL_PIXEL_MAP_A_TO_A:
            sizetoken=GL_PIXEL_MAP_A_TO_A;
            break;
        default:
            sizetoken=0;
            n=0;
    }

    glGetIntegerv(sizetoken, &n);
    ret=(float*)xalloc(sizeof(float)*n);
    glGetPixelMapfv(stuff->data, ret);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetPixelMapfvReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(float)*n, (char*)ret);
    }
    xfree(ret);
    return(client->noClientException);
}

int GLGetPixelMapuiv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetPixelMapuivReq);
    xGLXGetPixelMapuivReply reply;
    int n;
    unsigned int* ret;
    int sizetoken;

    REQUEST_SIZE_MATCH(xGLXGetPixelMapuivReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    switch (stuff->data)
    {
        case GL_PIXEL_MAP_I_TO_I:
            sizetoken=GL_PIXEL_MAP_I_TO_I;
            break;
        case GL_PIXEL_MAP_S_TO_S:
            sizetoken=GL_PIXEL_MAP_S_TO_S;
            break;
        case GL_PIXEL_MAP_I_TO_R:
            sizetoken=GL_PIXEL_MAP_I_TO_R;
            break;
        case GL_PIXEL_MAP_I_TO_G:
            sizetoken=GL_PIXEL_MAP_I_TO_G;
            break;
        case GL_PIXEL_MAP_I_TO_B:
            sizetoken=GL_PIXEL_MAP_I_TO_B;
            break;
        case GL_PIXEL_MAP_I_TO_A:
            sizetoken=GL_PIXEL_MAP_I_TO_A;
            break;
        case GL_PIXEL_MAP_R_TO_R:
            sizetoken=GL_PIXEL_MAP_R_TO_R;
            break;
        case GL_PIXEL_MAP_G_TO_G:
            sizetoken=GL_PIXEL_MAP_G_TO_G;
            break;
        case GL_PIXEL_MAP_B_TO_B:
            sizetoken=GL_PIXEL_MAP_B_TO_B;
            break;
        case GL_PIXEL_MAP_A_TO_A:
            sizetoken=GL_PIXEL_MAP_A_TO_A;
            break;
        default:
            sizetoken=0;
            n=0;
    }

    glGetIntegerv(sizetoken, &n);
    ret=(unsigned int*)xalloc(sizeof(unsigned int)*n);
    glGetPixelMapuiv(stuff->data, ret);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetPixelMapuivReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(unsigned int)*n, (char*)ret);
    }
    xfree(ret);
    return(client->noClientException);
}

int GLGetPixelMapusv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetPixelMapusvReq);
    xGLXGetPixelMapusvReply reply;
    int n;
    unsigned short* ret;
    int sizetoken;

    REQUEST_SIZE_MATCH(xGLXGetPixelMapusvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    switch (stuff->data)
    {
        case GL_PIXEL_MAP_I_TO_I:
            sizetoken=GL_PIXEL_MAP_I_TO_I;
            break;
        case GL_PIXEL_MAP_S_TO_S:
            sizetoken=GL_PIXEL_MAP_S_TO_S;
            break;
        case GL_PIXEL_MAP_I_TO_R:
            sizetoken=GL_PIXEL_MAP_I_TO_R;
            break;
        case GL_PIXEL_MAP_I_TO_G:
            sizetoken=GL_PIXEL_MAP_I_TO_G;
            break;
        case GL_PIXEL_MAP_I_TO_B:
            sizetoken=GL_PIXEL_MAP_I_TO_B;
            break;
        case GL_PIXEL_MAP_I_TO_A:
            sizetoken=GL_PIXEL_MAP_I_TO_A;
            break;
        case GL_PIXEL_MAP_R_TO_R:
            sizetoken=GL_PIXEL_MAP_R_TO_R;
            break;
        case GL_PIXEL_MAP_G_TO_G:
            sizetoken=GL_PIXEL_MAP_G_TO_G;
            break;
        case GL_PIXEL_MAP_B_TO_B:
            sizetoken=GL_PIXEL_MAP_B_TO_B;
            break;
        case GL_PIXEL_MAP_A_TO_A:
            sizetoken=GL_PIXEL_MAP_A_TO_A;
            break;
        default:
            sizetoken=0;
            n=0;
    }

    glGetIntegerv(sizetoken, &n);
    reply.n=n;
    n=((n+1)/2)*2;
    ret=(unsigned short*)xalloc(sizeof(unsigned short)*n);
    glGetPixelMapusv(stuff->data, ret);
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetPixelMapusvReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(unsigned short)*n, (char*)ret);
    }
    xfree(ret);
    return(client->noClientException);
}

int GLGetMaterialiv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetMaterialivReq);
    xGLXGetMaterialivReply reply;
    int n;
    int ret[4];

    REQUEST_SIZE_MATCH(xGLXGetMaterialivReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->face, n);
        swapl(&stuff->pname, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetMaterialiv(stuff->face, stuff->pname, ret);
    n=GLX_material_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetMaterialivReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(float)*n, (char*)ret);
    }
    return(client->noClientException);
}

int GLGetMapfv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetMapfvReq);
    xGLXGetMapfvReply reply;
    int n;
    float sret[4];
    float* ret;
    int dim;

    REQUEST_SIZE_MATCH(xGLXGetMapfvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->target, n);
        swapl(&stuff->query, n);
    }
    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (!ctx)
    {
        return (GLXBadContextState);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    switch (stuff->target)
    {
        case GL_MAP1_COLOR_4:
        case GL_MAP1_INDEX:
        case GL_MAP1_NORMAL:
        case GL_MAP1_TEXTURE_COORD_1:
        case GL_MAP1_TEXTURE_COORD_2:
        case GL_MAP1_TEXTURE_COORD_3:
        case GL_MAP1_TEXTURE_COORD_4:
        case GL_MAP1_VERTEX_3:
        case GL_MAP1_VERTEX_4:
            dim=1;
            break;
        case GL_MAP2_COLOR_4:
        case GL_MAP2_INDEX:
        case GL_MAP2_NORMAL:
        case GL_MAP2_TEXTURE_COORD_1:
        case GL_MAP2_TEXTURE_COORD_2:
        case GL_MAP2_TEXTURE_COORD_3:
        case GL_MAP2_TEXTURE_COORD_4:
        case GL_MAP2_VERTEX_3:
        case GL_MAP2_VERTEX_4:
            dim=2;
            break;
        default:
            dim=0;
            break;
    }
    switch (stuff->query)
    {
        case GL_COEFF:
            if (dim==1)
            {
                GLint uorder;
                glGetMapiv(stuff->target, GL_ORDER, &uorder);
                n=uorder;
            }
            else
            {
                GLint uorder;
                GLint vorder;
                glGetMapiv(stuff->target, GL_ORDER, &uorder);
                glGetMapiv(stuff->target, GL_ORDER, &vorder);
                n=uorder*vorder;
            }
            break;
        case GL_ORDER:
            n=dim;
            break;
        case GL_DOMAIN:
            n=2*dim;
            break;
        default:
            n=0;
            break;
    }

    if (n<4)
    {
        ret=sret;
    }
    else
    {
        ret=(float*)xalloc(sizeof(float)*n);
    }
    glGetMapfv(stuff->target, stuff->query, ret);

    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetMapfvReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(float)*n, (char*)ret);
    }
    if (ret != sret)
        xfree(ret);
    return(client->noClientException);
}

int GLGetMapiv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetMapivReq);
    xGLXGetMapivReply reply;
    int n;
    int sret[4];
    int* ret;
    int dim;

    REQUEST_SIZE_MATCH(xGLXGetMapivReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->target, n);
        swapl(&stuff->query, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    switch (stuff->target)
    {
        case GL_MAP1_COLOR_4:
        case GL_MAP1_INDEX:
        case GL_MAP1_NORMAL:
        case GL_MAP1_TEXTURE_COORD_1:
        case GL_MAP1_TEXTURE_COORD_2:
        case GL_MAP1_TEXTURE_COORD_3:
        case GL_MAP1_TEXTURE_COORD_4:
        case GL_MAP1_VERTEX_3:
        case GL_MAP1_VERTEX_4:
            dim=1;
            break;
        case GL_MAP2_COLOR_4:
        case GL_MAP2_INDEX:
        case GL_MAP2_NORMAL:
        case GL_MAP2_TEXTURE_COORD_1:
        case GL_MAP2_TEXTURE_COORD_2:
        case GL_MAP2_TEXTURE_COORD_3:
        case GL_MAP2_TEXTURE_COORD_4:
        case GL_MAP2_VERTEX_3:
        case GL_MAP2_VERTEX_4:
            dim=2;
            break;
        default:
            dim=0;
            break;
    }
    switch (stuff->query)
    {
        case GL_COEFF:
            if (dim==1)
            {
                GLint uorder;
                glGetMapiv(stuff->target, GL_ORDER, &uorder);
                n=uorder;
            }
            else
            {
                GLint uorder;
                GLint vorder;
                glGetMapiv(stuff->target, GL_ORDER, &uorder);
                glGetMapiv(stuff->target, GL_ORDER, &vorder);
                n=uorder*vorder;
            }
            break;
        case GL_ORDER:
            n=dim;
            break;
        case GL_DOMAIN:
            n=2*dim;
            break;
        default:
            n=0;
            break;
    }

    if (n<4)
    {
        ret=sret;
    }
    else
    {
        ret=(int*)xalloc(sizeof(int)*n);
    }
    glGetMapiv(stuff->target, stuff->query, ret);

    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetMapivReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    if (ret != sret)
        xfree(ret);
    return(client->noClientException);
}

int GLGetMapdv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetMapdvReq);
    xGLXGetMapdvReply reply;
    int n;
    double sret[4];
    double* ret;
    int dim;

    REQUEST_SIZE_MATCH(xGLXGetMapdvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->target, n);
        swapl(&stuff->query, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    switch (stuff->target)
    {
        case GL_MAP1_COLOR_4:
        case GL_MAP1_INDEX:
        case GL_MAP1_NORMAL:
        case GL_MAP1_TEXTURE_COORD_1:
        case GL_MAP1_TEXTURE_COORD_2:
        case GL_MAP1_TEXTURE_COORD_3:
        case GL_MAP1_TEXTURE_COORD_4:
        case GL_MAP1_VERTEX_3:
        case GL_MAP1_VERTEX_4:
            dim=1;
            break;
        case GL_MAP2_COLOR_4:
        case GL_MAP2_INDEX:
        case GL_MAP2_NORMAL:
        case GL_MAP2_TEXTURE_COORD_1:
        case GL_MAP2_TEXTURE_COORD_2:
        case GL_MAP2_TEXTURE_COORD_3:
        case GL_MAP2_TEXTURE_COORD_4:
        case GL_MAP2_VERTEX_3:
        case GL_MAP2_VERTEX_4:
            dim=2;
            break;
        default:
            dim=0;
            break;
    }
    switch (stuff->query)
    {
        case GL_COEFF:
            if (dim==1)
            {
                GLint uorder;
                glGetMapiv(stuff->target, GL_ORDER, &uorder);
                n=uorder;
            }
            else
            {
                GLint uorder;
                GLint vorder;
                glGetMapiv(stuff->target, GL_ORDER, &uorder);
                glGetMapiv(stuff->target, GL_ORDER, &vorder);
                n=uorder*vorder;
            }
            break;
        case GL_ORDER:
            n=dim;
            break;
        case GL_DOMAIN:
            n=2*dim;
            break;
        default:
            n=0;
            break;
    }

    if (n<4)
    {
        ret=sret;
    }
    else
    {
        ret=(double*)xalloc(sizeof(double)*n);
    }
    glGetMapdv(stuff->target, stuff->query, ret);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=2*n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapd(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetMapdvReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(double)*n, (char*)ret);
    }
    if (ret != sret)
        xfree(ret);
    return(client->noClientException);
}

int GLGetDoublev(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetDoublevReq);
    xGLXGetDoublevReply reply;
    int n;
    double ret[16];

    REQUEST_SIZE_MATCH(xGLXGetDoublevReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->data, n);
    }
    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (!ctx)
    {
        return (GLXBadContextState);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetDoublev(stuff->data, ret);
    n=GLX_get_size(stuff->data);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=2*n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapd(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapd(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetDoublevReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(double)*n, (char*)ret);
    }
    return(client->noClientException);
}

int GLGetTexGenfv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetTexGenfvReq);
    xGLXGetTexGenfvReply reply;
    int n;
    float ret[4];

    REQUEST_SIZE_MATCH(xGLXGetTexGenfvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->coord, n);
        swapl(&stuff->pname, n);
    }
    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (!ctx)
    {
        return (GLXBadContextState);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetTexGenfv(stuff->coord, stuff->pname, ret);
    n=GLX_texgen_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetTexGenfvReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}    

int GLGetTexGeniv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetTexGenivReq);
    xGLXGetTexGenivReply reply;
    int n;
    int ret[4];

    REQUEST_SIZE_MATCH(xGLXGetTexGenivReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->coord, n);
        swapl(&stuff->pname, n);
    }
    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (!ctx)
    {
        return (GLXBadContextState);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetTexGeniv(stuff->coord, stuff->pname, ret);
    n=GLX_texgen_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetTexGenivReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}    

int GLGetTexGendv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetTexGendvReq);
    xGLXGetTexGendvReply reply;
    int n;
    double ret[4];

    REQUEST_SIZE_MATCH(xGLXGetTexGendvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->coord, n);
        swapl(&stuff->pname, n);
    }
    ctx=LookupIDByType(stuff->context_tag, glContexts);
    if (!ctx)
    {
        return (GLXBadContextState);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetTexGendv(stuff->coord, stuff->pname, ret);
    n=GLX_texgen_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=2*n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapd(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapd(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetTexGenivReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}    

int GLGetTexParameterfv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetTexParameterfvReq);
    xGLXGetTexParameterfvReply reply;
    int n;
    float ret[4];

    REQUEST_SIZE_MATCH(xGLXGetTexParameterfvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->target, n);
        swapl(&stuff->pname, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetTexParameterfv(stuff->target, stuff->pname, ret);
    n=GLX_texparameter_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetTexParameterfvReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}    

int GLGetTexParameteriv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetTexParameterivReq);
    xGLXGetTexParameterivReply reply;
    int n;
    int ret[4];

    REQUEST_SIZE_MATCH(xGLXGetTexParameterivReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->target, n);
        swapl(&stuff->pname, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetTexParameteriv(stuff->target, stuff->pname, ret);
    n=GLX_texparameter_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetTexParameterivReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}    

int GLGetTexLevelParameterfv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetTexLevelParameterfvReq);
    xGLXGetTexLevelParameterfvReply reply;
    int n;
    float ret[4];

    REQUEST_SIZE_MATCH(xGLXGetTexLevelParameterfvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->target, n);
        swapl(&stuff->level, n);
        swapl(&stuff->pname, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetTexLevelParameterfv(stuff->target, stuff->level, stuff->pname, ret);
    n=GLX_texlevelparameter_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetTexLevelParameterfvReply), 
                  (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}    

int GLGetTexLevelParameteriv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetTexLevelParameterivReq);
    xGLXGetTexLevelParameterivReply reply;
    int n;
    int ret[4];

    REQUEST_SIZE_MATCH(xGLXGetTexLevelParameterivReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->target, n);
        swapl(&stuff->level, n);
        swapl(&stuff->pname, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetTexLevelParameteriv(stuff->target, stuff->level, stuff->pname, ret);
    n=GLX_texlevelparameter_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetTexLevelParameterivReply), 
                  (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}    

int GLGetTexEnvfv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetTexEnvfvReq);
    xGLXGetTexEnvfvReply reply;
    int n;
    float ret[4];

    REQUEST_SIZE_MATCH(xGLXGetTexEnvfvReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->target, n);
        swapl(&stuff->pname, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetTexEnvfv(stuff->target, stuff->pname, ret);
    n=GLX_tex_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
        if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetTexEnvfvReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}    

int GLGetTexEnviv(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXGetTexEnvivReq);
    xGLXGetTexEnvivReply reply;
    int n;
    int ret[4];

    REQUEST_SIZE_MATCH(xGLXGetTexEnvivReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->target, n);
        swapl(&stuff->pname, n);
    }
    __GLX_MAKE_CURRENT;

    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    glGetTexEnviv(stuff->target, stuff->pname, ret);
    n=GLX_tex_size(stuff->pname);
    reply.n=n;
    if (n==1)
    {
        reply.length=0;
        reply.singledata=ret[0];
    }
    else
    {
        reply.length=n;
    }

    if (client->swapped)
    {
        register int nn;
        register int i;
        swaps(&reply.sequenceNumber, nn);
        swapl(&reply.length, nn);
        swapl(&reply.singledata, nn);
        swapl(&reply.n, nn);
         if (n>1)
        {
            for (i=0;i<n;i++)
            {
                swapl(&ret[i], nn);
            }
        }
    }
    WriteToClient(client, sizeof(xGLXGetTexEnvivReply), (char*)&reply);
    if (n!=1)
    {
        WriteToClient(client, sizeof(int)*n, (char*)ret);
    }
    return(client->noClientException);
}    

int GLPixelStorei(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXPixelStoreiReq);

    REQUEST_SIZE_MATCH(xGLXPixelStoreiReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->pname, n);
        swapl(&stuff->param, n);
    }
    __GLX_MAKE_CURRENT;

    glPixelStorei( stuff->pname, stuff->param );

#if 0
    if ( ctx->gl_ctx->ErrorValue != GL_NO_ERROR )
    {
        return (BadRequest);
    }
    else
    {
        return (Success);
    }
#else
    return (Success);
#endif
}

int GLPixelStoref(ClientPtr client)
{
    XSMesaContext ctx;
    REQUEST(xGLXPixelStorefReq);

    REQUEST_SIZE_MATCH(xGLXPixelStorefReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->context_tag, n);
        swapl(&stuff->pname, n);
        swapl(&stuff->param, n);
    }
    __GLX_MAKE_CURRENT;

    glPixelStoref( stuff->pname, stuff->param );

    if ( ctx->gl_ctx->ErrorValue != GL_NO_ERROR )
    {
        return (BadRequest);
    }
    else
    {
        return (Success);
    }
}

int GLQueryServerString(ClientPtr client)
{
    int length;
    char buf[100];
    REQUEST(xGLXQueryServerStringReq);
    xGLXQueryServerStringReply reply;

    REQUEST_SIZE_MATCH(xGLXQueryServerStringReq);
    if (client->swapped)
    {
        register int n;
        swapl(&stuff->screen, n);
        swapl(&stuff->name, n);
    }
    reply.type=X_Reply;
    reply.sequenceNumber = client->sequence;
    switch (stuff->name)
    {
        case GLX_VENDOR:
            sprintf(buf, GLX_VENDOR_STRING);
            break;
        case GLX_VERSION:
            sprintf(buf, "%s GLX %s", GLX_PROTOCOL_VERSION, GLX_PACKAGE_VERSION );
            break;
        case GLX_EXTENSIONS:
            sprintf(buf, GLX_EXTENSIONS_STRING);
            break;
        default:
            client->errorValue=stuff->name;
            return (BadValue);
            break;
    }
    reply.n=strlen(buf)+1;
    reply.length=(reply.n+3)/4;
    length=reply.length;
    if (client->swapped)
    {
        register int n;
        swaps(&reply.sequenceNumber, n);
        swapl(&reply.length, n);
        swapl(&reply.n, n);
    }
    WriteToClient(client, sizeof(xGLXQueryServerStringReply), (char*)&reply);
    if (reply.n)
    {
        WriteToClient(client, 4*length, buf);
    }
    return(client->noClientException);
}
