/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "dumpToXyz.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <visu_tools.h>

/**
 * SECTION:dumpToXyz
 * @short_description: add an export capability of current positions.
 *
 * This provides a write routine to export V_Sim current
 * coordinates. It has several options to output or not hiddden nodes
 * or replicated nodes.
 */

static gboolean writeDataInXyz(FileFormat *format, const char* filename,
				 int width, int height, VisuData *dataObj,
				 guchar* imageData, GError **error,
				 voidDataFunc functionWait, gpointer data);
static gpointer waitData;
static voidDataFunc waitFunc;

DumpType* dumpToXyz_init()
{
  DumpType *xyz;
  char *typeXYZ[] = {"*.xyz", (char*)0};
#define descrXYZ _("Xyz file (current positions)")
  FileFormat* fmt;

  xyz = g_malloc(sizeof(DumpType));
  fmt = fileFormatNew(descrXYZ, typeXYZ);
  if (!fmt)
    {
      g_error("Can't initialize the XYZ dump module, aborting.\n");
    }

  xyz->bitmap = FALSE;
  xyz->fileType = fmt;
  xyz->writeFunc = writeDataInXyz;
  
  fileFormatAdd_propertyBoolean(fmt, "expand_box",
				_("Expand the bounding box"), TRUE);
  fileFormatAdd_propertyBoolean(fmt, "type_alignment",
				_("Export nodes sorted by elements"), FALSE);

  waitData = (gpointer)0;
  waitFunc = (voidDataFunc)0;

  return xyz;
}

static gboolean writeDataInXyz(FileFormat *format, const char* filename,
				 int width _U_, int height _U_, VisuData *dataObj,
				 guchar* imageData _U_, GError **error,
				 voidDataFunc functionWait, gpointer data)
{
  gchar *nom, *prevFile;
  OpenGLView *view;
  GList *prop;
  gchar* name, firstLine[256];
  gboolean expand, eleSort;
  gint nb;
  float xyz[3], ext[3], vertices[8][3];
  VisuDataIter iter;
  GString *output;
  OpenGLBoxBoundaries bc;
  gchar *bcStr[5] = {"periodic", "surface", "surface", "surface", "FreeBC"};

  g_return_val_if_fail(error && !*error, FALSE);

  expand  = FALSE;
  eleSort = FALSE;
  prop = fileFormatGet_propertiesList(format);
  while (prop)
    {
      name = fileFormatGet_propertyName((FileFormatProperty*)(prop->data));
      if (strcmp(name, "expand_box") == 0)
	expand  = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      else if (strcmp(name, "type_alignment") == 0)
	eleSort  = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      prop = g_list_next(prop);
    }

  waitData = data;
  waitFunc = functionWait;

  DBG_fprintf(stderr, "Dump XYZ: begin export of current positions...\n");

  output = g_string_new("");

  visuDataIter_new(dataObj, &iter);
  /* The number of elements will be prepended later... */
  /* The commentary line. */
  bc = visuDataGet_boundaryConditions(dataObj);
  if (bc != BOX_FREE)
    {
      visuDataGet_boxVertices(dataObj, vertices, expand);
      if (vertices[1][1] == 0.f && vertices[1][2] == 0.f &&
	  vertices[3][0] == 0.f && vertices[3][2] == 0.f &&
	  vertices[4][0] == 0.f && vertices[4][1] == 0.f)
	g_string_append_printf(output, "%s %g %g %g ", bcStr[bc],
			       vertices[1][0], vertices[3][1], vertices[4][2]);
      else
	g_warning("Can't export box, not orthogonal.");
    }
  nom = visuDataGet_file(dataObj, 0, (FileFormat**)0);
  if (nom)
    {
      prevFile = g_path_get_basename(nom);
      g_string_append_printf(output, "# V_Sim export to xyz from '%s'", prevFile);
      g_free(prevFile);
    }
  else
    {
      g_warning("Can't get the name of the file to export.");
      g_string_append_printf(output, "# V_Sim export to xyz");
    }
  g_string_append(output, "\n");

  view = visuDataGet_openGLView(dataObj);
  if (expand)
    visuDataGet_extension(dataObj, ext);
  else
    {
      ext[0] = 0.;
      ext[1] = 0.;
      ext[2] = 0.;
    }

  nb = 0;
  for (visuDataIter_startNumber(dataObj, &iter); iter.node && iter.element;
       (eleSort)?visuDataIter_next(dataObj, &iter):
       visuDataIter_nextNodeNumber(dataObj, &iter))
    {
      visuDataGet_nodePosition(dataObj, iter.node, xyz);
      if (iter.element->rendered && iter.node->rendered)
	{
	  xyz[0] += ext[0] * view->box->p2[0] + ext[1] * view->box->p4[0] +
	    ext[2] * view->box->p5[0];
	  xyz[1] += ext[1] * view->box->p4[1] + ext[2] * view->box->p5[1];
	  xyz[2] += ext[2] * view->box->p5[2];
	
	  nb += 1;
	  g_string_append_printf(output, "%s %15g %15g %15g\n", iter.element->name,
				 xyz[0], xyz[1], xyz[2]);
	}
    }

  switch (visuDataGet_unit(dataObj))
    {
    case unit_angstroem:
      sprintf(firstLine, " %d angstroem\n", nb);
      break;
    case unit_bohr:
      sprintf(firstLine, " %d atomic\n", nb);
      break;
    default:
      sprintf(firstLine, " %d\n", nb);
      break;
    }
  g_string_prepend(output, firstLine);

  g_file_set_contents(filename, output->str, -1, error);
  g_string_free(output, TRUE);

  return TRUE;
}
