/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresses ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "scalarFields.h"

#include <string.h>
#include <math.h>
#include <GL/gl.h>

#include <visu_tools.h>
#include <coreTools/toolFileFormat.h>
#include <coreTools/toolMatrix.h>
#include "surfaces_points.h"
#include "isoline.h"

#ifdef HAVE_CAIRO
#include <cairo.h>
#include <cairo-svg.h>
#include <cairo-pdf.h>
#endif

/**
 * SECTION:scalarFields
 * @short_description:Gives capabilities to load a scalar field.
 *
 * <para>A scalar field is represented by the given of datas on a
 * regular grid meshing the bounding box. Scalar field can be read
 * from several kind of files by adding load methods using
 * scalarFieldAdd_loadMethod(). The basic implementation gives access
 * to ASCII encoded files following a simple format.</para>
 *
 * <para>In coordination with #Plane and #Shade, scalar field can be
 * represented as coloured map calling scalarFieldDraw_map(). The
 * current implementation of interpolation is very limited since basic
 * linear approximation is used.</para>
 *
 * <para>If a structure file also contains a scalar field, when
 * loaded, it should add a #VisuData property called
 * #SCALAR_FIELD_DEFINED_IN_STRUCT_FILE using
 * visuDataAdd_property(). Then V_Sim will be able to handle the
 * structure file as a density file also.</para>
 */

#define MESH_FLAG             "meshType"
#define MESH_FLAG_UNIFORM     "uniform"
#define MESH_FLAG_NON_UNIFORM "nonuniform"

/**
 * ScalarField_struct:
 * @filename: the path to the file from which the scalar field
 *            has been read ;
 * @commentary: a commentary read from the file (must be in UTF-8) ;
 * @box: description of the associated bounding box ;
 * @nElements: number of points in each direction ;
 * @data: the values ;
 * @min: the minimum value ;
 * @max: the maximum value ;
 * @options: a GList of #Option values.
 *
 * The structure used to store a scalar field.
 */
struct ScalarField_struct
{
  /* The name of the file, where the data were read from. */
  gchar *filename;

  /* An associated commentary. */
  gchar *commentary;

  /* Description of the box. */
  float box[6];
  float fromXYZtoReducedCoord[3][3];

  /* Number of elements in each directions [x, y, z]. */
  int nElements[3];

  /* Mesh. */
  double *meshx;
  double *meshy;
  double *meshz;

  /* Datas. */
  double ***data;

  /* Minimum and maximum values. */
  double min, max, secondMin;

  /* Set to TRUE if the data can replicate by periodicity. */
  gboolean periodic;

  /* mesh type: uniform or non uniform */
  ScalarField_meshflag mesh_type;

  /* A GList to store some options (key, values) associated
     to the data. */
  GList *options;
};

enum
  {
    MAP_X,
    MAP_Y,
    MAP_Z,
    MAP_U,
    MAP_V,
    MAP_VISIBILITY,
    MAP_VALUE,
    MAP_R,
    MAP_G,
    MAP_B,
    MAP_A
  };

/* Local variables. */
static GList *loadMethods;

/* Local methods. */
static gboolean scalarFieldLoad_fromAscii(const gchar *filename, GList **fieldList, GError **error,
					  OptionTable *table);
static gint compareLoadPriority(gconstpointer a, gconstpointer b);


ScalarField* scalarFieldNew(const gchar *filename)
{
  ScalarField *field;

  g_return_val_if_fail(filename && filename[0], (ScalarField*)0);

  field               = g_malloc(sizeof(ScalarField));
  field->nElements[0] = 0;
  field->nElements[1] = 0;
  field->nElements[2] = 0;
  field->filename     = g_strdup(filename);
  field->commentary   = (gchar*)0;
  field->meshx        = (double*)0;
  field->meshy        = (double*)0;
  field->meshz        = (double*)0;
  field->data         = (double***)0;
  field->min          = G_MAXFLOAT;
  field->max          = -G_MAXFLOAT;
  field->secondMin    = G_MAXFLOAT;
  field->periodic     = FALSE;
  field->mesh_type    = uniform;
  field->options      = (GList*)0;

  return field;
}
void scalarFieldFree(ScalarField *field)
{
  int i, j;
  GList *tmplst;

  g_return_if_fail(field);

  if (field->filename)
    g_free(field->filename);

  if (field->commentary)
    g_free(field->commentary);

  if (field->meshx)
    g_free(field->meshx);

  if (field->meshy)
    g_free(field->meshy);

  if (field->meshz)
    g_free(field->meshz);

  if (field->data)
    {
      for (i = 0; i < field->nElements[0]; i++)
	{
	  for (j = 0; j < field->nElements[1]; j++)
	    g_free(field->data[i][j]);
	  g_free(field->data[i]);
	}
      g_free(field->data);
    }

  if (field->options)
    {
      tmplst = field->options;
      while(tmplst)
        {
          toolOptionsFree_option(tmplst->data);
          tmplst = g_list_next(tmplst);
        }
      g_list_free(field->options);
    }
}

gboolean scalarFieldLoad_fromFile(const gchar *filename, GList **fieldList,
				  GError **error, OptionTable *table)
{
  gboolean validFormat;
  GList *tmpLst;

  g_return_val_if_fail(filename, FALSE);
  g_return_val_if_fail(*fieldList == (GList*)0, FALSE);
  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);

  /* Try all supported format. */
  validFormat = FALSE;
  tmpLst = loadMethods;
  while(tmpLst && !validFormat)
    {
      DBG_fprintf(stderr, "Scalar Fields : try to open the file as a '%s'.\n",
		  ((ScalarFieldLoadStruct*)tmpLst->data)->name);
      validFormat = ((ScalarFieldLoadStruct*)tmpLst->data)->load(filename, fieldList, error, table);
      if (!validFormat && *error)
        {
          g_error_free(*error);
          *error = (GError*)0;
        }
      tmpLst = g_list_next(tmpLst);
    }
  
  if (!validFormat)
    g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_UNKNOWN_FORMAT, 
		_("unknown density/potential format.\n")); 
  return validFormat;
}


static gboolean scalarFieldLoad_fromAscii(const gchar *filename, GList **fieldList,
					  GError **error, OptionTable *table _U_)
{
  FILE *in;
  char rep[MAX_LINE_LENGTH], flag[MAX_LINE_LENGTH];
  char format[MAX_LINE_LENGTH], period[MAX_LINE_LENGTH];
  char *feed;
  gchar *comment;
  int res, i, j, k;
  int size[3];
  double box[6];
  ScalarField *field;
  gboolean periodic;
  ScalarField_meshflag meshtype;

  g_return_val_if_fail(filename, FALSE);
  g_return_val_if_fail(*fieldList == (GList*)0, FALSE);
  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);

  DBG_fprintf(stderr, "ScalarField : try to read '%s' as a ASCII scalar"
	      " field data file.\n", filename);

  in = fopen(filename, "r");
  if (!in)
    {
      g_set_error(error, G_FILE_ERROR, G_FILE_ERROR_ACCES,
		  _("impossible to open the file.\n"));
      return FALSE;
    }

  /* 1st line (comment) */
  (void)fgets(rep, MAX_LINE_LENGTH, in);
  rep[strlen(rep)-1] = 0; /* skipping \n */
  comment = g_locale_to_utf8(rep, -1, NULL, NULL, NULL);
  if (!comment)
    comment = g_strdup("");

  (void)fgets(rep, MAX_LINE_LENGTH, in);
  res = sscanf(rep, "%d %d %d", size, size + 1, size + 2);
  if (res != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  (void)fgets(rep, MAX_LINE_LENGTH, in);
  res = sscanf(rep, "%lf %lf %lf", box, box + 1, box + 2);
  if (res != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }
  (void)fgets(rep, MAX_LINE_LENGTH, in); 
  res = sscanf(rep, "%lf %lf %lf", box + 3, box + 4, box + 5);
  if (res != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  (void)fgets(rep, MAX_LINE_LENGTH, in);
  res = sscanf(rep, "%s %s", format, period);
  if (res < 1 || (strcmp(format, "xyz") && strcmp(format, "zyx")))
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }
  periodic = !strcmp(period, "periodic");

  /* OK, from now on, the format is supposed to be ASCII. */
  field = scalarFieldNew(filename);
  if (!field)
    {
      g_warning("impossible to create a ScalarField object.");
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  /* by default the meshtype is set to uniform to keep working previous version.*/
  meshtype = uniform;
  feed = fgets(rep, MAX_LINE_LENGTH, in);
  while (rep[0] == '#' && feed)
    {
      if (strncmp(rep + 2, MESH_FLAG, strlen(MESH_FLAG)) == 0)
	{
	  DBG_fprintf(stderr, "ScalarField: found flag '%s'.\n", MESH_FLAG);
	  res = sscanf(rep + 2 + strlen(MESH_FLAG) + 1, "%s", flag);
	  if (res == 1 && strcmp(flag, MESH_FLAG_UNIFORM) == 0)
	    meshtype = uniform;
	  else if (res == 1 && strcmp(flag, MESH_FLAG_NON_UNIFORM) == 0)
	    meshtype = nonuniform;
	  else if (res == 1)
	    {
              g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("wrong '%s' value for flag '%s'.\n"), flag, MESH_FLAG);
              fclose(in);
              return TRUE;
	    }
	}
      feed = fgets(rep, MAX_LINE_LENGTH, in);
    }

  scalarFieldSet_periodic(field, periodic);
  scalarFieldSet_commentary(field, comment);
  scalarFieldSet_meshtype(field, meshtype);
  scalarFieldSet_gridSize(field, size);
  scalarFieldSet_box(field, box);
  *fieldList = g_list_append(*fieldList, (gpointer)field);

  if (meshtype == nonuniform)
    {
      DBG_fprintf(stderr, "ScalarField : Start to read meshx.\n");

      for ( i = 0; i < size[0]; i++ )
        {
	  if (!feed)
	    {
	      g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("not enough meshx values.\n"));
	      fclose(in);
	      return TRUE;
	    }
          res = sscanf(rep, "%lf", &field->meshx[i]);
	  do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
          if (res != 1)
            {
              g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("impossible to read meshx values.\n"));
              fclose(in);
              return TRUE;
            }
        }

      DBG_fprintf(stderr, "ScalarField : Start to read meshy.\n");

      for ( j = 0; j < size[1]; j++ )
        {
	  if (!feed)
	    {
	      g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("not enough meshy values.\n"));
	      fclose(in);
	      return TRUE;
	    }
          res = sscanf(rep, "%lf", &field->meshy[j]);
	  do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
          if (res != 1)
            {
              g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("impossible to read meshy values.\n"));
              fclose(in);
              return TRUE;
            }
        }

      DBG_fprintf(stderr, "ScalarField : Start to read meshz.\n");

      for ( k = 0; k < size[2]; k++ )
        {
	  if (!feed)
	    {
	      g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("not enough meshz values.\n"));
	      fclose(in);
	      return TRUE;
	    }
          res = sscanf(rep, "%lf", &field->meshz[k]);
	  do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
          if (res != 1)
            {
              g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("impossible to read meshz values.\n"));
              fclose(in);
              return TRUE;
            }
        }
    }

  DBG_fprintf(stderr, "ScalarField : Start to read data.\n");

  field->min = G_MAXFLOAT;
  field->max = G_MINFLOAT;
  if(!strcmp(format, "xyz"))
    {
      for ( k = 0; k < size[2]; k++ ) 
	for ( j = 0; j < size[1]; j++ ) 
	  for ( i = 0; i < size[0]; i++ ) 
	    {
	      if (!feed)
		{
		  g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			      _("not enough density values.\n"));
		  fclose(in);
		  return TRUE;
		}

	      res = sscanf(rep, "%lf", &field->data[i][j][k]);

	      do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
	      if (res != 1)
		{
		  g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			      _("impossible to read density values.\n"));
		  fclose(in);
		  return TRUE;
		}
	      field->min = MIN(field->data[i][j][k], field->min);
	      field->max = MAX(field->data[i][j][k], field->max);
	    }
    }
  else
    {
      for ( i = 0; i < size[0]; i++ ) 
	for ( j = 0; j < size[1]; j++ ) 
	  for ( k = 0; k < size[2]; k++ ) 
	    {
	      if (!feed)
		{
		  g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			      _("not enough density values.\n"));
		  fclose(in);
		  return TRUE;
		}
	      res = sscanf(rep, "%lf", &field->data[i][j][k]);
	      do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
	      if (res != 1)
		{
		  g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			      _("impossible to read density values.\n"));
		  fclose(in);
		  return TRUE;
		}
	      field->min = MIN(field->data[i][j][k], field->min);
	      field->max = MAX(field->data[i][j][k], field->max);
	    }
    }
  
  fclose(in);
  return TRUE;
}

gchar* scalarFieldGet_commentary(ScalarField *field)
{
  g_return_val_if_fail(field, (gchar*)0);
  return field->commentary;
}
void scalarFieldSet_commentary(ScalarField *field, gchar* comment)
{
  g_return_if_fail(field);
  
  field->commentary = g_strdup(comment);
}

gboolean scalarFieldGet_periodic(ScalarField *field)
{
  g_return_val_if_fail(field, FALSE);
  return field->periodic;
}
void scalarFieldSet_periodic(ScalarField *field, gboolean periodic)
{
  g_return_if_fail(field);
  
  field->periodic = periodic;
}

ScalarField_meshflag scalarFieldGet_meshtype(ScalarField *field)
{
  g_return_val_if_fail(field, uniform);
  return field->mesh_type;
}

void scalarFieldSet_meshtype(ScalarField *field, ScalarField_meshflag meshtype)
{
  g_return_if_fail(field);

  field->mesh_type = meshtype;
}

gchar* scalarFieldGet_filename(ScalarField *field)
{
  g_return_val_if_fail(field, (gchar*)0);
  return field->filename;
}

void scalarFieldSet_gridSize(ScalarField *field, int grid[3])
{
  int i, j;
  
  g_return_if_fail(field);
  
  if (field->nElements[0] == grid[0] &&
      field->nElements[1] == grid[1] &&
      field->nElements[2] == grid[2])
    return;
  
  DBG_fprintf(stderr, "ScalarField: changing size from (%d ; %d ; %d)"
                      " to (%d ; %d ; %d).\n", field->nElements[0], field->nElements[1],
                      field->nElements[2], grid[0], grid[1], grid[2]);

  if (field->mesh_type == nonuniform)
    {
      DBG_fprintf(stderr, " |free the previous mesh allocation.\n");
      /* If mesh was already allocated, we free it. */
      if (field->meshx)
        g_free(field->meshx);

      if (field->meshy)
        g_free(field->meshy);

      if (field->meshz)
        g_free(field->meshz);
    }

  /* If data was already allocated, we free it. */
  if (field->data)
    {
      DBG_fprintf(stderr, " |free the previous data allocation.\n");
      for (i = 0; i < field->nElements[0]; i++)
	      {
          for (j = 0; j < field->nElements[1]; j++)
	          g_free(field->data[i][j]);
      	  g_free(field->data[i]);
      	}
      g_free(field->data);
    }

  /* We change the size and reallocate mesh and data. */
  field->nElements[0] = grid[0];
  field->nElements[1] = grid[1];
  field->nElements[2] = grid[2];

  if (field->mesh_type == nonuniform)
    {
      DBG_fprintf(stderr, "ScalarField: allocating meshx array.\n");
      field->meshx = g_malloc(sizeof(double) * grid[0]);

      DBG_fprintf(stderr, "ScalarField: allocating meshy array.\n");
      field->meshy = g_malloc(sizeof(double) * grid[1]);

      DBG_fprintf(stderr, "ScalarField: allocating meshz array.\n");
      field->meshz = g_malloc(sizeof(double) * grid[2]);
    }

  DBG_fprintf(stderr, "ScalarField: allocating data array.\n");
  field->data = g_malloc(sizeof(double **) * grid[0]);
  for(i = 0; i < grid[0]; i++)
    {
      field->data[i] = g_malloc(sizeof(double *) * grid[1]);
      for(j = 0; j < grid[1]; j++)
        field->data[i][j] = g_malloc(sizeof(double) * grid[2]);
    }
  DBG_fprintf(stderr, " | allocation done.\n");
}

void scalarFieldGet_minMax(ScalarField *field, double minmax[2])
{
  g_return_if_fail(field);
  
  minmax[0] = field->min;
  minmax[1] = field->max;
}
double scalarFieldGet_secondaryMin(ScalarField *field)
{
  int i, j, k;

  g_return_val_if_fail(field, 0.);
  
  /* In log scale, we need the second minimum value. */
  if (field->secondMin == G_MAXFLOAT)
    {
      for (i = 0; i < field->nElements[0]; i++)
	for (j = 0; j < field->nElements[1]; j++)
	  for (k = 0; k < field->nElements[2]; k++)
	    field->secondMin = (field->data[i][j][k] == field->min)?
	      field->secondMin:MIN(field->data[i][j][k], field->secondMin);
      DBG_fprintf(stderr, "Scalar Fields: second minimum is %g (%g < %g).\n",
		  field->secondMin, field->min, field->max);
    }
  return field->secondMin;
}
void scalarFieldSet_data(ScalarField *field, double *data, gboolean xyzOrder)
{
  int i, j, k, ii;
  
  g_return_if_fail(field && data);
  
  field->min = G_MAXFLOAT;
  field->max = -G_MAXFLOAT;
  ii = 0;
  if (xyzOrder)
    for (k = 0 ; k < field->nElements[2] ; k++)
      for (j = 0 ; j < field->nElements[1] ; j++)
	for (i = 0 ; i < field->nElements[0] ; i++)
	  {
	    field->data[i][j][k] = data[ii];
	    field->min = MIN(data[ii], field->min);
	    field->max = MAX(data[ii], field->max);
	    ii += 1;
	  }
  else
    for (i = 0 ; i < field->nElements[0] ; i++)
      for (j = 0 ; j < field->nElements[1] ; j++)
	for (k = 0 ; k < field->nElements[2] ; k++)
	  {
	    field->data[i][j][k] = data[ii];
	    field->min = MIN(data[ii], field->min);
	    field->max = MAX(data[ii], field->max);
	    ii += 1;
	  }
}

static void planeTo3D(float xyz[3], float uv[2], float basis[2][3], float origin[3])
{
  xyz[0] = origin[0] + uv[0] * basis[0][0] + uv[1] * basis[1][0];
  xyz[1] = origin[1] + uv[0] * basis[0][1] + uv[1] * basis[1][1];
  xyz[2] = origin[2] + uv[0] * basis[0][2] + uv[1] * basis[1][2];
}
static gboolean setValue(ScalarField *field, Shade *shade, matrixGet_scaleVal get_val,
			 double minmax[2], double logval, float drawnMinMax[2],
			 float extension[3], float xyz[3], float *val, float rgba[4])
{
  gboolean inside;
  double value;

  inside = scalarFieldGet_value(field, xyz, &value, extension);
  if (inside)
    {
      *val = (float)get_val(value, minmax, logval);
      shadeGet_valueTransformedInRGB(shade, rgba, *val);
      drawnMinMax[0] = MIN(drawnMinMax[0], (float)value);
      drawnMinMax[1] = MAX(drawnMinMax[1], (float)value);
    }
  return inside;
}

void drawMapAndLines(SurfacesPoints *points, Plane *plane, guint nIsoLines,
		     Shade *shade, float *rgb, float valMinMax[2])
{
  int i;
  GList *inter;
  float v, rgba[4];
  Line *line;

  glDisable(GL_CULL_FACE);
  glDisable(GL_LIGHTING);
/*   glShadeModel(GL_FLAT); */
/*   glPolygonMode(GL_FRONT_AND_BACK, GL_LINE); */
  DBG_fprintf(stderr, "ScalarFields: plot triangles (%d).\n", points->num_polys);
  for (i = 0; i < points->num_polys; i++)
    {
      if (points->poly_surf_index[i] > 0)
	{
	  glBegin(GL_TRIANGLE_STRIP);
	  glColor3fv(points->poly_points_data[points->poly_vertices[i][0]] + MAP_R);
	  glVertex3fv(points->poly_points_data[points->poly_vertices[i][0]]);
	  glColor3fv(points->poly_points_data[points->poly_vertices[i][1]] + MAP_R);
	  glVertex3fv(points->poly_points_data[points->poly_vertices[i][1]]);
	  glColor3fv(points->poly_points_data[points->poly_vertices[i][2]] + MAP_R);
	  glVertex3fv(points->poly_points_data[points->poly_vertices[i][2]]);
	  glEnd();
	}
    }
/*   for (i = 0; i < points.num_points; i++) */
/*     { */
/*       sprintf(str, "%6.4f", points.poly_points_data[i][MAP_VALUE]); */
/*       glRasterPos3fv(points.poly_points_data[i]); openGLText_drawChars(str); */
/*     } */
  DBG_fprintf(stderr, "ScalarFields: plot lines.\n");
  for (i = 1; i <= (int)nIsoLines; i++)
    {
      v = valMinMax[0] + (valMinMax[1] - valMinMax[0]) *
	(float)i / (float)(nIsoLines + 1);
      if (isolineBuild(&line, points, MAP_VALUE, MAP_VISIBILITY, v))
	{
	  if (rgb)
	    isolineDraw(line, rgb);
	  else
	    {
	      shadeGet_valueTransformedInRGB(shade, rgba, v);
	      rgba[0] = 1.f - rgba[0];
	      rgba[1] = 1.f - rgba[1];
	      rgba[2] = 1.f - rgba[2];
	      rgba[3] = 1.f;
	      isolineDraw(line, rgba);
	    }
	  isolineFree(line);
	}
    }

  DBG_fprintf(stderr, "ScalarFields: plot box.\n");
  glLineWidth(1.f);
  glColor3f(0.f, 0.f, 0.f);
  glBegin(GL_LINE_LOOP);
  for (inter = planeGet_intersection(plane); inter; inter = g_list_next(inter))
    glVertex3fv((float*)inter->data);
  glEnd();

  glEnable(GL_CULL_FACE);
  glEnable(GL_LIGHTING);
}
static void computeMap(SurfacesPoints *points, float viewport[4],
		       float valMinMax[2], float drawnMinMax[2],
		       ScalarField *field, Plane *plane, Shade *shade,
		       OpenGLView *view, float precision,
		       matrix_scaleflag scale, float *inputMinMax)
{
  matrixGet_scaleVal get_val;
  double logval;
  float xyz[2][3], center[3], uMinMax[2], vMinMax[2];
  float dist, coeff, *red, **data, a[2], *b, I[2], lambda;
  double minmax[2];
  GList *inter, *interRed;
  int alpha, i, j, k, n, uAlpha[2], vAlpha[2];
  int nU, nV, nPolys, nPoints, visible, hidden;
  int *linkedPoints;
  int iPolys[4];

  g_return_if_fail(field && shade && plane);
  g_return_if_fail(view && precision > 0);

  points->num_polys = 0;

  inter = planeGet_intersection(plane);
  /* In case of no intersections, we build an empty list. */
  if (!inter)
    return;
  
  /* We compute the plane basis. */
  planeGet_basis(plane, xyz, center);

  /* We look for the min and max position in the plane basis for the
     different intersections. */
  dist = 0.;
  uMinMax[0] = 0.;
  uMinMax[1] = 0.;
  vMinMax[0] = 0.;
  vMinMax[1] = 0.;
  for (interRed = (GList*)0; inter; inter = g_list_next(inter))
    {
      red = g_malloc(sizeof(float) * 2);
      red[0] = xyz[0][0] * (((float*)inter->data)[0] - center[0]) +
	xyz[0][1] * (((float*)inter->data)[1] - center[1]) +
	xyz[0][2] * (((float*)inter->data)[2] - center[2]);
      uMinMax[0] = MIN(red[0], uMinMax[0]);
      uMinMax[1] = MAX(red[0], uMinMax[1]);
      red[1] = xyz[1][0] * (((float*)inter->data)[0] - center[0]) +
	xyz[1][1] * (((float*)inter->data)[1] - center[1]) +
	xyz[1][2] * (((float*)inter->data)[2] - center[2]);
      vMinMax[0] = MIN(red[1], vMinMax[0]);
      vMinMax[1] = MAX(red[1], vMinMax[1]);
      interRed = g_list_append(interRed, red);
    }
  /* So, in plane coordinates, X range in uMinMax and Y in vMinMax. */

  /* We put by default (precision and gross = 1) 30 triangles along
     the longest line. */
  alpha = (int)(60. * (OpenGLViewGet_precision() / 2. + .5) *
                precision * (view->camera->gross / 2. + .5) / 100.);
  dist = sqrt(view->box->p7[0] * view->box->p7[0] +
	      view->box->p7[1] * view->box->p7[1] +
	      view->box->p7[2] * view->box->p7[2]);
/*   dist = MAX(uMinMax[1] - uMinMax[0], vMinMax[1] - vMinMax[0]); */
  uAlpha[0] = (int)(uMinMax[0] / dist * (float)alpha) - 2;
  uAlpha[1] = (int)(uMinMax[1] / dist * (float)alpha) + 4;
  vAlpha[0] = (int)(vMinMax[0] / dist * (float)alpha * 1.154700538379251448) - 2;
  vAlpha[1] = (int)(vMinMax[1] / dist * (float)alpha * 1.154700538379251448) + 3;
  if (!inputMinMax)
    scalarFieldGet_minMax(field, minmax);
  else
    {
      minmax[0] = inputMinMax[0];
      minmax[1] = inputMinMax[1];
    }
  drawnMinMax[0] = minmax[1];
  drawnMinMax[1] = minmax[0];

  /* In log scale, we need the second minimum value. */
  switch (scale)
    {
    case linear:
      logval = 0.;
      get_val = matrixGet_linear;
      break;
    case logarithm:
      logval = log10((scalarFieldGet_secondaryMin(field) - minmax[0]) /
		     (minmax[1] - minmax[0]));
      get_val = matrixGet_logarithm;
      break;
    case zeroCentredLog:
      logval = 0;
      get_val = matrixGet_zeroCenteredLog;
      break;
    default:
      logval = 0.;
      get_val = (matrixGet_scaleVal)0;
      break;
    }
  g_return_if_fail(get_val);

  DBG_fprintf(stderr, "ScalarFields: allocating storing values (%dx%d).\n",
	      vAlpha[0] + vAlpha[1], uAlpha[0] + uAlpha[1]);
  /* We store 8 additional data per point.
     - MAP_U: the x coordinate in plane ;
     - MAP_V: the y coordinate in plane ;
     - MAP_VISIBILITY: used or not ;
     - MAP_VALUE: field value ;
     - MAP_R...: RGBA. */
  isosurfacesPointsInit(points, 8);
  nU = uAlpha[1] - uAlpha[0];
  nV = vAlpha[1] - vAlpha[0];
  isosurfacesPointsAllocate(points, 1, (nV - 1) * (2 * (nU - 1) - 2), nV * nU);

  /* We use an hexagonal padding for equilateral triangles. */
  coeff =  dist / (float)alpha;
  nPoints = 0;
  viewport[0] = ((float)uAlpha[0] + 0.5f) * coeff;
  viewport[1] = (float)vAlpha[0] * coeff * 0.8660254f;
  viewport[2] = ((float)(uAlpha[1] - uAlpha[0]) + 0.5f) * coeff;
  viewport[3] = (float)(vAlpha[1] - vAlpha[0]) * coeff * 0.8660254f;
  for (i = 0; i < vAlpha[1] - vAlpha[0]; i++)
    {
      I[1] = (float)(i + vAlpha[0]) * coeff * 0.8660254f; /* the num coef is cos 30 */
      for (j = 0; j < uAlpha[1] - uAlpha[0]; j++)
	{
	  if (i%2)
	    I[0] = (float)(j + uAlpha[0]) * coeff;
	  else
	    I[0] = ((float)(j + uAlpha[0]) + 0.5f) * coeff;
	  planeTo3D(points->poly_points_data[nPoints] + MAP_X, I, xyz, center);
	  points->poly_points_data[nPoints][MAP_U] = I[0];
	  points->poly_points_data[nPoints][MAP_V] = I[1];
	  points->poly_points_data[nPoints][MAP_VISIBILITY] =
	    (float)setValue(field, shade, get_val, minmax, logval, drawnMinMax,
			    view->box->extension,
			    points->poly_points_data[nPoints] + MAP_X,
			    points->poly_points_data[nPoints] + MAP_VALUE,
			    points->poly_points_data[nPoints] + MAP_R);
	  points->poly_points_data[nPoints][MAP_A] = points->poly_points_data[nPoints][MAP_VALUE];
/* 	  DBG_fprintf(stderr, " | %d (%g, %g, %g) -> %g %g\n", */
/* 		      (gboolean)points->poly_points_data[nPoints][MAP_VISIBILITY], */
/* 		      points->poly_points_data[nPoints][MAP_X], */
/* 		      points->poly_points_data[nPoints][MAP_Y], */
/* 		      points->poly_points_data[nPoints][MAP_Z], */
/* 		      points->poly_points_data[nPoints][MAP_VALUE], */
/* 		      points->poly_points_data[nPoints][MAP_A]); */
	  nPoints += 1;
	}
    }

  linkedPoints = g_malloc(sizeof(int) * points->num_points);
  for (i = 0; i < points->num_points; i++)
    linkedPoints[i] = -1;
  nPolys = 0;
  points->num_polys_surf[0] = 0;
  points->num_polys_surf[1] = 0;
  data = points->poly_points_data;
  for (i = 0; i < vAlpha[1] - vAlpha[0] - 1; i++)
    {
      iPolys[3] = 0;
      for (j = 0; j < uAlpha[1] - uAlpha[0] - 1; j++)
	{
	  for (k = 1; k >= 0; k--)
	    {
	      iPolys[3] += 1;
	      iPolys[iPolys[3]%3] = (i + (i + k)%2) * nU + j;
	      if (iPolys[3] > 2)
		{
		  /* n is the number of visible points. */
		  n  = ((gboolean)data[iPolys[0]][MAP_VISIBILITY])?1:0;
		  n += ((gboolean)data[iPolys[1]][MAP_VISIBILITY])?1:0;
		  n += ((gboolean)data[iPolys[2]][MAP_VISIBILITY])?1:0;
		  switch (n)
		    {
		    case 1:
		      visible = ((gboolean)data[iPolys[0]][MAP_VISIBILITY])?
			0:((gboolean)data[iPolys[1]][MAP_VISIBILITY])?1:2;
		      linkedPoints[iPolys[(visible + 1) % 3]] = iPolys[visible];
		      linkedPoints[iPolys[(visible + 2) % 3]] = iPolys[visible];

		      points->poly_surf_index[nPolys] = 1;
		      points->num_polys_surf[0]      += 1;
		      break;
		    case 2:
		      hidden = (!(gboolean)data[iPolys[0]][MAP_VISIBILITY])?
			0:(!(gboolean)data[iPolys[1]][MAP_VISIBILITY])?1:2;
		      linkedPoints[iPolys[hidden]] = iPolys[(hidden + 1) % 3];

		      points->poly_surf_index[nPolys] = 1;
		      points->num_polys_surf[0]      += 1;
		      break;
		    case 3:
		      points->poly_surf_index[nPolys] = 1;
		      points->num_polys_surf[0]      += 1;
		      break;
		    default:
		      points->poly_surf_index[nPolys] = -1;
		    }
		  points->poly_num_vertices[nPolys] = 3;
		  points->poly_vertices[nPolys] = g_malloc(sizeof(int) * 3);
		  points->poly_vertices[nPolys][0] = iPolys[0];
		  points->poly_vertices[nPolys][1] = iPolys[1];
		  points->poly_vertices[nPolys][2] = iPolys[2];
		  nPolys += 1;
		}
	    }
	}
    }
  /* We now modify the position of the vertices defined in
     linkedPoints array. */
  data = points->poly_points_data;
  for (i = 0; i < points->num_points; i++)
    if (linkedPoints[i] >= 0)
      {
	a[0] = data[linkedPoints[i]][MAP_U];
	a[1] = data[linkedPoints[i]][MAP_V];
	b = data[i] + MAP_U;
	matrixGet_inter2DFromList(I, &lambda, a, b, interRed);
	b[0] = I[0];
	b[1] = I[1];
	planeTo3D(data[i] + MAP_X, b, xyz, center);
	data[i][MAP_VISIBILITY] = TRUE;
	data[i][MAP_VALUE] = data[linkedPoints[i]][MAP_VALUE];
	data[i][MAP_R] = data[linkedPoints[i]][MAP_R];
	data[i][MAP_G] = data[linkedPoints[i]][MAP_G];
	data[i][MAP_B] = data[linkedPoints[i]][MAP_B];
	data[i][MAP_A] = data[linkedPoints[i]][MAP_A];
      }
  g_free(linkedPoints);
#if DEBUG == 1
  isosurfacesPointsCheck(points);
#endif
  for (inter = interRed; inter; inter = g_list_next(inter))
    g_free(inter->data);
  g_list_free(interRed);

  valMinMax[0] = (float)get_val(drawnMinMax[0], minmax, logval);
  valMinMax[1] = (float)get_val(drawnMinMax[1], minmax, logval);
}

void scalarFieldDraw_map(ScalarField *field, Plane *plane, Shade *shade,
                         OpenGLView *view, float precision, float drawnMinMax[2],
			 matrix_scaleflag scale, guint nIsoLines, float *rgb,
			 float *inputMinMax)
{
  SurfacesPoints points;
  float valMinMax[2], viewport[4];
  #if DEBUG == 1
/*   gchar str[5]; */
  GTimer *timer;
  gulong fractionTimer;
  #endif

#if DEBUG == 1
  timer = g_timer_new();
  g_timer_start(timer);
#endif

  computeMap(&points, viewport, valMinMax, drawnMinMax,
	     field, plane, shade, view,
	     precision, scale, inputMinMax);
  DBG_fprintf(stderr, "ScalarFields: map computed.\n");
  drawMapAndLines(&points, plane, nIsoLines, shade, rgb, valMinMax);
  DBG_fprintf(stderr, "ScalarFields: map drawn.\n");
  isosurfacesPointsFree(&points);
  DBG_fprintf(stderr, "ScalarFields: memory free.\n");

#if DEBUG == 1
  g_timer_stop(timer);
  fprintf(stderr, "ScalarFields: Draw map in %g micro-s.\n",
	  g_timer_elapsed(timer, &fractionTimer)/1e-6);
  g_timer_destroy(timer);
#endif
}

gboolean scalarFieldExport_map(ScalarField *field, Plane *plane, Shade *shade,
			       OpenGLView *view, float precision,
			       float drawnMinMax[2], matrix_scaleflag scale,
			       guint nIsoLines, float *rgb, float *inputMinMax,
			       gchar *filename, ScalarField_exportFormat format,
			       GError **error)
{
#ifdef HAVE_CAIRO
#define fact 25.f
  SurfacesPoints points;
  float valMinMax[2], viewport[4], **data;
  int i, j;
  float v, rgba[4];
  Line *line;
  float *uvs;
  gint nVals;
  cairo_surface_t *surface;
  cairo_t *cr;
  cairo_status_t status;
  cairo_matrix_t mat = {fact, 0., 0., fact, 0., 0.};
  
  g_return_val_if_fail(error && *error == (GError*)0, FALSE);

  computeMap(&points, viewport, valMinMax, drawnMinMax,
	     field, plane, shade, view,
	     precision, scale, inputMinMax);

  DBG_fprintf(stderr, "Scalar Field: begin export to PDF/SVG.\n");

  switch (format)
    {
    case sf_svg:
      surface = cairo_svg_surface_create(filename,
					 (double)(viewport[2] * fact),
					 (double)(viewport[3] * fact));
      break;
    case sf_pdf:
      surface = cairo_pdf_surface_create(filename,
					 (double)(viewport[2] * fact),
					 (double)(viewport[3] * fact));
      break;
    default:
      surface = (cairo_surface_t*)0;
    }
  status = cairo_surface_status(surface);
  if (status != CAIRO_STATUS_SUCCESS)
    {
      *error = g_error_new(G_FILE_ERROR, G_FILE_ERROR_FAILED,
			   cairo_status_to_string(status));
      cairo_surface_destroy(surface);
      isosurfacesPointsFree(&points);
      return FALSE;
    }

  cr = cairo_create(surface);
  status = cairo_status(cr);
  if (status != CAIRO_STATUS_SUCCESS)
    {
      *error = g_error_new(G_FILE_ERROR, G_FILE_ERROR_FAILED,
			   cairo_status_to_string(status));
      cairo_destroy(cr);
      cairo_surface_destroy(surface);
      isosurfacesPointsFree(&points);
      return FALSE;
    }
  cairo_set_line_cap(cr, CAIRO_LINE_CAP_BUTT);
  cairo_set_line_join(cr, CAIRO_LINE_JOIN_BEVEL);
  cairo_set_line_width(cr, 0.01);
  mat.x0 = -(double)viewport[0] * fact;
  mat.y0 = -(double)viewport[1] * fact;
  cairo_set_matrix(cr, &mat);

  for (i = 0; i < points.num_polys; i++)
    {
      if (points.poly_surf_index[i] > 0)
	{
	  data = points.poly_points_data;
	  cairo_set_source_rgba(cr,
				(data[points.poly_vertices[i][0]][MAP_R] +
				 data[points.poly_vertices[i][1]][MAP_R] +
				 data[points.poly_vertices[i][2]][MAP_R]) / 3.f,
				(data[points.poly_vertices[i][0]][MAP_G] +
				 data[points.poly_vertices[i][1]][MAP_G] +
				 data[points.poly_vertices[i][2]][MAP_G]) / 3.f,
				(data[points.poly_vertices[i][0]][MAP_B] +
				 data[points.poly_vertices[i][1]][MAP_B] +
				 data[points.poly_vertices[i][2]][MAP_B]) / 3.f,
				1.f);
	  cairo_move_to(cr,
			data[points.poly_vertices[i][0]][MAP_U],
			data[points.poly_vertices[i][0]][MAP_V]);
	  cairo_line_to(cr,
			data[points.poly_vertices[i][1]][MAP_U],
			data[points.poly_vertices[i][1]][MAP_V]);
	  cairo_line_to(cr,
			data[points.poly_vertices[i][2]][MAP_U],
			data[points.poly_vertices[i][2]][MAP_V]);
	  cairo_line_to(cr,
			data[points.poly_vertices[i][0]][MAP_U],
			data[points.poly_vertices[i][0]][MAP_V]);	
	  cairo_fill_preserve(cr);
	  cairo_stroke(cr);
	}
    }

  /* Add isolines. */
  DBG_fprintf(stderr, "ScalarFields: export isolines.\n");
  for (i = 1; i <= (int)nIsoLines; i++)
    {
      v = valMinMax[0] + (valMinMax[1] - valMinMax[0]) *
	(float)i / (float)(nIsoLines + 1);
      if (isolineBuild(&line, &points, MAP_VALUE, MAP_VISIBILITY, v))
	{
	  DBG_fprintf(stderr, "ScalarFields: set color for value %g.\n", v);
	  if (!rgb)
	    {
	      shadeGet_valueTransformedInRGB(shade, rgba, v);
	      rgba[0] = 1.f - rgba[0];
	      rgba[1] = 1.f - rgba[1];
	      rgba[2] = 1.f - rgba[2];
	      rgba[3] = 1.f;
	      rgb = rgba;
	    }
	  cairo_set_source_rgb(cr, rgb[0], rgb[1], rgb[2]);

	  DBG_fprintf(stderr, "ScalarFields: project line on plane %p.\n",
		      (gpointer)plane);
	  uvs = isolineProject(line, plane, &nVals);
	  for (j = 0; j < nVals; j++)
	    {
	      cairo_move_to(cr, uvs[j * 4 + 0], uvs[j * 4 + 1]);
	      cairo_line_to(cr, uvs[j * 4 + 2], uvs[j * 4 + 3]);
	      cairo_stroke(cr);
	    }
	  g_free(uvs);

	  isolineFree(line);
	}
    }

  /* Add frame. */
  DBG_fprintf(stderr, "ScalarFields: export the frame.\n");
  uvs = planeGet_reducedIntersection(plane, &nVals);
  if (uvs)
    {
      cairo_set_source_rgb(cr, 0., 0., 0.);

      cairo_move_to(cr, uvs[(nVals - 1) * 2 + 0], uvs[(nVals - 1) * 2 + 1]);
      for (j = 0; j < nVals; j++)
	cairo_line_to(cr, uvs[j * 2 + 0], uvs[j * 2 + 1]);
      cairo_stroke(cr);
      g_free(uvs);
    }

  /* Add legend. */
/*   toolShadeExport(shade, (double)(viewport[2] * fact), */
/* 		  (double)(viewport[3] * fact)); */

  /* Finalising */
  cairo_show_page(cr);
  cairo_destroy(cr);
  cairo_surface_destroy(surface);
  isosurfacesPointsFree(&points);

  return TRUE;
#else
  g_error("Not compiled with CAIRO not able to export.");
  return FALSE;
#endif
}

gboolean scalarFieldGet_value(ScalarField *field, float xyz[3],
			      double *value, float extension[3])
{
  float redXyz[3], factor[3], pos;
  int l, m, n, ijk[3], dijk[3], nMax;
  int nval1, nval2;
  double *mesh;
  ScalarField_meshflag meshtype;

  /* Taken from ABINIT */
  float x1,x2,x3;

  g_return_val_if_fail(field, FALSE);

  meshtype = scalarFieldGet_meshtype(field);

  /* First, we transform the coordinates into reduced coordinates. */
  matrix_productVector(redXyz, field->fromXYZtoReducedCoord, xyz);

  /* We compute i, j, k. */
  for (l = 0; l < 3; l++)
    {
      /* If we are periodic and inside the extension, we put back in
	 the box. */
      if (field->periodic && redXyz[l] > -extension[l] &&
	  redXyz[l] < 1. + extension[l])
	redXyz[l] = fModulo(redXyz[l], 1);
      if (field->periodic)
	nMax = field->nElements[l];
      else
	nMax = field->nElements[l] - 1;

      switch (meshtype)
        {
        case uniform:
          pos = (float)nMax * redXyz[l];
          ijk[l] = (int)pos;
          factor[l] = pos - (float)ijk[l];
          break;
        case nonuniform:
	  mesh = (double*)0;
	  switch (l)
	    {
	    case 0:
	      mesh = scalarFieldGet_meshx(field);
	      break;
	    case 1:
	      mesh = scalarFieldGet_meshy(field);
	      break;
	    case 2:
	      mesh = scalarFieldGet_meshz(field);
	      break;
	    }
          nval1 = 0;
          nval2 = nMax-1;
	  n = 0;
          for (m = 0; m < nMax/2; m++)
            {
              n = (int)((nval2-nval1)/2);
              if (n == 0) {
                n = nval1;
                break;    }
              else
                n = nval1+n;
              if (redXyz[l] > mesh[n])
                nval1 = n;
              else
                nval2 = n;
            }
          ijk[l] = n;
          factor[l] = (redXyz[l]-mesh[n])/(mesh[n+1]-mesh[n]);
          break;
        default:
          g_warning("Wrong value for 'meshtype'.");
          return FALSE;
        }
	
      if (ijk[l] < 0 || redXyz[l] < 0.)
	return FALSE;
      if (ijk[l] >= nMax)
	return FALSE;
    }

  /* lower left is ijk. */
  /* upper right is dijk. */
  dijk[0] = (ijk[0] + 1) % field->nElements[0];
  dijk[1] = (ijk[1] + 1) % field->nElements[1];
  dijk[2] = (ijk[2] + 1) % field->nElements[2];
  
  /* weight is factor. */
  x1 = factor[0];
  x2 = factor[1];
  x3 = factor[2];

  /* calculation of the density value */
  *value  = 0.f;
  *value += field->data[ ijk[0]][ ijk[1]][ ijk[2]] *
    (1.f - x1) * (1.f - x2) * (1.f - x3);
  *value += field->data[dijk[0]][ ijk[1]][ ijk[2]] * x1 * (1.f - x2) * (1.f - x3);
  *value += field->data[ ijk[0]][dijk[1]][ ijk[2]] * (1.f - x1) * x2 * (1.f - x3);
  *value += field->data[ ijk[0]][ ijk[1]][dijk[2]] * (1.f - x1) * (1.f - x2) * x3;
  *value += field->data[dijk[0]][dijk[1]][ ijk[2]] * x1 * x2 * (1.f - x3);
  *value += field->data[ ijk[0]][dijk[1]][dijk[2]] * (1.f  -x1) * x2 * x3;
  *value += field->data[dijk[0]][ ijk[1]][dijk[2]] * x1 * (1.f - x2) * x3;
  *value += field->data[dijk[0]][dijk[1]][dijk[2]] * x1 * x2 * x3;

  return TRUE;
}
void scalarFieldSet_box(ScalarField *field, double box[6])
{
  int i;

  g_return_if_fail(field);

  DBG_fprintf(stderr, "ScalarField : set the bounding box.\n");

  /* Change the box. */
  for (i = 0; i < 6; i++)
    field->box[i] = box[i];

  /* Update the transformation matrix. */
  field->fromXYZtoReducedCoord[0][0] =
    1 / field->box[VISU_DATA_BOX_DXX];
  field->fromXYZtoReducedCoord[0][1] =
    - field->box[VISU_DATA_BOX_DYX] /
    field->box[VISU_DATA_BOX_DXX] /
    field->box[VISU_DATA_BOX_DYY];
  field->fromXYZtoReducedCoord[0][2] =
    - (field->box[VISU_DATA_BOX_DZX] /
       field->box[VISU_DATA_BOX_DXX] -
       field->box[VISU_DATA_BOX_DYX] *
       field->box[VISU_DATA_BOX_DZY] / 
       field->box[VISU_DATA_BOX_DXX] / 
       field->box[VISU_DATA_BOX_DYY] ) /
    field->box[VISU_DATA_BOX_DZZ];
  field->fromXYZtoReducedCoord[1][0] = 0.;
  field->fromXYZtoReducedCoord[1][1] =
    1 / field->box[VISU_DATA_BOX_DYY];
  field->fromXYZtoReducedCoord[1][2] =
    - field->box[VISU_DATA_BOX_DZY] /
    field->box[VISU_DATA_BOX_DYY] /
    field->box[VISU_DATA_BOX_DZZ];
  field->fromXYZtoReducedCoord[2][0] = 0.;
  field->fromXYZtoReducedCoord[2][1] = 0.;
  field->fromXYZtoReducedCoord[2][2] = 1 /
    field->box[VISU_DATA_BOX_DZZ];
}
void scalarFieldGet_box(ScalarField *field, double box[6])
{
  int i;

  g_return_if_fail(field);

  for (i = 0; i < 6; i++)
    box[i] = (double)field->box[i];
}
void scalarFieldGet_gridSize(ScalarField *field, int grid[3])
{
  int i;

  g_return_if_fail(field);

  for (i = 0; i < 3; i++)
    grid[i] = field->nElements[i];
}
double* scalarFieldGet_meshx(ScalarField *field)
{
  g_return_val_if_fail(field, (double*)0);
  return field->meshx;
}
double* scalarFieldGet_meshy(ScalarField *field)
{
  g_return_val_if_fail(field, (double*)0);
  return field->meshy;
}
double* scalarFieldGet_meshz(ScalarField *field)
{
  g_return_val_if_fail(field, (double*)0);
  return field->meshz;
}
double*** scalarFieldGet_data(ScalarField *field)
{
  g_return_val_if_fail(field, (double***)0);

  return field->data;
}

void scalarFieldSet_fitToBox(VisuData *data, ScalarField *field)
{
  double box[6];
  int i;

  g_return_if_fail(data);

  DBG_fprintf(stderr, "ScalarField: change the current box to fit to %p.\n",
	      (gpointer) data);
  for (i = 0; i < 6; i++)
    box[i] = (double)visuDataGet_boxGeometry(data, i);
  scalarFieldSet_box(field, box);
}  
GList* scalarFieldGet_allOptions(ScalarField *field)
{
  g_return_val_if_fail(field, (GList*)0);
  
  return g_list_copy(field->options);
}
void scalarFieldAdd_option(ScalarField *field, Option *option)
{
  g_return_if_fail(field && option);
  
  field->options = g_list_append(field->options, (gpointer)option);
}


/* Load method handling. */
void scalarFieldInit()
{
  char *type[] = {"*.pot", "*.dat", (char*)0};
  char *descr = _("Potential/density files");
  FileFormat *fmt;
  
  loadMethods = (GList*)0;
  
  fmt = fileFormatNew(descr, type);
  scalarFieldAdd_loadMethod("Plain ascii", scalarFieldLoad_fromAscii,
			    fmt, G_PRIORITY_LOW);
}

void scalarFieldAdd_loadMethod(const gchar *name, ScalarFieldLoadMethod method,
			       FileFormat *format, int priority)
{
  ScalarFieldLoadStruct *meth;

  g_return_if_fail(name && method && format);

  meth = g_malloc(sizeof(ScalarFieldLoadStruct));
  meth->name = g_strdup(name);
  meth->fmt = format;
  meth->load = method;
  meth->priority = priority;

  loadMethods = g_list_prepend(loadMethods, meth);
  loadMethods = g_list_sort(loadMethods, compareLoadPriority);
}

static gint compareLoadPriority(gconstpointer a, gconstpointer b)
{
  if (((ScalarFieldLoadStruct*)a)->priority <
      ((ScalarFieldLoadStruct*)b)->priority)
    return (gint)-1;
  else if (((ScalarFieldLoadStruct*)a)->priority >
	   ((ScalarFieldLoadStruct*)b)->priority)
    return (gint)+1;
  else
    return (gint)0;
}
GList* scalarFieldGet_allLoadMethods()
{
  return loadMethods;
}
