/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include <gtk/gtk.h>

#include "gtk_lineObjectWidget.h"
#include "gtk_stippleComboBoxWidget.h"
#include "gtk_colorComboBoxWidget.h"
#include <visu_tools.h>

/**
 * SECTION:lineobject
 * @short_description: Defines a specialised #GtkHBox to choose
 * all characteristic of lines.
 * @see_also: #ColorComboBox and #StippleComboBox
 *
 * <para>This widget looks like a #GtkComboBox and it displays a list
 * of preset patterns for line. These patterns are defined by a
 * guint16 value that is given to OpenGL for line stipple.</para>
 * <para>This widget can emit a #LineObject::stipple-selected
 * signal that is a wrapper around the #GtkComboBox::changed signal,
 * but it is emitted only when a stipple is selected and this stipple
 * is passed to the call back.</para>
 *
 * Since: 3.4
 */

enum {
  LINE_USE_CHANGED_SIGNAL,
  LINE_WIDTH_CHANGED_SIGNAL,
  LINE_STIPPLE_CHANGED_SIGNAL,
  LINE_COLOR_CHANGED_SIGNAL,
  LAST_SIGNAL
};

static void lineObject_dispose (GObject *obj);
static void lineObject_finalize(GObject *obj);

static guint lineObject_signals[LAST_SIGNAL] = { 0 };

struct _LineObject
{
  GtkVBox box;

  GtkWidget *label;
  GtkWidget *used;
  GtkWidget *width;
  GtkWidget *stipple;

  GtkWidget *rgRGB[3];
  GtkWidget *color;
  GtkWidget *btColor;

  gulong rgSignals[3];
  gulong colorSignal;

  /* Memory gestion. */
  gboolean dispose_has_run;
};

struct _LineObjectClass
{
  GtkComboBoxClass parent_class;

  void (*lineObject) (LineObject *box);
};

/* Local callbacks. */
static void onUseChanged(GtkCheckButton *check, LineObject *line);
static void onWidthChanged(GtkSpinButton *spin, LineObject *line);
static void onStippleChanged(StippleComboBox *wd, guint stipple, LineObject *line);
static void onRGBValueChanged(GtkRange *rg, LineObject *line);
static void onAddClicked(GtkButton *bt, LineObject *line);
static void onColorChanged(ColorComboBox *combo, Color *color, LineObject *line);

/* Local methods. */

G_DEFINE_TYPE(LineObject, lineObject, GTK_TYPE_VBOX)

static void lineObject_class_init(LineObjectClass *klass)
{
  DBG_fprintf(stderr, "Gtk LineObject: creating the class of the widget.\n");
  DBG_fprintf(stderr, "                     - adding new signals ;\n");
  /**
   * LineObject::use-changed:
   * @line: the #LineObject that emits the signal ;
   * @used: TRUE if the line is used.
   *
   * This signal is emitted when the usage check box is changed.
   *
   * Since: 3.4
   */
  lineObject_signals[LINE_USE_CHANGED_SIGNAL] =
    g_signal_new ("use-changed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		  G_STRUCT_OFFSET (LineObjectClass, lineObject),
		  NULL, 
		  NULL,                
		  g_cclosure_marshal_VOID__BOOLEAN,
		  G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
  /**
   * LineObject::width-changed:
   * @line: the #LineObject that emits the signal ;
   * @width: the new width.
   *
   * This signal is emitted when the width of the line is changed.
   *
   * Since: 3.4
   */
  lineObject_signals[LINE_WIDTH_CHANGED_SIGNAL] =
    g_signal_new ("width-changed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		  G_STRUCT_OFFSET (LineObjectClass, lineObject),
		  NULL, 
		  NULL,                
		  g_cclosure_marshal_VOID__INT,
		  G_TYPE_NONE, 1, G_TYPE_INT);
  /**
   * LineObject::stipple-changed:
   * @line: the #LineObject that emits the signal ;
   * @stipple: the new stipple pattern.
   *
   * This signal is emitted when the stipple pattern of the line is changed.
   *
   * Since: 3.4
   */
  lineObject_signals[LINE_STIPPLE_CHANGED_SIGNAL] =
    g_signal_new ("stipple-changed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		  G_STRUCT_OFFSET (LineObjectClass, lineObject),
		  NULL, 
		  NULL,                
		  g_cclosure_marshal_VOID__UINT,
		  G_TYPE_NONE, 1, G_TYPE_UINT);
  /**
   * LineObject::color-changed:
   * @line: the #LineObject that emits the signal ;
   * @color: the new color values (three RGB values).
   *
   * This signal is emitted when the colour of the line is changed.
   *
   * Since: 3.4
   */
  lineObject_signals[LINE_COLOR_CHANGED_SIGNAL] =
    g_signal_new ("color-changed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		  G_STRUCT_OFFSET (LineObjectClass, lineObject),
		  NULL, 
		  NULL,                
		  g_cclosure_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);
  
  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose = lineObject_dispose;
  G_OBJECT_CLASS(klass)->finalize = lineObject_finalize;
}

static void lineObject_dispose(GObject *obj)
{
  DBG_fprintf(stderr, "Gtk LineObject: dispose object %p.\n", (gpointer)obj);

  if (LINE_OBJECT(obj)->dispose_has_run)
    return;

  LINE_OBJECT(obj)->dispose_has_run = TRUE;
  /* Chain up to the parent class */
  G_OBJECT_CLASS(lineObject_parent_class)->dispose(obj);
}
static void lineObject_finalize(GObject *obj)
{
  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Gtk LineObject: finalize object %p.\n", (gpointer)obj);

  /* Chain up to the parent class */
  G_OBJECT_CLASS(lineObject_parent_class)->finalize(obj);

  DBG_fprintf(stderr, " | freeing ... OK.\n");
}


static void lineObject_init(LineObject *line)
{
  GtkWidget *vbox, *hbox, *label, *align, *table;
  char *rgb[3];
  char *rgbName[3] = {"scroll_r", "scroll_g", "scroll_b"};
  int i;

  DBG_fprintf(stderr, "Gtk LineObject: initializing new object (%p).\n",
	      (gpointer)line);

  gtk_box_set_spacing(GTK_BOX(line), 5);

  /********************/
  /* The header line. */
  /********************/
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(line), hbox, FALSE, FALSE, 0);

  /* The label. */
  line->label = gtk_label_new("");
  gtk_misc_set_alignment(GTK_MISC(line->label), 0., 0.5);
  gtk_widget_set_name(line->label, "label_head");
  gtk_box_pack_start(GTK_BOX(hbox), line->label, TRUE, TRUE, 2);

  /* The scale checkbox. */
  label = gtk_label_new(_("drawn:"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
  line->used = gtk_check_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), line->used, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(line->used), "toggled",
                   G_CALLBACK(onUseChanged), (gpointer)line);
  /* The scale width spin button. */
  align = gtk_alignment_new(1., 0.5, 1, 1);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 0, 5, 0);
  gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 0);
  label = gtk_label_new(_("width:"));
  gtk_container_add(GTK_CONTAINER(align), label);
  line->width = gtk_spin_button_new_with_range(1., 10., 1.);
  gtk_entry_set_width_chars(GTK_ENTRY(line->width), 2);
  gtk_box_pack_start(GTK_BOX(hbox), line->width, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(line->width), "value-changed",
		   G_CALLBACK(onWidthChanged), (gpointer)line);
  /* px for pixel. */
  label = gtk_label_new(_("px"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
  /* The stipple pattern. */
  align = gtk_alignment_new(1., 0.5, 1, 1);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 0, 5, 0);
  gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 3);
  line->stipple = stippleComboBox_new();
  gtk_container_add(GTK_CONTAINER(align), line->stipple);
  g_signal_connect(G_OBJECT(line->stipple), "stipple-selected",
		   G_CALLBACK(onStippleChanged), (gpointer)line);

  /********************/
  /* The colour line. */
  /********************/
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(line), hbox, FALSE, FALSE, 0);
  /* The table for the box colour. */
  table = gtk_table_new(3, 2, FALSE);
  gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 0);
  rgb[0] = _("R:");
  rgb[1] = _("G:");
  rgb[2] = _("B:");
  for (i = 0; i < 3; i++)
    {
      label = gtk_label_new(rgb[i]);
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, i, i + 1,
		       GTK_SHRINK, GTK_SHRINK, 5, 0);
      line->rgRGB[i] = gtk_hscale_new_with_range(0., 1., 0.001);
      gtk_scale_set_value_pos(GTK_SCALE(line->rgRGB[i]),
			      GTK_POS_RIGHT);
      gtk_range_set_value(GTK_RANGE(line->rgRGB[i]), 0.741456963);
      gtk_widget_set_name(line->rgRGB[i], rgbName[i]);
      gtk_table_attach(GTK_TABLE(table), line->rgRGB[i],
		       1, 2, i, i + 1,
		       GTK_EXPAND | GTK_FILL, GTK_SHRINK, 2, 0);
      line->rgSignals[i] = g_signal_connect(G_OBJECT(line->rgRGB[i]),
					    "value-changed",
					    G_CALLBACK(onRGBValueChanged),
					    (gpointer)line);
    }
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 5);
  label = gtk_label_new(_("color:"));
  line->color = colorComboBox_new(FALSE);
  line->colorSignal = g_signal_connect(G_OBJECT(line->color), "color-selected",
				       G_CALLBACK(onColorChanged), (gpointer)line);
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  colorComboBoxSet_printValues(COLOR_COMBOX(line->color), FALSE);
  gtk_box_pack_end(GTK_BOX(hbox), line->color, FALSE, FALSE, 0);
  align = gtk_alignment_new(0.5, 0.5, 0, 1);
  gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 0);
  line->btColor = gtk_button_new();
  gtk_container_add(GTK_CONTAINER(align), line->btColor);
  gtk_container_add(GTK_CONTAINER(line->btColor),
		    gtk_image_new_from_stock (GTK_STOCK_ADD, GTK_ICON_SIZE_MENU));
  label = gtk_label_new(_("<span size=\"small\"><i>(preset)</i></span>"));
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(line->btColor), "clicked",
		   G_CALLBACK(onAddClicked), (gpointer)line);

  gtk_widget_show_all(GTK_WIDGET(line));
}

GtkWidget* lineObjectNew(const gchar* label)
{
  LineObject *line;
  gchar *markup;

  DBG_fprintf(stderr, "Gtk LineObject: creating new object.\n");

  line = LINE_OBJECT(g_object_new(lineObject_get_type(), NULL));

  DBG_fprintf(stderr, "Gtk LineObject: build widgets.\n");
  markup = g_markup_printf_escaped("<b>%s</b>", label);
  gtk_label_set_markup(GTK_LABEL(line->label), markup);
  g_free(markup);

  return GTK_WIDGET(line);
}


static void onUseChanged(GtkCheckButton *check, LineObject *line)
{
  g_signal_emit(G_OBJECT(line), lineObject_signals[LINE_USE_CHANGED_SIGNAL],
		0, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check)), NULL);
}
static void onWidthChanged(GtkSpinButton *spin, LineObject *line)
{
  g_signal_emit(G_OBJECT(line), lineObject_signals[LINE_WIDTH_CHANGED_SIGNAL],
		0, (gint)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin)), NULL);
}
static void onStippleChanged(StippleComboBox *wd _U_, guint stipple, LineObject *line)
{
  g_signal_emit(G_OBJECT(line), lineObject_signals[LINE_STIPPLE_CHANGED_SIGNAL],
		0, stipple, NULL);
}
static void onRGBValueChanged(GtkRange *rg _U_, LineObject *line)
{
  float rgba[4];
  int i;

  g_return_if_fail(IS_LINE_OBJECT(line));

  rgba[0] = gtk_range_get_value(GTK_RANGE(line->rgRGB[0]));
  rgba[1] = gtk_range_get_value(GTK_RANGE(line->rgRGB[1]));
  rgba[2] = gtk_range_get_value(GTK_RANGE(line->rgRGB[2]));
  rgba[3] = 1.f;
  colorGet_byValues(&i, rgba[0], rgba[1], rgba[2], rgba[3]);

  g_signal_handler_block(G_OBJECT(line->color), line->colorSignal);
  if (i < 0)
    gtk_combo_box_set_active(GTK_COMBO_BOX(line->color), -1);
  else
    gtk_combo_box_set_active(GTK_COMBO_BOX(line->color), i + 1);
  g_signal_handler_unblock(G_OBJECT(line->color), line->colorSignal);

  gtk_widget_set_sensitive(line->btColor, (i < 0));

  g_signal_emit(G_OBJECT(line), lineObject_signals[LINE_COLOR_CHANGED_SIGNAL],
		0, (gpointer)rgba, NULL);
}
static void onAddClicked(GtkButton *bt _U_, LineObject *line)
{
  float rgba[4];
  int selected;
  Color *color;

  g_return_if_fail(IS_LINE_OBJECT(line));

  DBG_fprintf(stderr, "Gtk LineObject: adding a new color from ranges.\n");

  rgba[0] = gtk_range_get_value(GTK_RANGE(line->rgRGB[0]));
  rgba[1] = gtk_range_get_value(GTK_RANGE(line->rgRGB[1]));
  rgba[2] = gtk_range_get_value(GTK_RANGE(line->rgRGB[2]));
  rgba[3] = 1.f;
  color = colorAdd_floatRGBA(rgba, &selected);
  g_signal_handler_block(G_OBJECT(line->color), line->colorSignal);
  gtk_combo_box_set_active(GTK_COMBO_BOX(line->color), selected);
  g_signal_handler_unblock(G_OBJECT(line->color), line->colorSignal);
  gtk_widget_set_sensitive(line->btColor, FALSE);
}
static void onColorChanged(ColorComboBox *combo _U_, Color *color, LineObject *line)
{
  lineObjectSet_color(line, color->rgba);
}




void lineObjectSet_use(LineObject *line, gboolean status)
{
  g_return_if_fail(IS_LINE_OBJECT(line));

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(line->used), status);
}
void lineObjectSet_width(LineObject *line, gint width)
{
  g_return_if_fail(IS_LINE_OBJECT(line) && width > 0 && width < 11);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(line->width), (gdouble)width);
}
void lineObjectSet_color(LineObject *line, float rgb[3])
{
  gboolean change;

  g_return_if_fail(IS_LINE_OBJECT(line));

  g_signal_handler_block(G_OBJECT(line->rgRGB[0]), line->rgSignals[0]);
  g_signal_handler_block(G_OBJECT(line->rgRGB[1]), line->rgSignals[1]);
  g_signal_handler_block(G_OBJECT(line->rgRGB[2]), line->rgSignals[2]);

  change = FALSE;
  if ((float)gtk_range_get_value(GTK_RANGE(line->rgRGB[0])) != rgb[0])
    {
      change = TRUE;
      gtk_range_set_value(GTK_RANGE(line->rgRGB[0]), rgb[0]);
    }
  if ((float)gtk_range_get_value(GTK_RANGE(line->rgRGB[1])) != rgb[1])
    {
      change = TRUE;
      gtk_range_set_value(GTK_RANGE(line->rgRGB[1]), rgb[1]);
    }
  if ((float)gtk_range_get_value(GTK_RANGE(line->rgRGB[2])) != rgb[2])
    {
      change = TRUE;
      gtk_range_set_value(GTK_RANGE(line->rgRGB[2]), rgb[2]);
    }

  g_signal_handler_unblock(G_OBJECT(line->rgRGB[0]), line->rgSignals[0]);
  g_signal_handler_unblock(G_OBJECT(line->rgRGB[1]), line->rgSignals[1]);
  g_signal_handler_unblock(G_OBJECT(line->rgRGB[2]), line->rgSignals[2]);

  if (change)
    onRGBValueChanged((GtkRange*)0, line);
}
void lineObjectSet_stipple(LineObject *line, guint16 stipple)
{
  g_return_if_fail(IS_LINE_OBJECT(line));

  if (!stippleComboBoxSet_selectionByStipple(STIPPLE_COMBOX(line->stipple), stipple))
    {
      stippleComboBoxAdd_pattern(STIPPLE_COMBOX(line->stipple), stipple);
      stippleComboBoxSet_selectionByStipple(STIPPLE_COMBOX(line->stipple), stipple);
    }
}
