/*
   Help functions
   --------------------------------------------------------------------
   VCHE - Virtual Console Hex Editor

   Copyright (C) 1998, 1999 Diego Javier Grigna <diego@grigna.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "common.h"

static char *meaning[] = { 
 " NUL "," SOH "," STX "," ETX "," EOT "," ENQ "," ACK ",
 " BEL "," BS  "," HT  "," LF  "," VT  "," FF  "," CR  ",
 " SO  "," SI  "," DLE "," DC1 "," DC2 "," DC3 "," DC4 ",
 " NAK "," SYN "," ETB "," CAN "," EM  "," SUB "," ESC ",
 " FS  "," GS  "," RS  "," US  ","SPACE",
};

/*
 * Function prototypes
 */
static char *int_to_escape( int c);
static char *int_to_binary( int c);

/*
 * VCHE main help
 */
void help( void)
{
 lib_text( 

 "\n  VCHE - Virtual Console Hex Editor\n\n"

 "  Release " VCHE_VERSION "\n\n"
 "  If you need help, type \"man vche\" at the shell prompt.\n\n"

 "  If you have found a bug, have problems or comments, mail\n"
 "  me to <diego@grigna.com>\n"

 , " VCHE HELP ");

}

void show_gnu_license( void)
{
 lib_text(
 "\n  VCHE - Virtual Console Hex Editor\n"
 "  Copyright (C) 1998, 1999 Diego Javier Grigna <diego@grigna.com>\n\n"
 "  This program is free software; you can redistribute it and/or modify\n"
 "  it under the terms of the GNU General Public License as published by\n"
 "  the Free Software Foundation; either version 2 of the License, or\n"
 "  (at your option) any later version.\n\n"

 "  This program is distributed in the hope that it will be useful,\n"
 "  but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
 "  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
 "  GNU General Public License for more details.\n\n"

 "  You should have received a copy of the GNU General Public License\n"
 "  along with this program; if not, write to the Free Software\n"
 "  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA\n\n"

 , " GNU General Public License ");
}

/*
 * Shows the ascii table and let people navigate it.
 */
void ascii_table( void)
{
 char buf[ 128];
 int slines = vc_lines - 9; /* Number of lines to show */
 int endpos;
 int pos = 0;
 int i, j;
 int c;
#if defined(VCHE_RAW) || defined(VCHE_NC)
 char *bottom_line = "Use         to move           ENTER or F10 to QUIT";
 int ll;
#endif
#if defined(VCHE_NC)
 WINDOW *w; 
 WINDOW *w2; 
#endif

 endpos = 256 - slines;
 if( endpos < 0) endpos = 0;


#if defined(VCHE_VC)

 lib_save_screen();

 vc_box( 0, 0, vc_cols, vc_lines - 1, VC_COLOR_BACKCYAN);
 vc_do_title( " ASCII Table ");
 vc_put_bottom_line( "Use \x18 \x19 \x1A \x1B to move          ENTER or F10 to QUIT" );

 vc_puts( 7,  2, 65, "Ŀ");
 vc_puts( 7,  3, 65, " Dec  Oct  Hex  Character  Meaning  C Escape    Binary   ");
 vc_puts( 7,  4, 65, "Ĵ");
 vc_puts( 7, vc_lines - 4, 65, "");

#elif defined(VCHE_RAW)

 lib_save_screen();

 raw_box( 0, 0, vc_cols, vc_lines - 1, VC_COLOR_BACKCYAN);
 raw_do_title( " ASCII Table ");
 raw_put_bottom_line( bottom_line);

 ll = ( vc_cols - strlen( bottom_line)) / 2;

 term_enable_graf_map();
 raw_putc( ll + 24, vc_lines - 1, RAW_LINE_VERT);
 raw_putc( ll +  4, vc_lines - 1, RAW_UARROW   );
 raw_putc( ll +  6, vc_lines - 1, RAW_DARROW   );
 raw_putc( ll +  8, vc_lines - 1, RAW_RARROW   );
 raw_putc( ll + 10, vc_lines - 1, RAW_LARROW   );
 term_disable_graf_map();

 term_color( VC_COLOR_BACKCYAN);

 raw_puts( 9, 3, 59, "Dec   Oct   Hex   Character   Meaning   C Escape     Binary");

 ll = vc_lines - 4;

 term_enable_graf_map();

 raw_putc(  7,  2, RAW_LINE_TOPLEFT    );
 raw_putc( 71,  2, RAW_LINE_TOPRIGHT   );
 raw_putc(  7,  4, RAW_LTEE            );
 raw_putc( 71,  4, RAW_RTEE            );
 raw_putc(  7, ll, RAW_LINE_BOTTOMLEFT );
 raw_putc( 71, ll, RAW_LINE_BOTTOMRIGHT);

 for( i = 1; i < 64; i++) {
      raw_putc( 7 + i,  2, RAW_LINE_HORIZ);
      raw_putc( 7 + i,  4, RAW_LINE_HORIZ);
      raw_putc( 7 + i, ll, RAW_LINE_HORIZ);
 }

 for( i = 5; i < vc_lines - 4; i++) {
      raw_putc(  7, i, RAW_LINE_VERT);
      raw_putc( 71, i, RAW_LINE_VERT);
 }

 raw_putc( 13, 2, RAW_TTEE);
 raw_putc( 19, 2, RAW_TTEE);
 raw_putc( 25, 2, RAW_TTEE);
 raw_putc( 37, 2, RAW_TTEE);
 raw_putc( 47, 2, RAW_TTEE);
 raw_putc( 58, 2, RAW_TTEE);

 raw_putc(  7, 3, RAW_LINE_VERT);
 raw_putc( 13, 3, RAW_LINE_VERT);
 raw_putc( 19, 3, RAW_LINE_VERT);
 raw_putc( 25, 3, RAW_LINE_VERT);
 raw_putc( 37, 3, RAW_LINE_VERT);
 raw_putc( 47, 3, RAW_LINE_VERT);
 raw_putc( 58, 3, RAW_LINE_VERT);
 raw_putc( 71, 3, RAW_LINE_VERT);

 raw_putc( 13, 4, RAW_CROSS);
 raw_putc( 19, 4, RAW_CROSS);

 raw_putc( 25, 4, RAW_CROSS);
 raw_putc( 37, 4, RAW_CROSS);
 raw_putc( 47, 4, RAW_CROSS);
 raw_putc( 58, 4, RAW_CROSS);

 raw_putc( 13, ll, RAW_BTEE);
 raw_putc( 19, ll, RAW_BTEE);
 raw_putc( 25, ll, RAW_BTEE);
 raw_putc( 37, ll, RAW_BTEE);
 raw_putc( 47, ll, RAW_BTEE);
 raw_putc( 58, ll, RAW_BTEE);

 term_disable_graf_map();

#elif defined(VCHE_NC)

 w2 = nc_box( 0, 0, vc_cols, vc_lines, VC_COLOR_BACKCYAN);

 nc_put_bottom_line( w2, bottom_line);

 wattrset( w2, VC_COLOR_BACKGRAY);

 ll = ( vc_cols - strlen( bottom_line)) / 2;
 mvwaddch( w2, vc_lines - 1, ll + 24, ACS_VLINE);
 mvwaddch( w2, vc_lines - 1, ll +  4, ACS_UARROW);
 mvwaddch( w2, vc_lines - 1, ll +  6, ACS_DARROW);
 mvwaddch( w2, vc_lines - 1, ll +  8, ACS_RARROW);
 mvwaddch( w2, vc_lines - 1, ll + 10, ACS_LARROW);

 wrefresh( w2);
 
 w = nc_box( 0, 0, vc_cols, vc_lines - 1, VC_COLOR_BACKCYAN);
 nc_do_title( w, " ASCII Table ");

 ll = vc_lines - 4;

 wattrset( w, VC_COLOR_BACKCYAN);
 
 mvwaddnstr( w, 3, 9, "Dec   Oct   Hex   Character   Meaning   C Escape     Binary", 59);

 mvwaddch( w,  2,  7, ACS_ULCORNER);
 mvwaddch( w,  2, 71, ACS_URCORNER);
 mvwaddch( w,  4,  7,     ACS_LTEE);
 mvwaddch( w,  4, 71,     ACS_RTEE);
 mvwaddch( w, ll,  7, ACS_LLCORNER);
 mvwaddch( w, ll, 71, ACS_LRCORNER);

 for( i = 1; i < 64; i++) {
      mvwaddch( w,  2, 7 + i, ACS_HLINE);
      mvwaddch( w,  4, 7 + i, ACS_HLINE);
      mvwaddch( w, ll, 7 + i, ACS_HLINE);
 }

 for( i = 5; i < vc_lines - 4; i++) {
      mvwaddch( w, i,  7, ACS_VLINE);
      mvwaddch( w, i, 71, ACS_VLINE);
 }

 mvwaddch( w, 2, 13, ACS_TTEE);
 mvwaddch( w, 2, 19, ACS_TTEE);
 mvwaddch( w, 2, 25, ACS_TTEE);
 mvwaddch( w, 2, 37, ACS_TTEE);
 mvwaddch( w, 2, 47, ACS_TTEE);
 mvwaddch( w, 2, 58, ACS_TTEE);

 mvwaddch( w, 3,  7, ACS_VLINE);
 mvwaddch( w, 3, 13, ACS_VLINE);
 mvwaddch( w, 3, 19, ACS_VLINE);
 mvwaddch( w, 3, 25, ACS_VLINE);
 mvwaddch( w, 3, 37, ACS_VLINE);
 mvwaddch( w, 3, 47, ACS_VLINE);
 mvwaddch( w, 3, 58, ACS_VLINE);
 mvwaddch( w, 3, 71, ACS_VLINE);

 mvwaddch( w, 4, 13, ACS_PLUS);

 mvwaddch( w, 4, 19, ACS_PLUS);
 mvwaddch( w, 4, 25, ACS_PLUS);
 mvwaddch( w, 4, 37, ACS_PLUS);
 mvwaddch( w, 4, 47, ACS_PLUS);
 mvwaddch( w, 4, 58, ACS_PLUS);

 mvwaddch( w, ll, 13, ACS_BTEE);
 mvwaddch( w, ll, 19, ACS_BTEE);
 mvwaddch( w, ll, 25, ACS_BTEE);
 mvwaddch( w, ll, 37, ACS_BTEE);
 mvwaddch( w, ll, 47, ACS_BTEE);
 mvwaddch( w, ll, 58, ACS_BTEE);

 wrefresh( w);

#endif


 j = 5;

 while( 1) {
        if( pos > endpos) {
#if defined(VCHE_RAW)
            term_disable_graf_map();
#endif
            term_beep();
            pos = endpos;
        }

        if( pos < 0) {
#if defined(VCHE_RAW)
            term_disable_graf_map();
#endif
            term_beep();
            pos = 0;
        }

#if defined(VCHE_VC)
        if( pos == 0) {
            vc_color_putc( 10,  4, ''   , VC_COLOR_BACKCYAN );
            vc_color_putc( 10, vc_lines - 4, '\x1F', VC_COLOR_ATTENTION);
        } else 
            if( pos == endpos) {
                vc_color_putc( 10,  4, '\x1E', VC_COLOR_ATTENTION);
                vc_color_putc( 10, vc_lines - 4, ''   , VC_COLOR_BACKCYAN );
            } else {
                vc_color_putc( 10,  4, '\x1E', VC_COLOR_ATTENTION);
                vc_color_putc( 10, vc_lines - 4, '\x1F', VC_COLOR_ATTENTION);
            }

        for( i = pos; i < pos + slines; i++) {
             if( j > vc_lines - 5)
                 j = 5;
             if( i < 256)
                 sprintf( buf, " %3d  %03o  %02X       %c       %s     %s     %s  ", i, i, i, i, (i == 127) ? " DEL ": i <= 32 ? meaning[ i]: "     ", int_to_escape( i), int_to_binary( i));
             else
                 sprintf( buf, "                                                  ");

                 vc_puts( 7, j, 65, buf);
             j++;
        }

#elif defined(VCHE_RAW)

        if( pos == 0) {
            term_color( VC_COLOR_BACKCYAN);
            /* I have to put it here, because a bug or feature of rxvt */
            term_enable_graf_map(); 
            raw_putc( 10,  4, RAW_LINE_HORIZ );

            term_color( VC_COLOR_ATTENTION);
            term_enable_graf_map();
            raw_putc( 10, vc_lines - 4, RAW_DARROW);
        } else 
            if( pos == endpos) {
                term_color( VC_COLOR_ATTENTION);
                term_enable_graf_map(); 
                raw_putc( 10,  4, RAW_UARROW);

                term_color( VC_COLOR_BACKCYAN);
                term_enable_graf_map(); 
                raw_putc( 10, vc_lines - 4, RAW_LINE_HORIZ);
            } else {
                term_color( VC_COLOR_ATTENTION);
                term_enable_graf_map(); 
                raw_putc( 10,              4, RAW_UARROW);
                raw_putc( 10, vc_lines - 4, RAW_DARROW);
            }

        for( i = pos; i < pos + slines; i++) {
             if( j > vc_lines - 5)
                 j = 5;

             if( i < 256) {
                 sprintf( buf, "%3d   %03o   %02X        %c        %s      %s      %s"
                               , i, i, i, ( i > 31) && ( i < 127) ? i : '.'
                               , (i == 127) ? " DEL ": i <= 32 ? meaning[ i]: "     "
                               , int_to_escape( i)
                               , int_to_binary( i));
                 term_color( VC_COLOR_BACKCYAN);
                 term_disable_graf_map();
                 raw_puts( 9, j, 60, buf);
             }

             term_enable_graf_map();
             raw_putc( 13, j, RAW_LINE_VERT);
             raw_putc( 19, j, RAW_LINE_VERT);
             raw_putc( 25, j, RAW_LINE_VERT);
             raw_putc( 37, j, RAW_LINE_VERT);
             raw_putc( 47, j, RAW_LINE_VERT);
             raw_putc( 58, j, RAW_LINE_VERT);
             j++;
        }

#elif defined(VCHE_NC)

        if( pos == 0) {
            nc_color_wputc( w, 10,         4, ACS_HLINE , VC_COLOR_BACKCYAN );
            nc_color_wputc( w, 10, vc_lines - 4, ACS_DARROW, VC_COLOR_ATTENTION);
        } else 
            if( pos == endpos) {
                nc_color_wputc( w, 10,         4, ACS_UARROW, VC_COLOR_ATTENTION);
                nc_color_wputc( w, 10, vc_lines - 4, ACS_HLINE , VC_COLOR_BACKCYAN );
            } else {
                nc_color_wputc( w, 10,         4, ACS_UARROW, VC_COLOR_ATTENTION);
                nc_color_wputc( w, 10, vc_lines - 4, ACS_DARROW, VC_COLOR_ATTENTION);
            }

        for( i = pos; i < pos + slines; i++) {
             if( j > vc_lines - 5)
                 j = 5;

             if( i < 256) {
                 sprintf( buf, "%3d   %03o   %02X        %c        %s      %s      %s"
                           , i, i, i, ( i > 31) && ( i < 127) ? i : '.'
                           , (i == 127) ? " DEL ": i <= 32 ? meaning[ i]: "     "
                           , int_to_escape( i)
                           , int_to_binary( i));
                 wattrset( w, VC_COLOR_BACKCYAN);
                 mvwaddnstr( w, j, 9, buf, 60);
             }

             mvwaddch( w, j, 13, ACS_VLINE);
             mvwaddch( w, j, 19, ACS_VLINE);
             mvwaddch( w, j, 25, ACS_VLINE);
             mvwaddch( w, j, 37, ACS_VLINE);
             mvwaddch( w, j, 47, ACS_VLINE);
             mvwaddch( w, j, 58, ACS_VLINE);
             j++;
        }
        wrefresh( w);

#endif

        c = key_get();

        switch( c) {
                case '\n':
                case 'q' :
                case 'Q' :
                case VC_KEY_F10  :
#if defined(VCHE_RAW)
                                   term_disable_graf_map();
#endif
#if defined(VCHE_NC)
                                   delwin( w);
                                   delwin( w2);
                                   refresh();
#endif
#if defined(VCHE_VC) || defined(VCHE_RAW)
                                   lib_load_screen();
#endif
                                   return;
                case VC_KEY_UP   :
                case VC_KEY_LEFT : pos--;
                                   break;
                case VC_KEY_DOWN :
                case VC_KEY_RIGHT: pos++;
                                   break;
                case VC_KEY_PGUP : pos -= slines;
                                   break;
                case VC_KEY_PGDN : pos += slines;
                                   break;
                case VC_KEY_HOME : pos = 0;
                                   break;
                case VC_KEY_END  : pos = endpos;
                                   break;
        } /* end switch( c) */
 } /* end while( 1) */
}

/*
 * Return the C language escape char
 */
static char *int_to_escape( int c)
{

 switch( c) {
         case  0: return "'\\0'";
         case  7: return "'\\a'";
         case  8: return "'\\b'";
         case  9: return "'\\t'";
         case 10: return "'\\n'";
         case 11: return "'\\v'";
         case 12: return "'\\f'";
         case 13: return "'\\r'";
         case 92: return "'\\\\'";
         default: return "    ";
 }

}

/*
 * Convert from integer to binary
 */
static char *int_to_binary( int c)
{
 static char bbuf[ 16];
 register char *bp = bbuf + 8;

 do {
     *--bp = (c % 2) ? '1' : '0';
 } while( (c /= 2) != 0);

 while( bp != bbuf)
        *--bp = '0';

 return bbuf;
}

