/* -*- C++ -*-

  This file is part of ViPEC
  Copyright (C) 1991-2000 Johan Rossouw (jrossouw@alcatel.altech.co.za)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <SymbolBar.h>

#include <Strings.h>
#include <ToolBar.h>
#include <MainWindow.h>
#include <ToolButton.h>

#include <Vccs.h>
#include <Gyrator.h>
#include <SignalGnd.h>
#include <SignalPort.h>
#include <Resistor.h>
#include <Inductor.h>
#include <InductorQ.h>
#include <Capacitor.h>
#include <TLin2Port.h>
#include <TLin4Port.h>
#include <TLinPhysical.h>
#include <CLin.h>
#include <Block1Port.h>
#include <Block2Port.h>

#include <qpoint.h>
#include <qpixmap.h>

#include <iostream.h>

SymbolBar* SymbolBar::instance_ = 0;

//-----------------------------------------------------------------
SymbolBar::SymbolBar(const QString & label, QMainWindow* mainWindow, QMainWindow::ToolBarDock dock, bool newLine)
  : QToolBar(label, mainWindow, dock, newLine, Strings::SymbolBarName),
    selectedSymbolName_("")
{
  ASSERT( instance_ == 0 );
  instance_ = this;
}

//-----------------------------------------------------------------
SymbolBar::~SymbolBar()
{
  instance_ = 0;
}

//-----------------------------------------------------------------
SymbolBar* SymbolBar::instance()
{
  return instance_;
}

//-----------------------------------------------------------------
void SymbolBar::initialize()
{
  QPoint p(0,0);
  registerComponent( Vccs( p ) );
  registerComponent( Gyrator( p ) );
  registerComponent( Resistor( p ) );
  registerComponent( Capacitor( p ) );
  registerComponent( Inductor( p ) );
  registerComponent( InductorQ( p ) );
  registerComponent( TLin2Port( p ) );
  registerComponent( TLin4Port( p ) );
  registerComponent( TLinPhysical( p ) );
  registerComponent( CLin( p ) );
  registerComponent( SignalPort( p ) );
  registerComponent( SignalGnd( p ) );
  registerComponent( Block1Port( p ) );
  registerComponent( Block2Port( p ) );

  QMap<QString, QStringList>::Iterator it;
  for( it = catagoryMap_.begin(); it != catagoryMap_.end(); ++it )
    {
      QString name = Strings::translate( it.key() );
      ToolBar::instance()->addComponentCatagory( name );
    }

}

//-----------------------------------------------------------------
void SymbolBar::registerComponent(const Component& component)
{
  const QPixmap& pixmap = component.getPixmap();
  QString catagory = Strings::translate( component.getCatagory() );
  const QString& name = component.getName();
  catagoryMap_[catagory].append(name);
  iconMap_[name] = &pixmap;
}

//-----------------------------------------------------------------
void SymbolBar::newComponentCatagorySelected(const QString& catagory)
{
  clear();
  buttonList_.clear();
  selectedSymbolName_ = "";

  QStringList& list = catagoryMap_[catagory];
  for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it )
    {
      const QPixmap* pixmap = iconMap_[*it];
      QString toolTip = Strings::translate( *it );
      ToolButton* t = new ToolButton( *pixmap, *it, toolTip, this );
      t->setUsesBigPixmap(TRUE);
      t->setToggleButton(true);
      
      buttonList_.append( t );

      connect( t, SIGNAL( pressed( ToolButton* ) ),
	       this, SLOT( symbolClicked( ToolButton* ) ) );
    }
}

//-----------------------------------------------------------------
void SymbolBar::symbolClicked( ToolButton* button )
{
  const QString& name = button->getName();
  if ( button->isOn() )
    {
      resetSymbols();
      selectedSymbolName_ = name;
      button->setOn( TRUE );
    }
  else
    {
      selectedSymbolName_ = "";
    }
}

//-----------------------------------------------------------------
void SymbolBar::resetSymbols()
{
  ToolButton* button;
  for ( button=buttonList_.first(); button != 0; button=buttonList_.next() )
    {
      button->setOn( FALSE );
    }
  selectedSymbolName_ = "";
}

//-----------------------------------------------------------------
QString SymbolBar::getSelectedSymbolName()
{
  QString name = selectedSymbolName_;
  resetSymbols();
  return name;
}
