/* @(#) im_histplot: plot a 1xany or anyx1 image file as a max x any or 
 * @(#) any x max graph using these rules:
 * @(#)
 * @(#) - unsigned char
 * @(#) 	always output 256 
 * @(#) - other unsigned integer types
 * @(#) 	output 0 - max
 * @(#) - signed int types
 * @(#) 	min moved to 0, max moved to max + min.
 * @(#) - float types
 * @(#) 	min moved to 0, max moved to any (square output)
 * @(#)
 * @(#) usage:
 * @(#)
 * @(#)  	int 
 * @(#)  	im_histplot( hist, histplot )
 * @(#)  	IMAGE *hist, *histplot;
 * @(#) 
 * @(#) Returns non-zero on error
 *
 * Copyright: 1990, N. Dessipris.
 *
 * Author: Nicos Dessipris.
 * Written on: 09/07/1990
 * Modified on : 12/03/1991
 * 20/6/95 JC
 *	- rules rationalised
 *	- im_lineprof removed
 *	- rewritten
 * 13/8/99 JC
 *	- rewritten again for partial, rules redone
 * 19/9/99 JC
 *	- oooops, broken for >1 band
 * 26/9/99 JC
 *	- oooops, graph float was wrong
 * 17/11/99 JC
 *	- oops, failed for all 0's histogram 
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Normalise an image using the rules noted above.
 */
static int
normalise( IMAGE *in, IMAGE *out )
{
	IMAGE *t1 = im_open_local( out, "im_histplot:2", "p" );
	double min, max;

	if( in->Coding != IM_CODING_NONE ) {
		im_errormsg( "im_histplot: uncoded only" );
		return( -1 );
	}
	if( im_iscomplex( in ) ) {
		im_errormsg( "im_histplot: non-complex only" );
		return( -1 );
	}

	if( im_isuint( in ) ) {
		/* Trivial case.
		 */
		if( im_copy( in, out ) )
			return( -1 );
	}
	else if( im_isint( in ) ) {
		/* Move min up to 0.
		 */
		if( !t1 || im_min( in, &max ) ||
			im_lintra( 1.0, in, -min, t1 ) )
			return( -1 );
	}
	else {
		/* Float image: scale min--max to 0--any. Output square
		 * graph.
		 */
		int any;

		if( in->Xsize == 1 )
			any = in->Ysize;
		else
			any = in->Xsize;

		if( !t1 || im_min( in, &min ) || im_max( in, &max ) ||
			im_lintra( any/(max - min), in, 
				-min*any/(max - min), out ) )
			return( -1 );
	}

	return( 0 );
}

/* Make an n-band image from n 1 band images.
 */
static int
ndup( IMAGE *in, IMAGE *out, int n )
{
	IMAGE **arr;
	int i;

	if( n < 1 ) {
		im_errormsg( "ndup: bad number of images" );
		return( - 1 );
	}
	if( !(arr = IM_ARRAY( out, n, IMAGE * )) )
		return( -1 );
	for( i = 0; i < n; i++ )
		arr[i] = in;
	return( im_gbandjoin( arr, out, n ) );
}

/* Plot image.
 */
static int
plot( IMAGE *in, IMAGE *out )
{
	IMAGE *t1 = im_open_local( out, "im_histplot:3", "p" );
	IMAGE *t2 = im_open_local( out, "im_histplot:4", "p" );
	IMAGE *t2a = im_open_local( out, "im_histplot:4a", "p" );
	IMAGE *t3 = im_open_local( out, "im_histplot:4c", "p" );
	IMAGE *t4 = im_open_local( out, "im_histplot:5", "p" );
	double max;
	int tsize;

	/* Find range we will plot.
	 */
	if( !t1 || !t2 || !t2a || !t4 || im_max( in, &max ) )
		return( -1 );
	if( max < 0 ) {
		im_errormsg( "im_histplot: internal error #8254" );
		return( -1 );
	}
	if( in->BandFmt == IM_BANDFMT_UCHAR )
		tsize = 256;
	else
		tsize = ceil( max );

	/* Make sure we don't make a zero height image.
	 */
	if( tsize == 0 )
		tsize = 1;

	if( in->Xsize == 1 ) {
		/* Vertical graph.
		 */
		if( im_zoom( in, t1, tsize, 1 ) ||
			im_fgrey( t2, tsize, in->Ysize ) ||
			ndup( t2, t2a, in->Bands ) ||
			im_copy( t2a, t3 ) )
			return( -1 );
	}
	else {
		/* Horizontal graph.
		 */
		if( im_zoom( in, t1, 1, tsize ) ||
			im_fgrey( t2, tsize, in->Xsize ) ||
			ndup( t2, t2a, in->Bands ) ||
			im_rot270( t2a, t3 ) )
			return( -1 );
	}

	/* Plot!
	 */
	if( im_lintra( tsize - 1, t3, 0.0, t4 ) ||
		im_more( t1, t4, out ) )
		return( -1 );

	return( 0 );
}

int 
im_histplot( IMAGE *hist, IMAGE *histplot )
{
	IMAGE *norm = im_open_local( histplot, "im_histplot:1", "p" );

	if( !norm )
		return( -1 );
	if( hist->Xsize != 1 && hist->Ysize != 1 ) {
		im_errormsg( "im_histplot: Xsize or Ysize not 1" );
		return( -1 );
	}

	/* Normalise input to a UCHAR.
	 */
	if( normalise( hist, norm ) )
		return( -1 );
	
	/* And plot!
	 */
	if( plot( norm, histplot ) )
		return( -1 );
	
	return( 0 );
}

