
// filecheck.h
// 
// Copyright (c) 1998-2002 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
// 
// original version written by Kosh Banerjee

/*********************************************************************
* This class is used to get information about an existing file where *
* "file" is used in its broadest definition: an ordinary file, a     *
* device file, a directory, an open socket, a file descriptor, and   *
* links.                                                             *
*********************************************************************/

/*********************************************************************
* Include guard.                                                     *
*********************************************************************/
#ifndef FILECHECK_H
#define FILECHECK_H

/*********************************************************************
* Required include files.                                            *
*********************************************************************/
#include <unistd.h>
#include <stdlib.h>
#include <climits>
#include <sys/types.h>
#include <sys/stat.h>
#include <cstdio>
#include <cstring> // Need to use strerror();
#include <string>
#include <ctime>
#include <cerrno>
#include "vbutil.h"

/*********************************************************************
* Using the standard namespace.                                      *
*********************************************************************/
using namespace std;

/*********************************************************************
* errno is used to check the success of the calls to stat() and      *
* fstat().                                                           *
*********************************************************************/
extern int errno;

/*********************************************************************
* NEW_LINE_LOC is the index of the new line character in the C-style *
* string returned by ctime().                                        *
*********************************************************************/
const int NEW_LINE_LOC = 24;

/*********************************************************************
* Class definition.                                                  *
*********************************************************************/
class FileCheck
{

  private:

/*********************************************************************
* DATA MEMBERS:                                                      *
*                                                                    *
* NAME                  TYPE          DESCRIPTION                    *
* ------------------------------------------------------------------ *
* fileName              string        The name of the file (when     *
*                                     appropriate).                  *
* fileInfo              struct stat   An instance of a stat struct,  *
*                                     used to hold the information   *
*                                     about the file.                *
* fd                    int           An open file descriptor, for   *
*                                     which file information is      *
*                                     desired. Used as appropriate.  *
* baseName              string        The basename of the file.      *
* dirName               string        The directory of the file.     *
* valid                 bool          Is set to true if fileName is  *
*                                     a valid file, false otherwise. *
* fileErrno             int           If fileName is not a valid     *
*                                     file, then errno is assigned   *
*                                     to fileErrno.                  *
* fileStrError          string        If fileName is not a valid     *
*                                     file, then strerror(fileErrno) *
*                                     is assigned to fileStrError.   *
* realPath            string          The actual path to the file,   *
*                                     as defined by realpath().      *
*********************************************************************/
    string fileName;
    struct stat fileInfo;
    int fd;
    string baseName;
    string dirName;
    bool valid;
    int fileErrno;
    string fileStrError;
    //    string realPath;

/*********************************************************************
* Private methods.                                                   *
*********************************************************************/
    void init();
    void init(const int fd);
    void copyFileInfo(const struct stat *fInfo);
    void copyFileInfo(const struct stat& fInfo);
    bool checkValidAndWarn() const;
		void statToString(const struct stat *fInfo) const;

  public:

/*********************************************************************
* Constructors.                                                      *
*********************************************************************/
    FileCheck(const char *fileName);
    FileCheck(const string& fileName);
    FileCheck(const int fd);
    FileCheck(const FileCheck &file); // Copy constructor.

/*********************************************************************
* Accessor method to fetch the file's name.                          *
*********************************************************************/
    string getFileName() const;

/*********************************************************************
* Accessor method to fetch the file's file descriptor.               *
*********************************************************************/
    int getFD() const;

/*********************************************************************
* Mutator methods to set the file's name.                            *
*********************************************************************/
    void setFileName(const char *fileName);
    void setFileName(const string& fileName);

/*********************************************************************
* Prototypes to determine the file's type.                           *
*********************************************************************/
    bool isRegularFile() const;
    bool isDirectory() const;
    bool isCharacterSpecialDevice() const;
    bool isBlockSpecialDevice() const;
    bool isLink() const;
    bool isFIFO() const;
    bool isSocket() const;

/*********************************************************************
* Prototypes for fetching the various fields of this->fileInfo.      *
*********************************************************************/
    const char *getLastAccess() const;
    const char *getLastModification() const;
    const char *getLastINodeChange() const;
    // ino_t getINodeNumber() const;
    //uid_t getOwnerUserID() const;
    //gid_t getOwnerGroupID() const;
    off_t getFileSize() const;

/*********************************************************************
* This method returns the basename of the file.                      *
*********************************************************************/
    inline string getBaseName() const
    {
      return this->baseName;
    } // inline string getBaseName() const

/*********************************************************************
* This method returns the directory name of the file.                *
*********************************************************************/
    inline string getDirName() const
    {
      return this->dirName;
    } // inline string getDirName() const

/*********************************************************************
* This accessor method returns this->valid.                          *
*********************************************************************/
    inline bool getValid() const
    {
      return this->valid;
    } // inline bool getValid() const

/*********************************************************************
* This accessor method returns this->fileErrno.                      *
*********************************************************************/
    inline int getFileErrno() const
    {
      return this->fileErrno;
    } // inline int getFileErrno() const

/*********************************************************************
* This accessor method returns this->fileStrError.                   *
*********************************************************************/
    inline const string getFileStrErr() const
    {
      return this->fileStrError;
    } // inline const string getFileStrErr() const

/*********************************************************************
* This accessor method returns this->realPath.                       *
*********************************************************************
    inline const string getRealPath() const
    {
      return this->realPath;
    } // inline const string getRealPath() const
*/

}; // class FileCheck

#endif // FILECHECK_H
