
// myDB.cpp
// Member functions in myEnv and myDB classes
// Copyright (c) 1998-2010 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
// 
// original version written by Dongbo Hu

using namespace std;

#include "myDB.h"
#include "typedefs.h"
#include "vbutil.h"
#include <arpa/inet.h>
#include <cstdlib>
#include <cstring>

/******************************************************
 * Member functions in myEnv class
 ******************************************************/
const uint32 myEnv::log_size = 10 * 1024 * 1024;

// Default ctor does nothing
myEnv::myEnv() : env(DB_CXX_NO_EXCEPTIONS), stat(false)
{

}

// Ctor that accepts env home directory
myEnv::myEnv(const string& envName) : 
  env(DB_CXX_NO_EXCEPTIONS), envDir(envName), stat(false)
{
  open();
}

int myEnv::open(const string& envName)
{
  envDir = envName;
  return open();
}

/* Open db enviroment and a certain db.
 * Returns 0 if everything is ok;
 * returns an error code otherwise. */
int myEnv::open()
{
  if (stat)
    return 0;

  u_int32_t env_flags = DB_CREATE |  // If environment does not exist, create it.
    DB_INIT_LOCK  |                  // Initialize locking
    DB_INIT_LOG   |                  // Initialize logging
    DB_INIT_MPOOL |                  // Initialize the cache
    DB_INIT_TXN   |                  // Initialize transactions
    //DB_RECOVER    |
    DB_THREAD;                       // Feee-thread env handle

  env.set_error_stream(&cerr);                  // redirect debugging info to cerr
  env.set_encrypt("fyeo2006", DB_ENCRYPT_AES);  // hard-coded pass-phrase for encryption
  env.log_set_config(DB_LOG_IN_MEMORY, 1);      // enable in-memory logging
  env.set_lg_bsize(log_size);                   // set in-memory log size

  int foobar = env.open(envDir.c_str(), env_flags, 0);
  if (foobar) {
    cout << "Error open db enviroment: " << envDir << endl;
    return foobar;
  }

  stat = true;
  return 0;
}

/* Close db enviroment and a certain db.
 * Returns 0 if everything is ok;
 * returns an error code otherwise. */
int myEnv::close()
{
  if (!stat)
    return 0;

  int foo = env.close(0);
  if (foo) {
    cout << "Error closing db enviroment" << endl;
    return foo;
  }

  stat = false;
  return 0;
}

/******************************************************
 * Member functions in myDB class
 ******************************************************/
// Default ctor 
myDB::myDB() : dbp(NULL), stat(false)
{

}

// Ctor that builds a transactional db (requires db filename, env and some flags)
myDB::myDB(const string& inputName, myEnv& env_in, key_comp keyCompare, dup_flag dupsort, DBTYPE myType)
  : dbp(new Db(&env_in.getEnv(), DB_CXX_NO_EXCEPTIONS)),
    filename(inputName),
    stat(false) 
{
  /* Note that db-specified pass-phrase is not allowed in env, so the following line is illegal. */
  // dbp->set_encrypt("fyeo2006", DB_ENCRYPT_AES); 
  dbp->set_flags(DB_ENCRYPT);

  // duplicate key setup
  if (dupsort != no_dup) {
    dbp->set_flags(DB_DUPSORT);
    if (dupsort == sort_int)
      dbp->set_dup_compare(dupsort_int);
  }

  // If keyCompare is cmp_int and db is BTREE, compare keys by integer value.
  if (keyCompare == cmp_int && myType == DB_BTREE)
    dbp->set_bt_compare(compare_int);
    
  u_int32_t openFlag = DB_CREATE | DB_AUTO_COMMIT; // db open flags
  if (dbp->open(NULL, filename.c_str(), NULL, myType, openFlag, 0))
    cout << "Error opening database file: " << filename << endl;

  stat = true; // db open is successful  
}

/* Ctor that builds a stand-alone db table (does not require any 
 * enviroment parameter, but needs an additional openFlag to open db. */
myDB::myDB(const string& dbFile, key_comp keyCompare, dup_flag dupsort, 
	   u_int32_t openFlag, DBTYPE myType)
  : dbp(new Db(NULL, DB_CXX_NO_EXCEPTIONS)),   
    filename(dbFile), 
    stat(false)
{
  dbp->set_error_stream(&std::cerr);
  dbp->set_encrypt("fyeo2006", DB_ENCRYPT_AES); // hard-coded pass-phrase
  dbp->set_flags(DB_ENCRYPT); // must be called AFTER set_encrypt()!!!

  // duplicate key setup
  if (dupsort != no_dup) {
    dbp->set_flags(DB_DUPSORT);
    if (dupsort == sort_int)
      dbp->set_dup_compare(dupsort_int);
  }

  // If keyCompare is cmp_int and db is BTREE, compare keys by integer value.
  if (keyCompare == cmp_int && myType == DB_BTREE)
    dbp->set_bt_compare(compare_int);
    
  if (dbp->open(NULL, filename.c_str(), NULL, myType, openFlag, 0))
    cout << "Error opening database file: " << filename << endl;

  stat = true; // db open is successful  
}

// Initialize some flags before opening db w/o environment
void myDB::init()
{
  if (stat)
    return;

  dbp = new Db(NULL, DB_CXX_NO_EXCEPTIONS); 
  dbp->set_error_stream(&std::cerr);
  dbp->set_encrypt("fyeo2006", DB_ENCRYPT_AES); // hard-coded pass-phrase
  dbp->set_flags(DB_ENCRYPT); // must be called AFTER set_encrypt()!!!
}

/* Opens db in an environment. 
 * This function should be called if db is contructed by default ctor. */ 
int myDB::open(const string& inputName, myEnv& env_in, key_comp keyCompare, dup_flag dupsort, DBTYPE myType)
{
  if (stat)
    return 0;

  filename = inputName;
  dbp = new Db(&env_in.getEnv(), DB_CXX_NO_EXCEPTIONS);
  // Note that db-specified pass-phrase not allowed in env, so the following line is not allowed:
  // dbp->set_encrypt("fyeo2006", DB_ENCRYPT_AES); 
  dbp->set_flags(DB_ENCRYPT);

  // duplicate key setup
  if (dupsort != no_dup) {
    dbp->set_flags(DB_DUPSORT);
    if (dupsort == sort_int)
      dbp->set_dup_compare(dupsort_int);
  }

  // If keyCompare is cmp_int and db is BTREE, compare keys by integer value.
  if (keyCompare == cmp_int && myType == DB_BTREE)
    dbp->set_bt_compare(compare_int);
    
  u_int32_t openFlag = DB_CREATE | DB_AUTO_COMMIT; // db open flags
  int foo = dbp->open(NULL, filename.c_str(), NULL, myType, openFlag, 0);
  if (foo) {
    cout << "Error opening database file: " << filename << endl;
    return foo;
  }

  stat = true; // db open is successful  
  return 0;
}

/* Opens db w/o environment.
 * This function should be called if db is contructed by default ctor. */
int myDB::open(const string& inputName, key_comp keyCompare, dup_flag dupsort, 
	       u_int32_t openFlag, DBTYPE myType)
{
  if (stat)
    return 0;

  filename = inputName;
  dbp = new Db(NULL, DB_CXX_NO_EXCEPTIONS); 
  dbp->set_error_stream(&std::cerr);
  dbp->set_encrypt("fyeo2006", DB_ENCRYPT_AES); 
  dbp->set_flags(DB_ENCRYPT);

  // duplicate key setup
  if (dupsort != no_dup) {
    dbp->set_flags(DB_DUPSORT);
    if (dupsort == sort_int)
      dbp->set_dup_compare(dupsort_int);
  }

  // If keyCompare is cmp_int and db is BTREE, compare keys by integer value.
  if (keyCompare == cmp_int && myType == DB_BTREE)
    dbp->set_bt_compare(compare_int);
    
  int foo = dbp->open(NULL, filename.c_str(), NULL, myType, openFlag, 0);
  if (foo) {
    cout << "Error opening database file: " << filename << endl;
    return foo;
  }

  stat = true; // db open is successful  
  return 0;
}

/* Member function that closes db, called from the class destructor. */
int myDB::close()
{
  // Do nothing if dbp is not open
  if (!stat)
    return 0;

  // Close the db
  int foo =  dbp->close(0);
  if (foo) {
    cout << "Error closing db file: " << filename << endl;
    return foo;
  }

  stat = false;
  return 0;
}

/* This function returns the number of records in database, 
 * used to generate keys automatically. */
int32 myDB::countRec()
{
  int32 length = 0;
  // Get a cursor to the patient db
  Dbc *cursorp;
  getDb().cursor(NULL, &cursorp, 0);
  Dbt key, data;
  while (cursorp->get(&key, &data, DB_NEXT) == 0)
    length++;

  cursorp->close();
  return length;
}

/* This function compares keys arithmetically instead of lexically.
 * It is called when the db key is an int32 integer and we want to 
 * rank the records according to the key value. 
 * This function is copied from bdb "Getting Started" doc. */
int compare_int(Db*, const Dbt* dbt1, const Dbt* dbt2)
{
  int32 l, r;
  memcpy(&l, dbt1->get_data(), sizeof(int32)); 
  memcpy(&r, dbt2->get_data(), sizeof(int32)); 

  if (ntohs(1) == 1) {
    swap(&l);
    swap(&r);
  }

  return l - r;
}

/* Another compare_int function that does exactly the same thing as the previous one,
 * but according to /usr/include/db_cxx.h, this definition is deprecated already. 
 * DB and DBT should be replaced by Db and Dbt. */
int compare_int_deprc(DB*, const DBT* dbt1, const DBT* dbt2)
{
  int32 l = *(int32*) dbt1->data;
  int32 r = *(int32*) dbt2->data;

  if (ntohs(1) == 1) {
    swap(&l);
    swap(&r);
  }

  return l - r;
}

/* This function compares keys arithmetically instead of lexically.
 * It is called when the db key is a string, but key/data pair should be sorted by 
 * numeric value of the int32 integer that is converted from this string.
 * Once again, note that DB and DBT are both deprecated and should be replaced by Db and Dbt. */
int compare_str(DB*, const DBT* dbt1, const DBT* dbt2)
{
  int32 l = atol((const char*) dbt1->data);
  int32 r = atol((const char*) dbt2->data);

  if (ntohs(1) == 1) {
    swap(&l);
    swap(&r);
  }

  return l - r;
}

/* This function is used to sort duplicate keys in db table.
 * It is almost identical to compare_int, except that this function 
 * puts data that has larger value before the one that has smaller value. */
int dupsort_int(Db*, const Dbt* dbt1, const Dbt* dbt2)
{
  int32 l, r;
  memcpy(&l, dbt1->get_data(), sizeof(int32)); 
  memcpy(&r, dbt2->get_data(), sizeof(int32)); 

  if (ntohs(1) == 1) {
    swap(&l);
    swap(&r);
  }

  return r - l;
}
