
// gheaderinfo.cpp
// Copyright (c) 1998-2010 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
// 
// original version written by Dongbo Hu

#include "gheaderinfo.h"
#include <qstring.h>
#include <qmessagebox.h>

/* Basic constructor */
gHeaderInfo::gHeaderInfo()
{
  init();
}

/* This constructor takes VBMatrix as an argument */
gHeaderInfo::gHeaderInfo(VBMatrix inputG)
{
  init();
  getInfo(inputG);
}

/* Destructor */
gHeaderInfo::~gHeaderInfo()
{

}

/* Initialize parameters */
void gHeaderInfo::init()
{
  TR = -1;
  rowNum = colNum = 0;
  sampling = 0;
  condfxn="";
  condStat = false;
  condKey.clear();
  typeList.clear();  
  nameList.clear();
}

/* Collect TR, sampling, totalReps, condition function information 
 * because the .G and .preG files generated by Geoff's code doesn't include TR and sampling 
 * rate information, we cannot simply assume that the third line will hold TR and the 4th 
 * line will hold sampling rate value. It is only available in my header files.
 * The header is checked line by line for TR and sampling rate value. 
 * This is for compatibility of different header format between mine and Geoff's. 
 * When the header is eventually fixed, we can check a certain line instead of each line. */
void gHeaderInfo::getInfo(VBMatrix inputG)
{
  vector<string>tmpHeader = inputG.header;
  QString qLine, key, name, type;
  const char * charLine = 0;
  for (unsigned i = 0; i < tmpHeader.size(); i++) {
    if ((tmpHeader[i]).substr(0, 7) == "TR(ms):") {
      charLine = (tmpHeader[i]).data();
      qLine = charLine;
      TR = (qLine.right(qLine.length() - 9)).toInt();
    } 
    else if ((tmpHeader[i]).substr(0, 13) == "Sampling(ms):") {
      charLine = (tmpHeader[i]).data();
      qLine = charLine;
      sampling = (qLine.right(qLine.length() - 14)).toInt();
    }
    else if ((tmpHeader[i]).substr(0, 14) == "ConditionFile:") {
      charLine = (tmpHeader[i]).data();
      qLine = charLine;      
      condfxn = (string) (qLine.right(qLine.length() - 15)).ascii();
    }
    else if ((tmpHeader[i]).substr(0, 13) == "ConditionKey:") {
      charLine = (tmpHeader[i]).data();
      qLine = charLine;
      key = qLine.section('\t', 2, 2);
      condKey.Add(key.ascii());
    }
    else if ((tmpHeader[i]).substr(0, 10) == "Condition:") {
      charLine = (tmpHeader[i]).data();
      qLine = charLine;
      key = qLine.section('\t', 2, 2);
      condKey.Add(key.ascii());
    }
    else if ((tmpHeader[i]).substr(0, 10) == "Parameter:") { 
      charLine = (tmpHeader[i]).data();
      qLine = charLine;
      type = qLine.section('\t', 2, 2).left(1);
      name = qLine.section('\t', 3, 3);
      typeList.push_back(type.ascii());
      nameList.push_back(name.ascii());
    }
  }
  tmpHeader.clear();
  rowNum = inputG.m, colNum = inputG.n;
}

/* Check if the information in header is consistent or not */
bool gHeaderInfo::chkInfo(bool preG)
{
  if (typeList.size() != colNum || nameList.size() != colNum) {
    QMessageBox::critical(0, "G matrix Error", QString("The number of covariates in the header (%1) does \
not match the actual number of columns (%2)").arg(typeList.size()).arg(colNum));
    return false;
  }
  if (TR && sampling && TR % sampling != 0) {
    QMessageBox::critical(0, "G matrix Error", QString("TR in the header (%1) is not a multiple of \
sampling rate (%2)").arg(TR).arg(sampling));
    return false;
  }

  return chkCondfxn(preG);
}

/* Make sure condition function is in good shape */
bool gHeaderInfo::chkCondfxn(bool preG)
{
  if (!condfxn.length())
    return true;

  tokenlist condKeyInFile;
  condVector = new VB_Vector();
  int refStat = getCondVec(condfxn.c_str(), condKeyInFile, condVector);
  QString tmpString(condfxn.c_str());
  if (refStat == -1) { // Quit if the file isn't readable
    QMessageBox::warning(0, "Warning!", "Condition function in G matrix header not readable: " 
			  + tmpString);
    return true;
  }

  if (refStat == -2) {
    QMessageBox::critical(0, "Error!",  "Condition function has different number \
of keys in header and content: " + tmpString);
    return false;
  }

  if (refStat == 1) {
    QMessageBox::critical(0, "Error!", "Condition function defined in the header has different \
keys in header and content: " + tmpString); 
    return false;
  }

  if (condKey.size() != condKeyInFile.size()) {
    QMessageBox::critical(0, "Error!", QString("Condition function (%1) defined in the header has %2 condition \
keys, but there are %3 Condition lines").arg(condfxn.c_str()).arg(condKeyInFile.size()).arg(condKey.size()));
    return false;
  }

  // stop checking condition function if it is a preG file
  if (preG) {
    condStat = true;
    return true;
  }

  int condLen = condVector->getLength();
  // Make sure the number of elements in condition function is a multiple of total time points
  if (condLen % rowNum != 0) {
    QMessageBox::critical(0, "Error!", 
			  "Number of elements in condition function must be a multiple of \
total number of time points.");
    return false;
  }

  // Make sure condition function can be upsampled
  if (sampling && (TR / sampling) % (condLen / rowNum) != 0) {
    QString helpStr = "G matrix header error: Condition function can not be upsampled with the sampling rate.";
    QMessageBox::critical(0, "Error!", helpStr);
    return false;
  }

  condStat = true;
  return true;
}

/* Check G matrix file validity */
bool gHeaderInfo::read(const string& inputName, bool preG)
{
  if (!chkFileStat(inputName.c_str(), true))
    return false;

  string gName = inputName;  // previously converted to absolute path, but why???
  if (gMatrix.ReadMAT1Header(gName) || gMatrix.ReadMAT1(gName)) {
    QMessageBox::critical(0, "Error", "Invalid G matrix file format:" + QString(gName.c_str()));
    return false;
  }
  if (!gMatrix.m) {
    QMessageBox::critical(0, "Error", "G matrix file does not have any rows:" + QString(gName.c_str()));
    return false;
  }
  if (!gMatrix.n) {
    QMessageBox::critical(0, "Error", "G matrix file does not have any columns:" + QString(gName.c_str()));
    return false;
  }

  getInfo(gMatrix);
  if (!chkInfo(preG))
    return false;

  return true;
}

/* chkFileStat() makes sure a certain file exists and is regular file and readable.
 * If qtFlag is true, it also pops up a QT message box to tell the user check status */
bool chkFileStat(const char * inputName, bool qtFlag)
{
  FileCheck myCheck(inputName);
  if (access(inputName,R_OK)) {
    if (qtFlag)
      QMessageBox::critical(0, "Error", "File not exist: " + QString(inputName));
    return false;
  }
  return true;
}

