/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "htmlreportwriter.h"

#include "linklocator.h"

HtmlReportWriter::HtmlReportWriter() :
    m_row( 0 )
{
}

HtmlReportWriter::~HtmlReportWriter()
{
}

void HtmlReportWriter::begin( const QString& generator, const QString& title )
{
    m_lines.append( "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">" );
    m_lines.append( "<html xmlns=\"http://www.w3.org/1999/xhtml\">" );
    m_lines.append( "<head>" );
    m_lines.append( QString( "<title>%1</title>" ).arg( title ) );
    m_lines.append( "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\" />" );
    m_lines.append( QString( "<meta name=\"generator\" content=\"%1\" />" ).arg( generator ) );
    m_lines.append( "<style type=\"text/css\">" );
    m_lines.append( "body { font-family: sans-serif; }" );
    m_lines.append( "p { margin: 0.2em 0; }" );
    m_lines.append( "table { border-collapse: collapse; margin-top: 1em; }" );
    m_lines.append( "table.tb, table.tb td, table.tb th { border-style: solid; border-color: #888888; border-width: 0.1em; }" );
    m_lines.append( "table.tb td, table.tb th { padding: 0.2em 0.4em; }" );
    m_lines.append( "table.tp { margin-top: 0.6em; }" );
    m_lines.append( "table.tp td { padding: 0.1em 0em }" );
    m_lines.append( "table.th td { padding: 0.4em 0.4em; }" );
    m_lines.append( "p.footer { font-size: 0.8em; }" );
    m_lines.append( "</style>" );
    m_lines.append( "</head>" );
    m_lines.append( "<body>" );
    m_lines.append( QString( "<h1>%1</h1>" ).arg( title ) );
}

void HtmlReportWriter::end()
{
    m_lines.append( "</body>" );
    m_lines.append( "</html>" );
}

void HtmlReportWriter::writeHeader2( int id, const QString& text )
{
    QString anchor = QString( "<a name=\"#id%1\" />" ).arg( id );

    m_lines.append( QString( "<h2>%1%2</h2>" ).arg( anchor, text ) );
}

void HtmlReportWriter::writeHeader3( const QString& text )
{
    m_lines.append( QString( "<h3>%1</h3>" ).arg( text ) );
}

void HtmlReportWriter::writeParagraph( const QString& text )
{
    m_lines.append( QString( "<p>%1</p>" ).arg( text ) );
}

void HtmlReportWriter::beginSummary()
{
    m_lines.append( "<table class=\"tp\" cellspacing=\"0\" cellpadding=\"0\">" );
}

void HtmlReportWriter::writeSummaryItem( const QString& item, const QString& value )
{
    m_lines.append( QString( "<tr><td>%1</td><td>&nbsp;&nbsp;&nbsp;</td><td>%2</td></tr>" ).arg( item, value ) );
}

void HtmlReportWriter::endSummary()
{
    m_lines.append( "</table>" );
}

void HtmlReportWriter::beginTable()
{
    m_lines.append( "<table class=\"tb\" cellspacing=\"0\" cellpadding=\"4\" border=\"2\">" );

    m_row = 0;
}

void HtmlReportWriter::writeTableHeader( const QStringList& cells )
{
    m_lines.append( "<tr bgcolor=\"#dddddd\">" );
    m_lines.append( QString( "<th>%1</th>" ).arg( cells.join( "</th>\r\n<th>" ) ) );
    m_lines.append( "</tr>" );
}

void HtmlReportWriter::writeTableRow( int id, const QStringList& cells )
{
    QString anchor = QString( "<a name=\"#id%1\" />" ).arg( id );

    m_lines.append( ( m_row % 2 ) ? "<tr bgcolor=\"#eeeeee\">" : "<tr>" );
    m_lines.append( QString( "<td>%1%2</td>" ).arg( anchor, cells.join( "</td>\r\n<td>" ) ) );
    m_lines.append( "</tr>" );

    m_row++;
}

void HtmlReportWriter::endTable()
{
    m_lines.append( "</table>" );
}

void HtmlReportWriter::writeFooter( const QString& text )
{
    m_lines.append( "<br />" );
    m_lines.append( QString( "<p class=\"footer\">%1</p>" ).arg( text ) );
}

void HtmlReportWriter::writeRawHtml( const QString& text )
{
    m_lines.append( text );
}

QString HtmlReportWriter::name( const QString& text )
{
    return QString ( "<strong>%1</strong>" ).arg( LinkLocator::escape( text ) );
}

QString HtmlReportWriter::date( const QString& text )
{
    return QString ( "<strong>%1</strong>" ).arg( text );
}

QString HtmlReportWriter::item( const QString& text )
{
    return LinkLocator::escape( text );
}

QString HtmlReportWriter::value( const QString& text )
{
    return LinkLocator::convertToHtml( text, LinkLocator::NoInternalLinks );
}

QString HtmlReportWriter::titleName( const QString& text )
{
    return LinkLocator::escape( text );
}

QString HtmlReportWriter::result() const
{
    return m_lines.join( "\r\n" );
}
