/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "dashboardview.h"

#include <QTreeWidget>
#include <QHeaderView>
#include <QSettings>

#include "commands/commandmanager.h"
#include "data/datamanager.h"
#include "data/updatebatch.h"
#include "data/updateevent.h"
#include "models/folderwatchprovider.h"
#include "views/viewevent.h"
#include "viewmanager.h"
#include "connectionmanager.h"
#include "configdata.h"

DashboardView::DashboardView( QObject* parent, QWidget* parentWidget ) : View( parent ),
    m_selectedFolderId( 0 ),
    m_supressSelection( false ),
    m_activateReason( 0 )
{
    m_tree = new QTreeWidget( parentWidget );

    m_tree->setRootIsDecorated( false );
    m_tree->setItemsExpandable( false );
    m_tree->setUniformRowHeights( true );
    m_tree->header()->setMovable( false );

    m_provider = new FolderWatchProvider( this );
    m_provider->setupColumns( m_tree );

    QSettings* settings = connectionManager->serverSettings();
    settings->beginGroup( "Dashboard/Widths" );

    for ( int i = 0; i < m_tree->columnCount() - 1; i++ ) {
        int width = settings->value( QString( "Column%1" ).arg( i ) ).toInt();
        if ( width > 0 )
            m_tree->setColumnWidth( i, width );
    }

    settings->endGroup();

    connect( m_tree->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
        this, SLOT( selectionChanged() ) );
    connect( m_tree, SIGNAL( activated( const QModelIndex& ) ), this, SLOT ( activated( const QModelIndex& ) ) );

    m_tree->installEventFilter( this );
    m_tree->viewport()->installEventFilter( this );

    setMainWidget( m_tree );

    setAccess( NormalAccess );
}

DashboardView::~DashboardView()
{
    QSettings* settings = connectionManager->serverSettings();
    settings->beginGroup( "Dashboard/Widths" );

    for ( int i = 0; i < m_tree->columnCount() - 1; i++ ) {
        int width = m_tree->columnWidth( i );
        settings->setValue( QString( "Column%1" ).arg( i ), width );
    }

    settings->endGroup();
}

void DashboardView::initialUpdate()
{
}

void DashboardView::updateEvent( UpdateEvent* e )
{
    if ( e->unit() == UpdateEvent::Projects ) {
        m_provider->updateAll();
        populateDashboard();
        updateWatchedFolders();
    } else if ( e->unit() == UpdateEvent::Folder ) {
        viewManager->postViewEvent( NULL, ViewEvent::RecalculateWatches, e->id() );
    } else if ( e->unit() == UpdateEvent::Issue ) {
        const IssueRow* issue = dataManager->issues()->find( e->id() );
        if ( issue )
            viewManager->postViewEvent( NULL, ViewEvent::RecalculateWatches, issue->folderId() );
    }
}

void DashboardView::viewEvent( ViewEvent* e )
{
    switch ( e->action() ) {
        case ViewEvent::RecalculateWatches:
            m_provider->updateFolder( e->id() );
            populateDashboard();
            updateWatchedFolder( e->id() );
            break;

        case ViewEvent::UpdateFilters:
            m_provider->updateFolders( e->id() );
            populateDashboard();
            break;

        case ViewEvent::UpdateSettings:
            m_provider->updateAll();
            populateDashboard();
            break;

        default:
            break;
    }
    View::viewEvent( e );
}

bool DashboardView::eventFilter( QObject* obj, QEvent* e )
{
    if ( obj == m_tree || obj == m_tree->viewport() ) {
        QEvent::Type type = e->type();
        if ( type == QEvent::MouseButtonPress || type == QEvent::MouseButtonDblClick || type == QEvent::KeyPress )
            m_activateReason = type;
    }

    return View::eventFilter( obj, e );
}

void DashboardView::updateWatchedFolders()
{
    UpdateBatch* batch = NULL;

    RDB::IndexConstIterator<FolderRow> it( dataManager->folders()->index() );
    while ( it.next() ) {
        int folderId = it.key();
        if ( dataManager->folderWatchState( folderId ) != NotWatched ) {
            if ( dataManager->folderUpdateNeeded( folderId ) ) {
                if ( !batch ) {
                    batch = new UpdateBatch( -20 );
                    batch->setIfNeeded( true );
                }
                batch->updateFolder( folderId );
            }
        }
    }

    if ( batch )
        commandManager->execute( batch );
}

void DashboardView::updateWatchedFolder( int folderId )
{
    if ( dataManager->folderWatchState( folderId ) != NotWatched ) {
        if ( dataManager->folderUpdateNeeded( folderId ) ) {
            UpdateBatch* batch = new UpdateBatch( -20 );
            batch->setIfNeeded( true );
            batch->updateFolder( folderId );
            commandManager->execute( batch );
        }
    }
}

void DashboardView::populateDashboard()
{
    int folderId = m_selectedFolderId;
    QString filter = m_selectedFilter;

    m_supressSelection = true;

    m_provider->populateTree( m_tree );

    setSelectedWatch( folderId, filter );

    m_supressSelection = false;
}

void DashboardView::setSelectedWatch( int folderId, const QString& filter )
{
    if ( folderId != 0 ) {
        QAbstractItemModel* model = m_tree->model();
        for ( int i = 0; i < model->rowCount(); i++ ) {
            QModelIndex projectIndex = model->index( i, 0 );
            for ( int j = 0; j < model->rowCount( projectIndex ); j++ ) {
                QModelIndex folderIndex = model->index( j, 0, projectIndex );
                if ( folderIndex.data( FolderWatchProvider::FolderIdRole ).toInt() == folderId ) {
                    QModelIndex index;
                    if ( filter.isEmpty() ) {
                        index = folderIndex;
                    } else {
                        for ( int k = 0; k < model->rowCount( folderIndex ); k++ ) {
                            QModelIndex filterIndex = model->index( k, 0, folderIndex );
                            if ( filterIndex.data( FolderWatchProvider::FilterNameRole ).toString() == filter ) {
                                index = filterIndex;
                                break;
                            }
                        }
                    }
                    if ( index.isValid() ) {
                        m_tree->selectionModel()->setCurrentIndex( index, QItemSelectionModel::Current );
                        m_tree->selectionModel()->select( index, QItemSelectionModel::ClearAndSelect | QItemSelectionModel::Rows );
                    }
                    break;
                }
            }
        }
    }

    if ( m_selectedFolderId != folderId || m_selectedFilter != filter ) {
        m_supressSelection = true;

        m_tree->clearSelection();

        m_supressSelection = false;
    }
}

void DashboardView::selectionChanged()
{
    if ( m_supressSelection )
        return;

    QModelIndexList selection = m_tree->selectionModel()->selectedRows();
    if ( !selection.isEmpty() ) {
        QModelIndex index = selection.at( 0 );
        m_selectedFolderId = index.data( FolderWatchProvider::FolderIdRole ).toInt();
        m_selectedFilter = index.data( FolderWatchProvider::FilterNameRole ).toString();
    } else {
        m_selectedFolderId = 0;
        m_selectedFilter.clear();
    }

    emit selectedWatchChanged( m_selectedFolderId, m_selectedFilter );
}

void DashboardView::activated( const QModelIndex& index )
{
    if ( index.isValid() && ( m_activateReason != QEvent::MouseButtonPress || configData->windowLayout() == LayoutSinglePane ) ) {
        QModelIndex firstIndex = index.sibling( index.row(), 0 );

        int folderId = firstIndex.data( FolderWatchProvider::FolderIdRole ).toInt();
        QString filter = firstIndex.data( FolderWatchProvider::FilterNameRole ).toString();

        if ( folderId != 0 )
            viewManager->openFolderView( folderId, filter );
    }
}
