// Timer.cc --- break timer
//
// Copyright (C) 2001, 2002, 2003, 2004 Rob Caelers <robc@krandor.org>
// All rights reserved.
//
// Time-stamp: <2004-07-01 19:48:22 robc>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//

static const char rcsid[] = "$Id: Timer.cc,v 1.4 2004/07/01 18:32:05 rcaelers Exp $";

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.hh"

#include <sstream>
#include <stdio.h>
#include <math.h>

#include "Timer.hh"

#include "TimePredFactory.hh"
#include "TimePred.hh"
#include "TimeSource.hh"

#include "timeutil.h"


//! Constructs a new break timer.
/*!
 *  \param time_source this timer will obtain the current time from this
 *         source of time.
 */
Timer::Timer(TimeSource *time_source) :
  activity_timer(true),
  timer_enabled(false),
  timer_frozen(false),
  activity_state(ACTIVITY_UNKNOWN),
  timer_state(STATE_INVALID),
  previous_timer_state(STATE_INVALID),
  snooze_interval(60),
  snooze_on_active(true),
  snooze_inhibited(false),
  limit_enabled(true),
  limit_interval(600),
  autoreset_enabled(true),
  autoreset_interval(120),
  autoreset_interval_predicate(NULL),
  restore_enabled(true),
  elapsed_time(0),
  elapsed_idle_time(0),
  last_limit_time(0),
  last_limit_elapsed(0),
  last_start_time(0),
  last_reset_time(0),
  last_stop_time(0),
  next_reset_time(0),
  last_pred_reset_time(0),
  next_pred_reset_time(0),
  next_limit_time(0),
  total_overdue_time(0),
  time_source(time_source),
  activity_monitor(NULL),
  activity_sensitive(true)
{
}


//! Destructor
Timer::~Timer()
{
  delete autoreset_interval_predicate;
}


//! Enable activity event monitoring.
void
Timer::enable()
{
  TRACE_ENTER_MSG("Timer::disable", timer_id << timer_enabled);
  
  if (!timer_enabled)
    {
      timer_enabled = true;
      snooze_inhibited = false;
      snooze_on_active = true;
      stop_timer();

      if (autoreset_enabled && autoreset_interval != 0 && get_elapsed_time() == 0)
        {
          elapsed_idle_time = autoreset_interval;
        }
           
      if (get_elapsed_time() >= limit_interval)
        {
          // Break is overdue, force a snooze.
          last_limit_time = time_source->get_time();
          last_limit_elapsed = 0;
          compute_next_limit_time();
        }

      compute_next_predicate_reset_time();
    }

  TRACE_EXIT();
}


//! Disable activity event monitoring.
void
Timer::disable()
{
  TRACE_ENTER_MSG("Timer::disable", timer_id << timer_enabled);
  
  if (timer_enabled)
    {
      timer_enabled = false;
      stop_timer();

      last_start_time = 0;
      last_stop_time = 0;
      last_reset_time = 0;
      next_limit_time = 0;
      next_reset_time = 0;

      timer_state = STATE_INVALID;
      previous_timer_state = STATE_INVALID;
    }
  TRACE_EXIT();
}


//! Set Limit time in seconds.
/*!
 * \param limitTime time at which this timer reaches its limit.
 *
 */
void
Timer::set_limit(long limitTime)
{
  limit_interval = limitTime;

  if (get_elapsed_time() < limitTime)
    {
      // limit increased, pretend there was no limit-reached yet.
      // FIXME: Check if the timer would have been overdue with new settings.
      last_limit_time = 0;
      last_limit_elapsed = 0;
    }

  compute_next_limit_time();
}


//! Enable/Disable limiting
/*!
 * \param b indicates whether the timer limit must be enabled (\c true) or
 *          disabled (\c false).
 */
void
Timer::set_limit_enabled(bool b)
{
  limit_enabled = b;
  compute_next_limit_time();
}


void
Timer::set_snooze_interval(time_t t)
{
  snooze_interval = t;
}


void
Timer::inhibit_snooze()
{
  snooze_inhibited = true;
}


//! Enable/Disable auto-reset
/*!
 * \param b indicates whether the timer auto-reset must be enabled (\c true)
 *          or disabled (\c false).
 */
void
Timer::set_auto_reset_enabled(bool b)
{
  autoreset_enabled = b;
  compute_next_reset_time();
}


//! Set auto-reset time period.
/*!
 * \param resetTime after this amount of idle time the timer will reset itself.
 */
void
Timer::set_auto_reset(long resetTime)
{
  if (resetTime > autoreset_interval)
    {
      snooze_inhibited = false;
    }

  autoreset_interval = resetTime;
  compute_next_reset_time();
}


//! Marks timer sensitive or insensitive for activity
void
Timer::set_activity_sensitive(bool a)
{
  activity_sensitive = a;
  activity_state = ACTIVITY_UNKNOWN;

  if (!activity_sensitive)
    {
      // If timer is made insensitive, start it if
      // it has some elasped time. Other wise a DL will never
      // start (well, not until it resets...)
      int el = get_elapsed_time();
      if (el > 0 && el < limit_interval)
        {
          activity_state = ACTIVITY_ACTIVE;
        }
    }
}


//! Set auto-reset predicate.
/*!
 * \param predicate auto-reset predicate.
 */
void
Timer::set_auto_reset(string predicate)
{
  autoreset_interval_predicate = TimePredFactory::create_time_pred(predicate);
  compute_next_predicate_reset_time();
}



//! Compute the time the limit will be reached.
void
Timer::compute_next_limit_time()
{
  // default action.
  next_limit_time = 0;

  if (timer_enabled && last_limit_time > 0 && !snooze_on_active)
    {
      // Timer already reached limit
      if (!snooze_inhibited)
        {
          next_limit_time = last_limit_time + snooze_interval;
        }
    }
  else if (timer_enabled && timer_state == STATE_RUNNING && last_start_time != 0 &&
           limit_enabled && limit_interval != 0)
    { 
      // We are enabled, running and a limit != 0 was set.
      // So update our current Limit.

      if (last_limit_time > 0)
        {
          // Limit already reached.
          if (snooze_on_active && !snooze_inhibited)
            {
              next_limit_time = last_start_time - elapsed_time + last_limit_elapsed + snooze_interval;
            }
        }
      else
        {
          // new limit = last start time + limit - elapsed.
          next_limit_time = last_start_time + limit_interval - elapsed_time;
        }
    }
}


//! Compute the time the auto-reset must take place.
void
Timer::compute_next_reset_time()
{
  if (timer_enabled && timer_state == STATE_STOPPED && last_stop_time != 0 &&
      autoreset_enabled && autoreset_interval != 0)
    {
      // We are enabled, not running and a limit != 0 was set.

      // next reset time = last stop time + auto reset
      next_reset_time = last_stop_time + autoreset_interval - elapsed_idle_time;

      if (next_reset_time <= last_reset_time)
        {
          next_reset_time = 0;
        }
    }
  else
    {
      // Just in case....
      next_reset_time = 0;
    }
}


void
Timer::compute_next_predicate_reset_time()
{
  // This one ALWAYS sends a reset, also when the timer is disabled.
  
  if (autoreset_interval_predicate)
    {
      if (last_pred_reset_time == 0)
        {
          last_pred_reset_time = time_source->get_time();
        }
        
      autoreset_interval_predicate->set_last(last_pred_reset_time);
      next_pred_reset_time = autoreset_interval_predicate->get_next();
    }
}


//! Daily Reset.
void
Timer::daily_reset_timer()
{
  total_overdue_time = 0;
}


//! Reset and stop the timer.
void
Timer::reset_timer()
{
  TRACE_ENTER_MSG("Timer::reset", timer_id << timer_state);
  
  // Update total overdue.
  time_t elapsed = get_elapsed_time();
  if (elapsed > limit_interval)
    {
      total_overdue_time += (elapsed - limit_interval);
    }
  
  // Full reset.
  elapsed_time = 0;
  last_limit_time = 0;
  last_limit_elapsed = 0;
  last_reset_time = time_source->get_time();
  snooze_inhibited = false;
  snooze_on_active = true;
  
  if (timer_state == STATE_RUNNING)
    {
      // Pretend the timer just started.
      last_start_time = time_source->get_time();
      last_stop_time = 0;
      
      compute_next_limit_time();
      next_reset_time = 0;
      elapsed_idle_time = 0;
    }
  else
    {
      // Timer stopped.
      last_start_time = 0;
      next_reset_time = 0;
      next_limit_time = 0;

      if (autoreset_enabled && autoreset_interval != 0)
        {
          elapsed_idle_time = autoreset_interval;
        }
    }
      
  next_pred_reset_time = 0;
  compute_next_predicate_reset_time();
  TRACE_EXIT();
}


//! Start the timer.
void
Timer::start_timer()
{
  TRACE_ENTER_MSG("Timer::start_timer", timer_id << timer_state);
  
  if (timer_state != STATE_RUNNING)
    {
      TRACE_MSG("frozen = " <<  timer_frozen);
      
      // Set last start and stop times.
      if (!timer_frozen)
        {
          // Timer is not frozen, so let's start.
          last_start_time = time_source->get_time();
          elapsed_idle_time = 0;
        }
      else
        {
          // Update elapsed time.
          if (last_stop_time != 0)
            {
              elapsed_idle_time += (time_source->get_time() - last_stop_time);
            }
          last_start_time = 0;
        }
      
      last_stop_time = 0;
      next_reset_time = 0;
  
      // update state.
      timer_state = STATE_RUNNING;

      // When to generate a limit-reached-event.
      compute_next_limit_time();
    }
  TRACE_EXIT();
}


//! Stop the timer.
void
Timer::stop_timer()
{
  TRACE_ENTER_MSG("Timer::stop_timer", timer_id << timer_state);
  
  if (timer_state != STATE_STOPPED)
    {
      TRACE_MSG("last_start_time = " <<  last_start_time);

      // Update last stop time.
      last_stop_time = time_source->get_time();

      // Update elapsed time.
      if (last_start_time != 0)
        {
          // But only if we are running...

          elapsed_time += (last_stop_time - last_start_time);
        }
  
      // Reset last start time.
      last_start_time = 0;
  
      // Update state.
      timer_state = STATE_STOPPED;

      // When to reset the timer.
      compute_next_reset_time();

      // 
      compute_next_limit_time();
    }
  TRACE_EXIT();
}


//! Snoozes the Timer.
/*!
 *  When the limit of this timer was reached, the Limit Reached event will be
 *  re-sent.
 */
void
Timer::snooze_timer()
{
  if (timer_enabled)
    {
      // recompute.
      snooze_on_active = true;

      next_limit_time = 0;
      last_limit_time = time_source->get_time();
      last_limit_elapsed = get_elapsed_time();
      compute_next_limit_time();

      if (!activity_sensitive)
        {
          // Start the clock in case of insensitive timer.
          activity_state = ACTIVITY_ACTIVE;
        }
    }
}


void
Timer::freeze_timer(bool freeze)
{
  TRACE_ENTER_MSG("Timer::freeze_timer", timer_id << freeze << " " << timer_enabled << " " << activity_sensitive);
  
  if (timer_enabled && activity_sensitive)
    {
      TRACE_MSG("frozen "  << timer_frozen);
      if (freeze && !timer_frozen)
        {
          // FIXME: Why does this say "last_limit_time" ??? should it be last_start_time???
          if (last_limit_time != 0 && timer_state == STATE_RUNNING)
            {
              TRACE_MSG("reset");
              elapsed_time += (time_source->get_time() - last_start_time);
              last_start_time = 0;
            }
        }
      else if (!freeze && timer_frozen)
        {
          if (timer_state == STATE_RUNNING)
            {
              TRACE_MSG("reset");
              last_start_time = time_source->get_time();
              elapsed_idle_time = 0;
            }
        }
    }
  timer_frozen = freeze;
  TRACE_EXIT();
}


//! Returns the elapsed idle time.
time_t
Timer::get_elapsed_idle_time() const
{
  time_t ret = elapsed_idle_time;
  
  if (timer_enabled && last_stop_time != 0)
    {
      // We are not running.
      
      ret += (time_source->get_time() - last_stop_time);
    }

  return ret;
}


//! Returns the elapsed time.
time_t
Timer::get_elapsed_time() const
{
  TRACE_ENTER_MSG("Timer::get_elapsed_time", timer_id);
  
  time_t ret = elapsed_time;

  TRACE_MSG("enabled/last_start_time = " << timer_enabled << " " << last_start_time);
  
  if (timer_enabled && last_start_time != 0)
    {
      // We are running:
      // total elasped = elapes + (last start time - current time)

      ret += (time_source->get_time() - last_start_time);
    }
  else
    {
      // We are not running
      // total elapsed = elapsed
    }

  return ret;
}


time_t
Timer::get_total_overdue_time() const
{
  time_t ret = total_overdue_time;
  time_t elapsed = get_elapsed_time();
  if (elapsed > limit_interval)
    {
      ret += (elapsed - limit_interval);
    }

  return ret;
}

void
Timer::shift_time(int delta)
{
  if (last_limit_time > 0)
    {
      last_limit_time += delta;
    }
  
  if (last_start_time > 0)
    {
      last_start_time += delta;
    }

  if (last_reset_time > 0)
    {
      last_reset_time += delta;
    }

  if (last_stop_time > 0)
    {
      last_stop_time += delta;
    }

  compute_next_limit_time();
  compute_next_reset_time();
  compute_next_predicate_reset_time();
}


//! Activity callback from activity monitor.
void
Timer::activity_notify()
{
  if (timer_enabled)
    {
      if (activity_timer)
        {
          start_timer();
        }
      else
        {
          stop_timer();
        }
    }
}


//! Idle callback from activity monitor.
void
Timer::idle_notify()
{
  if (timer_enabled)
    {
      if (activity_timer)
        {
          stop_timer();
        }
      else
        {
          start_timer();
        }
    }
}


//! Perform timer processing.
/*! \param new_activity_state the current activity state as reported by the
 *         (global) activity monitor.
 *  \param info returns the state of the timer.
 */
void
Timer::process(ActivityState new_activity_state, TimerInfo &info)
{
  TRACE_ENTER_MSG("Timer::Process", timer_id << timer_id << " " << new_activity_state);
  time_t current_time= time_source->get_time();

  // Default event to return.
  info.event = TIMER_EVENT_NONE;
  info.idle_time = get_elapsed_idle_time();
  info.elapsed_time = get_elapsed_time();

  TRACE_MSG("idle = " << info.idle_time << " elap = " << info.elapsed_time);
  TRACE_MSG("enabled = " << timer_enabled << " " << activity_timer);
  TRACE_MSG("last_start_time " << last_start_time);
  
  if (activity_sensitive)
    {
      TRACE_MSG("is activity sensitive");
      
      // This timer responds to the activity monitoring.
      if (activity_monitor != NULL)
        {
          // The timer users its own activity monitor and ignore the 'global'
          // activity monitor state (new_activity_state)
          new_activity_state = activity_monitor->get_current_state();
          TRACE_MSG("state =" << new_activity_state);
        }
    }
  else
    {
      // This timer is activity insensitive. It periodically switches between
      // idle and active.

      TRACE_MSG("is not activity sensitive");
      
      if (activity_state != ACTIVITY_UNKNOWN)
        {
          TRACE_MSG("as = " << activity_state << " nas = " << new_activity_state << " el=" << get_elapsed_time());

          // When the timer has no elasped time, and it not running ->
          // use activity_state from activity monitor. This allows the
          // timer to start when the user becomes active.
          //
          // Otherwise, use the previous state: A timer that is running keep
          // running until it is set to idle below. An idle timer (after limit
          // was reached) remains idle.
          if (activity_state != ACTIVITY_ACTIVE && get_elapsed_time() == 0)
            {
              TRACE_MSG("new state1 = " << activity_state << " " << new_activity_state);
            }
          else
            {
              new_activity_state = activity_state;
              TRACE_MSG("new state2 = " << activity_state << " " << new_activity_state);
            }

          TRACE_MSG("state = " << new_activity_state);
          TRACE_MSG("time, next limit "
                    << current_time << " "
                    << next_limit_time << " "
                    << limit_interval << " "
                    << (next_limit_time - current_time)
                    );
        }
    }
  
  if (timer_enabled)
    {
      if (new_activity_state == ACTIVITY_ACTIVE && timer_state != STATE_RUNNING)
        {
          TRACE_MSG("activity_notify");
          activity_notify();
        }
      else if (new_activity_state != ACTIVITY_ACTIVE && timer_state == STATE_RUNNING)
        {
          TRACE_MSG("idle_notify");
          idle_notify();
        }
    }
  
  activity_state = new_activity_state;
  TRACE_MSG("activity_state = " << activity_state);

  if (autoreset_interval_predicate && next_pred_reset_time != 0 && current_time >= next_pred_reset_time)
    {
      // A next reset time was set and the current time >= reset time.
      // So reset the timer and send a reset event.
      reset_timer();

      last_pred_reset_time = time_source->get_time();
      next_pred_reset_time = 0;
      
      compute_next_predicate_reset_time();
      info.event = TIMER_EVENT_RESET;

    }
  else if (next_limit_time != 0 && current_time >=  next_limit_time)
    {
      // A next limit time was set and the current time >= limit time.
      next_limit_time = 0;
      last_limit_time = time_source->get_time();
      last_limit_elapsed = get_elapsed_time();

      snooze_on_active = true;

      compute_next_limit_time();
      
      info.event = TIMER_EVENT_LIMIT_REACHED;
      // Its very unlikely (but not impossible) that this will overrule
      // the EventStarted. Hey, shit happends.

      if (!activity_sensitive)
        {
          activity_state = ACTIVITY_IDLE;
          TRACE_MSG("limit reached, setting state = IDLE");
        }
    }
  else if (next_reset_time != 0 && current_time >=  next_reset_time)
    {
      // A next reset time was set and the current time >= reset time.
      
      next_reset_time = 0;
      
      bool natural = (limit_interval >=  get_elapsed_time());
      
      reset_timer();
      
      info.event = natural ? TIMER_EVENT_NATURAL_RESET : TIMER_EVENT_RESET;
      // Idem, may overrule the EventStopped.

      if (!activity_sensitive)
        {
          TRACE_MSG("reset reached, setting state = IDLE");
          activity_state = ACTIVITY_IDLE;
        }
    }
  else if (timer_enabled)
    {
      switch (timer_state)
        {
        case STATE_RUNNING:
          {
            if (info.event == TIMER_EVENT_NONE && previous_timer_state == STATE_STOPPED)
              {
                info.event = TIMER_EVENT_STARTED;
              }
          }
          break;
          
        case STATE_STOPPED:
          {
            if (info.event == TIMER_EVENT_NONE && previous_timer_state == STATE_RUNNING)
              {
                info.event = TIMER_EVENT_STOPPED;
              }
          }
          break;
          
        default:
          break;
        }
    }
  
  previous_timer_state = timer_state;
}


std::string
Timer::serialize_state() const
{
  stringstream ss;

  if (restore_enabled)
    {
      ss << timer_id << " " 
         << time_source->get_time() << " "
         << get_elapsed_time() << " "
         << last_pred_reset_time << " "
         << total_overdue_time << " "
         << snooze_inhibited << " "
         << last_limit_time << " "
         << last_limit_elapsed;
    }

  return ss.str();
}


bool
Timer::deserialize_state(std::string state)
{
  TRACE_ENTER("Timer::deserialize_state");
  istringstream ss(state);

  time_t saveTime = 0;
  time_t elapsed = 0;
  time_t lastReset = 0;
  time_t overdue = 0;
  time_t now = time_source->get_time();
  time_t llt = 0;
  time_t lle = 0;
  bool si = false;
  
  ss >> saveTime
     >> elapsed
     >> lastReset
     >> overdue
     >> si
     >> llt
     >> lle;

  TRACE_MSG(si << " " << llt << " " << lle);

  TRACE_MSG(snooze_inhibited);
  
  last_pred_reset_time = lastReset;
  total_overdue_time = overdue;
  elapsed_time = 0;
  last_start_time = 0;
  last_stop_time = 0;

  bool tooOld = ((autoreset_enabled && autoreset_interval != 0) && (now - saveTime >  autoreset_interval));

  if (! tooOld)
    {
      if (autoreset_enabled)
        {
          next_reset_time = now + autoreset_interval;
        }
      elapsed_time = elapsed;
      snooze_inhibited = si;
    }

  // overdue, so snooze
  if (get_elapsed_time() >= limit_interval)
    {
      last_limit_time = llt;
      last_limit_elapsed = lle;
      
      compute_next_limit_time();
    }

  compute_next_predicate_reset_time();

  TRACE_MSG("elapsed = " << elapsed_time);
  return true;
}


void
Timer::set_values(int elapsed, int idle)
{
  elapsed_time = elapsed;
  elapsed_idle_time = idle;

  last_start_time = 0;
  last_stop_time = 0;

  if (timer_state == STATE_RUNNING)
    {
      last_start_time = time_source->get_time();
    }
  else if (timer_state == STATE_STOPPED)
    {
      last_stop_time = time_source->get_time();
    }

  compute_next_limit_time();
  compute_next_reset_time();
  compute_next_predicate_reset_time();
}

void
Timer::set_state_data(const TimerStateData &data)
{
  time_t time_diff =  time_source->get_time() - data.current_time;

  elapsed_time = data.elapsed_time;
  elapsed_idle_time = data.elapsed_idle_time;
  last_pred_reset_time = data.last_pred_reset_time;
  total_overdue_time = data.total_overdue_time;

  last_limit_time = data.last_limit_time;
  last_limit_elapsed = data.last_limit_elapsed;
  snooze_inhibited = data.snooze_inhibited;

  if (last_pred_reset_time > 0)
    {
      last_pred_reset_time += time_diff;
    }
  
  if (last_limit_time > 0)
    {
      last_limit_time += time_diff;
    }

  last_start_time = 0;
  last_stop_time = 0;
  
  if (timer_state == STATE_RUNNING)
    {
      last_start_time = time_source->get_time();
    }
  else if (timer_state == STATE_STOPPED)
    {
      last_stop_time = time_source->get_time();
    }

  compute_next_limit_time();
  compute_next_reset_time();
  compute_next_predicate_reset_time();
}


void
Timer::get_state_data(TimerStateData &data)
{
  TRACE_ENTER("Timer::get_state_data");
  data.current_time = time_source->get_time();
  
  data.elapsed_time = get_elapsed_time();
  data.elapsed_idle_time = get_elapsed_idle_time();
  data.last_pred_reset_time = last_pred_reset_time;
  data.total_overdue_time = total_overdue_time;

  data.last_limit_time = last_limit_time;
  data.last_limit_elapsed = last_limit_elapsed;
  data.snooze_inhibited = snooze_inhibited;
  
  TRACE_MSG("elapsed = " << data.elapsed_time);
  TRACE_EXIT();
}

