/*
 * $Id: wx.c,v 1.26 2001/05/01 14:33:46 kg4ijb Exp $
 *
 * XASTIR, Amateur Station Tracking and Information Reporting
 * Copyright (C) 1999,2000  Frank Giannandrea
 * Copyright (C) 2000,2001  The Xastir Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Look at the README for more information on the program.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <string.h>

#include <Xm/XmAll.h>

#include "wx.h"
#include "main.h"
#include "xastir.h"
#include "interface.h"
#include "lang.h"
#include "db.h"


char wx_station_type[100];
char raw_wx_string[800];

#define MAX_WX_STRING 300
#define WX_TYPE 'z'

/* rain totals */
float rain[24];
float rain_00;
float rain_24;
float rain_base[24];
int rain_first[24];
int rain_check;

/* Other WX station data */
char wx_dew_point[10];
char wx_dew_point_on;
char wx_high_wind[10];
char wx_high_wind_on;
char wx_wind_chill[10];
char wx_wind_chill_on;
char wx_baro_inHg[10];
char wx_baro_inHg_on;
char wx_three_hour_baro[10];
char wx_three_hour_baro_on;
char wx_hi_temp[10];
char wx_hi_temp_on;
char wx_low_temp[10];
char wx_low_temp_on;
char wx_heat_index[10];
char wx_heat_index_on;


/***********************************************************/
/* return the hour of time                                 */
/***********************************************************/
int hour(void) {
    struct tm *time_now;
    time_t secs_now;
    char shour[5];

    secs_now=sec_now();
    time_now = localtime(&secs_now);
    (void)strftime(shour,4,"%H",time_now);
    return(atoi(shour));
}

/***********************************************************/
/* clear rain data                                         */
/***********************************************************/
void clear_rain_data(void) {
    int x;

    for (x=0;x<24;x++) {
        rain[x]=0.0;
        rain_base[x]=0.0;
        rain_first[x]=1;
    }
    rain_00=0.0;
    rain_24=0.0;
    rain_check=0;
}

/***********************************************************/
/* compute_rain - compute rain total from the total rain so*/
/* far                                                     */
/***********************************************************/

void compute_rain(float rain_total) {
    int h;
    int x;
    int i;
    float r;
    float r_first;

    r_first=0.0;
    h=hour();

    /* set base or first rain total for the hour */
    if (rain_first[h]) {
        rain_first[h]=0;
        rain_base[h]=rain_total;
        i=h+1;
        if (i>23)
            i=0;

        rain_first[i]=1;
        rain_check=0;
    } else {
        /* base has been set is it wrong? */
        if (rain_check<3) {
            rain_check++;
            /* is rain less then base? it shouldn't */
            if (rain_total < rain_base[h])
                rain_base[h]=rain_total;

            /* is the difference greater than 10 inches since last reading? it shouldn't */
            if (fabs(rain_total - rain_base[h]) > 10.0)
                rain_base[h]=rain_total;

        }
    }
    /* set total rain for this hour */
    rain[h]=rain_total-rain_base[h];

    /* set total rain in last 24 hours */
    r=0.0;
    i=h+1;
    if (i>23)
        i=0;

    for (x=0;x<=24;x++) {
        r+=rain[i++];
        if (i>23)
            i=0;
    }
    rain_24=r;

    /* set rain since midnight */
    r=0.0;
    for (x=1;x<=h;x++) {
        r+=rain[x];
    }
    rain_00=r;
}


/***********************************************************/
/* compute_gust - compute wind gusts                       */
/*                                                         */
/***********************************************************/
float compute_gust(float wx_speed, float last_speed, time_t *last_speed_time) {
    float computed_gust;

    /* make sure speed time is not zero */
    if(*last_speed_time==0)
        *last_speed_time=sec_now();

    /* set computed to last gust speed */
    computed_gust=last_speed;

    /* check to see if gust is with in 5 min*/
    if ((*last_speed_time+300) > sec_now()) {
        /* speed time within 5 min */
        /* see if gust is higher than last */
        if (wx_speed >= last_speed)
            computed_gust=wx_speed;
    } else {
        /* speed time over 5 min */
        /* since is was over 5 min reset gust */
        computed_gust=wx_speed;
        *last_speed_time=sec_now();
    }
    return(computed_gust);
}



/***********************************************************/
/* clear other wx data                                     */
/***********************************************************/
void clear_local_wx_data(void) {
    memset(wx_dew_point,0,sizeof(wx_dew_point));
    wx_dew_point_on = '\0';

    memset(wx_high_wind,0,sizeof(wx_high_wind));
    wx_high_wind_on = '\0';

    memset(wx_wind_chill,0,sizeof(wx_wind_chill));
    wx_wind_chill_on = '\0';

    memset(wx_baro_inHg,0,sizeof(wx_baro_inHg));
    wx_baro_inHg_on = '\0';

    memset(wx_three_hour_baro,0,sizeof(wx_three_hour_baro));
    wx_three_hour_baro_on = '\0';

    memset(wx_hi_temp,0,sizeof(wx_hi_temp));
    wx_hi_temp_on = '\0';

    memset(wx_low_temp,0,sizeof(wx_low_temp));
    wx_low_temp_on = '\0';

    memset(wx_heat_index,0,sizeof(wx_heat_index));
    wx_heat_index_on = '\0';
}



/**************************************************/
/* Check last WX data received - clear data if old*/
/**************************************************/
void wx_last_data_check(void) {
    DataRow *p_station;
    
    p_station = NULL;
    if (search_station_name(&p_station,my_callsign,1)) {
        if (p_station->weather_data != NULL)
            if (p_station->weather_data->wx_speed_sec_time+360 < sec_now())
                sprintf(p_station->weather_data->wx_gust,"%03d",0);
    }
}



/**************************************************/
/* RSW num convert                                */
/**************************************************/
int rswnc(unsigned char c) {
    return( (int)( (c>>4) & 0x0f) * 10 + (int)(c&0x0f) );
}



/*
 *  Decode Ultimeter 2000 weather data (Data logging mode)
 */
void decode_U2000_L(int from, unsigned char *data, WeatherRow *weather) {
    time_t last_speed_time;
    float last_speed;
    float computed_gust;
    char temp_data1[10];
    char *temp_conv;
    char format;

    last_speed = 0.0;
    last_speed_time = 0;
    computed_gust = 0.0;
    format = '\0';

    if (debug_level & 1)
        printf("APRS WX3 peet bros U2k (data logging mode): |%s|\n", (char *)data);

    weather->wx_type = WX_TYPE;
    strcpy(weather->wx_station,"U2k");

    /* get last gust speed */
    if (strlen(weather->wx_gust) > 0 && !from) {
        /* get last speed */
        last_speed = (float)atof(weather->wx_gust);
        last_speed_time = weather->wx_speed_sec_time;
    }

    // 006B 00 58
    // 00A4 00 46 01FF 380E 2755 02C1 03E8 ---- 0052 04D7    0001 007BM
    // ^       ^  ^    ^    ^         ^                      ^
    // 0       6  8    12   16        24                     40
    /* wind speed */
    if ((char)data[0] != '-') {
        substr(temp_data1,(char *)data,4);
        sprintf(weather->wx_speed,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
        if (from) {
            weather->wx_speed_sec_time = sec_now();
        } else {
            /* local station */
            computed_gust = compute_gust((float)atof(weather->wx_speed),last_speed,&last_speed_time);
            weather->wx_speed_sec_time = sec_now();
            sprintf(weather->wx_gust,"%03d",(int)computed_gust);
        }
    } else {
        if (!from)
            weather->wx_speed[0] = '\0';
    }

    /* wind direction */
    if ((char)data[6] != '-') {
        substr(temp_data1,(char *)data+6,2);
        sprintf(weather->wx_course,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/256.0)*360.0));
    } else {
        sprintf(weather->wx_course,"000");
        if (!from)
            weather->wx_course[0]='\0';
    }

    /* outdoor temp */
    if ((char)data[8] != '-') {
        substr(temp_data1,(char *)data+8,4);
        sprintf(weather->wx_temp,"%03d",(int)((float)(((unsigned long)strtol(temp_data1,&temp_conv,16)<<16)/65636)/10.0));
    } else {
        if (!from)
            weather->wx_temp[0]='\0';
    }

    /* rain total long term */
    if ((char)data[12] != '-') {
        substr(temp_data1,(char *)data+12,4);
        sprintf(weather->wx_rain_total,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
        if (!from) {
            /* local station */
            compute_rain((float)atof(weather->wx_rain_total));
            /*last hour rain */
            sprintf(weather->wx_rain,"%0.2f",rain[hour()]);
            /*last 24 hour rain */
            sprintf(weather->wx_prec_24,"%0.2f",rain_24);
            /* rain since midnight */
            sprintf(weather->wx_prec_00,"%0.2f",rain_00);
        }
    } else {
        if (!from)
            weather->wx_rain_total[0]='\0';
    }

    /* baro */
    if ((char)data[16] != '-') {
        substr(temp_data1,(char *)data+16,4);
        sprintf(weather->wx_baro,"%0.1f",(float)strtol(temp_data1,&temp_conv,16)/10.0);

        if (!from) {
            sprintf(wx_baro_inHg,"%0.1f",(atof(weather->wx_baro)*0.02953));
        }
    }

    /* outdoor humidity */
    if ((char)data[24] != '-') {
        substr(temp_data1,(char *)data+24,4);
        sprintf(weather->wx_hum,"%03d",(int)((float)strtol(temp_data1,&temp_conv,16)/10.0));
    } else {
        if (!from)
            weather->wx_hum[0]='\0';
    }

    /* todays rain total */
    if ((char)data[40] != '-') {
        if (from) {
            substr(temp_data1,(char *)data+40,4);
            sprintf(weather->wx_prec_00,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
        }
    } else {
        if (!from)
            weather->wx_prec_00[0] = '\0';
    }
}


/*
 *  Decode Ultimeter 2000 weather data (Packet mode)
 */
void decode_U2000_P(int from, unsigned char *data, WeatherRow *weather) {
    time_t last_speed_time;
    float last_speed;
    float computed_gust;
    char temp_data1[10];
    char *temp_conv;
    int len;

    last_speed      = 0.0;
    last_speed_time = 0;
    computed_gust   = 0.0;
    len = (int)strlen((char *)data);

    if (debug_level & 1)
        printf("APRS WX5 peet bros U2k Packet (Packet mode): |%s|\n",(char *)data);

    weather->wx_type = WX_TYPE;
    strcpy(weather->wx_station,"U2k");

    /* get last gust speed */
    if (strlen(weather->wx_gust) > 0 && !from) {
        /* get last speed */
        last_speed = (float)atof(weather->wx_gust);
        last_speed_time = weather->wx_speed_sec_time;
    }

    // $ULTW   0031 00 37 02CE  0069 ---- 0000 86A0 0001 ---- 011901CC   0000 0005
    //         ^       ^  ^     ^    ^                   ^               ^    ^
    //         0       6  8     12   16                  32              44   48

    /* wind speed peak over last 5 min */
    if ((char)data[0] != '-') {
        substr(temp_data1,(char *)data,4);
        if (from) {
            sprintf(weather->wx_gust,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
            /* this may be the only wind data */
            sprintf(weather->wx_speed,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
        } else {
            /* local station and may be the only wind data */
            if (len < 51) {
                sprintf(weather->wx_speed,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
                computed_gust = compute_gust((float)atof(weather->wx_speed),last_speed,&last_speed_time);
                weather->wx_speed_sec_time = sec_now();
                sprintf(weather->wx_gust,"%03d",(int)computed_gust);
            }
        }
    } else {
        if (!from)
            weather->wx_gust[0] = '\0';
    }

    /* wind direction */
    if ((char)data[6] != '-') {
        substr(temp_data1,(char *)data+6,2);
        sprintf(weather->wx_course,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/256.0)*360.0));
    } else {
        sprintf(weather->wx_course,"000");
        if (!from)
            weather->wx_course[0] = '\0';
    }

    /* outdoor temp */
    if ((char)data[8] != '-') {
        substr(temp_data1,(char *)data+8,4);
        sprintf(weather->wx_temp,"%03d",(int)((float)(((unsigned long)strtol(temp_data1,&temp_conv,16)<<16)/65636)/10.0));
    } else {
        if (!from)
            weather->wx_temp[0] = '\0';
    }
    /* todays rain total (on some units) */
    if ((char)(data[44]) != '-') {
        if (from) {
            substr(temp_data1,(char *)data+44,4);
            sprintf(weather->wx_prec_00,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
        }
    } else {
        if (!from)
            weather->wx_prec_00[0] = '\0';
    }

    /* rain total long term */
    if ((char)data[12] != '-') {
        substr(temp_data1,(char *)data+12,4);
        sprintf(weather->wx_rain_total,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
        if (!from) {
            /* local station */
            compute_rain((float)atof(weather->wx_rain_total));
            /*last hour rain */
            sprintf(weather->wx_rain,"%0.2f",rain[hour()]);
            /*last 24 hour rain */
            sprintf(weather->wx_prec_24,"%0.2f",rain_24);
            /* rain since midnight */
            sprintf(weather->wx_prec_00,"%0.2f",rain_00);
        }
    } else {
        if (!from)
            weather->wx_rain_total[0] = '\0';
    }

    /* baro */
    if ((char)data[16] != '-') {
        substr(temp_data1,(char *)data+16,4);
        sprintf(weather->wx_baro,"%0.1f",(float)strtol(temp_data1,&temp_conv,16)/10.0);

        if (!from) {
            sprintf(wx_baro_inHg,"%0.2f",(atof(weather->wx_baro)*0.02953));
        }
    } else {
        if (!from)
            weather->wx_baro[0] = '\0';
    }

    /* outdoor humidity */
    if ((char)data[32] != '-') {
        substr(temp_data1,(char *)data+32,4);
        sprintf(weather->wx_hum,"%03d",(int)((float)strtol(temp_data1,&temp_conv,16)/10.0));
    } else {
        if (!from)
            weather->wx_hum[0] = '\0';
    }

    /* 1 min wind speed avg */
    if (len > 48 && (char)(data[48]) != '-') {
        substr(temp_data1,(char *)data+48,4);
        sprintf(weather->wx_speed,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
        if (from) {
            weather->wx_speed_sec_time = sec_now();
        } else {
            /* local station */
            computed_gust = compute_gust((float)atof(weather->wx_speed),last_speed,&last_speed_time);
            weather->wx_speed_sec_time = sec_now();
            sprintf(weather->wx_gust,"%03d",(int)computed_gust);
        }
    } else {
        if (!from) {
            if (len > 48)
                weather->wx_speed[0] = '\0';
        }
    }
}


/*
 *  Decode Peet Bros U-II weather data
 */
void decode_Peet_Bros(int from, unsigned char *data, WeatherRow *weather, int type) {
    time_t last_speed_time;
    float last_speed;
    float computed_gust;
    char temp_data1[10];
    char *temp_conv;

    last_speed    = 0.0;
    computed_gust = 0.0;
    last_speed_time = 0;

    if (debug_level & 1)
        printf("APRS WX4 peet bros UII: |%s|\n",(char *)data);

    weather->wx_type = WX_TYPE;
    strcpy(weather->wx_station,"UII");

    // #  5 0B 75 0082 0082
    // *  7 00 76 0000 0000
    //    ^ ^  ^  ^
    //            rain [1/100 inch ?]
    //         outdoor temp
    //      wind speed [mph / km/h]
    //    wind dir

    /* wind direction */
    substr(temp_data1,(char *)data,1);
    sprintf(weather->wx_course,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/16.0)*360.0));

    /* get last gust speed */
    if (strlen(weather->wx_gust) > 0 && !from) {
        /* get last speed */
        last_speed = (float)atof(weather->wx_gust);
        last_speed_time = weather->wx_speed_sec_time;
    }

    /* wind speed */
    substr(temp_data1,(char *)data+1,2);
    if (type == APRS_WX4) {     // '#'  speed in mph
        sprintf(weather->wx_speed,"%03d",(int)(float)strtol(temp_data1,&temp_conv,16));
    } else {                    // '*'  speed in km/h    convert to mph
        sprintf(weather->wx_speed,"%03d",(int)(float)(strtol(temp_data1,&temp_conv,16)*0.62137));
    }

    if (from) {
        weather->wx_speed_sec_time = sec_now();
    } else {
        /* local station */
        computed_gust = compute_gust((float)atof(weather->wx_speed),last_speed,&last_speed_time);
        weather->wx_speed_sec_time = sec_now();
        sprintf(weather->wx_gust,"%03d",(int)computed_gust);
    }

    /* outdoor temp */
    if ((char)data[3] != '-') {
        substr(temp_data1,(char *)data+3,2);
        sprintf(weather->wx_temp,"%03d",(int)((float)strtol(temp_data1,&temp_conv,16)-56));
    } else {
        if (!from)
            weather->wx_temp[0] = '\0';
    }

    /* rain div by 100 for readings 0.1 inch */
    if ((char)data[5] != '-') {
        substr(temp_data1,(char *)data+5,4);
        sprintf(weather->wx_rain_total,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
        if (!from) {
            /* local station */
            compute_rain((float)atof(weather->wx_rain_total));
            /*last hour rain */
            sprintf(weather->wx_rain,"%0.2f",(float)rain[hour()]);
            /*last 24 hour rain */
            sprintf(weather->wx_prec_24,"%0.2f",rain_24);
            /* rain since midnight */
            sprintf(weather->wx_prec_00,"%0.2f",rain_00);
        }
    } else {
        if (!from)
            weather->wx_rain_total[0] = '\0';
    }
}






/***********************************************************/
/* wx fill data field                                      */
/* from: 0=local station, 1=regular decode (other stations)*/
/* type: type of WX packet                                 */
/* data: the packet of WX data                             */
/* fill: the station data to fill                          */
/***********************************************************/
void wx_fill_data(int from, int type, unsigned char *data, DataRow *fill) {
    time_t last_speed_time;
    float last_speed;
    float computed_gust;
    int  temp1;
    int  temp2;
    int  temp3;
    char temp[20];
    char temp_data1[10];
    char *temp_conv;
    int len;
    int t2;
    int hidx_temp;
    int rh2;
    int hi_hum;
    int heat_index;
    char format;
    WeatherRow *weather;

    last_speed=0.0;
    computed_gust=0.0;
    last_speed_time=0;
    format = '\0';

     len=(int)strlen((char *)data);

    weather = fill->weather_data;  // should always be defined
    switch (type) {


        case (APRS_WX3):
            if (debug_level & 1)
                printf("APRS WX3 peet bros U2k (data logging mode) %s:<%s>\n",fill->call_sign,(char *)data+2);

            weather->wx_type=WX_TYPE;
            strcpy(weather->wx_station,"U2k");

            /* get last gust speed */
            if (strlen(weather->wx_gust) > 0 && !from) {
                /* get last speed */
                last_speed=(float)atof(weather->wx_gust);
                last_speed_time=weather->wx_speed_sec_time;
            }

            /* wind speed */
            if ((char)data[2]!='-') {
                substr(temp_data1,(char *)data+2,4);
                sprintf(weather->wx_speed,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
                if (from) {
                    weather->wx_speed_sec_time = sec_now();
                } else {
                    /* local station */
                    computed_gust = compute_gust((float)atof(weather->wx_speed),last_speed,&last_speed_time);
                    weather->wx_speed_sec_time = sec_now();
                    sprintf(weather->wx_gust,"%03d",(int)computed_gust);
                }
            } else {
                if (!from)
                    weather->wx_speed[0]='\0';
            }

            /* wind direction */
            if ((char)data[8]!='-') {
                substr(temp_data1,(char *)data+8,2);
                sprintf(weather->wx_course,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/256.0)*360.0));
            } else {
                sprintf(weather->wx_course,"000");
                if (!from)
                    weather->wx_course[0]='\0';
            }

            /* outdoor temp */
            if ((char)data[10]!='-') {
                substr(temp_data1,(char *)data+10,4);
                sprintf(weather->wx_temp,"%03d",(int)((float)(((unsigned long)strtol(temp_data1,&temp_conv,16)<<16)/65636)/10.0));
            } else {
                if (!from)
                    weather->wx_temp[0]='\0';
            }

            /* rain total long term */
            if ((char)data[14]!='-') {
                substr(temp_data1,(char *)data+14,4);
                sprintf(weather->wx_rain_total,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
                if (!from) {
                    /* local station */
                    compute_rain((float)atof(weather->wx_rain_total));
                    /*last hour rain */
                    sprintf(weather->wx_rain,"%0.2f",rain[hour()]);
                    /*last 24 hour rain */
                    sprintf(weather->wx_prec_24,"%0.2f",rain_24);
                    /* rain since midnight */
                    sprintf(weather->wx_prec_00,"%0.2f",rain_00);
                }
            } else {
                if (!from)
                    weather->wx_rain_total[0]='\0';
            }

            /* baro */
            if ((char)data[18]!='-') {
                substr(temp_data1,(char *)data+18,4);
                sprintf(weather->wx_baro,"%0.1f",(float)strtol(temp_data1,&temp_conv,16)/10.0);

                if (!from) {
                    sprintf(wx_baro_inHg,"%0.1f",(atof(weather->wx_baro)*0.02953));
                }
            }

            /* outdoor humidity */
            if ((char)data[26]!='-') {
                substr(temp_data1,(char *)data+26,4);
                sprintf(weather->wx_hum,"%03d",(int)((float)strtol(temp_data1,&temp_conv,16)/10.0));
            } else {
                if (!from)
                    weather->wx_hum[0]='\0';
            }

            /* todays rain total */
            if ((char)data[42]!='-') {
                if (from) {
                    substr(temp_data1,(char *)data+42,4);
                    sprintf(weather->wx_prec_00,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
                }
            } else {
                if (!from)
                    weather->wx_prec_00[0]='\0';
            }

            break;



        case(APRS_WX4):
            if (debug_level & 1)
                printf("APRS WX4 peet bros UII %s:<%s>\n",fill->call_sign,(char *)data);

            weather->wx_type=WX_TYPE;
            strcpy(weather->wx_station,"UII");

            /* wind direction */
            substr(temp_data1,(char *)data+1,1);
            sprintf(weather->wx_course,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/16.0)*360.0));

            /* get last gust speed */
            if (strlen(weather->wx_gust) > 0 && !from) {
                /* get last speed */
                last_speed=(float)atof(weather->wx_gust);
                last_speed_time=weather->wx_speed_sec_time;
            }

            /* wind speed */
            substr(temp_data1,(char *)data+2,2);
            if ((char)data[0]=='#') {
                /* mph */
                sprintf(weather->wx_speed,"%03d",(int)(float)strtol(temp_data1,&temp_conv,16));
            } else {
                /* from kph to mph */
                sprintf(weather->wx_speed,"%03d",(int)(float)(strtol(temp_data1,&temp_conv,16)*0.62137));
            }

            if (from) {
                weather->wx_speed_sec_time = sec_now();
            } else {
                /* local station */
                computed_gust = compute_gust((float)atof(weather->wx_speed),last_speed,&last_speed_time);
                weather->wx_speed_sec_time = sec_now();
                sprintf(weather->wx_gust,"%03d",(int)computed_gust);
            }

            /* outdoor temp */
            if ((char)data[4]!='-') {
                substr(temp_data1,(char *)data+4,2);
                sprintf(weather->wx_temp,"%03d",(int)((float)strtol(temp_data1,&temp_conv,16)-56));
            } else {
                if (!from)
                    weather->wx_temp[0]='\0';
            }

            /* rain div by 100 for readings 0.1 inch */
            if ((char)data[6]!='-') {
                substr(temp_data1,(char *)data+6,4);
                sprintf(weather->wx_rain_total,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
                if (!from) {
                    /* local station */
                    compute_rain((float)atof(weather->wx_rain_total));
                    /*last hour rain */
                    sprintf(weather->wx_rain,"%0.2f",(float)rain[hour()]);
                    /*last 24 hour rain */
                    sprintf(weather->wx_prec_24,"%0.2f",rain_24);
                    /* rain since midnight */
                    sprintf(weather->wx_prec_00,"%0.2f",rain_00);
                }
            } else {
                if (!from)
                    weather->wx_rain_total[0]='\0';
            }
            break;



        case(APRS_WX5):
            if (debug_level & 1)
                printf("APRS WX5 peet bros U2k Packet (Packet mode) %s:<%s>\n",fill->call_sign,(char *)data);

            weather->wx_type=WX_TYPE;
            strcpy(weather->wx_station,"U2k");

            /* get last gust speed */
            if (strlen(weather->wx_gust) > 0 && !from) {
                /* get last speed */
                last_speed=(float)atof(weather->wx_gust);
                last_speed_time=weather->wx_speed_sec_time;
            }

            /* wind speed peak over last 5 min */
            if ((char)data[5]!='-') {
                substr(temp_data1,(char *)data+5,4);
                if (from) {
                    sprintf(weather->wx_gust,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
                    /* this may be the only wind data */
                    sprintf(weather->wx_speed,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
                } else {
                    /* local station and may be the only wind data */
                    if (len<56) {
                        sprintf(weather->wx_speed,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
                        computed_gust = compute_gust((float)atof(weather->wx_speed),last_speed,&last_speed_time);
                        weather->wx_speed_sec_time = sec_now();
                        sprintf(weather->wx_gust,"%03d",(int)computed_gust);
                    }
                }
            } else {
                if (!from)
                    weather->wx_gust[0]='\0';
            }

            /* wind direction */
            if ((char)data[11]!='-') {
                substr(temp_data1,(char *)data+11,2);
                sprintf(weather->wx_course,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/256.0)*360.0));
            } else {
                sprintf(weather->wx_course,"000");
                if (!from)
                    weather->wx_course[0]='\0';
            }

            /* outdoor temp */
            if ((char)data[13]!='-') {
                substr(temp_data1,(char *)data+13,4);
                sprintf(weather->wx_temp,"%03d",(int)((float)(((unsigned long)strtol(temp_data1,&temp_conv,16)<<16)/65636)/10.0));
            } else {
                if (!from)
                    weather->wx_temp[0]='\0';
            }
            /* todays rain total (on some units) */
            if ((char)data[49]!='-') {
                if (from) {
                    substr(temp_data1,(char *)data+49,4);
                    sprintf(weather->wx_prec_00,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
                }
            } else {
                if (!from)
                    weather->wx_prec_00[0]='\0';
            }

            /* rain total long term */
            if ((char)data[17]!='-') {
                substr(temp_data1,(char *)data+17,4);
                sprintf(weather->wx_rain_total,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
                if (!from) {
                    /* local station */
                    compute_rain((float)atof(weather->wx_rain_total));
                    /*last hour rain */
                    sprintf(weather->wx_rain,"%0.2f",rain[hour()]);
                    /*last 24 hour rain */
                    sprintf(weather->wx_prec_24,"%0.2f",rain_24);
                    /* rain since midnight */
                    sprintf(weather->wx_prec_00,"%0.2f",rain_00);
                }
            } else {
                if (!from)
                    weather->wx_rain_total[0]='\0';
            }

            /* baro */
            if ((char)data[21]!='-') {
                substr(temp_data1,(char *)data+21,4);
                sprintf(weather->wx_baro,"%0.1f",(float)strtol(temp_data1,&temp_conv,16)/10.0);

                if (!from) {
                    sprintf(wx_baro_inHg,"%0.2f",(atof(weather->wx_baro)*0.02953));
                }
            } else {
                if (!from)
                    weather->wx_baro[0]='\0';
            }

            /* outdoor humidity */
            if ((char)data[37]!='-') {
                substr(temp_data1,(char *)data+37,4);
                sprintf(weather->wx_hum,"%03d",(int)((float)strtol(temp_data1,&temp_conv,16)/10.0));
            } else {
                if (!from)
                    weather->wx_hum[0]='\0';
            }

            /* 1 min wind speed avg */
            if (len>53 && (char)(data[53]) != '-') {
                substr(temp_data1,(char *)data+53,4);
                sprintf(weather->wx_speed,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
                if (from) {
                    weather->wx_speed_sec_time = sec_now();
                } else {
                    /* local station */
                    computed_gust = compute_gust((float)atof(weather->wx_speed),last_speed,&last_speed_time);
                    weather->wx_speed_sec_time = sec_now();
                    sprintf(weather->wx_gust,"%03d",(int)computed_gust);
                }
            } else {
                if (!from) {
                    if (len>53)
                        weather->wx_speed[0]='\0';
                }
            }

            break;



        case(QM_WX):
            if (debug_level & 1)
                printf("Qualimetrics Q-Net %s:<%s>\n",fill->call_sign,(char *)data);

            weather->wx_type=WX_TYPE;
            strcpy(weather->wx_station,"Q-N");

            (void)sscanf((char *)data,"%19s %d %19s %d %19s %d",temp,&temp1,temp,&temp2,temp,&temp3);

            /* outdoor temp */
            sprintf(weather->wx_temp,"%03d",(int)(temp2/10.0));

            /* baro */
            sprintf(weather->wx_baro,"%0.1f",((float)temp3/100.0)*33.864);

            if (!from) {
                sprintf(wx_baro_inHg,"%0.2f",(float)temp3/100.0);
            }

            /* outdoor humidity */
            sprintf(weather->wx_hum,"%03d",temp1);

            if (!from) {
                weather->wx_gust[0]='\0';
                weather->wx_course[0]='\0';
                weather->wx_rain[0]='\0';
                weather->wx_prec_00[0]='\0';
                weather->wx_prec_24[0]='\0';
                weather->wx_rain_total[0]='\0';
                weather->wx_gust[0]='\0';
                weather->wx_speed[0]='\0';
            }
            break;



        case(PEET_COMPLETE):
            if (debug_level & 1)
                printf("peet bros U2k Packet (Complete Record Mode) %s:<%s>\n",fill->call_sign,(char *)data);

            if (!from) {
                /* decode only for local station */
                weather->wx_type=WX_TYPE;
                strcpy(weather->wx_station,"U2k");

                if ((char)data[12]!='-') {
                    substr(temp_data1,(char *)data+12,4);
                    sprintf(weather->wx_gust,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
                } else
                    weather->wx_gust[0]='\0';

                /* wind direction */
                if ((char)data[8]!='-') {
                    substr(temp_data1,(char *)data+8,4);
                    sprintf(weather->wx_course,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/256.0)*360.0));
                } else {
                    sprintf(weather->wx_course,"000");
                    weather->wx_course[0]='\0';
                }

                /* outdoor temp */
                if ((char)data[24]!='-') {
                    substr(temp_data1,(char *)data+24,4);
                    sprintf(weather->wx_temp,"%03d",(int)((float)(((unsigned long)strtol(temp_data1,&temp_conv,16)<<16)/65636)/10.0));
                } else
                    weather->wx_temp[0]='\0';

                /* todays rain total (on some units) */
                if ((char)data[28]!='-') {
                    substr(temp_data1,(char *)data+28,4);
                    sprintf(weather->wx_prec_00,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
                } else
                    weather->wx_prec_00[0]='\0';

               /* rain total long term */
                if ((char)data[432]!='-') {
                    substr(temp_data1,(char *)data+432,4);
                    sprintf(weather->wx_rain_total,"%0.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0);
                    /* Since local station only */
                    compute_rain((float)atof(weather->wx_rain_total));

                    /*last hour rain */
                    sprintf(weather->wx_rain,"%0.2f",rain[hour()]);

                    /*last 24 hour rain */
                    sprintf(weather->wx_prec_24,"%0.2f",rain_24);

                    /* rain since midnight */
                    sprintf(weather->wx_prec_00,"%0.2f",rain_00);
                } else
                    weather->wx_rain_total[0]='\0';

                /* baro */
                if ((char)data[32]!='-') {
                    substr(temp_data1,(char *)data+32,4);
                    sprintf(weather->wx_baro,"%0.1f",(float)strtol(temp_data1,&temp_conv,16)/10.0);
                } else
                    weather->wx_baro[0]='\0';

                /* outdoor humidity */
                if ((char)data[52]!='-') {
                    substr(temp_data1,(char *)data+52,4);
                    sprintf(weather->wx_hum,"%03d",(int)((float)strtol(temp_data1,&temp_conv,16)/10.0));
                } else
                    weather->wx_hum[0]='\0';

                /* dew point */
                if ((char)data[60]!='-') {
                    substr(temp_data1,(char *)data+60,4);
                    sprintf(wx_dew_point,"%03d",(int)((float)(((unsigned long)strtol(temp_data1,&temp_conv,16)<<16)/65636)/10.0));
                    wx_dew_point_on = '\1';
                }

                if ((char)data[4] !='-') {
                    substr(temp_data1,(char *)data+4,4);
                    sprintf(weather->wx_speed,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
                }

                /*high winds for today*/
                if ((char)data[248]!='-') {
                    substr(temp_data1,(char *)data+248,4);
                    sprintf(wx_high_wind,"%03d",(int)(((float)strtol(temp_data1,&temp_conv,16)/10.0)*0.62137));
                    wx_high_wind_on = '\1';
                }

                /*wind chill */
                if ((char)data[20]!='-') {
                substr(temp_data1,(char *)data+20,4);
                sprintf(wx_wind_chill,"%03d",(int)((float)(((unsigned long)strtol(temp_data1,&temp_conv,16)<<16)/65636)/10.0));
                wx_wind_chill_on = '\1';
            }

            /* Barometric Pressure inHg */
            if ((char)data[32]!='-') {
                substr(temp_data1,(char *)data+32,4);
                sprintf(wx_baro_inHg,"%2.2f",(float)strtol(temp_data1,&temp_conv,16)/100.0/3.38639);
                wx_baro_inHg_on = '\1';
            }

            /*3-Hr Barometric Change */
            if ((char)data[36]!='-') {
                substr(temp_data1,(char *)data+36,4);
                sprintf(wx_three_hour_baro,"%0.2f",(float)(((unsigned long)strtol(temp_data1,&temp_conv,16)<<16)/65636)/100.0/3.38639);
                wx_three_hour_baro_on = '\1';
            }

            /* High Temp for Today*/
            if ((char)data[276]!='-') {
                substr(temp_data1,(char *)data+276,4);
                sprintf(wx_hi_temp,"%03d",(int)((float)(((unsigned long)strtol(temp_data1,&temp_conv,16)<<16)/65636)/10.0));
                wx_hi_temp_on = '\1';
            } else
                wx_hi_temp_on = '\0';

            /* Low Temp for Today*/
            if ((char)data[100]!='-') {
                substr(temp_data1,(char *)data+100,4);
                sprintf(wx_low_temp,"%03d",(int)((float)(((unsigned long)strtol(temp_data1,&temp_conv,16)<<16)/65636)/10.0));
                wx_low_temp_on = '\1';
            } else
                wx_low_temp_on = '\0';

            /* Heat Index Calculation*/
            hi_hum=atoi(weather->wx_hum);
            rh2= atoi(weather->wx_hum);
            rh2=(rh2 * rh2);
            hidx_temp=atoi(weather->wx_temp);
            t2= atoi(weather->wx_temp);
            t2=(t2 * t2);

            if (hidx_temp >= 70) {
                heat_index=-42.379+2.04901523 * hidx_temp+10.1433127 * hi_hum-0.22475541 * hidx_temp * hi_hum-0.00683783 * t2-0.05481717 * rh2+0.00122874 * t2 * hi_hum+0.00085282 * hidx_temp * rh2-0.00000199 * t2 * rh2;
                sprintf (wx_heat_index,"%03d",heat_index);
                wx_heat_index_on = '\1';
            } else
                wx_heat_index_on = '\0';
        }

        break;



    case(RSWX200):
        if (!from) {
            if (debug_level & 1)
                printf("RSWX200 WX (binary)\n");

            weather->wx_type=WX_TYPE;
            strcpy(weather->wx_station,"RSW");

            switch (data[0]) {
                case 0x8f: /* humidity */
                    if (rswnc(data[20])<100)
                        sprintf(weather->wx_hum,"%03d",rswnc(data[20]));
                    else
                        sprintf(weather->wx_hum,"100");
                    break;

                case 0x9f: /* temp */
                    /* all data in C ?*/
                    sprintf(temp_data1,"%c%d%0.1f",((data[17]&0x08) ? '-' : '+'),(int)(data[17]&0x7),rswnc(data[16])/10.0);
                    /*printf("temp data: <%s> %d %d %d\n", temp_data1,((data[17]&0x08)==0x08),(data[17]&0x7),rswnc(data[16]));*/
                    sprintf(weather->wx_temp,"%03d",(int)(atof(temp_data1)*1.8)+32);
                    /*printf("Temp %s C %0.2f %03d\n",temp_data1,atof(temp_data1),(int)atof(temp_data1));
                    printf("Temp F %0.2f %03d\n",(atof(temp_data1)*1.8)+32,(int)(atof(temp_data1)*1.8)+32);
                    */
                    sprintf(temp_data1,"%c%ud%d.%ud",((data[18]&0x80) ? '-' : '+'),(unsigned int)(data[18]&0x70)>>4,(int)(data[18]&0x0f),(unsigned int)(data[17] & 0xf0) >> 4);
                    sprintf(wx_hi_temp,"%03d",(int)(atof(temp_data1)*1.8)+32);

                    sprintf(temp_data1,"%c%ud%d.%ud",((data[23]&0x80) ? '-' : '+'),(unsigned int)(data[23]&0x70)>>4,(int)(data[23]&0x0f),(unsigned int)(data[22] & 0xf0) >> 4);
                    sprintf(wx_low_temp,"%03d",(int)(atof(temp_data1)*1.8)+32);
                    break;

                case 0xaf: /* baro/dewpt */
                    // local baro pressure in mb?
                    // sprintf(weather->wx_baro,"%02d%02d",rswnc(data[2]),rswnc(data[1]));
                    // Sea Level Adjusted baro in mb
                    sprintf(weather->wx_baro,"%0d%02d%0.1f",(int)(data[5]&0x0f),rswnc(data[4]),((float)rswnc(data[3])/10.0));

                    sprintf(wx_baro_inHg,"%0.2f",(atof(weather->wx_baro)*0.02953));

                    /* dew point in C */
                    sprintf(wx_dew_point,"%03d",(int)(rswnc(data[18])*1.8)+32);
                    break;

                case 0xbf: /* Rain */
                    /* all data in mm ?*/
                    sprintf(temp_data1,"%02d%02d",rswnc(data[6]), rswnc(data[5]));
                    sprintf(weather->wx_rain_total,"%0.2f",atof(temp_data1)*0.03937);

                    /* Since local station only */
                    compute_rain((float)atof(weather->wx_rain_total));
                    /*last hour rain */
                    sprintf(weather->wx_rain,"%0.2f",rain[hour()]);

                    /*last 24 hour rain */
                    sprintf(weather->wx_prec_24,"%0.2f",rain_24);

                    /* rain since midnight */
                    sprintf(weather->wx_prec_00,"%0.2f",rain_00);
                    break;

                case 0xcf: /* Wind w/chill*/
                    /* get last gust speed */
                    if (strlen(weather->wx_gust) > 0) {
                        /* get last speed */
                        last_speed=(float)atof(weather->wx_gust);
                        last_speed_time=weather->wx_speed_sec_time;
                    }
                    /* all data in m/s */
                    /* average */
                    sprintf(temp_data1,"%01d%0.1f",(int)(data[5]&0xf),(float)(rswnc(data[4])/10));
                    sprintf(weather->wx_speed,"%03d",(int)(atof(temp_data1)*2.2369));
                    /* gust */
                    sprintf(temp_data1,"%01d%0.1f",(int)(data[2]&0xf),(float)(rswnc(data[1])/10));
                    /*sprintf(weather->wx_gust,"%03d",(int)(atof(temp_data1)*2.2369));*/

                    /* do computed gust */
                    computed_gust = compute_gust((float)(atof(temp_data1)*2.2369),last_speed,&last_speed_time);
                    weather->wx_speed_sec_time = sec_now();
                    sprintf(weather->wx_gust,"%03d",(int)computed_gust);

                    /* high gust */
                    sprintf(temp_data1,"%01d%0.1f",(int)(data[8]&0xf), (float)(rswnc(data[7])/10));
                    sprintf(wx_high_wind,"%03d",(int)(atof(temp_data1)*2.2369));
                    wx_high_wind_on = '\1';

                    sprintf(weather->wx_course,"%02d%01ud",rswnc(data[3]),(unsigned int)(data[2]&0xf0)>>4);

                    /* chill in C */
                    sprintf(temp_data1,"%c%d",((data[21]&0x20) ? '-' : '+'),rswnc(data[16]));
                    sprintf(wx_wind_chill,"%03d",(int)(atof(temp_data1)*1.8)+32);
                    wx_wind_chill_on = '\1';
                    break;
                default:
                    break;
            }


            if (strlen(weather->wx_hum) > 0 && strlen(weather->wx_temp) > 0) {
                /* Heat Index Calculation*/
                hi_hum=atoi(weather->wx_hum);
                rh2= atoi(weather->wx_hum);
                rh2=(rh2 * rh2);
                hidx_temp=atoi(weather->wx_temp);
                t2= atoi(weather->wx_temp);
                t2=(t2 * t2);

                if (hidx_temp >= 70) {
                    heat_index=(-42.379+2.04901523 * hidx_temp+10.1433127 * hi_hum-0.22475541 * hidx_temp *
                                hi_hum-0.00683783 * t2-0.05481717 * rh2+0.00122874 * t2 * hi_hum+0.00085282 *
                                hidx_temp * rh2-0.00000199 * t2 * rh2);
                    sprintf (wx_heat_index,"%03d",heat_index);
                } else
                    wx_heat_index[0] = '\0';
            }
        }
    }
}


/***********************************************************/
/* Decode WX data line                                     */
/* wx_line: raw wx data to decode                          */
/***********************************************************/
void wx_decode(unsigned char *wx_line, int port) {
    DataRow *p_station;
    int decoded;
    int find;
    int i;
    int len;
    char time_data[MAX_TIME];
    unsigned int check_sum;
    int max;
    WeatherRow *weather;

    find=0;
    len=(int)strlen((char *)wx_line);
    if (len>10 || ((int)wx_line[0]!=0 && port_data[port].data_type==1)) {
        if (search_station_name(&p_station,my_callsign,1)) {
            if (get_weather_record(p_station)) {    // DK7IN: only add record if we found something...
                weather = p_station->weather_data;
        
                decoded=0;
                /* Ok decode wx data */
                if ((char)wx_line[0]=='!' && (char)wx_line[1]=='!' && is_xnum_or_dash((char *)wx_line+2,40) && port_data[port].data_type==0) {
                    /* Found Peet Bros U2k */
                    /*printf("Found Peet Bros U2k WX:%s\n",wx_line+2);*/
                    strcpy(wx_station_type,langcode("WXPUPSI011"));
                    strcpy(raw_wx_string,(char *)wx_line);
                    strcpy(weather->wx_time,get_time(time_data));
                    weather->wx_sec_time=sec_now();
                    //weather->wx_data=1;
                    wx_fill_data(0,APRS_WX3,wx_line,p_station);
                    decoded=1;
                } else {
                    if ((((char)wx_line[0]=='#') || ((char)wx_line[0]=='*')) && is_xnum_or_dash((char *)wx_line+1,13) && port_data[port].data_type==0) {
                        /* Found Peet Bros raw U2 data */
                        strcpy(wx_station_type,langcode("WXPUPSI012"));
                        /*printf("Found Peet Bros raw U2 data WX#:%s\n",wx_line+1);*/
                        strcpy(raw_wx_string,(char *)wx_line);
                        strcpy(weather->wx_time,get_time(time_data));
                        weather->wx_sec_time=sec_now();
                        //weather->wx_data=1;
                        wx_fill_data(0,APRS_WX4,wx_line,p_station);
                        decoded=1;
                    } else {
                        if (strncmp("$ULTW",(char *)wx_line,5)==0 && is_xnum_or_dash((char *)wx_line+5,44) && port_data[port].data_type==0) {
                            /* Found Peet Bros raw U2 data */
                            strcpy(wx_station_type,langcode("WXPUPSI013"));
                            /*printf("Found Peet Bros Ultmeter Packet data WX#:%s\n",wx_line+5);*/
                            strcpy(raw_wx_string,(char *)wx_line);
                            weather->wx_sec_time=sec_now();
                            //weather->wx_data=1;
                            wx_fill_data(0,APRS_WX5,wx_line,p_station);
                            decoded=1;
                        } else {
                            if ((char)wx_line[2]==' ' && (char)wx_line[5]==' ' && (char)wx_line[8]=='/' && (char)wx_line[11]=='/'
                                    && (char)wx_line[14]==' ' && (char)wx_line[17]==':' && (char)wx_line[20]==':'
                                    && strncmp(" #0:",(char *)wx_line+23,4)==0 && port_data[port].data_type==0) {
                                find=0;
                                for (i=len;i>23 && !find;i--) {
                                    if ((int)wx_line[i]==0x03) {
                                        find=1;
                                        wx_line[i] = '\0';
                                    }
                                }
                                if (find) {
                                    /* found Qualimetrics Q-Net station */
                                    strcpy(wx_station_type,langcode("WXPUPSI016"));
                                    /*printf("Found Qualimetrics Q-Net station data WX#:%s\n",wx_line+23);*/
                                    strcpy(weather->wx_time,get_time(time_data));
                                    weather->wx_sec_time=sec_now();
                                    //weather->wx_data=1;
                                    wx_fill_data(0,QM_WX,wx_line+24,p_station);
                                    decoded=1;
                                }
                            } else {
                                if (strncmp("&CR&",(char *)wx_line,4)==0 && is_xnum_or_dash((char *)wx_line+5,44) && port_data[port].data_type==0) {
                                    strcpy(wx_station_type,langcode("WXPUPSI017"));
                                    strcpy(raw_wx_string,(char *)wx_line);
                                    weather->wx_sec_time=sec_now();
                                    //weather->wx_data=1;
                                    wx_fill_data(0,PEET_COMPLETE,wx_line,p_station);
                                    decoded=1;
                                } else {
                                    if (port_data[port].data_type==1) {
                                        /* binary data type */
                                        /* clear raw string */
                                        memset(raw_wx_string,0,sizeof(raw_wx_string));
                                        max=0;
                                        switch (wx_line[0]) {
                                        case 0x8f:
                                            max=34;
                                            break;

                                        case 0x9f:
                                            max=33;
                                            break;

                                        case 0xaf:
                                            max=30;
                                            break;

                                        case 0xbf:
                                            max=13;
                                            break;

                                        case 0xcf:
                                            max=26;
                                            break;

                                        default:
                                            break;
                                        }
                                        if (max>0) {
                                            check_sum=0;
                                            for (i=0; i<max;i++)
                                                check_sum+=(unsigned int)wx_line[i];

                                            if (wx_line[max] == (unsigned char)(0xff & check_sum)) {
                                                /* good RS WX-200 data */
                                                /*printf("GOOD %0X data\n",wx_line[0]);*/
                                                /* found RS WX-200 */
                                                strcpy(wx_station_type,langcode("WXPUPSI025"));
                                                strcpy(weather->wx_time,get_time(time_data));
                                                weather->wx_sec_time=sec_now();
                                                //weather->wx_data=1;
                                                wx_fill_data(0,RSWX200,wx_line,p_station);
                                                decoded=1;
                                            }
                                        }
                                    } else {
                                        if (debug_level & 1)
                                            printf("Unknown WX DATA:%s\n",(char *)wx_line);
                                    }
                                }
                            }
                        }
                    }
                }
                if (decoded) {
                    /* save data back */

if (begin_critical_section(&port_data_lock, "wx.c:wx_decode(1)" ) > 0)
    printf("port_data_lock, Port = %d\n", port);

                    port_data[port].decode_errors=0;

if (end_critical_section(&port_data_lock, "wx.c:wx_decode(2)" ) > 0)
    printf("port_data_lock, Port = %d\n", port);

                    statusline(langcode("BBARSTA032"),1);       // Decoded WX Data
                    /* redraw now */
                    //redraw_on_new_data=2;
                    redraw_on_new_data=1;
                    fill_wx_data();
                } else {
                    /* Undecoded packet */
                    memset(raw_wx_string,0,sizeof(raw_wx_string));

if (begin_critical_section(&port_data_lock, "wx.c:wx_decode(3)" ) > 0)
    printf("port_data_lock, Port = %d\n", port);

                    port_data[port].decode_errors++;    // We have errors in decoding
                    if (port_data[port].decode_errors>10) {
                        /* wrong data type? */
                        port_data[port].data_type++;    // Try another data type. 0=ascii, 1=wx binary
                        port_data[port].data_type&=0x01;
                        /*if (debug_level & 1)*/
                        printf("Data type %d\n",port_data[port].data_type);
                        port_data[port].decode_errors=0;
                    }

if (end_critical_section(&port_data_lock, "wx.c:wx_decode(4)" ) > 0)
    printf("port_data_lock, Port = %d\n", port);

                }
            }
        }
    }
}



/***********************************************************/
/* fill string with WX data for transmit                   */
/*                            */
/***********************************************************/

time_t wx_tx_data1(char *st) {
    DataRow *p_station;
    time_t wx_time;
    char temp[100];
    WeatherRow *weather;

    strcpy(st,"");
    wx_time = 0;
    if (search_station_name(&p_station,my_callsign,1)) {
        if (get_weather_record(p_station)) {    // station has wx data
            weather = p_station->weather_data;
            if (strlen(weather->wx_course) > 0 && strlen(weather->wx_speed) > 0) {
                /* we have enough wx_data */
                wx_time=weather->wx_sec_time;
                sprintf(st,"%s/%s",weather->wx_course,weather->wx_speed);
            } else {
                /* we don't have enough wx_data, must be from a Qualimetrics Q-Net */
                wx_time=weather->wx_sec_time;
                sprintf(st,".../...");
            }
            if (strlen(weather->wx_gust) > 0) {
                sprintf(temp,"g%s",weather->wx_gust);
                if (strlen(temp) > 4)
                    strcpy(temp,"g999");

                strcat(st,temp);
            } else
                strcat(st,"g...");

            if (strlen(weather->wx_temp) > 0) {
                sprintf(temp,"t%s",weather->wx_temp);
                strcat(st,temp);
            } else
                strcat(st,"t...");

            if (strlen(weather->wx_rain) > 0) {
                sprintf(temp,"r%03d",(int)(atof(weather->wx_rain)*100.0));
                if (strlen(temp)>4)
                    strcpy(temp,"r999");

                strcat(st,temp);
            } else
                strcat(st,"r...");

            if (strlen(weather->wx_prec_00) > 0) {
                sprintf(temp,"P%03d",(int)(atof(weather->wx_prec_00)*100.0));
                if (strlen(temp)>4)
                    strcpy(temp,"P999");

                strcat(st,temp);
            } else
                strcat(st,"P...");

            if (strlen(weather->wx_prec_24) > 0) {
                sprintf(temp,"p%03d",(int)(atof(weather->wx_prec_24)*100.0));
                if (strlen(temp)>4)
                    strcpy(temp,"p999");

                strcat(st,temp);
            } else
                strcat(st,"p...");

            if (strlen(weather->wx_hum) > 0) {
                if (atoi(weather->wx_hum)>99) {
                    sprintf(temp,"h00");
                } else
                    sprintf(temp,"h%02d",atoi(weather->wx_hum));

                strcat(st,temp);
            } else
                strcat(st,"h..");

            if (strlen(weather->wx_baro) > 0) {
                sprintf(temp,"b%05d",(int)(atof(weather->wx_baro)*10.0));
                if (strlen(temp)>6)
                    strcpy(temp,"b99999");

                strcat(st,temp);
            } else
                strcat(st,"b.....");

            sprintf(temp,"%c%s",weather->wx_type,weather->wx_station);
            strcat(st,temp);
        }
    }
    return(wx_time);
}


/***********************************************************/
/* transmit raw wx data                                    */
/*                                                         */
/***********************************************************/
void tx_raw_wx_data(void) {

    if (strlen(raw_wx_string)>10) {
        output_my_data(raw_wx_string,-1,0);
        if (debug_level & 1)
            printf("Sending Raw WX data <%s>\n",raw_wx_string);
    }
}
