/*
  XBubble - game.c
 
  Copyright (C) 2002  Ivan Djelic <ivan@savannah.gnu.org>
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <math.h>
#include <sys/time.h>

#include <X11/Xlib.h>
#include <X11/keysym.h>

#include "utils.h"
#include "data.h"
#include "rectangle.h"
#include "sprite.h"
#include "rgba.h"
#include "board.h"
#include "opponent.h"
#include "controls.h"
#include "dialog.h"
#include "game.h"

extern Display *display;
extern Window root, win;
extern int screen, depth;

extern GC dialog_gc, dialog_text_gc;
extern XFontStruct *dialog_font, *menu_font;
extern Pixmap win_bg, board_bg;
extern Pixmap diamond_on, diamond_off, diamond_on_mask, diamond_off_mask;

extern int scale, fps;
extern unsigned long frame_duration;

enum GameState { PLAYING, PAUSED, OVER, FINISHED, STOPPED };
enum GameComputerState { IDLE, THINKING, AIMING }; 

struct _Game {
  enum GameMode mode;
  enum GameState state;
  enum GameResult result;
  enum GameComputerState computer_state[2];
  int player_left[2], player_right[2], player_fire[2];
  int escape_pressed, pause_pressed, key_pressed, multi_player;
  int nb_boards, handicap[2], board_best_eval[2], board_best_angle[2];
  int round, computer_thinking_time[2], board_x[2], board_y[2], *score;
  unsigned int width, height;
  double canon_angle[2];
  Board board[2];
  Opponent opponent[2];
  GC board_gc;
  Pixmap board_pixmap[2];
  Window board_window[2], msg_box[2], diamond_box[2], tags[2];
};
Game current_game = NULL;

#define PERIOD        12
#define PLAYER1_TAG   "Player 1"
#define PLAYER2_TAG   "Player 2"
#define COMPUTER_TAG  "Computer"

char *player_name[][2] = {
  { PLAYER1_TAG, "" },
  { PLAYER1_TAG, PLAYER2_TAG },
  { PLAYER1_TAG, COMPUTER_TAG },
  { COMPUTER_TAG, COMPUTER_TAG }};

void display_diamonds( Game game, int n, int on ) {
  
  Pixmap back;
  int i, x, y, width, height, pos = 0;  
  GC gc;
  XGCValues gcv;
  unsigned long gcm;
  gcm = GCFunction | GCGraphicsExposures;
  gcv.graphics_exposures = False;
  gcv.function = GXcopy;
  gc = XCreateGC( display, root, gcm, &gcv);
  width = 2*scale;
  height = board_spacing(scale);
  x = game->board_x[n];
  y = game->board_y[n] - 3*board_spacing(scale)/2;
  back = XCreatePixmap( display, root, width, height, depth );
  XSetTSOrigin( display, dialog_gc, -x, -y );
  XFillRectangle( display, back, dialog_gc, 0, 0, width, height );
  XSetClipMask( display, gc, diamond_off_mask );
  for ( i = 0; i < game->score[n]-1; i++ ) {
    XSetClipOrigin( display, gc, pos, 0 );
    XCopyArea( display, diamond_off, back, gc, 0, 0, scale, height, pos, 0 );
    pos += scale;
  }
  if ( game->score[n] > 0 ) {
    XSetClipMask( display, gc, (( on )? diamond_on_mask : diamond_off_mask ));
    XSetClipOrigin( display, gc, pos, 0 );
    XCopyArea( display, (( on )? diamond_on : diamond_off ), back, gc,
	       0, 0, scale, height, pos, 0 );
  }
  XSetWindowBackgroundPixmap( display, game->diamond_box[n], back );
  XFreePixmap( display, back );
  XFreeGC( display, gc );
}

Game new_game( enum GameMode mode, int *handicap, int preload, int *colors,
	       int round, int *score, enum Level level ) {
  int i;
  unsigned long gcm;
  XGCValues gcv;
  Game game = (Game) xmalloc( sizeof( struct _Game ));
  game->state = PLAYING;
  game->mode = mode;
  game->score = score;
  game->round = round;
  for ( i = 0; i < 2; i++ ) {
    game->player_left[i] = 0;
    game->player_right[i] = 0;
    game->player_fire[i] = 0;    
    game->computer_state[i] = IDLE;
    game->computer_thinking_time[i] = 1000;
    game->canon_angle[i] = 0.0;
  }
  game->multi_player = ( mode != SINGLE_PLAYER );
  game->nb_boards = ( game->multi_player )? 2 : 1;
  /* create board(s) */
  for ( i = 0; i < game->nb_boards; i++ ) {
    game->board[i] = new_board( PERIOD, handicap[i], 1-game->multi_player );
    game->opponent[i] = new_opponent( game->board[i], level );
    if ( preload )
      load_bubbles( game->board[i], colors );
  }
  /* one board in single player mode */
  if ( ! game->multi_player ) {
    game->board_x[0] = game_win_width(scale)/2 - board_win_width(scale)/2;
    game->board_y[0] = game_win_height(scale)/2 - board_win_height(scale)/2 +
      board_spacing(scale)/2;
  }
  else { /* two boards in multi-player mode */
    game->board_x[1] = game_win_width(scale)/2 - board_win_width(scale) -
      board_spacing(scale)/2;
    game->board_y[1] = game_win_height(scale)/2 - board_win_height(scale)/2 +
      board_spacing(scale)/2;
    game->board_x[0] = game_win_width(scale)/2 + board_spacing(scale)/2;
    game->board_y[0] = game->board_y[1];
  }
  gcm = GCFunction | GCGraphicsExposures;
  gcv.graphics_exposures = False;
  gcv.function = GXcopy;
  game->board_gc = XCreateGC( display, root, gcm, &gcv);
  /* create windows */
  for ( i = 0; i < game->nb_boards; i++ ) {
    game->diamond_box[i] = 
      XCreateSimpleWindow( display, win, game->board_x[i],
			   game->board_y[i] - 3*board_spacing(scale)/2, 
			   2*scale, board_spacing(scale), 0, 0, 0 );
    display_diamonds( game, i, False );
    game->tags[i] = create_dialog( game->board_x[i] + board_win_width(scale)/2,
				   game->board_y[i] - board_spacing(scale),
				   player_name[mode][i], menu_font,
				   get_pixel( 0xff, 0xff, 0xff ), 0, 1.0 );
    game->board_window[i] =
      XCreateSimpleWindow( display, win, game->board_x[i],
			   game->board_y[i], board_win_width(scale),
			   board_win_height(scale), 0, 0, 0 );
    game->board_pixmap[i] = XCreatePixmap( display, root, 
					   board_win_width(scale),
					   board_win_height(scale), depth );
    draw_sprite_pool( get_board_sprite_pool(game->board[i]), 
		      game->board_pixmap[i], board_bg, board_win_width(scale),
		      board_win_height(scale));
    XSelectInput( display, game->board_window[i], ExposureMask );
    XSetWindowBackgroundPixmap( display, game->board_window[i], None );
    XMapWindow( display, game->board_window[i] );
    XMapWindow( display, game->diamond_box[i] );
  }
  return game;
}

void delete_game( Game game, int no_clean ) {
  int i;
  if ( no_clean )
    /* avoid game window flickering when board windows are destroyed */
    XSetWindowBackgroundPixmap( display, win, None );
  XFreeGC( display, game->board_gc );
  for ( i = 0; i < game->nb_boards; i++ ) {
    delete_opponent( game->opponent[i] );
    delete_board( game->board[i] );
    XFreePixmap( display, game->board_pixmap[i] );
    XDestroyWindow( display, game->board_window[i] );
    XDestroyWindow( display, game->diamond_box[i] );
    XDestroyWindow( display, game->tags[i] );
  }
  free(game);
  if ( no_clean )
    XSetWindowBackgroundPixmap( display, win, win_bg );
}

static void show_board_msg( Game game, int i, char *msg ) {
  game->msg_box[i] = create_dialog( game->board_x[i]+board_win_width(scale)/2,
				    game->board_y[i]+board_win_height(scale)/2,
				    msg, dialog_font, 
				    get_pixel( 0xff, 0xff, 0xff ), 2, 1.5 );
}

static void hide_game_msg( Game game ) {
  int i;
  for ( i = 0; i < game->nb_boards; i++ )
    XDestroyWindow( display, game->msg_box[i] );
}

static void redraw_game_window( Game game ) {
  int i, n;
  Rectangle *r;
  RectangleList rl;
  SpritePool sp;
  for ( n = 0; n < game->nb_boards; n++ ) {
    sp = get_board_sprite_pool( game->board[n] );
    /* only redraw some rectangles */
    rl = get_sprite_pool_redraw_list(sp);
    for ( i = 0; i < rl->size; i++ ) {
      r = &rl->element[i];
      XCopyArea( display, game->board_pixmap[n], game->board_window[n], 
		 game->board_gc, r->x1, r->y1, r->x2-r->x1, r->y2-r->y1,
		 r->x1, r->y1 );
    }
  }
}

static void process_x_events( Game game ) {
  XEvent event;
  Window window;
  KeySym keysym;
  int i;

  game->pause_pressed = 0;
  game->escape_pressed = 0;
  game->key_pressed = 0;

  while ( XPending(display) ) {
    XNextEvent( display, &event);
    switch ( event.type ) {

    case Expose:
      window = event.xexpose.window;
      /* redraw board windows only, game window has a background pixmap */
      for ( i = 0; i < game->nb_boards; i++ )
	if ( window == game->board_window[i] ) 
	  XCopyArea( display, game->board_pixmap[i], game->board_window[i], 
		     game->board_gc, event.xexpose.x, event.xexpose.y, 
		     event.xexpose.width, event.xexpose.height, 
		     event.xexpose.x, event.xexpose.y );
      break;
      
    case KeyRelease:
      keysym = XLookupKeysym( &event.xkey, 0);
      if ( PLAYER1_LEFT(keysym) )
	game->player_left[0] = 0;
      if ( PLAYER1_RIGHT(keysym) )
	game->player_right[0] = 0;
      if ( PLAYER1_FIRE(keysym) )
	game->player_fire[0] = 0;
      if ( PLAYER2_LEFT(keysym) )
	game->player_left[1] = 0;
      if ( PLAYER2_RIGHT(keysym) )
	game->player_right[1] = 0;
      if ( PLAYER2_FIRE(keysym) )
	game->player_fire[1] = 0;
      break;
      
    case KeyPress:
      keysym = XLookupKeysym( &event.xkey, 0);
      game->key_pressed = 1;
      if ( PLAYER1_LEFT(keysym) )
	game->player_left[0] = 1;
      if ( PLAYER1_RIGHT(keysym) )
	game->player_right[0] = 1;
      if ( PLAYER1_FIRE(keysym) )
	game->player_fire[0] = 1;
      if ( PLAYER2_LEFT(keysym) )
	game->player_left[1] = 1;
      if ( PLAYER2_RIGHT(keysym) )
	game->player_right[1] = 1;
      if ( PLAYER2_FIRE(keysym) )
	game->player_fire[1] = 1;
      /* escape & pause keys */
      if ( keysym == XK_Escape )
	game->escape_pressed = 1;
      if ( keysym == XK_p )
	game->pause_pressed = 1;
      break;
      
    default:
      break;
    }
  }
}

static void animate_game( Game game, int dt ) {
  int i, color;
  for ( i = 0; i < game->nb_boards; i++ ) { 
    animate_board( game->board[i], dt );
    redraw_sprite_pool( get_board_sprite_pool(game->board[i]), 
			game->board_pixmap[i], board_bg );
  }
  if (( game->multi_player )&&( game->state != OVER )) {
    /* exchange bubbles */
    while ( receive_bubble(game->board[0]) >= 0 ) {
      color = rnd( NB_COLORS );
      send_bubble( game->board[1], color);
    }
    while ( receive_bubble(game->board[1]) >= 0 ) {
      color = rnd( NB_COLORS );
      send_bubble( game->board[0], color);
    }
  }
}

static void player_move( Game game, int n ) {
  if ( game->player_left[n] )
    rotate_canon_left( game->board[n] );
  if ( game->player_right[n] )
    rotate_canon_right( game->board[n] );
  if ( game->player_fire[n] )
    fire_board_canon(game->board[n]);
}

static void computer_move( Game game, int n ) {
  int shift;
  /* if board not stable then restart from beginning */
  if (( ! ready_to_fire( game->board[n] ))||
      ( board_was_lowered( game->board[n] )))
    game->computer_state[n] = IDLE;
  game->computer_thinking_time[n] -= frame_duration/1000;
  /* think only when board is stable */
  if ( ready_to_fire( game->board[n] )) {
    switch ( game->computer_state[n] ) {
    case IDLE:
      reset_opponent( game->opponent[n], game->board[n]);
      game->computer_state[n] = THINKING;
      break;    
    case THINKING:
      if ( find_best_angle(game->opponent[n])) {
	game->board_best_angle[n] = get_best_angle(game->opponent[n]);
	game->board_best_eval[n] = get_best_eval(game->opponent[n]);
	game->computer_state[n] = AIMING;
      }
      break;
    case AIMING:
      /* if we are losing then wait */
      if ( game->board_best_eval[n] < 5 )
	break;
      /* if canon is correctly aimed and thinking time is elapsed then fire */
      if (( get_canon_angle( game->board[n] ) == game->board_best_angle[n] )&&
	  (( game->mode != PLAYER_VS_COMPUTER )||
	   ( game->computer_thinking_time[n] <= 0 ))) {
	fire_board_canon( game->board[n] );
	/* adjust our pace to our opponent's */
	game->computer_thinking_time[n] = 
	  get_last_fire_delay(game->board[0]) + 500;
      }
      /* else adjust canon angle */
      shift = game->board_best_angle[n] - get_canon_angle(game->board[n]);
      if ( shift != 0 ) {
	if ( shift > 0 )
	  rotate_canon_right( game->board[n] );
	else
	  rotate_canon_left( game->board[n] );
	/* if necessary, fine tune canon rotation ( yes that's cheating ) */
	if ( abs(shift) < ( frame_duration/1000.0 )*CANON_ROTATING_SPEED ) {
	  move_board_canon( game->board[n], 
			    (int) floor( abs(shift)/CANON_ROTATING_SPEED ));
	  canon_stop( game->board[n] );
	}
      }
    }
  }
}

static void frame_update() {
  int i, overflow1, overflow2;
  Game game = current_game;
  if ( game == NULL )
    return;
  process_x_events(game);
  switch ( game->state ) {
    
  case PLAYING:
    /* pause */
    if ( game->pause_pressed ) {
      game->state = PAUSED;
      for ( i = 0; i < game->nb_boards; i++ )
	show_board_msg( game, i, "Pause" );
      break;
    }
    /* abort with ESC key (or any key in demo mode) */
    if (( game->escape_pressed )||
	(( game->mode == DEMO )&&( game->key_pressed ))) {
      game->state = STOPPED;
      game->result = ABORTED;
      break;
    }

    switch ( game->mode ) {
    case TWO_PLAYERS:
      player_move( game, 1 );
    case SINGLE_PLAYER:
      player_move( game, 0 );
      break;
    case PLAYER_VS_COMPUTER: 
      player_move( game, 0 );
      computer_move( game, 1 );
      break;
    case DEMO:
      computer_move( game, 0 );
      computer_move( game, 1 );
      break;
    default:
      break;
    }
    animate_game( game, frame_duration/1000 );
    redraw_game_window(game);

    /* check if game is not over */
    if ( game->multi_player ) {
      overflow1 = board_overflow( game->board[0] );
      overflow2 = board_overflow( game->board[1] );
      if ( overflow1 || overflow2 ) {
	game->state = OVER;
	game->result = DRAW;
	if ( ! overflow1 ) {
	  game->result = PLAYER1_WON;
	  explode_board( game->board[0] );
	}
	if ( ! overflow2 ) {
	  game->result = PLAYER1_LOST;
	  explode_board( game->board[1] );
	}
      }
    }
    else { /* single player mode */
      if ( board_empty( game->board[0] )) {
	game->state = OVER;
	game->result = PLAYER1_WON;
	break;
      }
      if ( board_overflow( game->board[0] )) {
	game->state = OVER;
	game->result = PLAYER1_LOST;
      }
    }
    break;
    
  case PAUSED:
    if ( game->pause_pressed ) {
      game->state = PLAYING;
      hide_game_msg(game);
    }
    break;

  case OVER:
    /* wait for end of board animations ( quiet boards ) */
    if (( board_quiet( game->board[0] )) &&
	(( ! game->multi_player )||( board_quiet( game->board[1]))))
      game->state = FINISHED;
    else {
      animate_game( game, frame_duration/1000 );
      redraw_game_window(game);
    }
    break;
    
  default:
    break;
  }
}

static void timer_sleep( long ms ) {
  long i;
  for ( i = ms*fps/1000; i > 0; i-- )
    pause();
}

static void show_start_boxes( Game game ) {
  static char msg[20];
  sigset_t set;
  sigemptyset( &set );
  sigaddset( &set, SIGALRM );
  /* enter X critical section */
  sigprocmask( SIG_BLOCK, &set, NULL );
  switch ( game->mode ) {
  case SINGLE_PLAYER:
    sprintf( msg, "Stage %d", game->round );
    show_board_msg( game, 0, msg );
    break;
  case TWO_PLAYERS:
  case PLAYER_VS_COMPUTER:
    sprintf( msg, "Round %d", game->round );
    show_board_msg( game, 0, msg );
    show_board_msg( game, 1, msg );
    break;
  default:
    return;
  } 
  /* leave critical section */
  sigprocmask( SIG_UNBLOCK, &set, NULL );
  /* sleep for 2 seconds */
  timer_sleep( 2000 );
  /* enter X critical section */
  sigprocmask( SIG_BLOCK, &set, NULL );
  hide_game_msg(game);
  /* leave critical section */
  sigprocmask( SIG_UNBLOCK, &set, NULL );
}

static void show_game_result( Game game ) {
  int i, on = 1, winner = -1;
  sigset_t set;
  sigemptyset( &set );
  sigaddset( &set, SIGALRM );
  /* enter X critical section */
  sigprocmask( SIG_BLOCK, &set, NULL );
  switch ( game->result ) {
  case PLAYER1_WON:
    show_board_msg( game, 0, "Win" );
    if ( game->multi_player ) {
      show_board_msg( game, 1, "Lose" );
      game->score[0]++;
      winner = 0;
    }
    break;
  case PLAYER1_LOST:
    show_board_msg( game, 0, "Lose" );
    if ( game->multi_player ) {
      show_board_msg( game, 1, "Win" );
      game->score[1]++;
      winner = 1;
    }
    break;
  case DRAW:
    show_board_msg( game, 0, "Draw" );
    show_board_msg( game, 1, "Draw" );
    break;
  case ABORTED:
    return;
  }
  /* leave critical section */
  sigprocmask( SIG_UNBLOCK, &set, NULL );
  /* sleep for 4 seconds */
  for ( i = 0; i < 20; i++ ) {
    if ( winner >= 0 ) { 
      sigprocmask( SIG_BLOCK, &set, NULL );
      display_diamonds( game, winner, on );
      XClearWindow( display, game->diamond_box[winner] );
      sigprocmask( SIG_UNBLOCK, &set, NULL );
      on = 1 - on;
    }
    timer_sleep(200);
  }
  /* enter X critical section */
  sigprocmask( SIG_BLOCK, &set, NULL );
  hide_game_msg(game);
  /* leave critical section */
  sigprocmask( SIG_UNBLOCK, &set, NULL );
}

enum GameResult play_game( Game game ) {
  struct sigaction action;
  struct itimerval value;
  struct timeval interval;

  /* setup interval timer */
  interval.tv_sec = 0;
  interval.tv_usec = frame_duration;
  value.it_interval = interval;
  value.it_value = interval;

  /* prepare for catching SIGALRM signal */
  action.sa_handler = frame_update;
  action.sa_flags = 0;
  sigemptyset(&(action.sa_mask));

  current_game = game;
  game->state = STOPPED;

  if (( sigaction( SIGALRM, &action, NULL ) != 0 )||
      ( setitimer( ITIMER_REAL, &value, NULL) < 0 )) {
    perror( "play_game");
    exit(1);
  }
  show_start_boxes(game);
  /* start playing */
  game->state = PLAYING;

  /* loop until game is finished or aborted */
  while (( game->state != FINISHED )&&( game->state != STOPPED ))
    pause();
  if ( game->state == FINISHED )
    show_game_result(game);
  
  /* stop timer */
  interval.tv_usec = 0;
  value.it_interval = interval;
  value.it_value = interval;
  setitimer( ITIMER_REAL, &value, NULL);

  current_game = NULL;
  return game->result;
}
