/*
 *	Xenophilia GTK+ Theme Engine
 * 
 *  xeno_color.c:
 *		Routines for modifying colors and remapping the colors in XPM data
 *		to the colors in the GtkStyle.
 *
 *	Copyright  1999-2001 Johan Hanson <johan@tiq.com>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "xeno_rc_style.h"		/* for gradient configuration */
#include "xeno_style.h"
#include "xeno_color.h"
#include "xeno_images.h"

#include <math.h>
#include <stdio.h>

/*
 *	Globals
 */

const guchar xeno_dither_table[16][16] = {
	{ 0x00,0xc0,0x30,0xf0,0x0c,0xcc,0x3c,0xfc,0x03,0xc3,0x33,0xf3,0x0f,0xcf,0x3f,0xff },
	{ 0x80,0x40,0xb0,0x70,0x8c,0x4c,0xbc,0x7c,0x83,0x43,0xb3,0x73,0x8f,0x4f,0xbf,0x7f },
	{ 0x20,0xe0,0x10,0xd0,0x2c,0xec,0x1c,0xdc,0x23,0xe3,0x13,0xd3,0x2f,0xef,0x1f,0xdf },
	{ 0xa0,0x60,0x90,0x50,0xac,0x6c,0x9c,0x5c,0xa3,0x63,0x93,0x53,0xaf,0x6f,0x9f,0x5f },
	{ 0x08,0xc8,0x38,0xf8,0x04,0xc4,0x34,0xf4,0x0b,0xcb,0x3b,0xfb,0x07,0xc7,0x37,0xf7 },
	{ 0x88,0x48,0xb8,0x78,0x84,0x44,0xb4,0x74,0x8b,0x4b,0xbb,0x7b,0x87,0x47,0xb7,0x77 },
	{ 0x28,0xe8,0x18,0xd8,0x24,0xe4,0x14,0xd4,0x2b,0xeb,0x1b,0xdb,0x27,0xe7,0x17,0xd7 },
	{ 0xa8,0x68,0x98,0x58,0xa4,0x64,0x94,0x54,0xab,0x6b,0x9b,0x5b,0xa7,0x67,0x97,0x57 },
	{ 0x02,0xc2,0x32,0xf2,0x0e,0xce,0x3e,0xfe,0x01,0xc1,0x31,0xf1,0x0d,0xcd,0x3d,0xfd },
	{ 0x82,0x42,0xb2,0x72,0x8e,0x4e,0xbe,0x7e,0x81,0x41,0xb1,0x71,0x8d,0x4d,0xbd,0x7d },
	{ 0x22,0xe2,0x12,0xd2,0x2e,0xee,0x1e,0xde,0x21,0xe1,0x11,0xd1,0x2d,0xed,0x1d,0xdd },
	{ 0xa2,0x62,0x92,0x52,0xae,0x6e,0x9e,0x5e,0xa1,0x61,0x91,0x51,0xad,0x6d,0x9d,0x5d },
	{ 0x0a,0xca,0x3a,0xfa,0x06,0xc6,0x36,0xf6,0x09,0xc9,0x39,0xf9,0x05,0xc5,0x35,0xf5 },
	{ 0x8a,0x4a,0xba,0x7a,0x86,0x46,0xb6,0x76,0x89,0x49,0xb9,0x79,0x85,0x45,0xb5,0x75 },
	{ 0x2a,0xea,0x1a,0xda,0x26,0xe6,0x16,0xd6,0x29,0xe9,0x19,0xd9,0x25,0xe5,0x15,0xd5 },
	{ 0xaa,0x6a,0x9a,0x5a,0xa6,0x66,0x96,0x56,0xa9,0x69,0x99,0x59,0xa5,0x65,0x95,0x55 }
};

typedef struct {
	GdkBitmap	*bitmap;
	gint		ref_count;
} XenoImageMask;

static XenoImageMask xeno_image_masks[XENO_MASK_LAST] = {
	{ 0,0 }, { 0,0 }, { 0,0 }, { 0,0 }, { 0,0 }, { 0,0 }, {0,0}
};


/*
 *	XenoColor
 */
const static XenoColor xeno_color_black = { 0.0, 0.0, 0.0 };
const static XenoColor xeno_color_white = { 1.0, 1.0, 1.0 };

void xeno_color_shade (const XenoColor *src, gfloat k, XenoColor *dst) {
	if (k > 1.0) {
		k -= 1.0;
		dst->red	= (1.0 - src->red)   * k + src->red;
		dst->green	= (1.0 - src->green) * k + src->green;
		dst->blue	= (1.0 - src->blue)  * k + src->blue;
	} else {
		dst->red	= k * src->red;
		dst->green	= k * src->green;
		dst->blue	= k * src->blue;
	}
}

XENO_INLINE guint16 xeno_dither_component (gfloat f, gint shift, guchar dither) {
	gint32	value;
	
	shift += 8;
	value = f * ((1 << shift) - 1);
	if ((guchar)value > dither)
		value += 0x0100;
	
	value = value << (16 - shift);
	return CLAMP(value, 0x0000, 0x0ffff);
}

void xeno_color_dither (const XenoColor *src, const GdkVisual *visual, gint x, gint y, GdkColor *dst) {
	guchar dither = xeno_dither_table[x & 0x0f][y & 0x0f];
	dst->red   = xeno_dither_component (src->red,   visual->red_prec,   dither);
	dst->green = xeno_dither_component (src->green,	visual->green_prec,	dither);
	dst->blue  = xeno_dither_component (src->blue,	visual->blue_prec,  dither);
}


/*** Pixmap remapping functions ***/
typedef struct {
	const GtkStyle		*style, *parent_style;
	const XenoRcData	*rc_data;
	XenoShadowType		shadow_type;
	GtkStateType		state_type;
} XenoRemapContext;

static const guchar *
xeno_read_rgb (const guchar *stream, XenoRemapContext *ctx, XenoColor *dst)
{
	guchar			c;
	const GtkStyle	*style;
	const XenoRcData *rc_data;
	GtkStateType	state_type;
	gfloat	k, k2;
	
	style = ctx->style;
	
	c = *(stream++);
	if (c <= XENO_CORNER) {
		if (c <= XENO_COLOR_LAST) {
			/* assume the pen is a legal Gtk pen */
			state_type = (GtkStateType)(*(stream++));
		} else {
			/* lookup from a shadow */
			if (ctx->shadow_type > XENO_SHADOW_OUT) {
				const guint8 *pen = &xeno_shadows[ctx->shadow_type - XENO_SHADOW_ETCHED_IN].medium_ring[*(stream++)][c - XENO_TOP][0];
				c = pen[0];
				state_type = pen[1];
			} else {
				c = XENO_BG;
				state_type = ctx->state_type;
			}
		}
		if ((state_type & XENO_PARENT) != 0) {
			style = ctx->parent_style;
			state_type &= XENO_STATE_MASK;
		}
		if (state_type == (GtkStateType)XENO_CURRENT || ctx->state_type == GTK_STATE_INSENSITIVE) {
			state_type = ctx->state_type;
		} else if (state_type == (GtkStateType)XENO_PARENT_CURRENT) {
			state_type = GTK_STATE_NORMAL; /* differs from xeno_shadow_init() */
		}
		state_type &= XENO_GTK_STATE_MASK;
		
		if (c < XENO_WHITE) {
			const GdkColor *color;
			switch (c) {
			  case XENO_FG:
				color = style->fg;
				break;
			  case XENO_BASE:
				color = style->base;
				break;
			  case XENO_TEXT:
				color = style->text;
				break;
			  case XENO_LIGHT:
				color = style->light;
				break;
			  case XENO_DARK:
				color = style->dark;
				break;
			  case XENO_MID:
				color = style->mid;
				break;
			  case XENO_BG:
			  default:
				color = style->bg;
				break;
			}
			xeno_color_from_gdk (dst, &color[state_type]);
		} else {
			rc_data = XENO_STYLE_RC_DATA(style);
			if (c == XENO_GRADIENT_BG) {
				xeno_style_gradient_rgb (style, state_type, 2.0 - *(stream++) / 127.0, dst);
			} else {
				XenoStyleData *style_data;
				style_data = XENO_STYLE_DATA(style);
				if (style_data) {
					*dst = ((c == XENO_WHITE) ? style_data->white : style_data->black)[state_type];
				} else {
					*dst = (c == XENO_WHITE) ? xeno_color_white : xeno_color_black;
				}
			}
		}
	} else if (c == XENO_USER) {
		xeno_color_init (dst, *(stream++) / 255.0, *(stream++) / 255.0, *(stream++) / 255.0);
	} else if (c == XENO_PURE_WHITE) {
		*dst = xeno_color_white;
	} else if (c == XENO_PURE_BLACK) {
		*dst = xeno_color_black;
	} else if (c == XENO_THIN) {
		stream = xeno_read_rgb (stream, ctx, dst);
		if ((rc_data = ctx->rc_data) != NULL && rc_data->thickness < 2) {
			XenoColor dummy;
			return xeno_read_rgb (stream, ctx, &dummy);
		} else {
			return xeno_read_rgb (stream, ctx, dst);
		}
	} else if (c == XENO_SHADE) {
		stream = xeno_read_rgb (stream, ctx, dst);
		k = *(stream++) / 127.0;
		
		xeno_color_shade (dst, k, dst);
	} else {
		/* blend */
		XenoColor in[4], tmp;
		int n=2;
		
		k = *(stream++) / 255.0;
		
		if (c == XENO_BI) {
			stream = xeno_read_rgb(stream, ctx, &in[0]);
			stream = xeno_read_rgb(stream, ctx, &in[1]);
			xeno_color_blend (&in[0], &in[1], k, dst);
		} else {
			k2 = *(stream++) / 255.0;
			stream = xeno_read_rgb(stream, ctx, &in[0]);
			stream = xeno_read_rgb(stream, ctx, &in[1]);
			xeno_color_blend (&in[0], &in[1], k, &tmp);
			
			stream = xeno_read_rgb(stream, ctx, &in[2]);
			
			if (c == XENO_QUAD) {
				stream = xeno_read_rgb(stream, ctx, &in[3]);
				xeno_color_blend (&in[2], &in[3], k, dst);
				xeno_color_blend (&tmp, dst, k2, dst);
				n = 4;
			} else if (c == XENO_TRI) {
				xeno_color_blend (&tmp, &in[2], k2, dst);
				n = 3;
			} else {
				g_error("error in remapping stream\n");
			}
		}
	}
	
	return stream;
}


static char ** xeno_xpm_copy (const char **xpm) {
	gint w, h, nc, cpp, i;
	char **ret;
	sscanf(xpm[0], "%d %d %d %d", &w, &h, &nc, &cpp);
	ret = g_malloc(sizeof(gpointer) * (h + nc + 1));
	for (i=0; i<h+nc+1; i++)
		ret[i] = g_strdup(xpm[i]);

	return ret;
}

static void xeno_xpm_free (char **xpm) {
	gint w, h, nc, cpp, i;
	sscanf(xpm[0], "%d %d %d %d", &w, &h, &nc, &cpp);
	for (i=0; i<h+nc+1; i++)
		g_free(xpm[i]);
	g_free(xpm);
}


/* The main remapping function */
static void xeno_xpm_remap (const GtkStyle *style, const GtkStyle *style2,
							XenoShadowType shadow_type, GtkStateType state_type,
							const guchar *stream, char **xpm)
{
	static char hextab[] = "0123456789ABCDEF";
	XenoRemapContext ctx;
	const XenoRcData *rc_data;
	XenoColor	rgb /*, bg */;
	GdkColor color;
	guint	r, g, b;
	guint	i = 1;
	guchar	c;
	
	if (   shadow_type != XENO_SHADOW_NONE
		&& ((rc_data = XENO_STYLE_RC_DATA(style)) != NULL))
	{
		if (shadow_type == XENO_SHADOW_IN) {
			shadow_type = rc_data->shadow_in[state_type];
		} else {
			shadow_type = rc_data->shadow_out[state_type];
		}
	}
	
	ctx.style		 = style;
	ctx.parent_style = style2;
	ctx.shadow_type  = shadow_type;
	ctx.state_type	 = state_type;
	ctx.rc_data		 = rc_data;
	
	while ((c = *(stream++)) != 0) {
		if (xpm[i][0] != c) {
			g_warning("XPM and remapping stream do not match at color i==%d, remap==%d\n", i-1, c);
			return;
		}
		
		/* parse color entry */	
		stream = xeno_read_rgb (stream, &ctx, &rgb);
		xeno_color_to_gdk (&rgb, &color);
		
		r = color.red >> 8;
		g = color.green >> 8;
		b = color.blue >> 8;
		
		xpm[i][5]  = hextab[((r&0x00f0) >> 4) & 0x0000000f];
		xpm[i][6]  = hextab[  r				& 0x0000000f];
		xpm[i][7]  = hextab[((g&0x00f0) >> 4) & 0x0000000f];
		xpm[i][8]  = hextab[  g				& 0x0000000f];
		xpm[i][9]  = hextab[((b&0x00f0) >> 4) & 0x0000000f];
		xpm[i][10] = hextab[  b				& 0x0000000f];
		xpm[i][11] = 0;
		i++;
	}
}

static XenoImageMask *
xeno_mask_get (XenoImageType image_type)
{
	gint j;
	
	if (xeno_theme_pseudocolor) {
		j = xeno_low_images[image_type].mask_type;
	} else {
		j = xeno_images[image_type].mask_type;
	}
	
	return &xeno_image_masks[j];
}

GdkBitmap *
xeno_image_mask	(XenoImageType i)
{
	return xeno_mask_get (i)->bitmap;
}

void
xeno_image_ref (XenoImageType i)
{
	xeno_mask_get (i)->ref_count += 1;
}

void
xeno_image_unref (XenoImageType i)
{
	XenoImageMask *mask;
	
	mask = xeno_mask_get (i);
	
	if (--mask->ref_count <= 0 && mask->bitmap) {
		gdk_bitmap_unref (mask->bitmap);
		mask->bitmap = NULL;
	}
}

GdkPixmap *
xeno_pixmap_get (GdkWindow *window, GtkStyle *style,
				 const GtkStyle *style2, XenoImageType image_type)
{
	XenoStyleData	*data;
	const XenoImage	*image;
	XenoImageMask	*mask;
	GdkPixmap		*pixmap;
	GdkBitmap		**bitmap_ptr;
	char			**xpm;
	
	g_return_val_if_fail(window != NULL, NULL);
	g_return_val_if_fail(style  != NULL, NULL);
	g_return_val_if_fail(style2 != NULL, NULL);
	g_return_val_if_fail(XENO_STYLE_DATA(style) != NULL, NULL);
	
	data = XENO_STYLE_DATA(style);
	
	if ((pixmap = data->pixmaps[image_type])) {
		return pixmap;
	} else {
		if (xeno_theme_pseudocolor) {
			image = &xeno_low_images[image_type];
		} else {
			image = &xeno_images[image_type];
		}
		xpm = xeno_xpm_copy(image->xpm);
		xeno_xpm_remap (style, style2, image->shadow_type, image->state_type, image->remap, xpm);
		
		mask = &xeno_image_masks[image->mask_type];
		bitmap_ptr = &mask->bitmap;
		if (*bitmap_ptr != NULL)
			bitmap_ptr = NULL;
		
		pixmap	= gdk_pixmap_create_from_xpm_d (window, bitmap_ptr, NULL, xpm);
		
		if (pixmap)
			mask->ref_count += 1;
		
		data->pixmaps[image_type] = pixmap;
		xeno_xpm_free(xpm);
	}
	
	return pixmap;
}

/* Images */

XenoColor *
xeno_pixmap_get_average_color (GdkPixmap *pixmap, GdkVisual *visual, XenoColor *color) {
	GdkImage	*image;
	gdouble		red, green, blue, t;
	gint		x, y, width, height;
	guint		pixel;
	
	if (pixmap == (GdkPixmap *)GDK_PARENT_RELATIVE || visual->type != GDK_VISUAL_TRUE_COLOR)
		return NULL;
	
	gdk_window_get_size (pixmap, &width, &height);
	if ((image = gdk_image_get (pixmap, 0, 0, width, height)) == NULL)
		return NULL;
	
	red = green = blue = 0.0;
	for (y = 0; y < height; ++y) {
		for (x = 0; x < width; ++x) {
			pixel = gdk_image_get_pixel (image, x, y);
			red   += pixel & visual->red_mask;
			green += pixel & visual->green_mask;
			blue  += pixel & visual->blue_mask;
		}
	}
	gdk_image_destroy (image);
	
	t = 1.0 / (width * height);
	xeno_color_init (color, red*t/visual->red_mask, green*t/visual->green_mask, blue*t/visual->blue_mask);
	return color;
}

/*
GdkPixmap * xeno_pixmap_get (GdkWindow *window, GtkStyle *style,
							 const GtkStyle *parent_style, XenoImageType image_type, GdkBitmap *mask_p)
{
	XenoImage			*image_table;
	XenoImage			*image;
	XenoImageVariant	*variant;
	XenoImageMask		*mask;
	char				*xpm;
	
	XenoRcData			*rc_data;
	XenoStyleData		*style_data;
	
	g_return_val_if_fail(window != NULL, NULL);
	g_return_val_if_fail(style  != NULL, NULL);
	g_return_val_if_fail(parent_style != NULL, NULL);
	g_return_val_if_fail(XENO_STYLE_DATA(style) != NULL, NULL);
	g_return_val_if_fail(XENO_STYLE_RC_DATA(style) != NULL, NULL);
	
	rc_data = XENO_RC_DATA(style);
	style_data = XENO_STYLE_DATA(style);
	
	image_table = xeno_images;
	image = &image_table[image_type];
	variant = &image->variants[variant_type];
	mask = &xeno_masks[variant->mask_type];
	
	if ((pixmap = style_data->pixmaps[image_type]) == NULL) {
		xpm = xeno_xpm_copy (image->xpm);
		xeno_xpm_remap (style, parent_style, image->shadow_type, image->state_type, image->remap, xpm);
		bitmap_ptr = &mask->bitmap;
		if (*bitmap_ptr != NULL)
			bitmap_ptr = NULL;
		
		pixmap = gdk_pixmap_create_from_xpm_d (window, bitmap_ptr, NULL, xpm);
		
		if (pixmap)
			mask->ref_count += 1;
		
		style_data->pixmaps[image_type] = pixmap;
		style_data->npixmaps += 1;
		xeno_xpm_free(xpm);
	}
	
	if (mask_p)
		*mask_p = mask->bitmap;
	
	return pixmap;
}
*/

