/*
 *  Xeuklides  version 0.1.2 
 *  Copyright (c) Christian Obrecht 2000-2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <math.h>
#include <gtk/gtk.h>
#include "types.h"
#include "geometry.h"
#include "parser.tab.h"

extern int errorcount, undefined, width, height;

extern GtkWidget *drawing_area;

extern GdkPixmap *pixmap;

extern GdkGC *solid_gc, *dotted_gc, *dashed_gc;

extern GdkFont *font;

extern GdkColor black_col, darkgray_col, gray_col, lightgray_col, red_col, green_col, blue_col, cyan_col, magenta_col, yellow_col;

extern interrec inter_table[10][2];

void warning(char *);

void yyerror(char *);

double X1, Y1, X2, Y2, Unit, X_off, Y_off;

int X(double x)
{
  return (int)(X_off + Unit*(x-X1));
}

int Y(double y)
{
  return (int)(Y_off + Unit*(Y2-y));
}
 
void frame(double x1, double y1, double x2, double y2, double unit)
{
  double w, h;
  
  X1=x1; Y1=y1; X2=x2; Y2=y2; undefined = 0;
  w = X2 - X1; h = Y2 - Y1;
  if (width*h > height*w) {
    Unit = height/h;
    X_off = (width-w*Unit)/2;
    Y_off = 0;
  } else {
    Unit = width/w;
    X_off = 0;
    Y_off = (height-h*Unit)/2;
  }
}

void default_frame(void)
{
  if (undefined) frame(-2, -2, 8, 6, 1);
}

void setcolor_flag(int flag)
{
  GdkColor col;
  
  switch(flag) {
    case BLACK 		: col = black_col; break;
    case DARKGRAY	: col = darkgray_col; break;
    case GRAY		: col = gray_col; break;
    case LIGHTGRAY	: col = lightgray_col; break;
    case RED		: col = red_col; break;
    case GREEN		: col = green_col; break;
    case BLUE		: col = blue_col; break;
    case CYAN		: col = cyan_col; break;
    case MAGENTA	: col = magenta_col; break;
    case YELLOW		: col = yellow_col;
  } 
  
  gdk_gc_set_foreground(solid_gc,&col);
  gdk_gc_set_foreground(dotted_gc,&col);
  gdk_gc_set_foreground(dashed_gc,&col);
}

void change_font(char * lfd)
{
  GdkFont *fnt;
  fnt = gdk_font_load(lfd);
  if (fnt == NULL) warning("Unable to alloc font\n");
  else font = fnt;
}

void draw_point(_point* A, int flag)
{
  char * opt;
  
  default_frame();
  switch(flag) {
    case DOT : opt = "[dotstyle=*]"; break;
    case BOX : opt = "[dotstyle=square*]"; break;
    case CROSS : opt = "[dotstyle=x]"; break;
    case PLUS : opt = "[dotstyle=+]";
  }
  gdk_draw_rectangle(pixmap, solid_gc, TRUE, X(A->x)-1, Y(A->y)-1, 3, 3);
}

GdkGC * gc(int flag)
{
  switch(flag) {
    case FULL :  return solid_gc;
    case DOTTED : return dotted_gc; 
    case DASHED : return dashed_gc;
  }
}

void draw_vector(_vector* v, _point* A, int flag)
{
  double a, x, y;
  default_frame();
  a = V_angle(v); x = A->x+v->x; y = A->y+v->y;
  gdk_draw_line(pixmap, gc(flag), X(A->x), Y(A->y), X(x), Y(y));
  gdk_draw_line(pixmap, solid_gc, X(x), Y(y), X(x+.15*Cos(a+155)), Y(y+.15*Sin(a+155)));
  gdk_draw_line(pixmap, solid_gc, X(x), Y(y), X(x+.15*Cos(a-155)), Y(y+.15*Sin(a-155)));  
}

void draw_line(_line* l, int flag1, int flag2)
{
  double x1, y1, x2, y2, m;

  default_frame();
  if (ZERO(Sin(l->angle))) {
    x1 = X1; y1 = l->y; x2 = X2; y2 = l->y;
  } 
  else if (ZERO(Cos(l->angle))) {
    x1 = l->x ; y1 = Y1; x2 = l->x; y2 = Y2;
  }
  else {
    m = Tan(l->angle);
    x1 = (Y1-l->y)/m+l->x; y1 = Y1;
    if (x1 < X1) {
      x1 = X1; y1 = m*(X1-l->x)+l->y;
    }
    if (x1 > X2) {
      x1 = X2; y1 = m*(X2-l->x)+l->y;
    }
    x2 = (Y2-l->y)/m+l->x; y2 = Y2;
    if (x2 < X1) {
      x2 = X1; y2 = m*(X1-l->x)+l->y;
    }
    if (x2 > X2) {
      x2 = X2; y2 = m*(X2-l->x)+l->y;
    }
  }
  switch(flag2) {
    case HALFLINE :
      if (l->angle>=0) {
        x1 = l->x; y1 = l->y;
      }
      else {
	  x2 = l->x; y2 = l->y;
      }
      break;
    case BACKHALFLINE :
      if (l->angle>=0) {
        x2 = l->x; y2 = l->y;
      }
      else {
	  x1 = l->x; y1 = l->y;
      }      
  }
  if ((x1>=X1) && (x2<=X2) && (y1>=Y1) && (y2<=Y2))
    gdk_draw_line(pixmap, gc(flag1), X(x1), Y(y1), X(x2), Y(y2));
}

void draw_segment(_segment* s, int flag1, int flag2)
{
  double a;
  default_frame();
  gdk_draw_line(pixmap, gc(flag1), X(s->x1), Y(s->y1), X(s->x2), Y(s->y2));
  a = S_angle(s);
  if (flag2 == ARROW || flag2 == DOUBLEARROW) {
    gdk_draw_line(pixmap, solid_gc, X(s->x2), Y(s->y2), X(s->x2+.15*Cos(a+155)), Y(s->y2+.15*Sin(a+155)));
    gdk_draw_line(pixmap, solid_gc, X(s->x2), Y(s->y2), X(s->x2+.15*Cos(a-155)), Y(s->y2+.15*Sin(a-155)));
  }
  if (flag2 == BACKARROW || flag2 == DOUBLEARROW) {
    gdk_draw_line(pixmap, solid_gc, X(s->x1), Y(s->y1), X(s->x1+.15*Cos(a+25)), Y(s->y1+.15*Sin(a+25)));
    gdk_draw_line(pixmap, solid_gc, X(s->x1), Y(s->y1), X(s->x1+.15*Cos(a-25)), Y(s->y1+.15*Sin(a-25)));
  }
}

void draw_circle(_circle* c, int flag)
{
  int d;

  default_frame();
  d = (int)(2*Unit*c->radius); 
  gdk_draw_arc(pixmap, gc(flag), FALSE, X(c->x-c->radius), Y(c->y+c->radius), d , d, 0, 23040);
}

void draw_arc(_circle* c, double a1, double a2, int flag1, int flag2)
{
  int d;
  double l, r, x, y;

  default_frame();
  r = c->radius;
  d = (int)(2*Unit*r); 
  l = a2-a1;
  gdk_draw_arc(pixmap, gc(flag1), FALSE, X(c->x-r), Y(c->y+r),
    d, d, 64*a1, 64*((l>0)?l:(360+l)));
  if (flag2 == ARROW || flag2 == DOUBLEARROW) {
    x = c->x + r*Cos(a2); y = c->y + r*Sin(a2);
    gdk_draw_line(pixmap, solid_gc, X(x), Y(y), X(x+.15*Cos(a2-65)), Y(y+.15*Sin(a2-65)));
    gdk_draw_line(pixmap, solid_gc, X(x), Y(y), X(x+.15*Cos(a2-115)), Y(y+.15*Sin(a2-115)));
  }
  if (flag2 == BACKARROW || flag2 == DOUBLEARROW) {
    x = c->x + r*Cos(a1); y = c->y + r*Sin(a1);
    gdk_draw_line(pixmap, solid_gc, X(x), Y(y), X(x+.15*Cos(a1+65)), Y(y+.15*Sin(a1+65)));
    gdk_draw_line(pixmap, solid_gc, X(x), Y(y), X(x+.15*Cos(a1+115)), Y(y+.15*Sin(a1+115)));
  }
}

void draw_triangle(_point* A, _point* B, _point* C, int flag)
{
  GdkGC * tmp;
  default_frame();
  tmp = gc(flag);
  gdk_draw_line(pixmap, tmp, X(A->x), Y(A->y), X(B->x), Y(B->y));
  gdk_draw_line(pixmap, tmp, X(B->x), Y(B->y), X(C->x), Y(C->y));
  gdk_draw_line(pixmap, tmp, X(C->x), Y(C->y), X(A->x), Y(A->y));
}

void draw_quadrilateral(_point* A, _point* B, _point* C, _point* D, int flag)
{
  GdkGC * tmp;
  default_frame();
  tmp = gc(flag);
  gdk_draw_line(pixmap, tmp, X(A->x), Y(A->y), X(B->x), Y(B->y));
  gdk_draw_line(pixmap, tmp, X(B->x), Y(B->y), X(C->x), Y(C->y));
  gdk_draw_line(pixmap, tmp, X(C->x), Y(C->y), X(D->x), Y(D->y));
  gdk_draw_line(pixmap, tmp, X(D->x), Y(D->y), X(A->x), Y(A->y));
}

void draw_pentagon(_point* A, _point* B, _point* C, _point* D, _point* E, int flag)
{
  GdkGC * tmp;
  default_frame();
  tmp = gc(flag);
  gdk_draw_line(pixmap, tmp, X(A->x), Y(A->y), X(B->x), Y(B->y));
  gdk_draw_line(pixmap, tmp, X(B->x), Y(B->y), X(C->x), Y(C->y));
  gdk_draw_line(pixmap, tmp, X(C->x), Y(C->y), X(D->x), Y(D->y));
  gdk_draw_line(pixmap, tmp, X(D->x), Y(D->y), X(E->x), Y(E->y));
  gdk_draw_line(pixmap, tmp, X(E->x), Y(E->y), X(A->x), Y(A->y));
}

void draw_hexagon(_point* A, _point* B, _point* C, _point* D, _point* E, _point* F, int flag)
{
  GdkGC * tmp;
  default_frame();
  tmp = gc(flag);
  gdk_draw_line(pixmap, tmp, X(A->x), Y(A->y), X(B->x), Y(B->y));
  gdk_draw_line(pixmap, tmp, X(B->x), Y(B->y), X(C->x), Y(C->y));
  gdk_draw_line(pixmap, tmp, X(C->x), Y(C->y), X(D->x), Y(D->y));
  gdk_draw_line(pixmap, tmp, X(D->x), Y(D->y), X(E->x), Y(E->y));
  gdk_draw_line(pixmap, tmp, X(E->x), Y(E->y), X(F->x), Y(F->y));
  gdk_draw_line(pixmap, tmp, X(F->x), Y(F->y), X(A->x), Y(A->y));
}

void draw_P_label(char* s, _point* A, double l, double a)
{
  default_frame();
  gdk_draw_string(pixmap, font, solid_gc, X(A->x+l*Cos(a)) - gdk_string_width(font, s)/2,
                   Y(A->y+l*Sin(a)) + gdk_string_height(font, s)/2, s);
}

void draw_S_label(char* s, _segment* sg, double l, double a)
{
  default_frame();
  gdk_draw_string(pixmap, font, solid_gc, X((sg->x1+sg->x2)/2+l*Cos(a)) - gdk_string_width(font, s)/2,
                   Y((sg->y1+sg->y2)/2+l*Sin(a)) + gdk_string_height(font, s)/2, s);
}

void draw_P_N(double v, char* s, _point* A, double l, double a)
{
  char* tmp;
  
  default_frame();
  tmp = (char *)malloc(64);
  snprintf(tmp, 63, s, v);
  gdk_draw_string(pixmap, font, solid_gc, X(A->x+l*Cos(a)) - gdk_string_width(font, s)/2,
                   Y(A->y+l*Sin(a)) + gdk_string_height(font, s)/2, tmp);
  free(tmp);
}

void draw_S_N(double v, char* s, _segment* sg, double l, double a)
{
  char* tmp;
  
  default_frame();
  tmp = (char *)malloc(64);
  snprintf(tmp, 63, s, v);
  gdk_draw_string(pixmap, font, solid_gc, X((sg->x1+sg->x2)/2+l*Cos(a)) - gdk_string_width(font, s)/2,
                   Y((sg->y1+sg->y2)/2+l*Sin(a)) + gdk_string_height(font, s)/2, tmp);
  free(tmp);
}

void draw_P_NN(double v1, double v2, char* s, _point* A, double l, double a)
{
  char* tmp;
  
  default_frame();
  tmp = (char *)malloc(64);
  snprintf(tmp, 63, s, v1, v2);
  gdk_draw_string(pixmap, font, solid_gc, X(A->x+l*Cos(a)) - gdk_string_width(font, s)/2,
                   Y(A->y+l*Sin(a)) + gdk_string_height(font, s)/2, tmp);
  free(tmp);
}

void draw_S_NN(double v1, double v2, char* s, _segment* sg, double l, double a)
{
  char* tmp;
  
  default_frame();
  tmp = (char *)malloc(64);
  snprintf(tmp, 63, s, v1, v2);
  gdk_draw_string(pixmap, font, solid_gc, X((sg->x1+sg->x2)/2+l*Cos(a)) - gdk_string_width(font, s)/2,
                   Y((sg->y1+sg->y2)/2+l*Sin(a)) + gdk_string_height(font, s)/2, tmp);
  free(tmp);
}

void mark_S(_segment* sg, int flag)
{
  double x, y, a, c, s;
  
  default_frame();
  
  x = (sg->x1+sg->x2)/2; y = (sg->y1+sg->y2)/2; a = S_angle(sg); s = .15*Sin(a); c = .15*Cos(a);
  switch (flag) {
    case SIMPLE : gdk_draw_line(pixmap, solid_gc, X(x-s), Y(y+c), X(x+s), Y(y-c));
		  break;
    case DOUBLE : gdk_draw_line(pixmap, solid_gc, X(x-s-.2*c), Y(y+c-.2*s), X(x+s-.2*c), Y(y-c-.2*s));
		  gdk_draw_line(pixmap, solid_gc, X(x-s+.2*c), Y(y+c+.2*s), X(x+s+.2*c), Y(y-c+.2*s));
		  break;
    case TRIPLE : gdk_draw_line(pixmap, solid_gc, X(x-s-.4*c), Y(y+c-.4*s), X(x+s-.4*c), Y(y-c-.4*s));
		  gdk_draw_line(pixmap, solid_gc, X(x-s), Y(y+c), X(x+s), Y(y-c));
		  gdk_draw_line(pixmap, solid_gc, X(x-s+.4*c), Y(y+c+.4*s), X(x+s+.4*c), Y(y-c+.4*s));
		  break;
    case CROSS  : gdk_draw_line(pixmap, solid_gc, X(x-.7*(s+c)), Y(y+.7*(c-s)), X(x+.7*(s+c)), Y(y-.7*(c-s)));
		  gdk_draw_line(pixmap, solid_gc, X(x-.7*(s-c)), Y(y+.7*(c+s)), X(x+.7*(s-c)), Y(y-.7*(c+s)));
  }
}

void mark_A(_point* A, _point* B, _point* C, int flag)
{
  double x, y, a1, a2, a, l;
  
  default_frame();
  
  x = B->x; y = B->y; a1 = angle(A->x-x,A->y-y); a2 = angle(C->x-x,C->y-y);
  a = (a1+a2)/2; l = a2-a1;
  
  switch (flag) {
    case SIMPLE     : gdk_draw_arc(pixmap, solid_gc, FALSE, X(x-.5), Y(y+.5),
                                    Unit, Unit, 64*a1, 64*((l>0)?l:(360+l)));
                      break;
    case DOUBLE     : gdk_draw_arc(pixmap, solid_gc, FALSE, X(x-.55), Y(y+.55),
                                    1.1*Unit, 1.1*Unit, 64*a1, 64*((l>0)?l:(360+l)));
                      gdk_draw_arc(pixmap, solid_gc, FALSE, X(x-.45), Y(y+.45),
                                    .9*Unit, .9*Unit, 64*a1, 64*((l>0)?l:(360+l)));
                      break;
    case TRIPLE     : gdk_draw_arc(pixmap, solid_gc, FALSE, X(x-.42), Y(y+.42),
                                    .84*Unit, .84*Unit, 64*a1, 64*((l>0)?l:(360+l)));
                      gdk_draw_arc(pixmap, solid_gc, FALSE, X(x-.5), Y(y+.5),
                                    Unit, Unit, 64*a1, 64*((l>0)?l:(360+l)));
                      gdk_draw_arc(pixmap, solid_gc, FALSE, X(x-.58), Y(y+.58),
                                    1.16*Unit, 1.16*Unit, 64*a1, 64*((l>0)?l:(360+l)));
                      break;
    case ARROW      : gdk_draw_arc(pixmap, solid_gc, FALSE, X(x-.5), Y(y+.5),
                                    Unit, Unit, 64*a1, 64*((l>0)?l:(360+l)));
                      x += .5*Cos(a2); y += .5*Sin(a2);
                      gdk_draw_line(pixmap, solid_gc, X(x), Y(y), X(x+.15*Cos(a2-65)), Y(y+.15*Sin(a2-65)));
                      gdk_draw_line(pixmap, solid_gc, X(x), Y(y), X(x+.15*Cos(a2-115)), Y(y+.15*Sin(a2-115)));
                      break;
    case BACKARROW  : gdk_draw_arc(pixmap, solid_gc, FALSE, X(x-.5), Y(y+.5),
                                    Unit, Unit, 64*a1, 64*((l>0)?l:(360+l)));
                      x += .5*Cos(a1); y += .5*Sin(a1);
                      gdk_draw_line(pixmap, solid_gc, X(x), Y(y), X(x+.15*Cos(a1+65)), Y(y+.15*Sin(a1+65)));
                      gdk_draw_line(pixmap, solid_gc, X(x), Y(y), X(x+.15*Cos(a1+115)), Y(y+.15*Sin(a1+115)));
                      break;
    case DASH       : gdk_draw_arc(pixmap, solid_gc, FALSE, X(x-.5), Y(y+.5),
                                    Unit, Unit, 64*a1, 64*((l>0)?l:(360+l)));
                      gdk_draw_line(pixmap, solid_gc, X(x+.425*Cos(a)), Y(y+.425*Sin(a)), X(x+.575*Cos(a)), Y(y+.575*Sin(a)));
                      break;
    case RIGHT      : gdk_draw_line(pixmap, solid_gc, X(x+.3*Cos(a1)), Y(y+.3*Sin(a1)),
                                     X(x+.3*(Cos(a1)+Cos(a2))), Y(y+.3*(Sin(a1)+Sin(a2))));
                      gdk_draw_line(pixmap, solid_gc, X(x+.3*(Cos(a1)+Cos(a2))), Y(y+.3*(Sin(a1)+Sin(a2))),
                                     X(x+.3*Cos(a2)), Y(y+.3*Sin(a2)));
                      break;
    }
}
