/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2002 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#if WANT_HAMLIB
#include <gtk/gtk.h>
#include <math.h>
#include <hamlib/rig.h>

#include "hamlib-utils.h"
#include "types.h"
#include "utils.h"
#include "support.h"
#include "pixmaps/s0.xpm"
#include "pixmaps/s1.xpm"
#include "pixmaps/s2.xpm"
#include "pixmaps/s3.xpm"
#include "pixmaps/s4.xpm"
#include "pixmaps/s5.xpm"
#include "pixmaps/s6.xpm"
#include "pixmaps/s7.xpm"
#include "pixmaps/s8.xpm"
#include "pixmaps/s9.xpm"

typedef struct
{
	gchar **xpmdata;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
} typeLevel;

typeLevel S[] =
{
	{ s0_xpm, NULL, NULL },
	{ s1_xpm, NULL, NULL },
	{ s2_xpm, NULL, NULL },
	{ s3_xpm, NULL, NULL },
	{ s4_xpm, NULL, NULL },
	{ s5_xpm, NULL, NULL },
	{ s6_xpm, NULL, NULL },
	{ s7_xpm, NULL, NULL },
	{ s8_xpm, NULL, NULL },
	{ s9_xpm, NULL, NULL },
	{ NULL, NULL, NULL }
};

extern preferencestype preferences;
extern statetype state;
extern GtkWidget *mainwindow;
extern GList *riglist;
extern RIG *myrig;
GList *rigs = NULL;
GdkPixmap *pixmap = NULL;
gint svalue[10];

/* Append a new entry in the driver list. It is called by rig_list_foreach */
static gint riglist_make_list(const struct rig_caps *caps, gpointer data) {
	rigs = g_list_append(rigs, (gpointer)caps);
	return 1; /* !=0, we want them all ! */
}

/* Return a list with pointers of available drivers from hamlib */
GList *riglist_get_list(void) {
	gint status;

	rig_load_all_backends();
	status = rig_list_foreach(riglist_make_list, NULL);
	return rigs;
}

/* lookup hamlib rigid associated with model name */
gint get_rigid(gchar *rig) {
	gint i, numrigs;
	struct rig_caps *rcaps = NULL;

	numrigs = g_list_length(riglist);
	for (i=0; i<numrigs; i++) {
		rcaps = (struct rig_caps *)g_list_nth_data(riglist, i);
		if (g_strcasecmp(rig, rcaps->model_name) == 0)
			break;
	}
	if (i > 0) return rcaps->rig_model;
	else return(-1);
}

/* load xpm data from s-meter pixmaps */
void loadsmeter(GtkWidget *widget)
{
	gint i = 0;

	while (S[i].xpmdata)
	{
		S[i].pixmap = gdk_pixmap_create_from_xpm_d(widget->window,
			&S[i].mask, NULL, S[i].xpmdata);
		i++;
	}
}


/* copy a s-meter pixmap to pixmap depending on the value of smax */
static void draw_smeter(gint value)
{
	GtkWidget *drawingarea;
	GdkGC *gc;
	GdkRectangle rectangle;

	drawingarea = lookup_widget(mainwindow, "smeterdrawingarea");
	gc = drawingarea->style->fg_gc[GTK_STATE_NORMAL];
	/* clear background */
	gdk_draw_rectangle(pixmap, drawingarea->style->white_gc, TRUE, 0, 0,
		drawingarea->allocation.width, drawingarea->allocation.height);
	gdk_draw_pixmap(pixmap, gc, S[value].pixmap, 0, 0, 0, 0,
		drawingarea->allocation.width, drawingarea->allocation.height);
	rectangle.x = 0;
	rectangle.y = 0;
	rectangle.width = drawingarea->allocation.width;
	rectangle.height = drawingarea->allocation.height;
	gtk_widget_draw(drawingarea, &rectangle); /* see expose_event further down */
}

/* Retrieve frequency, mode and signal strength with hamlib commands.
 * In case of a timeout we return immediately, we would otherwise
 * block the user interface. We use 4 hamlib commands, which each have 200ms
 * timeout. The timer which calls get_riginfo does this every 350 ms.
 * Also, we never get info from a rig one after another, state.hlcounter counts
 * the number of hamlib commands and each command is invoked seperately
 * when this function is called. This is needed to let hamlib recover from errors.
 */
gint get_riginfo(void) {
	gint retcode, strength, spoint, i, smax = 0;
	freq_t freq;
	rmode_t rmode;
	pbwidth_t width;
	ptt_t ptt;
	GString *digits = g_string_new("");
	GtkWidget *frequencylabel;
	gchar *message;

	if (state.hlcounter == 0)
	{
		retcode = rig_get_freq(myrig, RIG_VFO_CURR, &freq);
		state.hlcounter++;
		if (retcode == RIG_OK)
		{
			if (preferences.round == 0)
				g_string_sprintf(digits, "%lld", freq);
			else
				g_string_sprintf(digits, "%lld", (long long)rintl((long double)freq/pow(10,preferences.round)));
			g_string_insert_c(digits, (digits->len) - 6 + preferences.round, '.');
			g_strstrip(digits->str);
			g_strdelimit(digits->str, " ", '0');
/*			digits->len = strlen(digits->str);
			if (digits->len == 7) g_string_prepend_c(digits, '0');*/
			state.rigfrequency = g_strdup(digits->str);
			if ((preferences.hamlib == 2) || (preferences.hamlib == 4))
			{
				frequencylabel = lookup_widget(mainwindow, "frequencylabel");
				digits = g_string_new(state.rigfrequency);
				digits = g_string_append(digits, " MHz");
				gtk_label_set_text(GTK_LABEL(frequencylabel), digits->str);
			}
		g_string_free(digits, TRUE);
		}
		else if (retcode != -RIG_ENAVAIL)
		{
			message = g_strdup_printf("Hamlib error %d: %s", retcode, rigerror(retcode));
			update_statusbar(message);
			g_free(message);
			return 1;
		}
	}

	if (state.hlcounter == 1)
	{
		retcode = rig_get_mode(myrig, RIG_VFO_CURR, &rmode, &width);
		state.hlcounter++;
		if (retcode == RIG_OK )
		{
			switch(rmode)
			{
				case RIG_MODE_AM:
					state.rigmode = g_strdup("AM");
				break;
				case RIG_MODE_CW:
					state.rigmode = g_strdup("CW");
				break;
				case RIG_MODE_USB:
					state.rigmode = g_strdup("USB");
				break;
				case RIG_MODE_LSB:
					state.rigmode = g_strdup("LSB");
				break;
				case RIG_MODE_RTTY:
					state.rigmode = g_strdup("RTTY");
				break;
				case RIG_MODE_FM:
					state.rigmode = g_strdup("FM");
				break;
				case RIG_MODE_WFM:
					state.rigmode = g_strdup("WFM");
				break;
				default:
					state.rigmode = g_strdup("UNKNOWN");
				break;
			}
		}
		else if (retcode != -RIG_ENAVAIL)
		{
			message = g_strdup_printf("Hamlib error %d: %s", retcode, rigerror(retcode));
			update_statusbar(message);
			g_free(message);
			return 1;
		}
	}

	/* are we transmitting or receiving ?*/
	if (state.hlcounter == 2)
	{
		retcode = rig_get_ptt(myrig, RIG_VFO_CURR, &ptt);
		state.hlcounter++;
		if (retcode == RIG_OK)
		{
			if (ptt == RIG_PTT_OFF)
				state.tx = FALSE;
			else
				state.tx = TRUE;
		}
		else if (retcode != -RIG_ENAVAIL)
		{
			message = g_strdup_printf("Hamlib error %d: %s", retcode, rigerror(retcode));
			update_statusbar(message);
			g_free(message);
			return 1;
		}
	}
			
	if (state.hlcounter == 3)
	{
		state.hlcounter = 0;
		if (!state.tx)
		{
			retcode = rig_get_strength(myrig, RIG_VFO_CURR, &strength);
			if (retcode == RIG_OK )
			{
				if (strength > 0) spoint = 9; 
				else if (strength < -60) spoint = 0;
				else spoint = (gint)ceil((strength+60)/6);

				svalue[state.scounter] = spoint;
				state.scounter++;
				if (state.scounter == 10) state.scounter = 0;

				/* find maximum of s-meter during last 3 seconds */
				for (i = 0; i < 10; i++)
					if (svalue[i] > smax) smax = svalue[i];
		
				if ((preferences.hamlib ==3) || (preferences.hamlib == 4))
					draw_smeter(smax);

				if (smax == 0) smax = 1;
				if ((g_strcasecmp(state.rigmode, "CW") == 0) || (g_strcasecmp(state.rigmode, "RTTY") == 0))
					state.rigrst = g_strdup_printf("5%d9", smax);
				else
					state.rigrst = g_strdup_printf("5%d", smax);
			}
			else if (retcode != -RIG_ENAVAIL)
			{
				message = g_strdup_printf("Hamlib error %d: %s", retcode, rigerror(retcode));
				update_statusbar(message);
				g_free(message);
				return 1;
			}

		}
		else 
		{
			if ((preferences.hamlib == 3) || (preferences.hamlib == 4))
				draw_smeter(0);
		}
	}

	return 1;
}

/* set appearance of some widgets dependent on preferences.hamlib */
void sethamlibwidgets(gint status) {
	GtkWidget *mhzlabel, *mhzbutton, *bandoptionmenu, *bandentry, *frequencypixmap,
		*frequencyhbox, *frequencyhandlebox, *frequencylabel, *modelabel, *modebutton,
		*modeoptionmenu, *modeentry, *rstlabel, *rstbutton, *smeterhandlebox, *smeterhbox,
		*smeterdrawingarea, *smeterpixmap;

	mhzlabel = lookup_widget(mainwindow, "mhzlabel");
	mhzbutton = lookup_widget(mainwindow, "mhzbutton");
	modelabel = lookup_widget(mainwindow, "modelabel");
	modebutton = lookup_widget(mainwindow, "modebutton");
	rstlabel = lookup_widget(mainwindow, "rstlabel");
	rstbutton = lookup_widget(mainwindow, "rstbutton");
	bandoptionmenu = lookup_widget(mainwindow, "bandoptionmenu");
	bandentry = lookup_widget(mainwindow, "bandentry");
	modeoptionmenu = lookup_widget(mainwindow, "modeoptionmenu");
	modeentry = lookup_widget(mainwindow, "modeentry");
	frequencypixmap = lookup_widget(mainwindow, "frequencypixmap");
	frequencyhbox = lookup_widget(mainwindow, "frequencyhbox");
	frequencyhandlebox = lookup_widget(mainwindow, "frequencyhandlebox");
	frequencylabel = lookup_widget(mainwindow, "frequencylabel");
	smeterhandlebox = lookup_widget(mainwindow, "smeterhandlebox");
	smeterhbox = lookup_widget(mainwindow, "smeterhbox");
	smeterdrawingarea = lookup_widget(mainwindow, "smeterdrawingarea");
	smeterpixmap = lookup_widget(mainwindow, "smeterpixmap");

	if (status == 0) /* hamlib disabled */
	{
		gtk_widget_show(mhzlabel);
		gtk_widget_hide(mhzbutton);
		gtk_widget_show(modelabel);
		gtk_widget_hide(modebutton);
		gtk_widget_show(rstlabel);
		gtk_widget_hide(rstbutton);
		gtk_widget_hide(frequencylabel);
		gtk_widget_hide(frequencypixmap);
		gtk_widget_hide(frequencyhbox);
		gtk_widget_hide(frequencyhandlebox);
		gtk_widget_hide(smeterhandlebox);
		gtk_widget_hide(smeterhbox);
		gtk_widget_hide(smeterdrawingarea);
		gtk_widget_hide(smeterpixmap);
	}
	else if (status == 1) /* hamlib enabled */
	{
		gtk_widget_hide(mhzlabel);
		gtk_widget_show(mhzbutton);
		gtk_widget_hide(modelabel);
		gtk_widget_show(modebutton);
		gtk_widget_hide(rstlabel);
		gtk_widget_show(rstbutton);
		gtk_widget_hide(bandoptionmenu);
		gtk_widget_hide(modeoptionmenu);
		gtk_widget_show(bandentry);
		gtk_widget_show(modeentry);
		preferences.bandseditbox = 1;
		preferences.modeseditbox = 1;
		gtk_widget_hide(frequencylabel);
		gtk_widget_hide(frequencypixmap);
		gtk_widget_hide(frequencyhbox);
		gtk_widget_hide(frequencyhandlebox);
		gtk_widget_hide(smeterhandlebox);
		gtk_widget_hide(smeterhbox);
		gtk_widget_hide(smeterdrawingarea);
		gtk_widget_hide(smeterpixmap);
	}
	else if (status == 2) /* hamlib enabled with frequency on statusbar*/
	{
		gtk_widget_hide(mhzlabel);
		gtk_widget_show(mhzbutton);
		gtk_widget_hide(modelabel);
		gtk_widget_show(modebutton);
		gtk_widget_hide(rstlabel);
		gtk_widget_show(rstbutton);
		gtk_widget_hide(bandoptionmenu);
		gtk_widget_hide(modeoptionmenu);
		gtk_widget_show(bandentry);
		gtk_widget_show(modeentry);
		preferences.bandseditbox = 1;
		preferences.modeseditbox = 1;
		gtk_widget_show(frequencylabel);
		gtk_widget_show(frequencypixmap);
		gtk_widget_show(frequencyhbox);
		gtk_widget_show(frequencyhandlebox);
		gtk_widget_hide(smeterhandlebox);
		gtk_widget_hide(smeterhbox);
		gtk_widget_hide(smeterdrawingarea);
		gtk_widget_hide(smeterpixmap);
	}
	else if (status == 3) /* hamlib enabled with s-meter on statusbar*/
	{
		gtk_widget_hide(mhzlabel);
		gtk_widget_show(mhzbutton);
		gtk_widget_hide(modelabel);
		gtk_widget_show(modebutton);
		gtk_widget_hide(rstlabel);
		gtk_widget_show(rstbutton);
		gtk_widget_hide(bandoptionmenu);
		gtk_widget_hide(modeoptionmenu);
		gtk_widget_show(bandentry);
		gtk_widget_show(modeentry);
		preferences.bandseditbox = 1;
		preferences.modeseditbox = 1;
		gtk_widget_hide(frequencylabel);
		gtk_widget_hide(frequencypixmap);
		gtk_widget_hide(frequencyhbox);
		gtk_widget_hide(frequencyhandlebox);
		gtk_widget_show(smeterhandlebox);
		gtk_widget_show(smeterhbox);
		gtk_widget_show(smeterdrawingarea);
		gtk_widget_show(smeterpixmap);
	}
	else if (status == 4) /* hamlib enabled with frequency and s-meter on statusbar*/
	{
		gtk_widget_hide(mhzlabel);
		gtk_widget_show(mhzbutton);
		gtk_widget_hide(modelabel);
		gtk_widget_show(modebutton);
		gtk_widget_hide(rstlabel);
		gtk_widget_show(rstbutton);
		gtk_widget_hide(bandoptionmenu);
		gtk_widget_hide(modeoptionmenu);
		gtk_widget_show(bandentry);
		gtk_widget_show(modeentry);
		preferences.bandseditbox = 1;
		preferences.modeseditbox = 1;
		gtk_widget_show(frequencylabel);
		gtk_widget_show(frequencypixmap);
		gtk_widget_show(frequencyhbox);
		gtk_widget_show(frequencyhandlebox);
		gtk_widget_show(smeterhandlebox);
		gtk_widget_show(smeterhbox);
		gtk_widget_show(smeterdrawingarea);
		gtk_widget_show(smeterpixmap);
	}
}

/* load s-meter pixmaps when main window is displayed */
void on_mainwindow_show(GtkWidget *widget, gpointer user_data)
{
	GtkWidget *drawingarea;

	if ((preferences.hamlib ==3) || (preferences.hamlib == 4))
	{
		drawingarea = lookup_widget(mainwindow, "smeterdrawingarea");
		loadsmeter(drawingarea);
	}
}

/* create a new backing pixmap for the s-meter whenever the window is resized */
gboolean on_smeterdrawingarea_configure_event(GtkWidget *widget,
	GdkEventConfigure *event, gpointer user_data)
{
	if ((preferences.hamlib ==3) || (preferences.hamlib == 4))
	{
		if (pixmap) gdk_pixmap_unref(pixmap);
		pixmap = gdk_pixmap_new(widget->window, widget->allocation.width, widget->allocation.height, -1);
	}
	return FALSE;
}

/* copy the background pixmap to the drawing area for the s-meter */
gboolean on_smeterdrawingarea_expose_event(GtkWidget *widget,
	GdkEventExpose *event, gpointer user_data)
{
	if ((preferences.hamlib ==3) || (preferences.hamlib == 4))
	gdk_draw_pixmap(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE(widget)], pixmap,
		event->area.x, event->area.y, event->area.x, event->area.y, event->area.width, event->area.height);
	return FALSE;
}
#endif
