/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <X11/Xlib.h>
#include <gtk/gtk.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>

#include "singit_macros.h"
#include "singit_config.h"
#include "singit_dialogs.h"
#include "singit_displayer_plugin.h"
#include "singit_wgt_karaoke.h"

void displayer_karaoke_window_init(void);
void displayer_karaoke_window_finish(void);

void displayer_karaoke_window_show(void);
void displayer_karaoke_window_hide(void);
void displayer_karaoke_window_toggle(void);

void displayer_karaoke_window_set_time(guint time, LSong *cur_song, GList *real_next);
void displayer_karaoke_window_config_update(const SingitConfigData *singit_config);

DisplayerPlugin karaoke_window_dp =
{
	NULL,
	NULL,
	0,

	"Karaoke window displayer",

	displayer_karaoke_window_init,
	displayer_karaoke_window_show,
	displayer_karaoke_window_hide,
	displayer_karaoke_window_toggle,
	displayer_karaoke_window_finish,
	NULL,
	displayer_karaoke_window_config_update,

	displayer_karaoke_window_set_time,

	NULL,
	NULL,
	NULL,
	NULL,

	NULL,
	NULL
};

DisplayerPlugin *get_dplugin_info(void)
{
	karaoke_window_dp.description = g_strdup("Karaoke window displayer");
	return &karaoke_window_dp;
}

static GtkWidget *displayer_karaoke_window_win = NULL;
static GtkWidget
		*singit_vbox,
			*singit_hbox,
			*singit_timestamp_status_bar,
			*singit_song_gtk_event_box, *singit_song_status_bar,
			*displayer_karaoke_window_karaoke_widget;
static GtkTooltips
		*singit_tooltips;

static gboolean displayer_karaoke_window_win_keypress(GtkWidget * w, GdkEventKey * event, gpointer data);

// Changed by program - not restored
typedef struct {

	GdkColormap *winColorMap;
	GdkColor     winPrivateColors[3];

	guint timestamp_context_id;
	guint song_context_id;
	guint messageCounter;
	gint jump_to_song;

	gboolean canChangeVisibility;
	gboolean isVisible;

	gchar *fileName;
	LSong *song;

	gboolean print_debug;
	gboolean toggle_yz;
	gboolean guess_synced_lyrics;
	gboolean show_jumping_ball;
	gboolean hide_if_not_found;
	gboolean auto_resize;

}
KaraokeWindowStatus;

KaraokeWindowStatus karaoke_window_status;

#define DEBUG(x)  { if (karaoke_window_status.print_debug) debug x; }
#define PERROR(x) { if (karaoke_window_status.print_debug) perror(x); }

void set_jumping_ball(void)
{
	if (karaoke_window_status.guess_synced_lyrics) {
		singit_karaoke_widget_set_jumping_ball(
			SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget),
			(l_song_guess_sync_lyrics(karaoke_window_status.song) &&
			karaoke_window_status.show_jumping_ball));
	}
	else {
		singit_karaoke_widget_set_jumping_ball(
			SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget),
			karaoke_window_status.show_jumping_ball);
	}
}

gboolean displayer_karaoke_window_change_visibility(gboolean visible, gboolean check)
{
	static gboolean shouldBeVisible = TRUE;
	if (!displayer_karaoke_window_win) { return FALSE; }
	if (!check) { shouldBeVisible = (!karaoke_window_status.hide_if_not_found || visible); }
	if (!karaoke_window_status.canChangeVisibility && !shouldBeVisible) { return FALSE; }
	if (shouldBeVisible == karaoke_window_status.isVisible) { return FALSE; }

	#ifdef CODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_change_visibility]\n"));
	#endif

	if (shouldBeVisible) { gtk_widget_show(displayer_karaoke_window_win); }
	else { gtk_widget_hide(displayer_karaoke_window_win); }
	karaoke_window_status.isVisible = shouldBeVisible;
	return TRUE;
}

static void set_statusbar_messages(gint time)
{
	gint new_message_count;
	gchar timeStamp[10];

	if (!displayer_karaoke_window_win) { return; }

	timeStamp[9] = '\0';
	new_message_count = (time / 1000);
	if (new_message_count < 0) { new_message_count = 0; }
	else
		if (!(time % 1000 == 0)) { new_message_count++; }
	if (karaoke_window_status.messageCounter == new_message_count) { return; }

	if (karaoke_window_status.timestamp_context_id != 0) {
		while (karaoke_window_status.messageCounter > new_message_count) {
			gtk_statusbar_pop(GTK_STATUSBAR(singit_timestamp_status_bar), karaoke_window_status.timestamp_context_id);
			karaoke_window_status.messageCounter--;
		}
	}
	else {
		karaoke_window_status.timestamp_context_id =
			gtk_statusbar_get_context_id(GTK_STATUSBAR(singit_timestamp_status_bar), "timestamps");
		karaoke_window_status.messageCounter = 0;
	}
	while (karaoke_window_status.messageCounter < new_message_count) {
		karaoke_window_status.messageCounter++;
		sprintf(timeStamp, " - %.2i:%.2i ", karaoke_window_status.messageCounter / 60, karaoke_window_status.messageCounter % 60);
		gtk_statusbar_push
			(GTK_STATUSBAR(singit_timestamp_status_bar), karaoke_window_status.timestamp_context_id, timeStamp);
	}
}

static void set_status_filename(gchar *fileName)
{
	gchar *tmp;
	if (!displayer_karaoke_window_win) { return; }
	if (karaoke_window_status.song_context_id) {
		gtk_statusbar_pop(GTK_STATUSBAR(singit_song_status_bar), karaoke_window_status.song_context_id);
	}
	else {
		karaoke_window_status.song_context_id =
			gtk_statusbar_get_context_id(GTK_STATUSBAR(singit_song_status_bar), "filename");
	}
	tmp = g_strconcat(" ", fileName, NULL);
	gtk_statusbar_push(GTK_STATUSBAR(singit_song_status_bar), karaoke_window_status.song_context_id, tmp);
	gtk_tooltips_set_tip(singit_tooltips, singit_song_gtk_event_box, fileName, NULL);
	g_free(tmp);
}

void displayer_karaoke_window_config_update(const SingitConfigData *singit_config)
{
	karaoke_window_status.print_debug = singit_config->debugEnable;
	karaoke_window_status.toggle_yz = singit_config->toggleYZ;
	karaoke_window_status.guess_synced_lyrics = singit_config->guessSyncLyrics;
	karaoke_window_status.show_jumping_ball = singit_config->showJumpingBall;
	karaoke_window_status.hide_if_not_found = singit_config->hideIfNotFound;
	karaoke_window_status.auto_resize = singit_config->autoResize;

	singit_karaoke_widget_freeze
		(SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget));
	singit_karaoke_widget_set_color(
		SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget),
		skwc_text, singit_config->textColor);
	singit_karaoke_widget_set_color(
		SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget),
		skwc_active, singit_config->activeColor);
	singit_karaoke_widget_set_color(
		SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget),
		skwc_background, singit_config->backgroundColor);
	singit_karaoke_widget_set_font(
		SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget),
		singit_config->lyricFontName);
	set_jumping_ball();
	singit_karaoke_widget_thaw
		(SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget));
}

void displayer_karaoke_window_set_time(guint time, LSong *cur_song, GList *real_next)
{
	LSong *my_song = NULL;
	GList *item;

	if (displayer_karaoke_window_win && GTK_WIDGET_VISIBLE(displayer_karaoke_window_win))
	{
		my_song = l_song_attach(cur_song);
		if (!my_song) { return; }

		GDK_THREADS_ENTER();
		if ((my_song != karaoke_window_status.song) ||
			(strcmp(my_song->song_filename, karaoke_window_status.song->song_filename) != 0))
		{
			if (l_song_lyrics_found(my_song)) {
				set_status_filename(my_song->lyric_filename);
			}
			singit_karaoke_widget_freeze
				(SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget));
			singit_karaoke_widget_set_song
				(SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget), my_song);
			singit_karaoke_widget_set_time
				(SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget), time);

			l_song_detach(karaoke_window_status.song, TRUE);
			karaoke_window_status.song = my_song;
			set_jumping_ball();

			singit_karaoke_widget_thaw
				(SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget));
		}
		else {
			singit_karaoke_widget_set_time
				(SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget), time);
			if (l_song_lyrics_found(my_song)) {
				item = (real_next) ? real_next : my_song->last_token;
				if (item) { set_statusbar_messages(tTime(item) - time); }
				l_song_detach(my_song, TRUE);
			}
		}
		GDK_THREADS_LEAVE();
	}
}

static gboolean displayer_karaoke_window_win_keypress(GtkWidget * w, GdkEventKey * event, gpointer data)
{
	gint volume, value;
	LSong *cur_song = l_song_attach(karaoke_window_status.song);

        switch(event->keyval) {
                case GDK_Escape: // exit plugin
			karaoke_window_dp.disable(&karaoke_window_dp);
                        break;
		case GDK_a:
			if (event->state & GDK_CONTROL_MASK) { singit_about_show(); }
			break;
                case GDK_b:
                        xmms_remote_playlist_next(karaoke_window_dp.xmms_session);
                        break;
                case GDK_c:
			if (event->state & GDK_CONTROL_MASK) {  }
			else { xmms_remote_pause(karaoke_window_dp.xmms_session); }
                        break;
		case GDK_e:
			if (event->state & GDK_CONTROL_MASK) { singit_editor_show(); }
			break;
#		ifdef HAVE_ID3
		case GDK_i:
			if (event->state & GDK_CONTROL_MASK) { singit_id3dlg_show(); }
			break;
#		endif
		case GDK_p:
			if (event->state & GDK_CONTROL_MASK) { singit_config_show(); }
			break;
                case GDK_v:
                        xmms_remote_stop(karaoke_window_dp.xmms_session);
                        break;
                case GDK_x:
                        xmms_remote_play(karaoke_window_dp.xmms_session);
                        break;
		case GDK_y:
			if (karaoke_window_status.toggle_yz)
				{ xmms_remote_playlist_prev(karaoke_window_dp.xmms_session); }
			break;
                case GDK_z:
			if (!karaoke_window_status.toggle_yz)
				{ xmms_remote_playlist_prev(karaoke_window_dp.xmms_session); }
                        break;
		case GDK_KP_Add:
			volume = xmms_remote_get_main_volume(karaoke_window_dp.xmms_session);
			if (volume < 100) { volume++; }
			xmms_remote_set_main_volume(karaoke_window_dp.xmms_session, volume);
			break;
		case GDK_KP_Subtract:
			volume = xmms_remote_get_main_volume(karaoke_window_dp.xmms_session);
			if (volume > 0) { volume--; }
			xmms_remote_set_main_volume(karaoke_window_dp.xmms_session, volume);
			break;
		case GDK_KP_Enter:
			if (karaoke_window_status.jump_to_song == 0) { break; }
			if (xmms_remote_get_playlist_pos(karaoke_window_dp.xmms_session) !=
				(karaoke_window_status.jump_to_song - 1)) {
					xmms_remote_set_playlist_pos(karaoke_window_dp.xmms_session,
						(karaoke_window_status.jump_to_song - 1));
			}
		case GDK_KP_Decimal:
			karaoke_window_status.jump_to_song = 0;
			break;
		case GDK_KP_0:
		case GDK_KP_1:
		case GDK_KP_2:
		case GDK_KP_3:
		case GDK_KP_4:
		case GDK_KP_5:
		case GDK_KP_6:
		case GDK_KP_7:
		case GDK_KP_8:
		case GDK_KP_9:
		case GDK_0:
		case GDK_1:
		case GDK_2:
		case GDK_3:
		case GDK_4:
		case GDK_5:
		case GDK_6:
		case GDK_7:
		case GDK_8:
		case GDK_9:
			value = event->keyval - GDK_0;
			if (value < 0 || value > 9)
				value = event->keyval - GDK_KP_0;

			if (event->state & GDK_CONTROL_MASK) {
				enable_plugin(((value + 9) % 10));
			}
			else {
				karaoke_window_status.jump_to_song = (karaoke_window_status.jump_to_song * 10) + value;
				if (karaoke_window_status.jump_to_song > xmms_remote_get_playlist_length
					(karaoke_window_dp.xmms_session)) { karaoke_window_status.jump_to_song = value; }
			}
			break;
                default:
			// printf("%x - %x\n", event->state, event->keyval);
                        break;
        }

	l_song_detach(cur_song, TRUE);
        return TRUE;
}

void displayer_karaoke_window_finish(void)
{
	gint posX, posY;

	#ifdef CODECODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_finish]\n"));
	#endif

	if (displayer_karaoke_window_win) {
		if (displayer_karaoke_window_win->window) {
			gdk_window_get_root_origin(displayer_karaoke_window_win->window, &posX, &posY);
			singit_config_save_plugin_position("karaoke_window", posX, posY, -1, -1);
		}
		gtk_widget_destroy(displayer_karaoke_window_win);
		karaoke_window_status.messageCounter = 0;
		karaoke_window_status.timestamp_context_id = 0;
	}

	if (karaoke_window_status.fileName) {
		g_free(karaoke_window_status.fileName);
		karaoke_window_status.fileName = NULL;
	}

	l_song_detach(karaoke_window_status.song, TRUE);
}

gint karaoke_window_delete_event( GtkWidget *widget, GdkEvent  *event, gpointer data )
{
	karaoke_window_dp.disable(&karaoke_window_dp);
	return (FALSE);
}

static void displayer_karaoke_window_build_dialog(void)
{
	gint posX, posY;
	GtkWidget *tmp_win;

	singit_tooltips = gtk_tooltips_new ();

	tmp_win = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_realize(tmp_win);
	gtk_window_set_title(GTK_WINDOW(tmp_win), SINGIT_VERSION_STRING);
	gtk_window_set_policy(GTK_WINDOW(tmp_win), TRUE, FALSE, TRUE);
	gtk_signal_connect(GTK_OBJECT(tmp_win), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &displayer_karaoke_window_win);
	gtk_signal_connect(GTK_OBJECT(tmp_win), "delete_event", GTK_SIGNAL_FUNC(karaoke_window_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(tmp_win), "key-press-event", GTK_SIGNAL_FUNC(displayer_karaoke_window_win_keypress), NULL);
	gtk_container_border_width(GTK_CONTAINER(tmp_win), 5);
	gtk_object_set_data (GTK_OBJECT(tmp_win), "tooltips", singit_tooltips);

	singit_vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(tmp_win), singit_vbox);
	gtk_widget_show(singit_vbox);

	displayer_karaoke_window_karaoke_widget = singit_karaoke_widget_new();
	gtk_box_pack_start(GTK_BOX(singit_vbox), displayer_karaoke_window_karaoke_widget, FALSE, FALSE, 0);
	singit_karaoke_widget_set_show_empty_lines (
		SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget), TRUE);
	singit_karaoke_widget_set_lines (SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget), 5);
	singit_karaoke_widget_set_toplines (SINGIT_KARAOKE_WIDGET(displayer_karaoke_window_karaoke_widget), 2);
	gtk_widget_show(displayer_karaoke_window_karaoke_widget);

	singit_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(singit_vbox), singit_hbox, FALSE, FALSE, 1);
	gtk_widget_show(singit_hbox);

	singit_timestamp_status_bar = gtk_statusbar_new();
	gtk_box_pack_start (GTK_BOX(singit_hbox), singit_timestamp_status_bar, FALSE, FALSE, 0);
	gtk_widget_show(singit_timestamp_status_bar);

	singit_song_gtk_event_box = gtk_event_box_new();
	gtk_container_add (GTK_CONTAINER(singit_hbox), singit_song_gtk_event_box);
	gtk_widget_show (singit_song_gtk_event_box);

	singit_song_status_bar = gtk_statusbar_new();
	gtk_container_add (GTK_CONTAINER (singit_song_gtk_event_box), singit_song_status_bar);
	gtk_widget_show(singit_song_status_bar);

	karaoke_window_status.isVisible = FALSE;

	karaoke_window_status.timestamp_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(singit_timestamp_status_bar), "timestamps");
	gtk_statusbar_push(GTK_STATUSBAR(singit_timestamp_status_bar),
		karaoke_window_status.timestamp_context_id, "         ");

	if (singit_config_load_plugin_position("karaoke_window", &posX, &posY, NULL, NULL)) {
		gtk_widget_set_uposition(tmp_win, posX, posY);
	}
	else
		gtk_window_set_position
			(GTK_WINDOW(tmp_win), GTK_WIN_POS_CENTER);

	displayer_karaoke_window_win = tmp_win;
}

void displayer_karaoke_window_init(void)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_init] : Start\n"));
	#endif

	karaoke_window_status.timestamp_context_id = 0;
	karaoke_window_status.song_context_id = 0;
	karaoke_window_status.messageCounter = 0;
	karaoke_window_status.canChangeVisibility = TRUE;
	karaoke_window_status.fileName = NULL;
	karaoke_window_status.song = NULL;
	karaoke_window_status.jump_to_song = 0;

	displayer_karaoke_window_build_dialog();

	#ifdef CODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_init] : Finish\n"));
	#endif
}

void displayer_karaoke_window_show(void)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_show]\n"));
	#endif

	if (displayer_karaoke_window_win) {
		if (!GTK_WIDGET_VISIBLE(displayer_karaoke_window_win))
			{ gtk_widget_show(displayer_karaoke_window_win); }
		else { gdk_window_raise(displayer_karaoke_window_win->window); }
	}
}

void displayer_karaoke_window_hide(void)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_hide]\n"));
	#endif

	if (displayer_karaoke_window_win) {
		if (GTK_WIDGET_VISIBLE(displayer_karaoke_window_win)) {
			gtk_widget_hide(displayer_karaoke_window_win);
		}
	}
}

void displayer_karaoke_window_toggle(void)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_karaoke_window.c [displayer_karaoke_window_toggle]\n"));
	#endif

	if (displayer_karaoke_window_win) {
		if (gdk_window_is_visible(displayer_karaoke_window_win->window))
			{ gtk_widget_hide(displayer_karaoke_window_win); }
		else  { gtk_widget_show(displayer_karaoke_window_win); }
	}
}
