/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <pthread.h>

#ifdef HAVE_SCHED_SETSCHEDULER
#include <sched.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <X11/Xlib.h>

#include <stdlib.h>
#include <stdio.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include "singit_macros.h"
#include "singit_macros_private.h"
#include "singit_song_private.h"
#include "singit_config_private.h"
#include "dlg_singit_config.h"

#include "singit_main.h"
#include "singit_plugin_scanner.h"
#include "editor_singit_main.h"
#include "dlg_singit_about.h"

#ifdef HAVE_ID3
#include "dlg_singit_id3.h"
#endif

#if ((defined CODEDEBUG) && (defined USE_MTRACE))
#include "mcheck.h"
#endif

extern GtkObject *singit_config;
extern VisPlugin singit_vp;

SingitStatus singit_status = { NULL, NULL, 0, 0, { 0 }, { 0 }, { 0 }, FALSE, FALSE, 0, FALSE };

gboolean song_changed = TRUE;

void singit_main_set_timeout();

/***********************************************************************
*
*	Display thread and helper function(s)
*	This thread controls the visual updates of the displayers
*
***********************************************************************/

void set_new_lyric_line(gint time, gint state, LSong* cur_song)
{
	GList *item = NULL;

	if ((state == 1) && (cur_song->active_token == cur_song->last_token)) { return; }
	if ((state == -1) && (cur_song->active_token == NULL)) { return; }

#	ifdef CODEDEBUG
	DEBUG(("singit_main.c [set_new_lyric_line] : "), 9);
#	endif

	// Find current token and next line token
	item = l_song_find_current_token(cur_song, time, state);
	singit_status.next_lyric_line = l_song_find_next_lyric_line(cur_song, item, TRUE, NULL);

	// Set the new tokens
	cur_song->active_token = item;

	#ifdef CODEDEBUG
        DEBUG((" Ok\n"), 9);
        #endif

//	printf("t: %i / %i\n", time, tTime(item));
}

void *display_thread_func (void *args)
{
	gint time, old_time = -1;
	gint status = 0;
	LSong* cur_song = NULL;
	gboolean real_change = FALSE;
	gboolean hickup = FALSE;

#	ifdef CODEDEBUG
	DEBUG(("singit_main.c [display_thread_func] : Thread started\n"), 9);
#	endif

	cur_song = l_song_attach(getSCD(singit_config)->song);

#	ifdef HAVE_SCHED_SETSCHEDULER
	if (xmms_check_realtime_priority())
	{
		struct sched_param sparam;
		sparam.sched_priority = sched_get_priority_max(SCHED_OTHER);
		pthread_setschedparam(pthread_self(), SCHED_OTHER, &sparam);
	}
#	endif


	while (!singit_status.kill_display_thread)
	{
		// To prevent "unlimited" ifs we first schedule the thread
/*
#	ifdef __USE_GNU
		if (pthread_yield() != 0)
			{ xmms_usleep(100); }
#	else
#		ifdef HAVE_SCHED_H
		if (sched_yield() != 0)
			{ xmms_usleep(100); }
#		else
		xmms_usleep(100);
#		endif
#	endif
*/
		if (l_song_lyrics_found(cur_song)) {
			if (getSCD(singit_config)->useXMMSuSleep) { xmms_usleep(1000); }
		}
		else {
			xmms_usleep(10000);
		}

		time = xmms_remote_get_output_time(singit_vp.xmms_session);

		// This took me nearly 4 months to find out
		// xmms return "-1" time while changing songs which crashed the plugin - damn
		// Before I installed loops to skip "some time" - sometimes it worked sometimes not ;-(
		// The soncond item seems to be an error in xmms (not sure)
		if (time < 0) { continue; }
		if (time == 0 && (old_time != time) && !song_changed) {
			if (!hickup) {
				hickup = TRUE;
				continue;
			}
		}

		if (old_time != time)
			singit_editor_set_time(time);

		if (song_changed && !real_change) {
			pthread_mutex_lock(&singit_status.lyrics_rw_mutex);
			real_change = song_changed;
			cur_song = l_song_detach(cur_song, TRUE);
			cur_song = l_song_attach(getSCD(singit_config)->song);
			song_changed = FALSE;
			pthread_mutex_unlock(&singit_status.lyrics_rw_mutex);
		}

		if (l_song_lyrics_found(cur_song)) {
			status = inl_l_song_is_time_ok(cur_song, time);
			if (status != 0) {
#				ifdef CODEDEBUG
				DEBUG((" %.2i:%.2i - ", time / 60000, (time / 1000) % 60), 9);
#				endif
				set_new_lyric_line(time, status, cur_song);
			}
		}

		if (get_dis_plugin_enabled_list(TRUE) && (l_song_lyrics_found(cur_song) || real_change || singit_status.initialize_plugins))
		{
			if (real_change || singit_status.initialize_plugins) {
				singit_status.initialize_plugins = FALSE;
				GDK_THREADS_ENTER();
				(l_song_lyrics_found(cur_song)) ? dis_plugin_show(-1) : dis_plugin_hide(-1);
				GDK_THREADS_LEAVE();
				real_change = FALSE;
				plugins_set_time(time, cur_song, singit_status.next_lyric_line);
				old_time = time;
				if (singit_status.kill_display_thread) { break; }
			}

			if (old_time != time) {
				pthread_mutex_lock(&singit_status.time_update_mutex);
				plugins_set_time(time, cur_song, singit_status.next_lyric_line);
				pthread_mutex_unlock(&singit_status.time_update_mutex);
			}

			pthread_mutex_lock(&singit_status.config_rw_mutex);
			if (!singit_status.initialize_plugins && singit_status.config_update)
			{
				GDK_THREADS_ENTER();
				dis_plugin_update(-1);
				GDK_THREADS_LEAVE();
				singit_status.config_update = FALSE;
			}
			pthread_mutex_unlock(&singit_status.config_rw_mutex);
		}

		old_time = time;
	}

	cur_song = l_song_detach(cur_song, TRUE);

#	ifdef CODEDEBUG
	DEBUG(("singit_main.c [display_thread_func] : Thread ended\n"), 9);
#	endif

        pthread_exit(0);
}


/***********************************************************************
*
*	Check thread and helper functions
*	This thread checks, if the contents of a file has changed
*	All funtions are inlined because they are used only
*	once and are just seperated for overview purposes
*
***********************************************************************/

inline gboolean check_lyric_filenames(LSong *check_song)
{
        gchar *buffer, *fileName, *fileBase;
	gchar **directories, **extensions, *text;
        gint i = 0, j = 0;
	gboolean result = FALSE;

#	ifdef CODEDEBUG
	DEBUG(("singit_main.c [check_filenames]\n"), 9);
#	endif

	fileBase = strdup(g_basename(check_song->song_filename));
	buffer = strrchr(fileBase, '.');
	if (buffer != NULL) { buffer[1] = '\0'; }

	singit_status.next_lyric_line = NULL;

	text = g_strconcat(getSCD(singit_config)->basePath, ",", g_dirname(check_song->song_filename), "/", 0);
	directories = g_strsplit(text, ",", 0);
	g_free(text);
	extensions = g_strsplit(getSCD(singit_config)->lyricExtension, ",", 0);
	while ((directories[i]) && (!result)) {
		j = 0;
		while ((extensions[j]) && (!result)) {
			if (directories[i][0] == '~') {
				if (directories[i][1] == '~') {
					buffer = &directories[i][2];
					fileName = g_strconcat(g_dirname(check_song->song_filename), buffer, fileBase, extensions[j], NULL);
				}
				else {
					buffer = &directories[i][1];
					fileName = g_strconcat(g_get_home_dir(), buffer, fileBase, extensions[j], NULL);
				}
			}
			else { fileName = g_strdup(g_strconcat(directories[i], fileBase, extensions[j], NULL)); }
#			ifdef CODEDEBUG
			DEBUG(("     %s\n", fileName), 9);
#			endif
			result = l_song_load_lyrics(check_song, fileName);
			if (result != l_song_text_found(check_song))
				{ g_print("singit_main.c [check_lyric_filenames] : Error\n"); }
			g_free(fileName);
			j++;
		}
		i++;
	}
	g_strfreev(directories);
	g_strfreev(extensions);
	g_free(fileBase);

	return result;
}

inline gboolean get_lsong_lyrics(LSong *check_song)
{
	gboolean result;

#	ifdef CODEDEBUG
	DEBUG(("singit_main.c [get_lsong_lyrics]\n"), 9);
#	endif

	result = check_lyric_filenames(check_song);

#	ifdef HAVE_ID3
	if (!result) {
#		ifdef CODEDEBUG
		DEBUG(("       %s\n", check_song->song_filename), 9);
#		endif
		result = l_song_load_lyrics(check_song, check_song->song_filename);
	}
#	endif

	return result;
}

inline gboolean has_song_changed(LSong *cur_song)
{
	static gint previous_length = -1;
	gchar *fileName = NULL;
	gint pos, length;
	gboolean lyric_changed = FALSE;

	// Get all important values from xmms to check if the song changed
	pos = xmms_remote_get_playlist_pos(singit_vp.xmms_session);
	length = xmms_remote_get_playlist_time(singit_vp.xmms_session, pos);
	fileName = xmms_remote_get_playlist_file(singit_vp.xmms_session, pos);

	// fileName == NULL when plugin closed
	if (!fileName) { return FALSE; }

	lyric_changed = (length != previous_length);
	if (!lyric_changed) {
		if (!singit_status.fileName) { lyric_changed = TRUE; }
		else { lyric_changed = (strcmp(singit_status.fileName, fileName) != 0); }
	}
	if (cur_song && (!lyric_changed)) { lyric_changed = l_song_lyrics_changed(cur_song); }
        if (lyric_changed) {
		g_free(singit_status.fileName);
		singit_status.fileName = fileName;
               	previous_length = length;
       	}
	else { g_free(fileName); }

	return lyric_changed;
}

/*
 * We have two song pointers
 * One contains the current song, the other one is used for checks
 * We don't need to alloc song "objects" for every check
 */
void *check_thread_func (void *args)
{
	enum {
		CUR_SONG = 0,
		CHECK_SONG
	};
	LSong* songs[2], *xchg_song;
	gboolean lyrics_changed;
	gint time;

	// Get two song objects
	getSCD(singit_config)->song = songs[CUR_SONG] = l_song_new(NULL);
	songs[CHECK_SONG] = l_song_new(NULL);

#	ifdef CODEDEBUG
	DEBUG(("singit_main.c [check_thread_func] : Thread started\n"), 9);
#	endif

	while (!singit_status.kill_display_thread) {
		time = xmms_remote_get_output_time(singit_vp.xmms_session);

		if (time >= 0) {
			pthread_mutex_lock(&singit_status.lyrics_rw_mutex);
			lyrics_changed = has_song_changed(songs[CUR_SONG]);
			if (lyrics_changed) {
				l_song_set_song_filename(songs[CHECK_SONG], singit_status.fileName);
				get_lsong_lyrics(songs[CHECK_SONG]);

				pthread_mutex_lock(&singit_status.time_update_mutex);
				xchg_song = songs[CUR_SONG];
				getSCD(singit_config)->song = songs[CUR_SONG] = songs[CHECK_SONG];
				l_song_detach(xchg_song, TRUE);
				songs[CHECK_SONG] = l_song_new(NULL);

				singit_status.next_lyric_line =
					(l_song_lyrics_found(songs[CUR_SONG])) ? inl_l_song_get_next_token(songs[CUR_SONG]) : NULL;
				pthread_mutex_unlock(&singit_status.time_update_mutex);
				song_changed = TRUE;
#				ifdef CODEDEBUG
				DEBUG(("singit_main.c [check_thread_func] : New song\n"), 9);
#				endif
			}
			pthread_mutex_unlock(&singit_status.lyrics_rw_mutex);
		}
		xmms_usleep(300000);
	}

	// Remove all song references
	getSCD(singit_config)->song = NULL;
	l_song_detach(songs[CUR_SONG], TRUE);
	l_song_detach(songs[CHECK_SONG], TRUE);
	songs[CUR_SONG] = songs[CHECK_SONG] = NULL;

#	ifdef CODEDEBUG
	DEBUG(("singit_main.c [check_thread_func] : Thread ended\n"), 9);
#	endif

        pthread_exit(0);
}

void singit_main_init(gboolean initialize_plugins)
{
#	ifdef CODEDEBUG
	DEBUG(("singit_main.c [singit_main_init]\n"), 9);
#	endif

	singit_status.attachments++;
	if (singit_status.attachments > 1) {
		if (initialize_plugins) {
			pthread_mutex_lock(&singit_status.time_update_mutex);
			plugins_initialize();
			pthread_mutex_unlock(&singit_status.time_update_mutex);
			singit_status.initialize_plugins = TRUE;
		}
		else { singit_main_set_timeout(); }
		return;
	}

#	if ((defined CODEDEBUG) && (defined USE_MTRACE))
	g_print("singit_main.c [singit_main_init] : mtrace\n");
	mtrace();
#	endif

	pthread_mutex_init(&singit_status.config_rw_mutex, NULL);
	pthread_mutex_init(&singit_status.lyrics_rw_mutex, NULL);
	pthread_mutex_init(&singit_status.time_update_mutex, NULL);

#	ifdef ENABLE_NLS
	setlocale (LC_ALL, "");
	bindtextdomain (PACKAGE, SINGIT_LOCALE_DIR);
#	endif

	if (!singit_config_attach()) {
		singit_config_new();
		singit_config_load();
	}

	plugins_init();
	if (initialize_plugins) {
		pthread_mutex_lock(&singit_status.time_update_mutex);
		plugins_initialize();
		pthread_mutex_unlock(&singit_status.time_update_mutex);
		singit_status.initialize_plugins = TRUE;
	}
	singit_main_set_timeout();
}

void singit_main_finish(gboolean finish_plugins)
{
#	ifdef CODEDEBUG
	DEBUG(("singit_main.c [singit_main_finish]\n"), 9);
#	endif

	singit_status.attachments--;
	if (singit_status.attachments > 0) {
		if (finish_plugins) {
			while (!pthread_mutex_trylock(&singit_status.time_update_mutex)) {
				GDK_THREADS_LEAVE();
				while (g_main_iteration(FALSE));
				GDK_THREADS_ENTER();
			}
			plugins_finalize();
			pthread_mutex_unlock(&singit_status.time_update_mutex);
		}
		else { singit_main_set_timeout(); }
		return;
	}

	/*
	 * Little bit tricky to cleanup
	 */
	if (singit_status.display_thread || singit_status.check_thread) {
		// First: Tell the thread to shutdown
		singit_status.kill_display_thread = TRUE;

		pthread_join(singit_status.check_thread, NULL);

		/*
		 * Second: Tell GDK to process all pending events
		 * Since we are in a event call and have the GDK_LOCK
		 * we have to release it first
		 */
		GDK_THREADS_LEAVE();

		// Process all pending events
		while(g_main_iteration(FALSE));

		/*
		 * Finally: Wait for the thread to shut down
		 * and reclaim the GDK_LOCK
		 */
		pthread_join(singit_status.display_thread, NULL);

		GDK_THREADS_ENTER();

		singit_status.kill_display_thread = FALSE;
		singit_status.display_thread = 0;
	}

	plugins_finish();

	pthread_mutex_destroy(&singit_status.config_rw_mutex);
	pthread_mutex_destroy(&singit_status.lyrics_rw_mutex);

	singit_config_hide();
	singit_about_hide();
#ifdef HAVE_ID3
	singit_id3dlg_hide();
#endif
	singit_editor_hide();

	if (singit_status.fileName) {
		g_free(singit_status.fileName);
		singit_status.fileName = NULL;
	}

	singit_config_detach(TRUE);

        singit_status.next_lyric_line = NULL;
	singit_status.attachments = 0;

#	if ((defined CODEDEBUG) && (defined USE_MTRACE))
	g_print("singit_main.c [singit_main_finish] : muntrace\n");
	muntrace();
#	endif

}

void singit_main_set_timeout()
{
	if (!get_dis_plugin_enabled_list(TRUE) && singit_status.display_thread &&
		!singit_editor_is_realized()) {

		singit_status.kill_display_thread = TRUE;

		pthread_join(singit_status.check_thread, NULL);
		pthread_join(singit_status.display_thread, NULL);
		singit_status.kill_display_thread = FALSE;
		singit_status.display_thread = 0;
		singit_status.next_lyric_line = NULL;
	}
	else {
		if ((get_dis_plugin_enabled_list(TRUE) || singit_editor_is_realized()) &&
			!singit_status.display_thread)
		{
			singit_status.kill_display_thread = FALSE;

			if (pthread_create(&singit_status.check_thread, NULL,
				check_thread_func, NULL))
			{
				singit_status.check_thread = 0;
				printf("Unable to create check thread\n");
			}

			if (pthread_create(&singit_status.display_thread, NULL,
				display_thread_func, NULL))
			{
				singit_status.display_thread = 0;
				printf("Unable to create display thread\n");
			}
		}
	}
}

void enable_plugin(gint plugin)
{
	set_dis_plugin_status_by_number(plugin, TRUE);
	if (getSCD(singit_config)->song && l_song_lyrics_found(getSCD(singit_config)->song)) { dis_plugin_show(plugin); }
}
