/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/***********************************************************************
*
*	Display thread and helper function(s)
*	This thread controls the visual updates of the displayers
*
***********************************************************************/

#include <pthread.h>
#include <errno.h>

#ifdef HAVE_SCHED_SETSCHEDULER
#include <sched.h>
#endif

#include <gtk/gtk.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include "singit_debug.h"
#include "singit/config_gen.h"
#include "singit/song.h"
#include "singit_song_private.h"

#include "singit_main.h"
#include "singit_plugin_scanner.h"
#include "editor_singit_main.h"


extern VisPlugin singit_vp;

static void set_new_lyric_line(gint time, gint state, SingitSong* cur_song)
{
	GList *item = NULL;

	if ((state == 1) && (cur_song->active_token == cur_song->last_token)) { return; }
	if ((state == -1) && (cur_song->active_token == NULL)) { return; }

#ifdef CODEDEBUG
	DEBUG(8, ("singit_main.c [set_new_lyric_line] : "));
#endif

	// Find current token and next line token
	item = singit_song_find_current_token(cur_song, time, state);
	STATUS->next_lyric_line = singit_song_find_next_lyric_line(cur_song, item, TRUE, NULL);

	// Set the new tokens
	cur_song->active_token = item;

#ifdef CODEDEBUG
        DEBUG(8, (" Ok\n"));
        #endif
}

void *display_thread_func (void *args)
{
	gint time = 0;
	gint status = 0;
	SingitSong* cur_song = NULL;
	gboolean hickup = FALSE;
	gboolean status_change, xmms_music_playing, have_lyrics;

#ifdef CODEDEBUG
	DEBUG(6, ("singit_main.c [display_thread_func] : Thread started\n"));
#endif

	cur_song = singit_song_attach(STATUS->song);

#ifdef HAVE_SCHED_SETSCHEDULER
	if (xmms_check_realtime_priority())
	{
		struct sched_param sparam;
		sparam.sched_priority = sched_get_priority_max(SCHED_OTHER);
		pthread_setschedparam(pthread_self(), SCHED_OTHER, &sparam);
	}
#endif

	singit_framerate_counter_set_limit
		(STATUS->singit_framerate_counter,
			(GET_SCD->framerateLimitEnabled)
			? GET_SCD->framerateLimit : 0);
	singit_framerate_counter_start
		(STATUS->singit_framerate_counter);

	while (!STATUS->kill_threads)
	{
		// Is some kind of state changing
		status_change = (STATUS->song_changed ||
				STATUS->config_update ||
				STATUS->initialize_plugins);

		// Does xmms really plays some tunes
		xmms_music_playing = ((xmms_remote_is_paused(singit_vp.xmms_session) == FALSE) &&
				(STATUS->playing == TRUE));

		// Do we have lyrics
		have_lyrics = ((cur_song != NULL) && (singit_song_lyrics_found(cur_song) == TRUE));

#ifdef CODEDEBUG
		DEBUG_AREA(DLV_ALL,
		g_print("SC: %s / CU: %s / IP: %s  ==> %s  -\\\n",
			((STATUS->song_changed == TRUE) ? "TRUE " : "FALSE"),
			((STATUS->config_update == TRUE) ? "TRUE " : "FALSE"),
			((STATUS->initialize_plugins == TRUE) ? "TRUE " : "FALSE"),
			((status_change == TRUE) ? "TRUE " : "FALSE"));
		g_print("XP: %s / SSP: %s                 ==> %s  --> %s\n",
			((xmms_remote_is_paused(singit_vp.xmms_session) == FALSE) ? "TRUE " : "FALSE"),
			((STATUS->playing == TRUE) ? "TRUE " : "FALSE"),
			((xmms_music_playing == TRUE) ? "TRUE " : "FALSE"),
			((!status_change && (!xmms_music_playing || !have_lyrics)) ? "TRUE " : "FALSE"));
		g_print("HS: %s /  HL: %s                 ==> %s  -/\n\n",
			((cur_song != NULL) ? "TRUE " : "FALSE"),
			((singit_song_lyrics_found(cur_song) == TRUE) ? "TRUE " : "FALSE"),
			((have_lyrics == TRUE) ? "TRUE " : "FALSE"));
		);
#endif

		// If nothing important is ongoing sleep a bit
		if (!status_change && (!xmms_music_playing || !have_lyrics))
		{
			xmms_usleep(10000);
		}
		if (!status_change && GET_SCD->useXMMSuSleep)
		{
			xmms_usleep(1000);
		}

		time = xmms_remote_get_output_time(singit_vp.xmms_session);

		// This took me nearly 4 months to find out
		// xmms return "-1" time while changing songs which crashed the plugin - damn
		// Before I installed loops to skip "some time" - sometimes it worked sometimes not ;-(
		// The second item seems to be an error in xmms (not sure)
		if (time < 0) { continue; }
		if ((time == 0) && (STATUS->old_time != time) && !STATUS->song_changed) {
			if (!hickup) {
				hickup = TRUE;
				continue;
			}
		}

		if (STATUS->old_time != time)
			singit_editor_set_time(time);

		if (STATUS->song_changed) {
#ifdef CODEDEBUG
			DEBUG(6, ("singit_main.c [display_thread_func]: New song\n"));
#endif
			pthread_mutex_lock(&STATUS->lyrics_rw_mutex);
			singit_song_detach(&cur_song);
			cur_song = singit_song_attach(STATUS->song);
			cur_song = singit_song_attach(STATUS->song);
			STATUS->song_changed = FALSE;
			STATUS->old_time = -1;
			pthread_mutex_unlock(&STATUS->lyrics_rw_mutex);
			if (is_dis_plugin_running(-1)) {
				GDK_THREADS_ENTER();
				dis_plugin_set_song(-1, cur_song);
				GDK_THREADS_LEAVE();
			}
		}

		if (cur_song == NULL)
			{ continue; }

		if (singit_song_lyrics_found(cur_song)) {
			status = inl_singit_song_is_time_ok(cur_song, time);
			if (status != 0) {
#ifdef CODEDEBUG
				DEBUG(8, (" %.2i:%.2i - ", time / 60000, (time / 1000) % 60));
#endif
				set_new_lyric_line(time, status, cur_song);
			}
		}

		if (!STATUS->kill_threads && is_dis_plugin_running(-1) &&
			(singit_song_lyrics_found(cur_song) || STATUS->config_update))
		{
			if (STATUS->config_update)
			{
				pthread_mutex_lock(&STATUS->config_rw_mutex);
				singit_framerate_counter_set_limit
					(STATUS->singit_framerate_counter,
						(GET_SCD->framerateLimitEnabled)
						? GET_SCD->framerateLimit : 0);

				GDK_THREADS_ENTER();
				dis_plugin_update(-1);
				GDK_THREADS_LEAVE();

				STATUS->config_update = FALSE;
				pthread_mutex_unlock(&STATUS->config_rw_mutex);
			}

			if (STATUS->old_time != time) {
				dis_plugin_set_time(-1, time, STATUS->next_lyric_line);
			}
		}

		STATUS->old_time = time;
		singit_framerate_counter_new_frame
			(STATUS->singit_framerate_counter);
	}

	singit_framerate_counter_reset(STATUS->singit_framerate_counter);

	singit_song_detach(&cur_song);

#ifdef CODEDEBUG
	DEBUG(6, ("singit_main.c [display_thread_func] : Thread ended\n"));
#endif

	STATUS->display_thread = 0;
        pthread_exit(0);
}
