/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <stdio.h>

#include "clevelparticleengine.h"
#include "maths.h"
#include "math.h"

enum { X, Y, Z };

typedef struct _TParticle
{
	gfloat vector[3];

	gfloat position;
	gfloat energy;

	gfloat color[3];
}
TParticle;

#define ppos(a,b) (a->vector[b] * a->position)

CLevelParticleEngine::CLevelParticleEngine(guint max_level_part)
{
	active_particles = 0;
	free_particles = 0;
	max_level_particles = max_level_part;
	list_length = 0;
	scaler = 1.0;
	srand(time(0));
}

CLevelParticleEngine::~CLevelParticleEngine()
{
	GSList *particle = active_particles;
	while (particle != 0) {
		g_free(particle->data);
		particle = g_slist_next(particle);
	}
	g_slist_free(active_particles);

	particle = free_particles;
	while (particle != 0) {
		g_free(particle->data);
		particle = g_slist_next(particle);
	}
	g_slist_free(free_particles);
}

gdouble CLevelParticleEngine::calc_level(gint16 freq_data[256])
{
	gint16 c = 0;
        gdouble level = 0;
        for(c = 0; c < 256; c++)
        {
                level = level + (gdouble) freq_data[c];
        }
        if(level > (512*256))
        {
                level = 512*256;
        }
	return (gdouble)(level / (512 * 256));
}

void CLevelParticleEngine::calc(gint16 freq_data[2][256], gfloat *pcLevel)
{
	if (level == 0) {
		level[0] = calc_level(freq_data[0]);
		level[1] = calc_level(freq_data[1]);
		level[2] = (level[0] + level[1]) / 2;
	}
	else {
		level[0] = pcLevel[0];
		level[1] = pcLevel[1];
		level[2] = pcLevel[2];
	}
//	g_print("Level: %f / %f - %f\n", level[0], level[1], (level[0] + level[1]) / 2);
}

TParticle* new_paticle(gdouble level, TParticle* particle)
{
	const float u = 16.0f;
	const float r = 28.0f;

	// Gesamtgroesse des Raumes
	const float src[] = {
		-u, u, r, 1.0f,
		 u, u, r, 1.0f,
		 u,-u, r, 1.0f,
		-u,-u, r, 1.0f
	};

	CMatrix m;
	TParticle* result;

	if (particle == NULL) { result = g_new(TParticle, 1); }
	else { result = particle; }

	// Beliebige Rotations-Matrix
	m.Unit();
	m.MulRotX(RandFloat() * PI2);
	m.MulRotY(RandFloat() * PI2);
	m.MulRotZ(RandFloat() * PI2);

	// Ein Partikel positionieren
	m.MulVector(&src[0], &result->vector[0]);
	UnitVector(&result->vector[0]);

	// Farbe bestimmen
	result->color[0] = 0.2f + level;
	result->color[1] = 0.2f + level;
	result->color[2] = 0.1f;

	result->position = 0.0f;
	result->energy = level;

	return result;
}

void CLevelParticleEngine::draw(bool beat)
{
#define BEAT_LOCK (0)
#define MAX_NEG_ENERGY (-0.2f)
#define ENERGY_MUL (3.0f)
	static gint last_beat = 0;

	GSList *particles = active_particles;
	GSList *prev_slist = 0;
	GLfloat adj_texsize = 0.8f;

	glPushMatrix();
	glShadeModel(GL_SMOOTH);

	glBlendFunc(GL_SRC_ALPHA,GL_ONE); // Reset the default blend mode
	glEnable(GL_BLEND);
	glEnable(GL_TEXTURE_2D);

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

	texture->bind();

//	if (particles) { g_print("%f\n", ((TParticle*) particles->data)->position); }
	while (particles != NULL)
	{
		TParticle* my_particle = (TParticle*) particles->data;

		// Draw
		glBegin(GL_TRIANGLE_STRIP);
			glColor3fv(my_particle->color);

			glTexCoord2d(1,1); // Top Right
			glVertex3f(ppos(my_particle, X) + adj_texsize,
				ppos(my_particle, Y) + adj_texsize, ppos(my_particle, Z));
			glTexCoord2d(0,1); // Top Left
			glVertex3f(ppos(my_particle, X) - adj_texsize,
				ppos(my_particle, Y) + adj_texsize, ppos(my_particle, Z));
			glTexCoord2d(1,0); // Bottom Right
			glVertex3f(ppos(my_particle, X) + adj_texsize,
				ppos(my_particle, Y) - adj_texsize, ppos(my_particle, Z));
			glTexCoord2d(0,0); // Bottom Left
			glVertex3f(ppos(my_particle, X) - adj_texsize,
				ppos(my_particle, Y) - adj_texsize, ppos(my_particle, Z));
		glEnd();

		// Move
		if (last_beat == 0) {
			my_particle->position += my_particle->energy;
			if (beat) {
				my_particle->position += 0.2;
				last_beat = BEAT_LOCK;
			}
			if (my_particle->energy >= 0) {
				if (my_particle->energy > 0.01f) { my_particle->energy -= (my_particle->energy / 3.5f); }
				else { my_particle->energy = -0.01f; }
			}
			else {
				my_particle->color[0] = 0.2f;

				if (my_particle->energy > MAX_NEG_ENERGY)
					{ my_particle->energy += (my_particle->energy / 3.5f); }
				else { my_particle->energy -= 0.03; }
			}
		}

		if (my_particle->position < 0) {
			GSList *next = g_slist_next(particles);
			if (prev_slist != 0) { prev_slist->next = next; }
			particles->next = free_particles;
			free_particles = particles;
			particles = next;
		}
		else {
			prev_slist = particles;
			particles = g_slist_next(particles);
		}
	}
	glPopMatrix();

	glDisable(GL_BLEND);
	glDisable(GL_TEXTURE_2D);

	if (last_beat > 0) { last_beat--; }

	gfloat mul = 1.0;
	if (beat) { mul = 1.5; }
	gint counter = (gint) (pow(level[2] * mul , 2) * max_level_particles + 1);

	// create new particles
	for (; counter > 0; counter--) {
		TParticle* new_particle;

		if (free_particles != 0) { // if there are free particles take one
			GSList *first_element = free_particles;
			free_particles = free_particles->next;
			first_element->next = active_particles;
			active_particles = first_element;

			new_particle  = new_paticle(level[2] * mul * ENERGY_MUL, (TParticle*) active_particles->data);
		}
		else { // create new ones
			new_particle  = new_paticle(level[2]  * mul * ENERGY_MUL, NULL);
			active_particles = g_slist_prepend(active_particles, new_particle);
		}
		list_length++;
	}
}
