/*
 *      $COPYRIGHT$
 *
 *	$Id: xmpi_dtprint.cc,v 1.2 2000/02/09 13:51:57 bbarrett Exp $
 *
 *	Function:	- print MPI datatype
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lam.h"
#include "mpitrace.h"
#include "xmpi.h"
#include "xmpi_dtprint.h"

/*
 * constants and macros
 */
#define DISP(f, x)	((f) ? ((x) & 0xFFFF) : (x))

/*
 * local functions
 */
static void indent(int);

static void print_basic(int, struct trdtype**, int);

static void print_datatype(struct trdtype**, int);

static void print_contig(struct trdtype **, int);

static void print_hvector(int, struct trdtype **, int, int);

static void print_hindexed(int, struct trdtype**, int, int);

static void print_struct(struct trdtype **, int);

/*
 * local variables
 */
static int colcount;		       /* print column count */

static int nlifconst;		       /* newline if contructor */

static char fmtbuf[80];		       /* formatting buffer */

static char *prefix;		       /* prefix for each line */

static char *obuf;		       /* output buffer or NULL */

static int identchar;		       /* # chars per indent level */

/*
 * basic datatypes
 */
const char *dtbasic[] = {
  "MPI_CONTIG", "MPI_VECTOR", "MPI_HVECTOR",
  "MPI_INDEX", "MPI_HINDEX", "MPI_STRUCT",
  "MPI_BYTE", "MPI_CHAR", "MPI_UNSIGNED_CHAR",
  "MPI_SHORT", "MPI_UNSIGNED_SHORT", "MPI_INT",
  "MPI_UNSIGNED", "MPI_LOGICAL", "MPI_LONG",
  "MPI_UNSIGNED_LONG", "MPI_FLOAT", "MPI_DOUBLE",
  "MPI_COMPLEX", "MPI_UB", "MPI_LB",
  "MPI_PACKED", "MPI_2INT", "MPI_2FLOAT",
  "MPI_2DOUBLE", "MPI_FLOAT_INT",
  "MPI_DOUBLE_INT", "MPI_LONG_INT",
  "MPI_SHORT_INT", "MPI_LONG_DOUBLE",
  "MPI_LONG_DOUBLE_INT", "MPI_INTEGER",
  "MPI_CHARACTER", "MPI_REAL",
  "MPI_DOUBLE_PRECISION", "MPI_DOUBLE_COMPLEX",
  "MPI_2INTEGER", "MPI_2REAL",
  "MPI_2DOUBLE_PRECISION", "MPI_LONG_LONG_INT",
  "MPI_INTEGER1", "MPI_INTEGER2", "MPI_INTEGER4",
  "MPI_INTEGER8", "MPI_REAL4", "MPI_REAL8",
  "MPI_REAL16",
};

/*
 *	xmpi_dtypeprint
 *
 *	Function:	- print datatype trace
 *	Accepts:	- datatype index
 *			- trace datatype buffer (may be NULL if basic datatype)
 *			- number of chars to indent by
 *			- string prefix for each output line
 *			- output buffer (NULL means output to stdout)
 */
void
xmpi_dtypeprint(int4 dtype, char *buf, int4 indent, 
		char *pfix, char *outbuf)
{
  struct trdtype *p;		       /* datatype trace */

  identchar = indent;
  prefix = pfix;
  obuf = outbuf;
/*
 * Recursively (depth-first) print datatype if trace buffer is non-NULL.
 */
  if (buf) {
    p = (struct trdtype *) buf;
    p++;
    colcount = 0;
    nlifconst = 0;
    if (obuf)
      *obuf = '\0';

    print_datatype(&p, 0);
  }
/*
 * Check for illegal datatype.
 */
  else if (dtype < 0) {
    if (obuf) {
      sprintf(obuf, "%sinvalid datatype: %d", prefix, dtype);
    } else {
      printf("%sinvalid datatype: %d\n", prefix, dtype);
    }
  }
/*
 * Handle basic datatypes.
 */
  else {
    if (obuf) {
      strcpy(obuf, prefix);
      strcat(obuf, dtbasic[dtype]);
    } else {
      printf("%s%s\n", prefix, dtbasic[dtype]);
    }
  }
}

/*
 *	print_datatype
 *
 *	Function:	- print current datatype
 *			- recursive function
 *	Accepts:	- trace record (inout)
 *			- nesting level
 */
static void
print_datatype(struct trdtype **dtrbuf, int level)
{
  int format;			       /* datatype format */

  format = (*dtrbuf)->trd_dtype;

  switch (format) {

  case XMPI_BYTE:
  case XMPI_CHAR:
  case XMPI_UCHAR:
  case XMPI_SHORT:
  case XMPI_USHORT:
  case XMPI_INT:
  case XMPI_UINT:
  case XMPI_LOGICAL:
  case XMPI_LONG:
  case XMPI_ULONG:
  case XMPI_FLOAT:
  case XMPI_DOUBLE:
  case XMPI_COMPLEX:
  case XMPI_UB:
  case XMPI_LB:
  case XMPI_PACKED:
  case XMPI_2INT:
  case XMPI_2FLOAT:
  case XMPI_2DOUBLE:
  case XMPI_FLOAT_INT:
  case XMPI_DOUBLE_INT:
  case XMPI_LONG_INT:
  case XMPI_SHORT_INT:
  case XMPI_LONG_DOUBLE:
  case XMPI_LONG_DOUBLE_INT:
  case XMPI_INTEGER:
  case XMPI_CHARACTER:
  case XMPI_REAL:
  case XMPI_DOUBLE_PRECISION:
  case XMPI_DOUBLE_COMPLEX:
  case XMPI_2INTEGER:
  case XMPI_2REAL:
  case XMPI_2DOUBLE_PRECISION:
  case XMPI_LONG_LONG_INT:
  case XMPI_INTEGER1:
  case XMPI_INTEGER2:
  case XMPI_INTEGER4:
  case XMPI_INTEGER8:
  case XMPI_REAL4:
  case XMPI_REAL8:
  case XMPI_REAL16:
    print_basic(format, dtrbuf, level);
    break;

  case XMPI_CONTIG:
    print_contig(dtrbuf, level);
    break;

  case XMPI_VECTOR:
    print_hvector(format, dtrbuf, level, 1);
    break;

  case XMPI_HVECTOR:
    print_hvector(format, dtrbuf, level, 0);
    break;

  case XMPI_INDEXED:
    print_hindexed(format, dtrbuf, level, 1);
    break;

  case XMPI_HINDEXED:
    print_hindexed(format, dtrbuf, level, 0);
    break;

  case XMPI_STRUCT:
    print_struct(dtrbuf, level);
    break;

  default:
    break;
  }
}

/*
 *	print_basic
 *
 *	Function:	- print basic datatype record
 *	Accepts:	- datatype label
 *			- trace record (inout)
 *			- nesting level
 */
static void
print_basic(int dtype, struct trdtype **dtrace, int nlev)
{
  nlifconst = 0;
  indent(nlev);
  if (obuf) {
    sprintf(fmtbuf, "%s\n", dtbasic[dtype]);
    strcat(obuf, fmtbuf);
  } else {
    printf("%s\n", dtbasic[dtype]);
  }
  colcount = 0;
  ++(*dtrace);
}

/*
 *	print_contig
 *
 *	Function:	- print contiguous datatype record
 *	Accepts:	- trace record (inout)
 *			- nesting level
 */
static void
print_contig(struct trdtype **dtrace, int nlev)
{
  indent(nlev);

  sprintf(fmtbuf, "%s (%d)", dtbasic[XMPI_CONTIG], (*dtrace)->trd_count);

  if (obuf)
    strcat(obuf, fmtbuf);
  else
    printf("%s", fmtbuf);

  colcount += strlen(fmtbuf);
  nlifconst = 1;

  ++(*dtrace);

  print_datatype(dtrace, nlev + 1);
}

/*
 *	print_hvector
 *
 *	Function:	- print vector or hvector datatype record
 *	Accepts:	- datatype label
 *			- trace record (inout)
 *			- nesting level
 *			- vector flag
 */
static void
print_hvector(int dtype,  struct trdtype **dtrace, 
	      int nlev, int fl_vec)
{
  indent(nlev);

  sprintf(fmtbuf, "%s (%d x %d, %d)", dtbasic[dtype],
	  (*dtrace)->trd_count, (*dtrace + 1)->trd_length,
	  DISP(fl_vec, (*dtrace + 1)->trd_stride));

  if (obuf)
    strcat(obuf, fmtbuf);
  else
    printf("%s", fmtbuf);

  colcount += strlen(fmtbuf);
  nlifconst = 1;

  *dtrace += 2;

  print_datatype(dtrace, nlev + 1);
}

/*
 *	print_hindexed
 *
 *	Function:	- print indexed or hindexed datatype record
 *	Accepts:	- datatype label
 *			- trace record (inout)
 *			- nesting level
 *			- indexed flag
 */
static void
print_hindexed(int dtype, struct trdtype **dtrace, 
	       int nlev, int fl_idx)
{
  int i;			       /* favourite index */

  int count;			       /* datatype count */

  indent(nlev);

  count = (*dtrace)->trd_count;
  ++(*dtrace);

  sprintf(fmtbuf, "%s (%d)", dtbasic[dtype], count);

  if (obuf)
    strcat(obuf, fmtbuf);
  else
    printf("%s", fmtbuf);

  colcount += strlen(fmtbuf);

  for (i = 0; i < count; ++i) {
    sprintf(fmtbuf, " (%d, %d)", (*dtrace)->trd_length,
	    DISP(fl_idx, (*dtrace)->trd_disp));

    if (obuf)
      strcat(obuf, fmtbuf);
    else
      printf("%s", fmtbuf);

    colcount += strlen(fmtbuf);
    ++(*dtrace);
  }

  nlifconst = 1;

  print_datatype(dtrace, nlev + 1);
}

/*
 *	print_struct
 *
 *	Function:	- print struct datatype record
 *	Accepts:	- trace record (inout)
 *			- nesting level
 */
static void
print_struct(struct trdtype **dtrace, int nlev)
{
  int i;			       /* favourite index */

  int count;			       /* datatype count */

  indent(nlev);

  count = (*dtrace)->trd_count;
  ++(*dtrace);

  sprintf(fmtbuf, "%s (%d)\n", dtbasic[XMPI_STRUCT], count);

  if (obuf)
    strcat(obuf, fmtbuf);
  else
    printf("%s", fmtbuf);

  colcount = 0;
  nlifconst = 0;

  for (i = 0; i < count; ++i) {

    indent(nlev + 1);

    sprintf(fmtbuf, "(%d, %d)",
	    (*dtrace)->trd_length, (*dtrace)->trd_disp);

    if (obuf)
      strcat(obuf, fmtbuf);
    else
      printf("%s", fmtbuf);

    colcount += strlen(fmtbuf);
    ++(*dtrace);

    print_datatype(dtrace, nlev + 2);
  }
}

/*
 *	indent
 *
 *	Function:	- indent line according to level
 *	Accepts:	- indentation level
 */
static void
indent(int lev)
{
  lev *= identchar;

  if (nlifconst) {
    if (obuf)
      strcat(obuf, "\n");
    else
      printf("\n");

    nlifconst = 0;
    colcount = 0;
  }
  if (obuf) {
    if (colcount == 0)
      strcat(obuf, prefix);
    if (colcount < lev) {
      for (; colcount < lev; ++colcount)
	strcat(obuf, " ");
    } else if (colcount > 0) {
      strcat(obuf, " ");
    }
  } else {
    if (colcount == 0)
      printf("%s", prefix);
    if (colcount < lev) {
      for (; colcount < lev; ++colcount)
	printf(" ");
    } else if (colcount > 0) {
      printf(" ");
    }
  }
}
