/* XPENGUINS_APPLET - gnome panel implementation of xpenguins
 * Copyright (C) 1999, 2000  Robin Hogan
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Robin Hogan <R.J.Hogan@reading.ac.uk>
 * Project started 22 November 1999
 * This file last updated 3 June 2001
 */

#include <config.h>
#include <applet-widget.h>
#include <gdk/gdkx.h>

#include "sitter_button.xpm"

/* Button pixmaps look much nicer at small sizes if custom pixmaps are
 * used rather than just scaling down */
#include "play-pause-button.xpm"
#include "play-pause-button-small.xpm"
#include "play-pause-button-tiny.xpm"
#include "stop-button.xpm"
#include "stop-button-small.xpm"
#include "stop-button-tiny.xpm"

#include "xpenguins.h"
#include "applet.h"


/* Load theme and handle errors */
int
load_theme(XPenguinData *xp)
{
  char *error = xpenguins_load_theme(xp->theme_name, &(xp->theme));
  
  if (error) {
    GString *str = g_string_new(_("Error loading theme `"));
    g_string_append(str, xp->theme_name);
    g_string_append(str, _("'\n"));
    g_string_append(str, error);
    xp->error_dialog = gnome_warning_dialog(str->str);
    gtk_signal_connect(GTK_OBJECT(xp->error_dialog), "destroy",
		       GTK_SIGNAL_FUNC(gtk_widget_destroyed),
		       &xp->error_dialog);
    g_string_free(str, TRUE);
    return (xp->loaded = FALSE);
  }
  return (xp->loaded = TRUE);
}


/* Timeout callback to update penguins */
int
service_xpenguins(gpointer data)
{
  XPenguinData *xp = data;
  int i=1;
  if (xp->paused) {
    xpenguins_pause_frame();
  }
  else {
    if (!xpenguins_frame()) {
      /* No toons drawn! Must be end of exit sequence */
      gtk_timeout_remove(xp->timeout);
      xpenguins_exit();
      xp->active = FALSE;
    }
  }
  return i;
}

/* Callback for play-pause button on the applet */
static void
playpause_xpenguins(GtkWidget *widget, gpointer data)
{
  XPenguinData *xp = data;
  if (!xp->active) {
    xp->paused = FALSE;
    if (!xp->loaded) {
      if (!load_theme(xp)) {
	/* Failed to load theme! */
	return;
      }
      xp->loaded = TRUE;
      if (!xp->overridetheme) {
	xp->speed = 1000/(xp->theme).delay;
	xp->number = (xp->theme).total;
      }
    }
    xpenguins_set_number(xp->number);
    if (xpenguins_start(NULL))
      return;
    xp->timeout = gtk_timeout_add(1000/xp->speed, service_xpenguins, xp);
    xp->active = TRUE;
  }
  else if (xp->paused) {
    xp->paused = FALSE;
  }
  else {
    xp->paused = TRUE;
  }
  return;
}

/* Callback for the stop button on the applet */
static void
stop_xpenguins(GtkWidget *widget, gpointer data)
{
  XPenguinData *xp = data;
  if (xp->active) {
    xp->paused = FALSE;
    xpenguins_set_number(0);
  }
}

/* Save the current settings to file */
static void
property_save(gchar *path, XPenguinData *xp)
{
  gnome_config_push_prefix(path);

  gnome_config_set_string("xpenguins/theme", xp->theme_name);
  gnome_config_set_bool("xpenguins/override_theme", xp->overridetheme);
  gnome_config_set_int("xpenguins/number", xp->number);
  gnome_config_set_int("xpenguins/frame_rate", xp->speed);
  gnome_config_set_bool("xpenguins/ignore_popups", xp->ignorepopups);
  gnome_config_set_bool("xpenguins/no_blood", xp->noblood);

  gnome_config_pop_prefix();
  gnome_config_sync();
  gnome_config_drop_all();
}

/* Load the current settings from file */
static void
property_load(gchar *path, XPenguinData *xp)
{
  gnome_config_push_prefix(path);

  xp->theme_name = gnome_config_get_string("xpenguins/theme=Penguins");
  xp->overridetheme = gnome_config_get_bool("xpenguins/override_theme=FALSE");
  xp->number = gnome_config_get_int("xpenguins/number=10");
  xp->speed = gnome_config_get_int("xpenguins/frame_rate=20");
  xp->ignorepopups = gnome_config_get_bool("xpenguins/ignore_popups=FALSE");
  xp->noblood = gnome_config_get_bool("xpenguins/no_blood=FALSE");

  gnome_config_pop_prefix();
}

/* Exit cleanly and delete data */
static void
destroy_xpenguins_widget(GtkWidget *widget, gpointer data)
{
  XPenguinData *xp = data;
  if (xp->active) {
    xpenguins_exit();
  }
  g_free((gpointer) xp);
}

/* Save session */
static gint
save_xpenguins_session(GtkWidget *widget, gchar *privcfgpath,
		       gchar *globcfgpath, gpointer data)
{
  XPenguinData *xp = data;
  property_save(privcfgpath, xp);
  return FALSE;
  widget = NULL;
  globcfgpath = NULL;
}

/* Bring up help documentation */
static void
help(AppletWidget *applet, gpointer data)
{
  GnomeHelpMenuEntry help_entry = { "xpenguins_applet",
				    "index.html" };
  gnome_help_display(NULL, &help_entry);
}

/* Bring up the "About" window */
static void
about(AppletWidget *applet, gpointer data)
{
  static const char *authors[] = { "Robin Hogan <R.J.Hogan@reading.ac.uk>", 
				   NULL };
  GtkWidget *about_box;

  about_box = gnome_about_new (_("XPenguins Applet"),
			       _(VERSION),
			       _("Copyright (C) 2000, 2001 "),
			       authors,
			       _("Animates a friendly family of"
				 " penguins in your root window."),
			       NULL);

  gtk_widget_show(about_box);
}


/* Display one of the three applet buttons */
static gint
make_pixmap_button(GtkWidget **button, GtkWidget **button_pixmap,
		   char **xpm, gint width, gint height, guint border)
{
  if (*button_pixmap) {
    gtk_widget_destroy(*button_pixmap);
  }    
  *button_pixmap
    = gnome_pixmap_new_from_xpm_d_at_size(xpm,
					  width - border*2,
					  height - border*2);
  gtk_container_add(GTK_CONTAINER(*button), *button_pixmap);
  GTK_WIDGET_UNSET_FLAGS(*button, GTK_CAN_DEFAULT);
  GTK_WIDGET_UNSET_FLAGS(*button, GTK_CAN_FOCUS);
  gtk_widget_set_usize(GTK_WIDGET(*button), width, height);
  gtk_widget_show(*button_pixmap);
  return 0;
}

/* Create the three buttons in the applet */
static gint
make_pixmap_buttons(GtkWidget *w, gint size, gpointer data)
{
  XPenguinData *xp = data;
  char **my_stop_xpm = stop_xpm;
  char **my_play_pause_xpm = play_pause_xpm;
  guint border = size/12 - 1;

  if (border < 2) {
    /* A tiny panel */
    my_stop_xpm = stop_tiny_xpm;
    my_play_pause_xpm = play_pause_tiny_xpm;
  }
  else if (border == 2) {
    /* A small panel */
    my_stop_xpm = stop_small_xpm;
    my_play_pause_xpm = play_pause_small_xpm;
  }
  else if (border > 4) {
    /* A huge panel */
    border = 4;
  }

  make_pixmap_button(&(xp->stop), &(xp->stop_pixmap), my_stop_xpm,
		     (16*size)/48, (16*size)/48, border);
  make_pixmap_button(&(xp->playpause), &(xp->playpause_pixmap),
		     my_play_pause_xpm,
		     (32*size)/48, (16*size)/48, border);
  make_pixmap_button(&(xp->button), &(xp->button_pixmap),
		     sitter_button_xpm, (32*size)/48, (32*size)/48, 1);

  return 0;
}

int
main (int argc, char **argv)
{
  XPenguinData *xp;
  GtkWidget *applet;
  GtkTooltips *tooltip;
  gint size;

  /* Now the GNOME stuff */
  bindtextdomain (PACKAGE, GNOMELOCALEDIR);
  textdomain (PACKAGE);

  /* Create a new applet */
  applet_widget_init("xpenguins_applet", VERSION, argc,
		     argv, NULL, 0, NULL);
  applet = applet_widget_new("xpenguins_applet");
  if (!applet)
    g_error (_("Can't create xpenguins applet!"));

  gtk_widget_realize(applet);

  size = applet_widget_get_panel_pixel_size (APPLET_WIDGET (applet));

  /* Allocate memory to hold all the applet data */
  xp = (XPenguinData *) g_new(XPenguinData, 1);

  /* Initialise the applet data */
  xp->prop_window = NULL;

  /* All the data that we may have saved in a config file */
  property_load(APPLET_WIDGET(applet)->privcfgpath, xp);

  xpenguins_verbose = 0;

  xp->active = FALSE;
  xp->loaded = FALSE;
  xp->paused = FALSE;
  xp->send_event = FALSE;
  xp->next_theme_name = NULL;
  xp->theme_list = NULL;
  xp->theme_info = NULL;

  xpenguins_ignorepopups((char) xp->ignorepopups);

  /* Register general applet callbacks */
  gtk_signal_connect(GTK_OBJECT(applet),"destroy",
		     GTK_SIGNAL_FUNC(destroy_xpenguins_widget),
		     xp);

  gtk_signal_connect(GTK_OBJECT(applet),"save_session",
		     GTK_SIGNAL_FUNC(save_xpenguins_session),
		     xp);

  gtk_signal_connect(GTK_OBJECT(applet),"change_pixel_size",
		     GTK_SIGNAL_FUNC(make_pixmap_buttons),
		     xp);

  applet_widget_register_stock_callback(APPLET_WIDGET(applet),
					"properties",
					GNOME_STOCK_MENU_PROP,
					_("Properties..."),
					property_show, xp);

  applet_widget_register_stock_callback(APPLET_WIDGET(applet),
					"help",
					GNOME_STOCK_PIXMAP_HELP,
					_("Help"),
					help, NULL);

  applet_widget_register_stock_callback(APPLET_WIDGET (applet),
					"about",
					GNOME_STOCK_MENU_ABOUT,
					_("About..."),
					about, NULL);

  xp->vbox = gtk_vbox_new(FALSE,0);
  xp->hbox = gtk_hbox_new(FALSE,0);

  /* Make the property, play-pause and stop buttons */
  xp->button = gtk_button_new();
  xp->stop = gtk_button_new();
  xp->playpause = gtk_button_new();

  xp->button_pixmap = NULL;
  xp->stop_pixmap = NULL;
  xp->playpause_pixmap = NULL;

  make_pixmap_buttons(NULL, size, xp);

  gtk_widget_show(xp->button);
  gtk_widget_show(xp->stop);
  gtk_widget_show(xp->playpause);

  /* Set the callbacks for actions on the various buttons */
  gtk_signal_connect(GTK_OBJECT(xp->button),"clicked",
		     GTK_SIGNAL_FUNC(property_show), xp);
  gtk_signal_connect(GTK_OBJECT(xp->playpause),"clicked",
		     GTK_SIGNAL_FUNC(playpause_xpenguins), xp);
  gtk_signal_connect(GTK_OBJECT(xp->stop),"clicked",
		     GTK_SIGNAL_FUNC(stop_xpenguins), xp);

  /* Pack it all in and show */
  gtk_box_pack_start(GTK_BOX(xp->vbox),xp->button, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(xp->vbox),xp->hbox, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(xp->hbox),xp->stop, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(xp->hbox),xp->playpause, FALSE, FALSE, 0);

  gtk_widget_show(xp->vbox);
  gtk_widget_show(xp->hbox);

  applet_widget_add(APPLET_WIDGET(applet), xp->vbox);

  tooltip = gtk_tooltips_new();
  gtk_tooltips_set_tip(tooltip, applet, _("XPenguins controller"), NULL);

  gtk_widget_show(applet);
  applet_widget_gtk_main ();
  exit(0);
}

