/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef XSID_HAVE_NOTHROW
#include <new>
#endif

#include <qcheckbox.h>
#include <qlabel.h>
#include <qlcdnumber.h>
#include <qslider.h>
#include <qspinbox.h>

#include "MainDialog.h"
#include "AudioDialog.h"
#include "EmuDialog.h"
#include "FilterDialog.h"
#include "HistoryDialog.h"
#include "HVSC_Dialog.h"
#include "MixerDialog.h"
#include "PlaylistDialog.h"
#include "PlaylistEditDialog.h"
#include "PlaylistOptDialog.h"
#include "StilDialog.h"
#include "WaveViewDialog.h"

#include "ConfigC.h"
#include "ConfigFile.h"
#include "HVSC_Config.h"
#include "filenames.h"
#include "Playlist.h"
#include "PlaylistCheck.h"
#include "Player.h"
#include "songlendb/File.h"
#include "songlendb/MD5.h"

#include "widgets/PicButton.h"
#include "widgets/TimeLCD.h"

#include "images/prev.xpm"
#include "images/stop.xpm"
#include "images/play.xpm"
#include "images/pause.xpm"
#include "images/ff.xpm"
#include "images/next.xpm"

#include "images/stop_p.xpm"
#include "images/pause_p.xpm"
#include "images/ff_p.xpm"

#include "images/prev_l.xpm"
#include "images/stop_l.xpm"
#include "images/play_l.xpm"
#include "images/pause_l.xpm"
#include "images/ff_l.xpm"
#include "images/next_l.xpm"

#include "images/prev_l_p.xpm"
#include "images/stop_l_p.xpm"
#include "images/play_l_p.xpm"
#include "images/next_l_p.xpm"

// !!!
#include <iostream.h>
#include <iomanip.h>

// --------------------------------------------------------------------------

MainDialog::MainDialog(QWidget* parent, const char* name)
: MainDialogData(parent,name)
{
    // Init pixmap pointers.
    
    prevPic = new QPixmap((const char**)prev_xpm);
    stopPic = new QPixmap((const char**)stop_xpm);
    playPic = new QPixmap((const char**)play_xpm);
    pausePic = new QPixmap((const char**)pause_xpm);
    ffPic = new QPixmap((const char**)ff_xpm);
    nextPic = new QPixmap((const char**)next_xpm);

    stop_pPic = new QPixmap((const char**)stop_p_xpm);
    pause_pPic = new QPixmap((const char**)pause_p_xpm);
    ff_pPic = new QPixmap((const char**)ff_p_xpm);
    
    prev_lPic = new QPixmap((const char**)prev_l_xpm);
    stop_lPic = new QPixmap((const char**)stop_l_xpm);
    play_lPic = new QPixmap((const char**)play_l_xpm);
    pause_lPic = new QPixmap((const char**)pause_l_xpm);
    ff_lPic = new QPixmap((const char**)ff_l_xpm);
    next_lPic = new QPixmap((const char**)next_l_xpm);
    
    prev_l_pPic = new QPixmap((const char**)prev_l_p_xpm);
    stop_l_pPic = new QPixmap((const char**)stop_l_p_xpm);
    play_l_pPic = new QPixmap((const char**)play_l_p_xpm);
    next_l_pPic = new QPixmap((const char**)next_l_p_xpm);

    // Finish dialog setup.
    
    setMinimumSize(width(),96);
    setIcon(*mainIcon);

//    infoField->setFont(QFont("Courier",12,QFont::Normal,false,QFont::Latin1));
    infoField->setText("");

    QPopupMenu *fileMenu = new QPopupMenu(this);
    CHECK_PTR( fileMenu );
    fileMenu->insertItem( "&Open file(s)", this, SLOT(openFile()), CTRL+Key_O );
    fileMenu->insertItem( "&Save as...", this, SLOT(saveAs()), CTRL+Key_S );
    fileMenu->insertSeparator();
    fileMenu->insertItem( "Exit", this, SLOT(quit()), CTRL+Key_Q );

    QPopupMenu *configMenu = new QPopupMenu(this);
    CHECK_PTR( configMenu );
    configMenu->insertItem( "Audio", this, SLOT(openAudioDialog()) );
    configMenu->insertItem( "Emulator", this, SLOT(openEmulatorDialog()) );
    configMenu->insertItem( "SID filter", this, SLOT(openFilterDialog()) );
    configMenu->insertItem( "HVSC", this, SLOT(openHVSC_Dialog()) );

    QPopupMenu *playlistAddMenu = new QPopupMenu(this);
    CHECK_PTR( playlistAddMenu );
    playlistAddMenu->insertItem( "Directory", this, SLOT(addDirToPlaylist()) );
    playlistAddMenu->insertItem( "Tree", this, SLOT(addTreeToPlaylist()) );
    playlistAddMenu->insertItem( "Hotlist", this, SLOT(addHotlistToPlaylist()) );

    QPopupMenu *playlistMenu = new QPopupMenu(this);
    CHECK_PTR( playlistMenu );
    playlistMenu->insertItem( "&View", this, SLOT(openPlaylistDialog()), CTRL+Key_V );
    playlistMenu->insertItem( "&Edit", this, SLOT(openPlaylistEditDialog()), CTRL+Key_E );
    playlistMenu->insertSeparator();
    playlistMenu->insertItem( "&Load...", this, SLOT(loadPlaylist()), CTRL+Key_L );
    playlistMenu->insertItem( "Save as...", this, SLOT(savePlaylistAs()) );
    playlistMenu->insertSeparator();
    playlistMenu->insertItem( "&Add tune", this, SLOT(addTuneToPlaylist()), CTRL+Key_A );
    playlistMenu->insertItem( "Add...", playlistAddMenu );
    playlistMenu->insertSeparator();
    playlistMenu->insertItem( "Update", this, SLOT(updatePlaylist()) );
    playlistMenu->insertSeparator();
    playlistMenu->insertItem( "Options", this, SLOT(openPlaylistOptions()) );

    QPopupMenu *hotlistMenu = new QPopupMenu(this);
    CHECK_PTR( hotlistMenu );
    hotlistMenu->insertItem( "&Mark tune", this, SLOT(markTune()), CTRL+Key_M );
    hotlistMenu->insertItem( "Mark all", this, SLOT(markPlaylist()) );
    hotlistMenu->insertSeparator();
    hotlistMenu->insertItem( "Save as...", this, SLOT(saveHotlist()) );

    extraMenu = new QPopupMenu(this);
    CHECK_PTR( extraMenu );
    extraMenu->setCheckable(true);
    stilID = extraMenu->insertItem( "STIL view", this, SLOT(toggleStilDialog()) );
    songLenDbID = extraMenu->insertItem( "Songlen DB", this, SLOT(toggleSongLengthDB()) );
    extraMenu->insertItem( "History", this, SLOT(openHistoryDialog()) );
    extraMenu->insertItem( "Mixer", this, SLOT(openMixerDialog()) );
    waveViewID = extraMenu->insertItem( "Oscilloscope", this, SLOT(openWaveViewDialog()) );

    QPopupMenu *helpMenu = new QPopupMenu(this);
    CHECK_PTR( helpMenu );
    helpMenu->insertItem( "SIDPLAY", this, SLOT(about()), CTRL+Key_H );
    helpMenu->insertItem( "Qt", this, SLOT(aboutQt()) );

    menuBar = new QMenuBar(this);
    CHECK_PTR( menuBar );
    menuBar->insertItem( "&File", fileMenu );
    menuBar->insertItem( "&Config", configMenu );
    menuBar->insertItem( "&Playlist", playlistMenu );
    menuBar->insertItem( "&Hotlist", hotlistMenu );
    menuBar->insertItem( "&Extra", extraMenu );
    menuBar->insertSeparator();
    menuBar->insertItem( "&About", helpMenu );

    timeLCD->setSmallDecimalPoint( FALSE );
    timeLCD->setNumDigits( 5 );
    timeLCD->setMode( QLCDNumber::DEC );
    timeLCD->setSegmentStyle( QLCDNumber::Filled );
    timeLCD->setFrameShadow( QLCDNumber::Sunken );
    timeLCD->setFrameShape( QLCDNumber::Panel );
    
    // The currently active file name.
    sidFileNameFull = sidFileNameHVSC = "";

    hotlist = new Playlist;

    myFileDlg = 0;  // do that when we get access to the config file
    
    myAudioDlg = new AudioDialog(this);
    myEmuDlg = new EmuDialog(this);
    myFilterDlg = new FilterDialog(this);
    myMixerDlg = new MixerDialog(this);
    // Note: StilDialog is _NOT_ created as a child widget.
    // The 'this' pointer is provided in order to allow the dialog
    // check main window isVisible() and show()/hide() itself
    // appropriately.
    //
    // This is a workaround for a bug in KDE 1.1.2's kwm which
    // does not hide/show child windows correctly.
    myStilDlg = new StilDialog(this);
    myHVSC_Dlg = new HVSC_Dialog(this);
    myWaveViewDlg = new WaveViewDialog(this);
    myHistoryDlg = new HistoryDialog(this);
    myPlaylistDlg = new PlaylistDialog(this);
    myPlaylistEditDlg = new PlaylistEditDialog(this);
    
    connect(myAudioDlg,SIGNAL(changed(const AudioConfig&)),this,SLOT(receiveAudioConfig(const AudioConfig&)));
    myFilterDlg->connect(myEmuDlg,SIGNAL(changed(const emuConfig&)),myFilterDlg,SLOT(receiveConfig(const emuConfig&)));
    myEmuDlg->connect(myFilterDlg,SIGNAL(changed(const emuConfig&)),myEmuDlg,SLOT(receiveConfig(const emuConfig&)));
    connect(myEmuDlg,SIGNAL(changed(const emuConfig&)),this,SLOT(receiveEmuConfig(const emuConfig&)));
    connect(myFilterDlg,SIGNAL(changed(const emuConfig&)),this,SLOT(receiveEmuConfig(const emuConfig&)));
    connect(myHVSC_Dlg,SIGNAL(changed(const HVSC_Config&)),this,SLOT(receiveHVSC_Config(const HVSC_Config&)));
    connect(myMixerDlg,SIGNAL(changed(const MixerConfig&)),this,SLOT(receiveMixerConfig(const MixerConfig&)));
    connect(myHistoryDlg,SIGNAL(playHistoryItem(const QString&)),this,SLOT(playHistoryFile(const QString&)));

    playerButtonsOff();
    subSongSlider->setFocus();
}

MainDialog::~MainDialog()
{
    delete hotlist;
    
    delete next_l_pPic;
    delete play_l_pPic;
    delete stop_l_pPic;
    delete prev_l_pPic;

    delete next_lPic;
    delete ff_lPic;
    delete pause_lPic;
    delete play_lPic;
    delete stop_lPic;
    delete prev_lPic;

    delete ff_pPic;
    delete pause_pPic;
    delete stop_pPic;

    delete nextPic;
    delete ffPic;
    delete pausePic;
    delete playPic;
    delete stopPic;
    delete prevPic;
}

void MainDialog::setup(ConfigFile* cfg, const QString& path, Player* p)
{
    config = cfg;
    configPath = path;
    player = p;
    
    // Receive 'play' and 'stop' signals from player and playlist dialogs.
    // We are responsible for starting/stopping songs.
    connect(player,SIGNAL( playListItem(const PlaylistItem&) ),this,SLOT( playPlaylistFile(const PlaylistItem&) ));
    connect(player,SIGNAL( stopPlaylistPlay() ),this,SLOT( playerButtonsForceStop() ));
    connect(myPlaylistDlg,SIGNAL( playlistPlayRequest(const PlaylistItem&) ),this,SLOT( playPlaylistFile(const PlaylistItem&) ));
    connect(myPlaylistEditDlg,SIGNAL( playlistEditPlayRequest(const PlaylistItem&) ),this,SLOT( playPlaylistFile(const PlaylistItem&) ));
    // Let player receive 'playnewpos' signals from playlist dialog.
    connect(myPlaylistDlg,SIGNAL( playlistPlayNewPosRequest(int) ),player,SLOT( playNextFromList(int) ));
    // Let playlist dialog receive 'play' signal from player.
    // That way it can update the current item.
    connect(player,SIGNAL( playerPlayRequest(uint) ),myPlaylistDlg,SLOT( newListPos(uint) ));
    // Let playlist dialog receive signals from playlist editor.
    // That way it can update the current item or listbox.
    connect(myPlaylistEditDlg,SIGNAL( playlistEditPlayRequest(uint) ),myPlaylistDlg,SLOT( newListPos(uint) ));
    connect(myPlaylistEditDlg,SIGNAL( playlistEditDelRequest(uint) ),myPlaylistDlg,SLOT( deleteListPos(uint) ));
    connect(myPlaylistEditDlg,SIGNAL( playlistEditClearRequest() ),myPlaylistDlg,SLOT( clear() ));
    connect(myPlaylistEditDlg,SIGNAL( playlistEditUpdateRequest(uint) ),myPlaylistDlg,SLOT( updateListPos(uint) ));
    
    // Give player access to a few widgets.
    player->link(this,timeLCD,myPlaylistDlg,myWaveViewDlg);
    // Give playlist dialogs access to list in player.
    myPlaylistEditDlg->takeList( player->getPlaylist() );
    myPlaylistDlg->takeList( player->getPlaylist() );
    
    myFilterDlg->setConfigDefault( player->getEmuConfigDefault() );  // *not* redundant
    // Order is important with current implementation.
    setEmuConfig( config->getEmuConfig() );
    myHVSC_Dlg->setConfig( config->getHVSC_Config() );
    myStilDlg->setConfig( config->getHVSC_Config() );
    setAudioConfig( config->getAudioConfig() );

    // Mixer - Since we do not save the entire MixerConfig to disk,
    // we copy relevant values from the emuConfig.
    // The mixer knows the emuConfig constants.
    MixerConfig newMixerConfig = config->getMixerConfig();
    newMixerConfig.channels = player->getEmuConfig().channels;
    newMixerConfig.mixingMode = player->getEmuConfig().volumeControl;
    newMixerConfig.panningMode = player->getEmuConfig().autoPanning;
    myMixerDlg->setConfig(newMixerConfig);
    // Make sure volume levels get set as well.
    receiveMixerConfig(newMixerConfig);

    // Playlist dialog
    config->setGroup(ConfigC::groups[ConfigC::Playlist]);
    config->restoreGeom(myPlaylistDlg);
    config->getValue(ConfigC::keys[ConfigC::PlaytimeDefault],playtimeDefault);
    config->getValue(ConfigC::keys[ConfigC::FadeOutDefault],fadeoutDefault);
    if ( config->getBool( ConfigC::keys[ConfigC::SongLenDB] ))
        setSongLenDbEnabled( initSongLengthDB() );

    // Playlist editor
    config->setGroup(ConfigC::groups[ConfigC::PlaylistEditor]);
    config->restoreGeom(myPlaylistEditDlg);
    
    myHistoryDlg->load(configPath+'/'+HISTORY_FILE_NAME);

    // Misc
    config->setGroup(ConfigC::groups[ConfigC::Misc]);
    menuBar->setItemEnabled(waveViewID,config->getBool(ConfigC::keys[ConfigC::ExpertMode]));
    if ( config->getBool( ConfigC::keys[ConfigC::STIL_Viewer] ))
        setStilInfoEnabled( initStilViewClass() );
    config->getValue( ConfigC::keys[ConfigC::Usage], usage );

    // Main window
    config->setGroup(ConfigC::groups[ConfigC::MainWindow]);
    config->restoreGeom(this);

    // Create file selector with defaults from config file.
    config->setGroup(ConfigC::groups[ConfigC::Selector]);
    QString cwd = config->getString(ConfigC::keys[ConfigC::Dir]);
    QString nameFilter = config->getString(ConfigC::keys[ConfigC::NameFilter]);
    myFileDlg = new QFileDialog(cwd,nameFilter,this,"Selector_file_dialog",false);
    config->restoreGeom(myFileDlg);
    myFileDlg->setCaption("SIDPLAY: Sidtune selector");
    myFileDlg->setMode(QFileDialog::ExistingFile);
    myFileDlg->setIcon(*mainIcon);
    connect(myFileDlg,SIGNAL(fileHighlighted(const QString&)),this,SLOT(playFile(const QString&)));
    if ( config->getBool(ConfigC::keys[ConfigC::Open]) )
        openFile();
}

bool MainDialog::writeConfig()
{
    // Misc
    config->setGroup(ConfigC::groups[ConfigC::Misc]);
    long int mileage = usage+player->getEmuEngine()->getSecondsTotal();
    config->setEntry( ConfigC::keys[ConfigC::Usage], mileage);
    config->setEntry( ConfigC::keys[ConfigC::STIL_Viewer], isStilInfoEnabled() );

    AudioConfig thisAudioConfig = myAudioDlg->getConfig();
    // Always save user`s sane fragmentation settings.
    extern AudioConfig bakAudioConfig;
    thisAudioConfig.bufSize = bakAudioConfig.bufSize;
    thisAudioConfig.maxFrags = bakAudioConfig.maxFrags;
    thisAudioConfig.fragSize = bakAudioConfig.fragSize;
    
    config->setAudioConfig(thisAudioConfig);
    config->setEmuConfig( player->getEmuConfig() );
    config->setHVSC_Config( myHVSC_Dlg->getConfig() );
    config->setMixerConfig( myMixerDlg->getConfig() );

    // Playlist
    config->setGroup( ConfigC::groups[ConfigC::Playlist] );
    config->saveGeom(myPlaylistDlg);
    config->setEntry( ConfigC::keys[ConfigC::PlaytimeDefault], playtimeDefault);
    config->setEntry( ConfigC::keys[ConfigC::FadeOutDefault], fadeoutDefault);
    config->setEntry( ConfigC::keys[ConfigC::SongLenDB], isSongLenDbEnabled() );
    // Playlist editor
    config->setGroup(ConfigC::groups[ConfigC::PlaylistEditor]);
    config->saveGeom(myPlaylistEditDlg);
    
    // Selector
    config->setGroup(ConfigC::groups[ConfigC::Selector]);
    config->saveGeom(myFileDlg);
    config->setEntry( ConfigC::keys[ConfigC::Dir], myFileDlg->dirPath() );
    config->setEntry( ConfigC::keys[ConfigC::NameFilter], myFileDlg->selectedFilter() );
    config->setEntry( ConfigC::keys[ConfigC::Open], myFileDlg->isVisible() );

    // Dialog geometries.
    config->setGroup(ConfigC::groups[ConfigC::MainWindow]);
    config->saveGeom(this);
        
    bool ret = config->save();
    if (!ret)
    {
        QMessageBox error;
        error.setIcon(QMessageBox::Warning);
        error.setText("Could not save configuration file!");
        error.adjustSize();
        error.show();
    }
    return ret;
}

/*
 * [ms]: If this way of receiving config data from dialogs (and sending
 *       data to other dialogs) on the fly turns out to be buggy, it
 *       could be changed at any time by turning the dialogs into modal
 *       dialogs and adding common OK/Apply/Cancel buttons.
 *       Currently the worst thing that can happen is that Qt signals
 *       pass on to the next dialog until all dialogs have the same
 *       config.
 */

void MainDialog::setAudioConfig(const AudioConfig& inConfig)
{
    player->pauseHard();

    // Let the audio dialog check/adjust the config.
    myAudioDlg->setConfig(inConfig);
    player->init( myAudioDlg->getConfig() );

    // Now retrieve the possibly changed config from the driver.
    AudioConfig goodAudioConfig = player->getAudioConfig();
    myAudioDlg->setConfig(goodAudioConfig);
    
    // Set dialogs that directly depend on the emuConfig.
    myEmuDlg->setConfig( player->getEmuConfig() );
    myFilterDlg->setConfig( player->getEmuConfig() );
    
    // Here update the dialog buttons according to the settings the
    // audio driver accepted.
    myAudioDlg->setButtons();

    myWaveViewDlg->setConfig(goodAudioConfig);
    
    MixerConfig newMixerConfig = myMixerDlg->getConfig();
    // The mixer knows the emuConfig constants.
    newMixerConfig.channels = player->getEmuConfig().channels;
    newMixerConfig.mixingMode = player->getEmuConfig().volumeControl;
    newMixerConfig.panningMode = player->getEmuConfig().autoPanning;
    myMixerDlg->setConfig(newMixerConfig);
    // Make sure volume levels get set as well.
    receiveMixerConfig(newMixerConfig);
    
    player->resume();
}

void MainDialog::receiveAudioConfig(const AudioConfig& inConfig)
{
    setAudioConfig(inConfig);
}

void MainDialog::setEmuConfig(const emuConfig& inConfig)
{
    emuConfig myEmuConfig = inConfig;
    player->getEmuEngine()->setConfig(myEmuConfig);
    player->getEmuEngine()->getConfig(myEmuConfig);  // get a good copy
    
    // Set dialogs that directly depend on the emuConfig.
    // Not: myMixerDlg.
    myEmuDlg->setConfig(myEmuConfig);
    myFilterDlg->setConfig(myEmuConfig);
}

void MainDialog::receiveEmuConfig(const emuConfig& inConfig)
{
    emuConfig myEmuConfig = inConfig;
    player->getEmuEngine()->setConfig(myEmuConfig);
    player->getEmuEngine()->getConfig(myEmuConfig);  // get a good copy

    // Do not write back emuConfig to any dialog.
    // But:
    myFilterDlg->paintFilterFunc();
    
    MixerConfig newMixerConfig = myMixerDlg->getConfig();
    // The mixer knows the emuConfig constants.
    newMixerConfig.channels = myEmuConfig.channels;
    newMixerConfig.mixingMode = myEmuConfig.volumeControl;
    newMixerConfig.panningMode = myEmuConfig.autoPanning;
    myMixerDlg->setConfig(newMixerConfig);
}

void MainDialog::receiveHVSC_Config(const HVSC_Config& inConfig)
{
    myStilDlg->setConfig(inConfig);
    
    if ( isStilInfoEnabled() )
        setStilInfoEnabled( initStilViewClass() );
    
    // Hide a possibly open STIL dialog.
    if ( myStilDlg->isVisible() && !isStilInfoEnabled() )
        myStilDlg->hide();
    
    if ( isSongLenDbEnabled() )
        setSongLenDbEnabled( initSongLengthDB() );
}

void MainDialog::receiveMixerConfig(const MixerConfig& inConfig)
{
    emuConfig myEmuConfig;
    player->getEmuEngine()->getConfig(myEmuConfig);  // get a good copy
    
    if ((myEmuConfig.volumeControl!=inConfig.mixingMode) ||
        (myEmuConfig.autoPanning!=inConfig.panningMode))
    {
        myEmuConfig.volumeControl = inConfig.mixingMode;
        myEmuConfig.autoPanning = inConfig.panningMode;
        player->getEmuEngine()->setConfig(myEmuConfig);
        player->getEmuEngine()->getConfig(myEmuConfig);
        myEmuDlg->setConfig(myEmuConfig);
    }

    for (int v=0; v < MixerDialog::voices; v++)
    {
        // In mono mode, set voice volume regardless of whether mixing
        // mode is active.
        if (myEmuConfig.channels == SIDEMU_MONO)
        {
            player->getEmuEngine()->setVoiceVolume(v+1,255,0,
                                       inConfig.volTotal[v]&inConfig.muteMask[v]);
        }
        else
        {
            if (myEmuConfig.volumeControl == SIDEMU_VOLCONTROL)
            {
                player->getEmuEngine()->setVoiceVolume(v+1,inConfig.volHQ[v].l,inConfig.volHQ[v].r,
                                           inConfig.volTotal[v]&inConfig.muteMask[v]);
            }
            else if ((myEmuConfig.volumeControl==SIDEMU_FULLPANNING) &&
                     (myEmuConfig.autoPanning==SIDEMU_NONE))
            {
                player->getEmuEngine()->setVoiceVolume(v+1,inConfig.volFP[v].l,inConfig.volFP[v].r,
                                           inConfig.volTotal[v]&inConfig.muteMask[v]);
            }
            else  // STEREOSURROUND or AUTOPANNING
            {
                // Adjust only the master volume level.
                // This is not clean because we don't use the emuEngine-defaults.
                if (myEmuConfig.volumeControl == SIDEMU_STEREOSURROUND)
                {
                    player->getEmuEngine()->setVoiceVolume(v+1,255,255,inConfig.volTotal[v]&inConfig.muteMask[v]);
                }
                else
                {
                    if ((v&1)==0)
                        player->getEmuEngine()->setVoiceVolume(v+1,255,0,inConfig.volTotal[v]&inConfig.muteMask[v]);
                    else
                        player->getEmuEngine()->setVoiceVolume(v+1,0,255,inConfig.volTotal[v]&inConfig.muteMask[v]);
                }
            }
        }
    }
}

// --------------------------------------------------------------------------

void MainDialog::closeEvent(QCloseEvent* e)
{
    programExitHook();
    e->accept();
    
    myStilDlg->close();
    delete myStilDlg;    // StilDialog is NOT a child
}

void MainDialog::hideEvent(QHideEvent*)
{
    if ( myStilDlg->isVisible() )
        myStilDlg->hide();
}

void MainDialog::resizeEvent(QResizeEvent*)
{
    int w = width();
    int h = height();
    // 5,75,350,115 (main 360,217)
    infoField->setGeometry(5,75,w-10,h-102);
    if ( (h-102)<8 )
        infoField->hide();
    else if ( !infoField->isVisible() )
        infoField->show();
    playlistCheckBox->move(15,h-22);  //  15,195
    loopCheckBox->move(140,h-22);     // 140,195
    randomCheckBox->move(250,h-22);   // 250,195
}

void MainDialog::playerButtonsOff()
{
    prevBtn->setPixmap(*prevPic);
    stopBtn->setPixmap(*stopPic);
    playBtn->setPixmap(*playPic);
    ffBtn->setPixmap(*ffPic);
    nextBtn->setPixmap(*nextPic);
}

void MainDialog::playerButtonsForceStart()
{
    // Make them look as if playing.
    setPrevBtnState();
    stopBtn->setPixmap(*stop_lPic);
    playBtn->setPixmap(*pause_lPic);
    ffBtn->setPixmap(*ff_lPic);
    setNextBtnState();
}

void MainDialog::playerButtonsForceStop()
{
    // Make them look as if stopped.
    setPrevBtnState();
    stopBtn->setPixmap(*stopPic);
    playBtn->setPixmap(*play_lPic);
    ffBtn->setPixmap(*ffPic);
    setNextBtnState();
}

void MainDialog::showSidTuneInfo()
{
    sidTuneInfo mySidTuneInfo =    player->getSidTuneInfo();
    QString tmp;
    if ( mySidTuneInfo.numberOfInfoStrings == 3 )
    {
        tmp =    "Name      : ";
        tmp += mySidTuneInfo.infoString[0];
        tmp += "\nAuthor    : ";
        tmp += mySidTuneInfo.infoString[1];
        tmp += "\nCopyright : ";
        tmp += mySidTuneInfo.infoString[2];
    }
    else
    {
        for ( int infoi = 0; infoi < mySidTuneInfo.numberOfInfoStrings; infoi++ )
        {
            if (infoi > 0)
                tmp += "\n";
            tmp += "Info      : ";
            tmp += mySidTuneInfo.infoString[infoi];
        }
    }
    QString tmp2;
    tmp2.sprintf("\nSongs     : %d (Startsong: %d)",
             mySidTuneInfo.songs, 
             mySidTuneInfo.startSong);
    tmp += tmp2;
    tmp2.sprintf("\nSong speed: %s",mySidTuneInfo.speedString);
    tmp += tmp2;
    tmp2.sprintf("\nAddresses : $%04x, $%04x, $%04x",
             mySidTuneInfo.loadAddr,
             mySidTuneInfo.initAddr,
             mySidTuneInfo.playAddr);
    tmp += tmp2;
    tmp2.sprintf("\nFormat    : %s",mySidTuneInfo.formatString);
    tmp += tmp2;
    infoField->setText(tmp);
            
    songLCD->display(mySidTuneInfo.currentSong);
}

void MainDialog::clearSidTuneInfo()
{
    QString tmp 
         = "Name      : \n";
    tmp += "Author    : \n";
    tmp += "Copyright : \n";
    tmp += "Addresses : \n";
    tmp += "Songs     : \n";
    tmp += "Song speed: \n";
    tmp += "Format    : \n";
    infoField->setText(tmp);
}

// --------------------------------------------------------------------------

/*
 * Called before application is quit.
 */
void MainDialog::programExitHook()
{
    player->stop();
    writeConfig();     // ignore return value
    myHistoryDlg->save();
    
    bool forceExit = false;
    while ( player->getPlaylist()->isModified() && !forceExit )
    {
        QMessageBox mb( "SIDPLAY",
                       "You are about to quit.\n\n"
                       "Playlist has not been saved yet!\n"
                       "Do you want to save it?",
                       QMessageBox::Information,
                       QMessageBox::Yes | QMessageBox::Default | QMessageBox::Escape,
                       QMessageBox::No,
                       0);
        mb.setButtonText( QMessageBox::Yes, "Yes" );
        mb.setButtonText( QMessageBox::No, "No" );
        switch( mb.exec() ) 
        {
         case QMessageBox::Yes:
            savePlaylistAs();
            break;
         case QMessageBox::No:
            forceExit = true;
            break;
        }
    }
    
    forceExit = false;
    while ( hotlist->isModified() && !forceExit )
    {
        QMessageBox mb( "SIDPLAY",
                       "You are about to quit.\n\n"
                       "Hotlist has not been saved yet!\n"
                       "Do you want to save it?",
                       QMessageBox::Information,
                       QMessageBox::Yes | QMessageBox::Default | QMessageBox::Escape,
                       QMessageBox::No,
                       0);
        mb.setButtonText( QMessageBox::Yes, "Yes" );
        mb.setButtonText( QMessageBox::No, "No" );
        switch( mb.exec() ) 
        {
         case QMessageBox::Yes:
            saveHotlist();
            break;
         case QMessageBox::No:
            forceExit = true;
            break;
        }
    }
}

void MainDialog::quit()
{
    programExitHook();
    qApp->quit();
}

void MainDialog::openFile()
{
    if ( myFileDlg!=0 )
    {
        myFileDlg->show();
    }
}

void MainDialog::saveAs()
{
    if ( player->getSidTune()->getStatus() )
    {
        // Get last dir from config file.
        config->setGroup(ConfigC::groups[ConfigC::Selector]);
        QString lastSaveDir = config->getString(ConfigC::keys[ConfigC::SaveDir]);

        QString fileName = QFileDialog::getSaveFileName(lastSaveDir);
        if ( !fileName.isEmpty() && !player->getSidTune()->savePSIDfile(fileName,true) )
        {
            QMessageBox error;
            error.setIcon(QMessageBox::Warning);
            error.setText("Could not save file!");
            error.adjustSize();
            error.show();
        }
        // Determine dir name and save it in config file.
        int slashPos = fileName.findRev( QDir::separator() );
        if ( slashPos >= 0 )
            fileName.truncate(slashPos);
        config->setGroup(ConfigC::groups[ConfigC::Selector]);
        config->setEntry(ConfigC::keys[ConfigC::SaveDir],fileName);
    }
}

// ----------------------------------------------------------- Playlist stuff

void MainDialog::loadPlaylist()
{
    bool forceLoad = false;
    while ( player->getPlaylist()->isModified() && !forceLoad )
    {
        QMessageBox mb( "SIDPLAY",
                       "Playlist has not been saved yet!\n"
                       "Do you want to save it?",
                       QMessageBox::Information,
                       QMessageBox::Yes | QMessageBox::Default | QMessageBox::Escape,
                       QMessageBox::No,
                       0);
        mb.setButtonText( QMessageBox::Yes, "Yes" );
        mb.setButtonText( QMessageBox::No, "No" );
        switch( mb.exec() ) 
        {
         case QMessageBox::Yes:
            savePlaylistAs();
            break;
         case QMessageBox::No:
            forceLoad = true;
            break;
        }
    }

    QString fileName = QFileDialog::getOpenFileName(fetchLastDir(ConfigC::groups[ConfigC::Playlist]),
                                                    "*.spl",this,0,"SIDPLAY: Load playlist");
    if ( !fileName.isEmpty() )
    {
        QApplication::setOverrideCursor( Qt::waitCursor );
        player->getPlaylist()->load(fileName);
        storeLastDir(ConfigC::groups[ConfigC::Playlist],fileName);
        QApplication::restoreOverrideCursor();
        bool goodList = PlaylistCheck::checkBaseDir( this, player->getPlaylist(),
                                                     myFileDlg );
        if ( !goodList )
            player->getPlaylist()->clear();
        myPlaylistEditDlg->takeList( player->getPlaylist() );
        myPlaylistDlg->takeList( player->getPlaylist() );
        if ( goodList )
            myPlaylistDlg->show();
    }
}

// slot
void MainDialog::savePlaylistAs()
{
    QString fileName = QFileDialog::getSaveFileName(fetchLastDir(ConfigC::groups[ConfigC::Playlist]),
                                                    "*.spl",this,0,"SIDPLAY: Save playlist");
    if ( !fileName.isEmpty() )
    {
        QApplication::setOverrideCursor( Qt::waitCursor );
        bool ret = player->getPlaylist()->save(fileName);
        storeLastDir(ConfigC::groups[ConfigC::Playlist],fileName);
        QApplication::restoreOverrideCursor();
        if (!ret)
        {
            QMessageBox error;
            error.setIcon(QMessageBox::Warning);
            error.setText("Could not save playlist!");
            error.adjustSize();
            error.show();
        }
    }
}

const QString& MainDialog::fetchLastDir(const char* group)
{
    config->setGroup(group);
    return config->getString(ConfigC::keys[ConfigC::Dir]);
}
 
void MainDialog::storeLastDir(const char* group, QString& fileName)
{
    // Determine dir name and save it in config file.
    int slashPos = fileName.findRev( QDir::separator() );
    if ( slashPos >= 0 )
        fileName.truncate(slashPos);
    config->setGroup(group);
    config->setEntry(ConfigC::keys[ConfigC::Dir],fileName);
}

void MainDialog::openPlaylistDialog()
{
    myPlaylistDlg->show();
}

void MainDialog::openPlaylistEditDialog()
{
    myPlaylistEditDlg->show();
}

void MainDialog::openPlaylistOptions()
{
    PlaylistOptDialog* myPlaylistOptDlg = new PlaylistOptDialog(this,"Playlist Options Dialog",true);
    myPlaylistOptDlg->playtimeSpinBox->setValue( playtimeDefault );
    myPlaylistOptDlg->fadeoutSpinBox->setValue( fadeoutDefault );
    bool oldIsSongLenDbEnabled = isSongLenDbEnabled();
    myPlaylistOptDlg->songlenDBcheckbox->setChecked( isSongLenDbEnabled() );
    int result = myPlaylistOptDlg->exec();
    if ( result==QDialog::Accepted )
    {
        playtimeDefault = myPlaylistOptDlg->playtimeSpinBox->text().toInt();
        fadeoutDefault = myPlaylistOptDlg->fadeoutSpinBox->text().toInt();
        if ( myPlaylistOptDlg->songlenDBcheckbox->isChecked() &&
             !oldIsSongLenDbEnabled )
            setSongLenDbEnabled( initSongLengthDB() );
    }
    delete myPlaylistOptDlg;
}

// slot
void MainDialog::addTuneToPlaylist()
{
    if ( player->getSidTune()->getStatus() )
    {
        sidTuneInfo mySidTuneInfo = player->getSidTuneInfo();
        PlaylistItem* item = new PlaylistItem;
        item->fileNameString = sidFileNameFull;
        item->titleString = mySidTuneInfo.infoString[0];
        item->authorString = mySidTuneInfo.infoString[1];
        item->copyrightString = mySidTuneInfo.infoString[2];
        item->subtune = mySidTuneInfo.currentSong;
        item->songs = mySidTuneInfo.songs;
        item->fadeout = fadeoutDefault;
        if ( isSongLenDbEnabled() )
        {
            item->time = querySongLengthDB(player->getSidTune(),mySidTuneInfo.currentSong);
        }
        else
            item->time = playtimeDefault;
        addTuneToPlaylistMain(item);
    }
}

void MainDialog::addTuneToPlaylistMain(PlaylistItem* item)
{
    QApplication::setOverrideCursor( Qt::waitCursor );
    player->getPlaylist()->add(item);
    myPlaylistDlg->add(item);
    myPlaylistEditDlg->add(item);
    QApplication::restoreOverrideCursor();
}

void MainDialog::addDirToPlaylist()
{
    addDirMain(false);  // not recursive
}

void MainDialog::addTreeToPlaylist()
{
    addDirMain(true);  // recursive
}

void MainDialog::addDirMain(bool recurse)
{
    
    QString dirName = 
        QFileDialog::getExistingDirectory(fetchLastDir(ConfigC::groups[ConfigC::Selector]),this,0,
                                          "SIDPLAY: Open directory to be added");
    if ( dirName.isEmpty() )
    {
        return;  // User cancelled. 
    }
    storeLastDir(ConfigC::groups[ConfigC::Selector],dirName);
    SidTuneMod* mySidLoader = new SidTuneMod(0);  // unchecked alloc
    QApplication::setOverrideCursor( Qt::waitCursor );
    addDirTraverse(dirName,recurse,mySidLoader);
    QApplication::restoreOverrideCursor();
    delete mySidLoader;
}

void MainDialog::addDirTraverse(QString dirName, bool recurse, SidTuneMod* mySidLoader)
{
    QDir* startDir = new QDir(dirName);
       CHECK_PTR(startDir);

    const QFileInfoList* list = startDir->entryInfoList();
    QListIterator<QFileInfo> it( *list );  // create list iterator
   
    for ( it.toFirst(); it.current(); ++it )
    {
        QFileInfo* fi = it.current();
        if ( recurse && fi->isDir() && fi->fileName()!="." && fi->fileName()!=".." )
        {
            addDirTraverse(fi->absFilePath(),recurse,mySidLoader);
        }
        else if ( fi->isFile() && mySidLoader->open(fi->absFilePath()) )
        {
            sidTuneInfo tmpInfo;
            mySidLoader->getInfo(tmpInfo);
            PlaylistItem* item = new PlaylistItem;
            item->fileNameString = fi->absFilePath();
            item->titleString = tmpInfo.infoString[0];
            item->authorString = tmpInfo.infoString[1];
            item->copyrightString = tmpInfo.infoString[2];
            item->subtune = tmpInfo.startSong;
            item->songs = tmpInfo.songs;
            item->fadeout = fadeoutDefault;
            if ( isSongLenDbEnabled() )
            {
                item->time = querySongLengthDB(mySidLoader,tmpInfo.startSong);
            }
            else
                item->time = playtimeDefault;
            addTuneToPlaylistMain(item);
        }
    }
    delete startDir;
}

// slot
void MainDialog::updatePlaylist()
{
    PlaylistCheck::updateEntries( this, player->getPlaylist(), myPlaylistDlg, 
                                 myPlaylistEditDlg, myFileDlg );
}

// slot
void MainDialog::markTune()
{
    if ( player->getSidTune()->getStatus() )
    {
        sidTuneInfo mySidTuneInfo = player->getSidTuneInfo();
        PlaylistItem* item = new PlaylistItem;
        item->fileNameString = sidFileNameFull;
        item->titleString = mySidTuneInfo.infoString[0];
        item->authorString = mySidTuneInfo.infoString[1];
        item->copyrightString = mySidTuneInfo.infoString[2];
        item->subtune = mySidTuneInfo.currentSong;
        item->songs = mySidTuneInfo.songs;
        PlaylistItem i = player->getCurPlaylistParams();
        item->fadeout = i.fadeout;
        item->time = i.time;
        QApplication::setOverrideCursor( Qt::waitCursor );
        hotlist->add(item);
        QApplication::restoreOverrideCursor();
    }
}

// slot
void MainDialog::markPlaylist()
{
    QApplication::setOverrideCursor( Qt::waitCursor );
    // Add all playlist items to hotlist.
    QListIterator<PlaylistItem> it(player->getPlaylist()->list);
    for ( it.toFirst(); it.current(); ++it ) 
    {
        PlaylistItem* item = new PlaylistItem;
        *item = *it.current();
        hotlist->add(item);
    }
    QApplication::restoreOverrideCursor();
}

// slot
void MainDialog::addHotlistToPlaylist()
{
    QApplication::setOverrideCursor( Qt::waitCursor );
    // Add all hotlist list items to playlist.
    QListIterator<PlaylistItem> it(hotlist->list);
    for ( it.toFirst(); it.current(); ++it ) 
    {
        PlaylistItem* item = new PlaylistItem;
        *item = *it.current();
        player->getPlaylist()->add(item);
        myPlaylistDlg->add(item);
        myPlaylistEditDlg->add(item);
    }
    // Clear hotlist.
    hotlist->clear();
    QApplication::restoreOverrideCursor();
}
 
// slot
void MainDialog::saveHotlist()
{
    // We use the last playlist open/save dir.
    QString fileName = QFileDialog::getSaveFileName(fetchLastDir(ConfigC::groups[ConfigC::Playlist]),
                                                    "*.spl",this,0,"SIDPLAY: Save hotlist");
    if ( !fileName.isEmpty() )
    {
        QApplication::setOverrideCursor( Qt::waitCursor );
        bool ret = hotlist->save(fileName);
        storeLastDir(ConfigC::groups[ConfigC::Playlist],fileName);
        QApplication::restoreOverrideCursor();
        if (!ret)
        {
            QMessageBox error;
            error.setIcon(QMessageBox::Warning);
            error.setText("Could not save hotlist!");
            error.adjustSize();
            error.show();
        }
    }
}

// --------------------------------------------------------------------------

void MainDialog::about()
{
    QString verCat = "SIDPLAY/X11   Version ";
    verCat += VERSION;
    verCat += "\n\n";
    verCat += "Music player and SID chip emulator ";
    verCat += player->getEmuEngine()->getVersionString();
    verCat += "\nby Michael Schwendt <sidplay@geocities.com>\n\n";
    
    verCat += mySTIL.getVersion();
        
    verCat += "\nListening mileage: ";
    long int mileage = usage+player->getEmuEngine()->getSecondsTotal();
    int days = mileage/(3600L*24);
    mileage -= days*(3600L*24);
    int hours = mileage/3600L;
    mileage -= hours*3600L;
    int minutes = mileage/60;
    mileage -= minutes*60;
    QString timeCat;
    timeCat.sprintf("%d days, %d hours, %d minutes",days,hours,minutes);
    verCat += timeCat;
    QMessageBox::about(this,"About SIDPLAY",verCat);
}

void MainDialog::aboutQt()
{
    QMessageBox::aboutQt(this);
}

void MainDialog::openAudioDialog()
{
    myAudioDlg->setButtons();
    myAudioDlg->show();
}

void MainDialog::openEmulatorDialog()
{
    myEmuDlg->show();
}

void MainDialog::openFilterDialog()
{
    myFilterDlg->show();
}

void MainDialog::openMixerDialog()
{
    myMixerDlg->show();
}

void MainDialog::openWaveViewDialog()
{
    if ( !myWaveViewDlg->isShown() )
        myWaveViewDlg->show();
}

void MainDialog::openHVSC_Dialog()
{
    myHVSC_Dlg->show();
}

void MainDialog::openHistoryDialog()
{
    myHistoryDlg->show();
}

// --------------------------------------------------------------------------

/* slot */
void MainDialog::setPlaylistEnabled(bool value)
{
    player->enablePlaylist(value);
}

/* slot */
void MainDialog::setLoopPlayEnabled(bool value)
{
    player->enablePlaylistLoop(value);
}

/* slot */
void MainDialog::setRandomPlayEnabled(bool value)
{
    player->enablePlaylistRand(value);
}

/* slot */
void MainDialog::newSubSong(int newSong)
{
    if ( player->getSidTune()->getStatus() && 
         newSong>=1 && newSong<=player->getSidTuneInfo().songs )
    {
        player->pause();
        resetTimeDisplay();
        player->initSong(newSong);
        //??? player->enablePlaylist(false);
        playlistCheckBox->setChecked(false);
        player->resume();
        if ( player->isReady() )
        {
            showSidTuneInfo();
            showStilInfo();
            setPrevBtnState();
            setNextBtnState();
        }
    }
    else
    {
        player->stop();
        playerButtonsOff();
    }
}

void MainDialog::updateSubSongSlider()
{
    if ( player->getSidTune()->getStatus() )
    {
        // Block signals here, or otherwise the slider would emit
        // newValue(int) and activate the new newSubSong(int) slot.
        subSongSlider->blockSignals(true);
        subSongSlider->setRange(1,player->getSidTuneInfo().songs);
        subSongSlider->setSteps(1,1);
        subSongSlider->setValue(player->getSidTuneInfo().currentSong);
        subSongSlider->blockSignals(false);
    }
}

void MainDialog::resetTimeDisplay()
{
    player->getEmuEngine()->resetSecondsThisSong();
    timeLCD->resetDisplay();
}

void MainDialog::startLoaded()
{
    updateSubSongSlider();
    player->start();
    if ( player->isPlaying() )
        playerButtonsForceStart();
    showSidTuneInfo();
}

// ----------------------------------------------------------- Player buttons

void MainDialog::prevSongPressed()
{
    if ( player->havePrevSong() )
    {
        prevBtn->setPixmap(*prev_l_pPic);
        resetTimeDisplay();
        player->pause();
        player->initPrevSong();
        //??? player->enablePlaylist(false);
        playlistCheckBox->setChecked(false);
        updateSubSongSlider();
        showSidTuneInfo();
        showStilInfo();
        if ( player->isReady() )
               setNextBtnState();
        else
            playerButtonsOff();
        player->resume();
    }
}

void MainDialog::prevSongReleased()
{
    setPrevBtnState();
}

void MainDialog::setPrevBtnState()
{
    if ( player->havePrevSong() )
        prevBtn->setPixmap(*prev_lPic);
    else
        prevBtn->setPixmap(*prevPic);
}

void MainDialog::nextSongPressed()
{
    if ( player->haveNextSong() )
    {
        nextBtn->setPixmap(*next_l_pPic);
        resetTimeDisplay();
        player->pause();
        player->initNextSong();
        //??? player->enablePlaylist(false);
        playlistCheckBox->setChecked(false);
        updateSubSongSlider();
        showSidTuneInfo();
        showStilInfo();
        if ( player->isReady() )
            setPrevBtnState();
        else
            playerButtonsOff();
        player->resume();
    }
}

void MainDialog::nextSongReleased()
{
    setNextBtnState();
}

void MainDialog::setNextBtnState()
{
    if ( player->haveNextSong() )
        nextBtn->setPixmap(*next_lPic);
    else
        nextBtn->setPixmap(*nextPic);
}

void MainDialog::stopSongPressed()
{
    if ( player->isReady() && player->isPlaying() )
    {
        stopBtn->setPixmap(*stop_l_pPic);
        playBtn->setPixmap(*play_lPic);
        ffBtn->setPixmap(*ffPic);
        player->stop();
        //??? player->enablePlaylist(false);
        playlistCheckBox->setChecked(false);
    }
}

void MainDialog::stopSongReleased()
{
    stopBtn->setPixmap(*stopPic);
}

void MainDialog::playSongPressed()
{
    if ( !player->haveSidTune() )
        return;
    if ( player->isStopped() )
    {
        playBtn->setPixmap(*play_l_pPic);
        stopBtn->setPixmap(*stop_lPic);
        resetTimeDisplay();
        player->start();
    }
    else if ( player->isPaused() )
    {
        playBtn->setPixmap(*play_l_pPic);
        player->resume();
    }
    else if ( player->isPlaying() && !player->isForwarding() )
    {
        playBtn->setPixmap(*pause_pPic);
        player->pause();
    }
}

void MainDialog::playSongReleased()
{
    if ( !player->haveSidTune() )
        return;
    if ( player->isPaused() )
    {
        playBtn->setPixmap(*play_lPic);
        ffBtn->setPixmap(*ffPic);
    }
    else if ( player->isPlaying() && !player->isForwarding() )
    {
        playBtn->setPixmap(*pause_lPic);
        ffBtn->setPixmap(*ff_lPic);
    }
    // Handle player error state here.
}

void MainDialog::forwardSongPressed()
{
    if ( player->isForwarding() )
    {
        playBtn->setPixmap(*pause_lPic);
        ffBtn->setPixmap(*ff_lPic);
        player->playNormalSpeed();
    }
    else if ( player->isPlaying() )
    {
        playBtn->setPixmap(*pausePic);
        ffBtn->setPixmap(*ff_pPic);
        player->playFastForward();
    }
}

void MainDialog::forwardSongReleased()
{
    if ( player->isForwarding() )
        ffBtn->setPixmap(*ff_pPic);
    else if ( player->isPlaying() )
        ffBtn->setPixmap(*ff_lPic);
}

// --------------------------------------------------------------- HVSC stuff

void MainDialog::createHVSCfileName(const QString& fileName)
{
    // Is filename anywhere in HVSC tree?
    if (fileName.find(myHVSC_Dlg->getConfig().hvscRootPath.absPath()) == 0)
    {
        // Create filename which is relative to the HVSC root.
        sidFileNameHVSC = 
            fileName.right(fileName.length()-
                           myHVSC_Dlg->getConfig().hvscRootPath.absPath().length());
    }
    else
    {
        sidFileNameHVSC = "";
    }
}

// --------------------------------------------------------------- STIL stuff

bool MainDialog::initStilViewClass()
{
    // Let STIL View class perform the check for the STIL database file.
    if ( mySTIL.setBaseDir( myHVSC_Dlg->getConfig().hvscRootPath.absPath() ) )
        return true;
    
    QMessageBox error;
    error.setIcon( QMessageBox::Warning );
    error.setText( mySTIL.getErrorStr() );
    error.adjustSize();
    error.show();
    return false;
}

void MainDialog::toggleStilDialog()
{
    if ( isStilInfoEnabled() )
    {
        // Disable STIL View.
        setStilInfoEnabled( false );
        // Hide a possibly open STIL View dialog.
        if ( myStilDlg->isVisible() )
            myStilDlg->hide();
    }
    else  // if !isStilInfoEnabled()
    {
        // Enabled STIL View (if init is successful).
        setStilInfoEnabled( initStilViewClass() );
        // Try to show STIL entry for currently active file.
        showStilInfo();
    }
}

void MainDialog::showStilInfo()
{
    // STIL View not enabled or no HVSC-relative file active?
    if ( !isStilInfoEnabled() || sidFileNameHVSC.isEmpty() )
    {
        // Nothing to show. Hence hide a possibly open STIL View dialog.
        myStilDlg->hide();
        return;
    }
    
    int song = player->getSidTuneInfo().currentSong;
    if (myHVSC_Dlg->getConfig().showForFile)
        song = 0;              // 0 means: file-global STIL entry
    
    bool stilIsOkay = true;    // we want to avoid successive errors
    // These should be converted to QString as well.
    char* globalComment;
    char* stilEntry;
    char* bugEntry;
    // Now retrieve the individual STIL entries.
    if (stilIsOkay)
    {
        globalComment = mySTIL.getGlobalComment(sidFileNameHVSC);
        stilIsOkay = !mySTIL.hasCriticalError();
    }
    if (stilIsOkay)
    {
        stilEntry = mySTIL.getEntry(sidFileNameHVSC,song);
        stilIsOkay = !mySTIL.hasCriticalError();
    }
    if (stilIsOkay)
    {
        bugEntry = mySTIL.getBug(sidFileNameHVSC,song);
        stilIsOkay = !mySTIL.hasCriticalError();
    }
    if (stilIsOkay)
    {
        // Only show STIL View if main window is visible.
        myStilDlg->setLabels(globalComment,stilEntry,bugEntry);
    }
    else
    {
        // Disable STIL View.
        // Protect the user from any further critical errors.
        setStilInfoEnabled(false);
        myStilDlg->hide();
        
        QMessageBox error;
        error.setIcon(QMessageBox::Critical);
        error.setText(mySTIL.getErrorStr());
        error.adjustSize();
        error.show();
    }
}

// ------------------------------------------------------ Songlength DB stuff

bool MainDialog::initSongLengthDB()
{
    QString fileName = configPath + '/' + SONGLENGTHS_DB_FILE_NAME;
    
    if ( !QFile::exists(fileName) )
        fileName = myHVSC_Dlg->getConfig().hvscRootPath.absPath() + HVSC_DOC_DIR_NAME + '/' + SONGLENGTHS_DB_FILE_NAME;

    if ( player->getSongLengthDB()->init(fileName ) )
        return true;

    QMessageBox error;
    error.setIcon( QMessageBox::Warning );
    error.setText( player->getSongLengthDB()->getErrorStr() );
    error.adjustSize();
    error.show();
    return false;
}
 
void MainDialog::toggleSongLengthDB()
{
    if ( isSongLenDbEnabled() )
    {
        // Disable songlength database.
        setSongLenDbEnabled(false);
        player->enableSongLengthDB(false);
    }
    else
    {
        // Enable songlength database (if init is successful).
        setSongLenDbEnabled( player->enableSongLengthDB( initSongLengthDB() ));
    }
}

int MainDialog::querySongLengthDB(SidTuneMod* mySidLoader, int song)
{
    MD5 myMD5;
    mySidLoader->createMD5(myMD5);
    myMD5.finish();
    
    // Print fingerprint.
    QString digest, tmp;
    for (int di = 0; di < 16; ++di)
        digest += tmp.sprintf("%02x",(int)myMD5.getDigest()[di]);

    SongLengthDBitem sli;
    // No other error handling yet.
    bool ret = player->getSongLengthDB()->getSongLength(digest,song,sli);
#ifdef XSID_WB_DEBUG
    if ( !ret )
        cerr << player->getSongLengthDB()->getErrorStr() << endl;
#endif
    return sli.playtime;
}

// ------------------------------------------------------------------- Player

void MainDialog::playFile(const QString& fileName)
{
    playFileMain(fileName,0,true,false);
}

void MainDialog::playHistoryFile(const QString& fileName)
{
    playFileMain(fileName,0,false,true);
}

void MainDialog::playPlaylistFile(const PlaylistItem& item)
{
    playFileMain(item.fileNameString,item.subtune,true,true);
    player->setPlaylistParams(item);
    playlistCheckBox->setChecked(true);
}

void MainDialog::playFileMain(const QString& fileName, int song,
                              bool putInHistory, bool enablePlaylist)
{
    QFile testFile(fileName);
    // Who would use '-' for something else than stdin?
    if ( fileName.isEmpty() || ( fileName!="-" && !testFile.exists() ))
        return;
    loadSidTune(fileName,song,putInHistory);
    playlistCheckBox->setChecked(enablePlaylist);
    startLoaded();
}

void MainDialog::loadSidTune(const QString& fileName, int overrideStartSong,
                             bool putInHistory)
{
    player->stop();
    resetTimeDisplay();

    // Make a copy for later use.
    sidFileNameFull = fileName;
    createHVSCfileName(fileName);
        
    bool loadSucceeded = false;
    
    // sidTune::open() does not support standard input.
    if (fileName == "-")
    {
        ubyte* fileBuf = new ubyte[maxSidtuneFileLen];
        if (fileBuf == 0)
        {
            // If we are out of memory at this point,
            // it is very unlikely that anything else works.
            QMessageBox error;
            error.setIcon(QMessageBox::Critical);
            error.setText("Out of memory.");
            error.adjustSize();
            error.show();
            return;
        }
        udword fileLen = 0;
        ubyte datb;
        while (cin.get(datb) && fileLen<maxSidtuneFileLen)
            fileBuf[fileLen++] = datb;
        loadSucceeded = player->getSidTune()->load(fileBuf,fileLen);
        delete[] fileBuf;
    }
    else
    {
        loadSucceeded = player->getSidTune()->open(fileName);
    }

    PlaylistItem pi = player->getCurPlaylistParams();
    pi.fadeout = fadeoutDefault;
    pi.time = playtimeDefault;
    player->setPlaylistParams(pi);

    if ( !loadSucceeded )
    {
        playerButtonsOff();
        clearSidTuneInfo();
        
        infoField->setText(player->getSidTuneInfo().statusString);
        // With too many errors one has to click too much if
        // message boxes are used here.
    }
    
    if ( player->getSidTune()->getStatus() )
    {
        int song = (overrideStartSong != 0) ? overrideStartSong : player->getSidTuneInfo().startSong;
        player->initSong(song);
        
        showStilInfo();
        
        // Add to history if not from there.
        if ( putInHistory )
        {
            sidTuneInfo mySidTuneInfo = player->getSidTuneInfo();
            QString tmp = mySidTuneInfo.nameString;
            tmp += " (";
            tmp += mySidTuneInfo.authorString;
            tmp += ")";
            myHistoryDlg->add(fileName,tmp);
        }
    }
}

