/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef XSIDPLAY_PLAYER_H
#define XSIDPLAY_PLAYER_H

class AudioDriver;
class ConfigFile;
class MainDialog;
class TimeLCD;
class Playlist;
class PlaylistDialog;
class WaveViewDialog;

#include <qobject.h>

#include <sidplay/player.h>

#include "AudioConfig.h"
#include "Fading.h"
#include "PlaylistItem.h"
#include "myTimer.h"
#include "songlendb/File.h"
#include "songlendb/SidTuneMod.h"

class Player : public QObject
{
    Q_OBJECT
        
 public:
    Player();
    ~Player();

    void readConfig(ConfigFile*);
    void writeConfig(ConfigFile*);
    
    void link(MainDialog*, TimeLCD*, PlaylistDialog*, WaveViewDialog*);
    bool init(const AudioConfig&);
         
    void start();
    void playFastForward();
    void playNormalSpeed();
    void pause();
    void pauseHard();
    void resume();
    void stop();

    bool isReady() const;
    bool isPaused() const;
    bool isStopped() const;
    bool isPlaying() const;
    bool isForwarding() const;

    bool haveSidTune() const;
    bool havePrevSong();
    bool haveNextSong();
    
    void initSong(int);
    void initNextSong();
    void initPrevSong();
    
    emuEngine* getEmuEngine() const;
    const emuConfig& getEmuConfig();
    const emuConfig& getEmuConfigDefault() const;

    const AudioConfig& getAudioConfig() const;
    
    SidTuneMod* getSidTune() const;
    const sidTuneInfo& getSidTuneInfo();

    SongLengthDB* getSongLengthDB() const;
    bool enableSongLengthDB(bool value = true);
        
    Playlist* getPlaylist() const;
    void setPlaylistParams(const PlaylistItem& item);
    const PlaylistItem& getCurPlaylistParams();
    void enablePlaylist(bool value = true);
    void enablePlaylistLoop(bool value = true);
    void enablePlaylistRand(bool value = true);
    void playFromList(uint index);

 protected:
    emuEngine* pEmuEngine;
    emuConfig myEmuConfig;
    emuConfig myEmuConfigDefault;
    
    SidTuneMod* pSidTune;  // derived sidtune loader
    sidTuneInfo mySidTuneInfo;

    AudioDriver* pAudioDrv;
    AudioConfig myAudioConfig;
    
    SongLengthDB* pSongLengthDB;
    
    Playlist* pPlaylist;
    
    struct PlayerState
    {
        bool ready;      // whether all components are initialized 
                         // and the last operation on a sidtune has been
                         // successful
        bool playing;
        bool paused;
        bool forwarding;
        bool stopped;
    } state;

    bool openAudioDriver();
    void closeAudioDriver();

    void initAudioBufferSystem();
    bool openAudioBufferSystem();
    void freeAudioBufferSystem();

    void fillBuffer(ubyte*,udword);

    // Whether the player at the end of a song asks the playlist
    // dialog to activate a new song.
    bool playlistDriven;
    bool loopPlaylist;
    bool randPlaylist;
    PlaylistItem curPlaylistItem;
    
    bool driverIsOpen;

    FadingOut myFadeOut;

    myTimer playbackTimer;

    MainDialog* myMainDlg;
    TimeLCD* myTimeLCD;
    PlaylistDialog* myPlaylistDlg;
    WaveViewDialog* myWaveViewDlg;
    
    // Mostly unimplemented! Multi-buffering system.
    
#define maxBuffers 256 
    // worst case: 256*256 = 64 KB
    
    bool audioBuffersReady;
    int buffers;
    int multiBufferSize;
    int currentBuffer,
         nextBuffer;
    bool bufferFlag[maxBuffers];
    ubyte* pBuffer[maxBuffers];
    int bufferCount[maxBuffers];
    int lastBufferCount;
    
 public slots:
    bool playNextFromList(int delta);
    void timerJob();
    
 signals:
    void playListItem(const PlaylistItem&);
    void playerPlayRequest(uint);
    void stopPlaylistPlay();
    
 private:
    // prevent copying
    Player(const Player&);
    Player& operator=(Player&);  
};

#endif  /* XSIDPLAY_PLAYER_H */
