/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "AudioDriverALSA.h"

#ifdef XSID_ALSA_DEBUG
#include <iostream>
#include <iomanip>
using namespace std;
#endif

AudioDriverALSA::AudioDriverALSA()
{
#ifdef XSID_ALSA_DEBUG
  cout << "AudioDriverALSA::AudioDriverALSA" << endl;
#endif
  _audioHandle=NULL;
  outOfOrder();
}

AudioDriverALSA::~AudioDriverALSA()
{ 
#ifdef XSID_ALSA_DEBUG
  cout << "AudioDriverALSA::~AudioDriverALSA" << endl;
#endif
  close();
}

bool AudioDriverALSA::open(const AudioConfig& inConfig)
{
#ifdef XSID_ALSA_DEBUG
  cout << "AudioDriverALSA::open" << endl;
#endif
  static AudioConfig lastConfig;

  if ( haveStream && inConfig==config && inConfig==lastConfig ) {
    if (snd_pcm_pause(_audioHandle, 0)<0) {
      errorString = "ERROR: cannot open again";
      return false;
    }
    return true;
  } else unload();

  // Copy input parameters. May later be replaced with driver defaults.
  config = inConfig;

  snd_pcm_uframes_t    buffer_frames;
  snd_pcm_hw_params_t *hw_params = 0;

  if (snd_pcm_open (&_audioHandle, "default", SND_PCM_STREAM_PLAYBACK, 0)<0)
  {
    errorString = "ERROR: Could not open audio device.";
    return false;
  }

  if (snd_pcm_hw_params_malloc (&hw_params)<0)
  {
    errorString = "ERROR: could not malloc hwparams.";
    return false;
  }
 
  if (snd_pcm_hw_params_any (_audioHandle, hw_params)<0)
  {
    errorString = "ERROR: could not initialize hw params";
    return false;
  }
 
  if (snd_pcm_hw_params_set_access (_audioHandle, hw_params, SND_PCM_ACCESS_RW_INTERLEAVED)<0)
  {
    errorString = "ERROR: could not set access type";
    return false;
  }

  snd_pcm_format_t alsamode;
  switch (config.precision)
  {
    case AudioConfig::BITS_8:
      if (config.encoding==AudioConfig::UNSIGNED_PCM) alsamode = SND_PCM_FORMAT_U8;
      else alsamode = SND_PCM_FORMAT_S8;
      break;
    case AudioConfig::BITS_16:
      if (config.encoding==AudioConfig::UNSIGNED_PCM) alsamode = SND_PCM_FORMAT_U16;
      else alsamode = SND_PCM_FORMAT_S16;
      break;
    default:
      errorString = "ERROR: set desired number of bits for audio device.";
      return false;
  }

  if (snd_pcm_hw_params_set_format (_audioHandle, hw_params, alsamode)<0)
  {
    errorString = "ERROR: could not set sample format";
    return false;
  }
   
  if (snd_pcm_hw_params_set_channels (_audioHandle, hw_params, config.channels)<0)
  {
    errorString = "ERROR: could not set channel count";
    return false;
  }
   
  unsigned int rate = (unsigned int)config.frequency;
  if (snd_pcm_hw_params_set_rate_near (_audioHandle, hw_params, &rate, 0)<0)
  {
    errorString = "ERROR: could not set sample rate";
    return false;
  }

  _alsa_to_frames_divisor = config.channels * config.precision / 8;
  buffer_frames = config.bufSize;
  if (snd_pcm_hw_params_set_period_size_near(_audioHandle, hw_params, &buffer_frames, 0)<0)
  {
    errorString = "ERROR: could not set period size near";
    return false;
  }
  unsigned int periods=32; /// just for testing
  if (snd_pcm_hw_params_set_periods_near(_audioHandle, hw_params, &periods, 0)<0)
  {
    errorString = "ERROR: could not set periods near";
    return false;
  }

  if (snd_pcm_hw_params (_audioHandle, hw_params)<0)
  {
    errorString = "ERROR: could not set hw parameters";
    return false;
  }

  snd_pcm_hw_params_free (hw_params);
  hw_params = 0;
 
  if (snd_pcm_prepare (_audioHandle)<0)
  {
    errorString = "ERROR: could not prepare audio interface for use";
    return false;
  }

  errorString = "None";
  haveStream=true;
  lastConfig = config;

  return true;
}

void AudioDriverALSA::close()
{
#ifdef XSID_ALSA_DEBUG
  cout << "AudioDriverALSA::close" << endl;
#endif
  if (_audioHandle != NULL )
  {
    snd_pcm_close(_audioHandle);
    _audioHandle = NULL;
    outOfOrder ();
  }
  haveStream = false;
}

void AudioDriverALSA::unload()
{
#ifdef XSID_ALSA_DEBUG
  cout << "AudioDriverALSA::unload" << endl;
#endif
  if ( !haveStream ) return;

  outOfOrder(); 
}

bool AudioDriverALSA::reset()
{
#ifdef XSID_ALSA_DEBUG
  cout << "AudioDriverALSA::reset" << endl;
#endif
  if (snd_pcm_pause(_audioHandle, 1)<0) {
    errorString = "ERROR: cannot reset";
    return false;
  }

  return true;
}

void AudioDriverALSA::play(void* buffer, unsigned long int bufferSize)
{
  if (snd_pcm_writei (_audioHandle, buffer, bufferSize / _alsa_to_frames_divisor) == -EPIPE)
   snd_pcm_prepare (_audioHandle); // Underrun
}

void AudioDriverALSA::outOfOrder()
{
#ifdef XSID_ALSA_DEBUG
  cout << "AudioDriverALSA::outOfOrder" << endl;
#endif
  // Reset everything.
  haveStream = false;
  errorString = "None";
}
