/*  TideEditor class.  */

/*****************************************************************************\

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

\*****************************************************************************/



#include "tideEditor.h"
#include "tideEditorHelp.h"
#include "tideDialog.h"
#include "tideList.h"



//  This is the alpha sort function for qsort.

static int compare_names (const void *a, const void *b)
{
    char low_left[NAME_LENGTH], low_right[NAME_LENGTH];


    STATION_LIST left = *(STATION_LIST *)(a);
    STATION_LIST right = *(STATION_LIST *)(b);

    for (int i = 0 ; i < strlen (left.name) ; i++)
        low_left[i] = tolower (left.name[i]);
    low_left[strlen (left.name)] = 0;

    for (int i = 0 ; i < strlen (right.name) ; i++)
        low_right[i] = tolower (right.name[i]);
    low_right[strlen (right.name)] = 0;

    return (strcmp (low_left, low_right));
}



TideEditor::TideEditor (QWidget * parent, const char *name):
  QMainWindow (parent, name)
{
    double slat = -90.0;
    double nlat = 90.0;
    double wlon = -180.0;
    double elon = 180.0;


    //  Get the users default settings (right now just color and position
    //  format.

    envin ();


    //  Make the map.

    map = new Map (this);

    setCentralWidget (map);


    QWhatsThis::add (map, mapText);


    //  Button, button, who's got the buttons?

    QPixmap leftIcon = QPixmap (icon_left);

    QMimeSourceFactory::defaultFactory ()->setPixmap ("icon_left", leftIcon);
    QPixmap rightIcon = QPixmap (icon_right);

    QMimeSourceFactory::defaultFactory ()->setPixmap ("icon_right",
        rightIcon);
    QPixmap upIcon = QPixmap (icon_up);

    QMimeSourceFactory::defaultFactory ()->setPixmap ("icon_up", upIcon);
    QPixmap downIcon = QPixmap (icon_down);

    QMimeSourceFactory::defaultFactory ()->setPixmap ("icon_down", downIcon);



    QToolBar *tools = new QToolBar (this);

    QPixmap quitIcon = QPixmap (icon_quit);

    bQuit = new QToolButton (quitIcon, "Quit", QString::null, this,
        SLOT (slotQuit ()), tools);
    QWhatsThis::add (bQuit, quitText);
    QMimeSourceFactory::defaultFactory ()->setPixmap ("icon_quit", quitIcon);


    QPixmap zoomOutIcon = QPixmap (zoomout);

    bZoomOut = new QToolButton (zoomOutIcon, "Zoom Out", QString::null, this,
        SLOT (slotZoomOut ()), tools);
    QWhatsThis::add (bZoomOut, zoomOutText);
    QMimeSourceFactory::defaultFactory ()->setPixmap ("zoomout", zoomOutIcon);


    QPixmap resetIcon = QPixmap (icon_reset);

    bReset = new QToolButton (resetIcon, "Reset", QString::null, this,
        SLOT (slotReset ()), tools);
    QWhatsThis::add (bReset, resetText);
    QMimeSourceFactory::defaultFactory ()->setPixmap ("icon_reset",
        resetIcon);


    bLeft = new QToolButton (leftIcon, "Move West", QString::null, this,
        SLOT (slotLeft ()), tools);
    QWhatsThis::add (bLeft, leftText);
    QMimeSourceFactory::defaultFactory ()->setPixmap ("icon_left",
        leftIcon);


    bUp = new QToolButton (upIcon, "Move North", QString::null, this,
        SLOT (slotUp ()), tools);
    QWhatsThis::add (bUp, upText);
    QMimeSourceFactory::defaultFactory ()->setPixmap ("icon_up",
        upIcon);


    bDown = new QToolButton (downIcon, "Move South", QString::null, this,
        SLOT (slotDown ()), tools);
    QWhatsThis::add (bDown, downText);
    QMimeSourceFactory::defaultFactory ()->setPixmap ("icon_down",
        downIcon);


    bRight = new QToolButton (rightIcon, "Move East", QString::null, this,
        SLOT (slotRight ()), tools);
    QWhatsThis::add (bRight, rightText);
    QMimeSourceFactory::defaultFactory ()->setPixmap ("icon_right",
        rightIcon);


    QPixmap searchIcon = QPixmap (search);
    bSearch =
        new QToolButton (searchIcon, "Station Search", QString::null,
        this, SLOT (slotSearch ()), tools);
    QWhatsThis::add (bSearch, searchText);
    QMimeSourceFactory::defaultFactory ()->setPixmap ("search", searchIcon);


    QPixmap prefsIcon = QPixmap (prefs);
    bPrefs =
        new QToolButton (prefsIcon, "Preferences", QString::null,
        this, SLOT (slotPrefs ()), tools);
    QWhatsThis::add (bPrefs, prefsText);
    QMimeSourceFactory::defaultFactory ()->setPixmap ("prefs", prefsIcon);


    QPixmap disasterIcon = QPixmap (disaster);
    bDisaster =
        new QToolButton (disasterIcon, "Abort Editing and Revert File", 
        QString::null, this, SLOT (slotDisaster ()), tools);
    QWhatsThis::add (bDisaster, disasterText);
    QMimeSourceFactory::defaultFactory ()->setPixmap ("disaster",
        disasterIcon);


    (void) QWhatsThis::whatsThisButton (tools);


    //  Setup the file menu.

    QAction *fileQuitAction;

    fileQuitAction =
        new QAction ("Quit", "&Quit", CTRL + Key_Q, this, "quit");
    connect (fileQuitAction, SIGNAL (activated ()), qApp,
        SLOT (closeAllWindows ()));


    QPopupMenu *file = new QPopupMenu (this);

    menuBar ()->insertItem ("&File", file);
    fileQuitAction->addTo (file);


    //  Setup the help menu.  I like leaving the About Qt in since this is
    //  a really nice package - and it's open source.

    QPopupMenu *help = new QPopupMenu (this);

    menuBar ()->insertSeparator ();
    menuBar ()->insertItem ("&Help", help);
    help->insertItem ("&About", this, SLOT (about ()), Key_F1);
    help->insertItem ("About &Qt", this, SLOT (aboutQt ()));


    if (getenv ("HFILE_PATH") == NULL)
    {
        fprintf (stderr, "\nUnable to find environment variable HFILE_PATH\n");
        exit (-1);
    }


    strcpy (hfile, getenv ("HFILE_PATH"));


    //  Disaster recovery.  Copy the file to a temp file in the same directory
    //  as the DB.  If everything goes south we can replace the file.  If not,
    //  we make this the .bak file at the end.

    FILE *fp, *fp1;
    if ((fp = fopen (hfile, "rb")) == NULL)
    {
        fprintf (stderr, "\nUnable to open Harmonic Constituent Database %s\n",
            hfile);
        exit (-1);
    }

    char string[512];
    sprintf (string, "%s.running", hfile);
    if ((fp1 = fopen (string, "wb")) == NULL)
    {
        fprintf (stderr, "Unable to create disaster recovery file %s\n", 
            string);
        exit (-1);
    }


    //  Read and write the whole file.

    int size;
    while (size = fread (string, 1, sizeof (string), fp))
        fwrite (string, 1, size, fp1);

    fclose (fp);
    fclose (fp1);


    if (!open_tide_db (hfile))
    {
        fprintf (stderr, "\nUnable to open Harmonic Constituent Database %s\n",
            getenv ("HFILE_PATH"));
        exit (-1);
    }

    db = get_tide_db_header ();


    //  Connect to the signals from the map class.

    connect (map, SIGNAL (leftMouseSignal (QMouseEvent *)), this, 
        SLOT (slotLeftMouse (QMouseEvent *)));
    connect (map, SIGNAL (midMouseSignal (QMouseEvent *)), this, 
        SLOT (slotMidMouse (QMouseEvent *)));
    connect (map, SIGNAL (rightMouseSignal (QMouseEvent *)), this, 
        SLOT (slotRightMouse (QMouseEvent *)));
    connect (map, SIGNAL (moveMouseSignal (QMouseEvent *)), this, 
        SLOT (slotMouseMove (QMouseEvent *)));
    connect (map, SIGNAL (redrawSignal ()), this, SLOT (slotRedraw ()));


    //  Set the zero zoom level map extents.

    map->setExtents (0, slat, nlat, wlon, elon);


    number_of_records = 0;
    station_list = NULL;


    statusBar ()->message (VERSION);


    //  Make the tide station list dialog.

    tideList = new TideList (this, "Tide Station List");
}



TideEditor::~TideEditor ()
{
    slotQuit ();
}



//  Get the users defaults.

void
TideEditor::envin ()
{

    /* Set Defaults so the if keys don't exist the parms are defined */

    strcpy (pos_format, "hdms");
    refStationColor = QColor (255, 0, 0);
    subStationColor = QColor (0, 0, 255);


//RWL if on windows store user prefs in the registry like any well behaved 
//program

#ifdef WIN32
	
    HKEY        hkey = NULL;
    DWORD       dwDisposition;

    DWORD       dwType, dwSize;
    char        strRGB[12];
    int         r, g, b;
    long        rtnVal;


    sprintf(strRGB,"%d,%d,%d", 255, 0, 0);

    rtnVal = RegCreateKeyEx (HKEY_CURRENT_USER, 
        TEXT ("Software\\NAVO\\TideEditor"), 0, NULL, 0, KEY_READ | KEY_WRITE,
        NULL, &hkey, &dwDisposition);
    if(rtnVal == ERROR_SUCCESS)
    {
        if (dwDisposition == REG_OPENED_EXISTING_KEY)
        {
            dwType = REG_SZ;
            dwSize = 10;
            RegQueryValueEx (hkey, TEXT ("PositionFormat"), NULL, &dwType, 
                (unsigned char *) pos_format, &dwSize);

            dwType = REG_SZ;
            dwSize = 12;;
            RegQueryValueEx (hkey, TEXT ("RefStationColor"), NULL, &dwType, 
	        (unsigned char *) strRGB, &dwSize);
            sscanf ((char *)strRGB, "%d,%d,%d", &r, &g, &b);
            refStationColor.setRgb (r, g, b);

            dwType = REG_SZ;
            dwSize = 12;;
            RegQueryValueEx (hkey, TEXT ("SubStationColor"), NULL, &dwType, 
	        (unsigned char *) strRGB, &dwSize);
            sscanf ((char *)strRGB, "%d,%d,%d", &r, &g, &b);
            subStationColor.setRgb (r, g, b);
        }
        RegCloseKey(hkey);
    }

#else

    if (getenv ("HOME") != NULL)
    {
        char string[128], info[128];
        FILE *fp;

        sprintf (string, "%s/.tideEditor", getenv ("HOME"));

        if ((fp = fopen (string, "r")) != NULL)
        {
            int r, g, b;
            while (fgets (string, sizeof (string), fp))
            {
                if (strchr (string, '=') != NULL)
                    strcpy (info, (strchr (string, '=') + 1));

                if (strstr (string, "[POSITION FORMAT]"))
                    sscanf (info, "%s", pos_format);

                if (strstr (string, "[REFERENCE STATION COLOR]"))
                {
                    sscanf (info, "%d,%d,%d", &r, &g, &b);
                    refStationColor = QColor (r, g, b);
                }

                if (strstr (string, "[SUBORDINATE STATION COLOR]"))
                {
                    sscanf (info, "%d,%d,%d", &r, &g, &b);
                    subStationColor = QColor (r, g, b);
                }
            }

            fclose (fp);
            return;
        }
    }
#endif
}




//  Save the users defaults.

void
TideEditor::envout ()
{
//RWL save the user params in the registry if windows or in hidden HOME dir file if Unix

#ifdef WIN32
	
    HKEY        hkey = NULL;
    DWORD       dwDisposition;
	
    DWORD       dwType, dwSize;
    char        strRGB[12];

    if (RegCreateKeyEx (HKEY_CURRENT_USER, TEXT ("Software\\NAVO\\TideEditor"),
        0, NULL, 0, KEY_READ | KEY_WRITE, NULL, &hkey, &dwDisposition) == 
        ERROR_SUCCESS)
    {
        dwType = REG_SZ;
        dwSize = strlen (pos_format);
        RegSetValueEx (hkey, TEXT ("PositionFormat"), NULL, dwType, 
            (unsigned char *) pos_format, dwSize);
	
        dwType = REG_SZ;
        sprintf ((char *) strRGB, "%d,%d,%d", refStationColor.red(), 
            refStationColor.green(), refStationColor.blue());

        dwSize = strlen (strRGB);
        RegSetValueEx (hkey, TEXT("RefStationColor"), NULL, dwType, 
	    (unsigned char *)strRGB, dwSize);
        RegCloseKey(hkey);

        dwType = REG_SZ;
        sprintf ((char *) strRGB, "%d,%d,%d", subStationColor.red(), 
            subStationColor.green(), subStationColor.blue());

        dwSize = strlen (strRGB);
        RegSetValueEx (hkey, TEXT("SubStationColor"), NULL, dwType, 
	    (unsigned char *)strRGB, dwSize);
        RegCloseKey(hkey);
    }

#else

    char string[128];

    if (getenv ("HOME") != NULL)
    {
        FILE *fp;
        sprintf (string, "%s/.tideEditor", getenv ("HOME"));
        if ((fp = fopen (string, "w")) != NULL)
        {
            fprintf (fp, "[POSITION FORMAT] = %s\n", pos_format);

            int r, g, b;
            refStationColor.rgb (&r, &g, &b);
            fprintf (fp, "[REFERENCE STATION COLOR] = %d,%d,%d\n", r, g, b);

            subStationColor.rgb (&r, &g, &b);
            fprintf (fp, "[SUBORDINATE STATION COLOR] = %d,%d,%d\n", r, g, b);

            fclose (fp);
            return;
        }
    }
#endif
}




//  Kicked from the map class.  Plot the stations.

void 
TideEditor::slotRedraw ()
{
    plotStations ();
}



//  Kicked from the map class.

void 
TideEditor::slotLeftMouse (QMouseEvent * e)
{
}



//  Kicked from the map class.  Edit the nearest tide station data.

void 
TideEditor::slotMidMouse (QMouseEvent * e)
{
    double lat, lon;


    map->untranslateCoordinates (&lat, &lon, e->x (), e->y ());


    /*  Adjust for 180 crossing if needed.  */

    if (lon > 180.0) lon -= 360.0;

    get_nearest_partial_tide_record (lat, lon, &rec.header);


    read_tide_record (rec.header.record_number, &rec);


    TideDialog *tideDialog = new TideDialog (this, "Tide Constituent Editor", 
        rec, this);
}



//  Kicked from the map class.  Add a tide station at the pointer.

void 
TideEditor::slotRightMouse (QMouseEvent * e)
{
    //  Ask if Reference or Subordinate.

    QMessageBox mb ("Add Tide Station",
        "Will the new station be a reference station or a subordinate station?\n\n",
        QMessageBox::NoIcon,
        QMessageBox::Yes | QMessageBox::Default,
        QMessageBox::No,
        QMessageBox::Cancel | QMessageBox::Escape,
        0);
    mb.setButtonText (QMessageBox::Yes, "Reference");
    mb.setButtonText (QMessageBox::No, "Subordinate");

    int result = mb.exec ();

    if (result == QMessageBox::Cancel) return;


    map->untranslateCoordinates (&rec.header.latitude, &rec.header.longitude,
        e->x (), e->y ());


    //  Default stuff.

    rec.header.record_number = -1;
    rec.header.reference_station = -1;
    rec.header.tzfile = 0;
    rec.header.name[0] = 0;
    rec.country = 0;
    rec.pedigree = 0;
    rec.source[0] = 0;
    rec.restriction = 0;
    rec.comments[0] = 0;
    rec.units = 0;
    rec.datum_offset = 0.0;
    rec.datum = 0;
    rec.zone_offset = 0;
    rec.expiration_date = 0;
    rec.months_on_station = 0;
    rec.last_date_on_station = 0;
    rec.confidence = 0;
    for (int i = 0 ; i < MAX_CONSTITUENTS ; i++)
    {
        rec.amplitude[i] = 0.0;
        rec.epoch[i] = 0.0;
    }
    rec.level_units = 0;
    rec.avg_level_units = 0;
    rec.direction_units = 0;
    rec.min_time_add = 0;
    rec.min_level_add = 0.0;
    rec.min_level_multiply = 0.0;
    rec.min_avg_level = 0.0;
    rec.min_direction = 361;
    rec.max_time_add = 0;
    rec.max_level_add = 0.0;
    rec.max_level_multiply = 0.0;
    rec.max_avg_level = 0.0;
    rec.max_direction = 361;
    rec.flood_begins = rec.ebb_begins = NULLSLACKOFFSET; /* DWF 2003-06-27 */


    //  Subordinate station.

    if (result == QMessageBox::No)
    {
        rec.header.record_type = 2;


        TIDE_RECORD temp_rec;
        int count = 0;
        QStringList strlist;


        //  Look for displayed Reference stations to use as a Reference
        //  station for this Subordinate station.

        for (int i = 0 ; i < number_of_records ; i++)
        {
            get_partial_tide_record (station_list[i].recnum, &temp_rec.header);
            if (temp_rec.header.reference_station == -1)
            {
                strlist += station_list[i].name;
                count++;
            }
        }


        bool ok = FALSE;
        QString res = QInputDialog::getItem (tr ("Reference Stations"),
            tr ("Please select the new reference station"), strlist, 0, 
            FALSE, &ok, (QWidget *) this);

        if (ok) 
        {
            char station[NAME_LENGTH];
            strcpy (station, res);

            int num = find_station (station);

            rec.header.reference_station = num;
        }
        else
        {
            return;
        }
    }
    else
    {
        rec.header.record_type = 1;
    }


    TideDialog *tideDialog = new TideDialog (this, "Add Tide Record", 
        rec, this);
}



//  Kicked from the map class.  Convert pixels to position and then update the
//  lat/lon info in the status bar.

void
TideEditor::slotMouseMove (QMouseEvent *e)
{
    char ltstring[25], lnstring[25], hem;
    double lat, lon, deg, min, sec;
    double slat, nlat, wlon, elon, full_slat, full_nlat, full_wlon, full_elon;
    int zoom_level;


    zoom_level = map->getExtents (&slat, &nlat, &wlon, &elon, &full_slat, 
        &full_nlat, &full_wlon, &full_elon);


    int rx = e->x ();
    int ry = e->y ();


    map->untranslateCoordinates (&lat, &lon, rx, ry);


    if (lat < slat) lat = slat;
    if (lat > nlat) lat = nlat;

    if (lon < wlon) lon = wlon;
    if (lon > elon) lon = elon;

    if (lon < -180.0) lon += 360.0;
    if (lon > 180.0) lon -= 360.0;

    strcpy (ltstring, fixpos (lat, &deg, &min, &sec, &hem, "lat", pos_format));
    strcpy (lnstring, fixpos (lon, &deg, &min, &sec, &hem, "lon", pos_format));

    QString string;
    QTextOStream (&string) << "Latitude :" << ltstring << " Longitude: " 
        <<lnstring;

    statusBar ()->message (string);
}




//  Plot the stations on the map.

void 
TideEditor::plotStations ()
{
    double slat, nlat, wlon, elon, full_slat, full_nlat, full_wlon, full_elon;
    int zoom_level;


    zoom_level = map->getExtents (&slat, &nlat, &wlon, &elon, &full_slat, 
        &full_nlat, &full_wlon, &full_elon);


    size = (int) ((float) zoom_level * 1.5) + 2;
    half_size = size / 2;


    //  Free memory.

    for (int i = 0 ; i < number_of_records ; i++)
    {
        if (station_list[i].name) free (station_list[i].name);
    }
    if (station_list) free (station_list);
    station_list = NULL;
    number_of_records = 0;


    //  Plot the stations.

    for (int i = 0 ; i < db.number_of_records ; i++)
    {
        get_partial_tide_record (i, &rec.header);


        int x, y;
        if (map->translateCoordinates (rec.header.latitude, 
            rec.header.longitude, &x, &y))
        {
            if (rec.header.reference_station >= 0)
            {
                map->drawCircle (x - half_size, y - half_size, size, size, 360,
                    subStationColor);
            }
            else
            {
                map->drawCircle (x - half_size, y - half_size, size, size, 360,
                    refStationColor);
            }


            //  Allocate memory for the visible stations.

            station_list = (STATION_LIST *) realloc (station_list, 
                (number_of_records + 1) * sizeof (STATION_LIST));
            station_list[number_of_records].name = 
                (char *) malloc (strlen (rec.header.name) + 1);

            if (station_list == NULL)
            {
                perror ("Allocating station list memory in tideEditor.cpp");
                exit (-1);
            }

            station_list[number_of_records].recnum = i;
            strcpy (station_list[number_of_records].name, rec.header.name);
            station_list[number_of_records].latitude = rec.header.latitude;
            station_list[number_of_records].longitude = rec.header.longitude;

            number_of_records++;
        }
    }


    //  Sort the names then update the station list.

    ::qsort (station_list, number_of_records, sizeof (STATION_LIST), 
    &compare_names);

    tideList->updateTideList ();
}



//  Flags a station by coloring it green.

void 
TideEditor::highlightStation (double lat, double lon, 
                              unsigned char record_type)
{
    int x, y;

    switch (record_type)
    {
        case 99:
            if (map->translateCoordinates (lat, lon, &x, &y))
                map->drawCircle (x - half_size, y - half_size, size, size, 360,
                Qt::green);
            break;


        case 1:
            if (map->translateCoordinates (lat, lon, &x, &y))
                map->drawCircle (x - half_size, y - half_size, size, size, 360,
                refStationColor);
            break;


        case 2:
            if (map->translateCoordinates (lat, lon, &x, &y))
                map->drawCircle (x - half_size, y - half_size, size, size, 360,
                subStationColor);
            break;
    }
}


  
void
TideEditor::redrawMap ()
{
    map->redrawMap ();
}



void
TideEditor::setRedrawZeroLevel ()
{
    db = get_tide_db_header ();

    map->setRedrawZeroLevel ();
}



//  Using the keys to move around.

void
TideEditor::keyPressEvent (QKeyEvent * e)
{
    switch (e->key ())
    {
        case Key_Left:
            map->moveMap (MAP_LEFT);
            break;
        case Key_Up:
        case Key_PageUp:
            map->moveMap (MAP_UP);
            break;
        case Key_Right:
            map->moveMap (MAP_RIGHT);
            break;
        case Key_Down:
        case Key_PageDown:
            map->moveMap (MAP_DOWN);
            break;
        default:
            return;
    }
    e->accept ();
}



//  A bunch of slots.

void 
TideEditor::slotQuit ()
{
    close_tide_db ();

    for (int i = 0 ; i < number_of_records ; i++)
    {
        if (station_list[i].name) free (station_list[i].name);
    }
    if (station_list) free (station_list);


    char string[512], string2[512];
    sprintf (string, "%s.running", hfile);
    sprintf (string2, "%s.bak", hfile);

    rename (string, string2);

    envout ();

    exit (0);
}



void 
TideEditor::slotDisaster ()
{
    close_tide_db ();

    for (int i = 0 ; i < number_of_records ; i++)
    {
        if (station_list[i].name) free (station_list[i].name);
    }
    if (station_list) free (station_list);


    char string[512];
    sprintf (string, "%s.running", hfile);

    rename (string, hfile);

    exit (0);
}



void
TideEditor::slotRefColor ()
{
    QColor temp;


    temp = QColorDialog::getColor (refStationColor, this);
    if (temp.isValid ()) 
    {
        refStationColor = temp;

        plotStations ();
    }
}



void
TideEditor::slotSubColor ()
{
    QColor temp;


    temp = QColorDialog::getColor (subStationColor, this);

    if (temp.isValid ())
    {
        subStationColor = temp;

        plotStations ();
    }
}



void
TideEditor::slotPrefs ()
{
    prefsD = new QDialog (this, 0, FALSE, WDestructiveClose);
    prefsD->setCaption ("Preferences");

    QVBox *vbox = new QVBox (prefsD);
    vbox->resize (400, 250);
    vbox->setMargin (5);
    vbox->setSpacing (5);

    bGrp = new QButtonGroup (1, QGroupBox::Horizontal, "Position Format", 
        vbox);
    bGrp->resize (250, 250);
    bGrp->setExclusive (TRUE);
    QWhatsThis::add (bGrp, bGrpText);
    connect (bGrp, SIGNAL (clicked (int)), this, SLOT (slotPrefsClicked ()));

    (void) new QRadioButton ("Hemisphere Degrees Minutes Seconds.decimal", 
        bGrp);
    (void) new QRadioButton ("Hemisphere Degrees Minutes.decimal", bGrp);
    (void) new QRadioButton ("Hemisphere Degrees.decimal", bGrp);
    (void) new QRadioButton ("+/-Degrees Minutes Seconds.decimal", bGrp);
    (void) new QRadioButton ("+/-Degrees Minutes.decimal", bGrp);
    (void) new QRadioButton ("+/-Degrees.decimal", bGrp);

    if (!strcmp (pos_format, "hdms")) bGrp->setButton (0);
    if (!strcmp (pos_format, "hdm")) bGrp->setButton (1);
    if (!strcmp (pos_format, "hd")) bGrp->setButton (2);
    if (!strcmp (pos_format, "dms")) bGrp->setButton (3);
    if (!strcmp (pos_format, "dm")) bGrp->setButton (4);
    if (!strcmp (pos_format, "d")) bGrp->setButton (5);

    QPushButton *refColorButton = new QPushButton ("Reference Station Color", 
        vbox);
    QWhatsThis::add (refColorButton, refColorButtonText);
    connect (refColorButton, SIGNAL (clicked ()), this, 
        SLOT (slotRefColor ()));

    QPushButton *subColorButton = new QPushButton ("Subordinate Station Color",
        vbox);
    QWhatsThis::add (subColorButton, subColorButtonText);
    connect (subColorButton, SIGNAL (clicked ()), this, 
        SLOT (slotSubColor ()));

    QPushButton *closeButton = new QPushButton ("Close", vbox);
    QWhatsThis::add (closeButton, closePrefsText);
    connect (closeButton, SIGNAL (clicked ()), this, SLOT (slotClosePrefs ()));

    prefsD->show ();
}



//  A station search dialog.

void 
TideEditor::slotSearch ()
{
    searchD = new QDialog (this, 0, FALSE, WDestructiveClose);
    searchD->setCaption ("Station Search");

    QVBox *vbox = new QVBox (searchD);
    vbox->resize (800, 250);
    vbox->setMargin (5);
    vbox->setSpacing (5);

    QString names = QString ("");
    (void) new QLabel ("Station Name:", vbox);
    searchName = new QLineEdit (names, vbox);
    QWhatsThis::add (searchName, searchNameText);

    connect (searchName, SIGNAL (textChanged (const QString &)), this, 
        SLOT (slotTextChanged ()));



    searchList = new QListBox (vbox);
    QWhatsThis::add (searchList, searchListText);

    connect (searchList, SIGNAL (selected (int)), this, 
            SLOT (slotSelected ()));
    connect (searchList, SIGNAL (highlighted (int)), this, 
            SLOT (slotHighlighted ()));

    QPushButton *closeButton = new QPushButton ("Close", vbox);
    QWhatsThis::add (closeButton, closeSearchText);
    connect (closeButton, SIGNAL (clicked ()), this, 
        SLOT (slotCloseSearch ()));

    searchD->show ();
}



void 
TideEditor::slotTextChanged ()
{
    char name[NAME_LENGTH];


    searchList->clear ();

    strcpy (name, searchName->text ());

    if (strlen (name) >= 3)
    {
        int num;
        while ((num = search_station (name)) != -1)
        {
            TIDE_RECORD temp_rec;
            double deg, min, sec;
            char hem;

            get_partial_tide_record (num, &temp_rec.header);

            QString lats = 
                QString (fixpos (temp_rec.header.latitude, &deg, &min, &sec, 
                &hem, "lat", pos_format));
            QString lons = 
                QString (fixpos (temp_rec.header.longitude, &deg, &min, &sec, 
                &hem, "lon", pos_format));

            QString line = QString ("%1, %2   : %3")
                .arg (lats, 13)
                .arg (lons, 14)
                .arg (temp_rec.header.name);

            searchList->insertItem (line, -1);
        }
    }
}




void 
TideEditor::slotSelected ()
{
    char name[NAME_LENGTH], item[256];
    strcpy (item, searchList->currentText ());

    strcpy (name, (strchr (item, ':') + 2));

    int record_number = find_station (name);

    TIDE_RECORD lrec;
    read_tide_record (record_number, &lrec);

    TideDialog *tideDialog = new TideDialog (this, "Tide Constituent Editor", 
        lrec, this);
}



void 
TideEditor::slotHighlighted ()
{
    TIDE_RECORD temp_rec;

    get_partial_tide_record (lastrec, &temp_rec.header);

    highlightStation (temp_rec.header.latitude, temp_rec.header.longitude, 
        temp_rec.header.record_type);

    char name[NAME_LENGTH], item[256];
    strcpy (item, searchList->currentText ());

    strcpy (name, (strchr (item, ':') + 2));

    int record_number = find_station (name);
    lastrec = record_number;

    get_partial_tide_record (lastrec, &temp_rec.header);

    highlightStation (temp_rec.header.latitude, temp_rec.header.longitude, 99);
}



void
TideEditor::slotPrefsClicked ()
{
    switch (bGrp->id (bGrp->selected ()))
    {
        case 0:
        default:
            strcpy (pos_format, "hdms");
            break;

        case 1:
            strcpy (pos_format, "hdm");
            break;

        case 2:
            strcpy (pos_format, "hd");
            break;

        case 3:
            strcpy (pos_format, "dms");
            break;

        case 4:
            strcpy (pos_format, "dm");
            break;

        case 5:
            strcpy (pos_format, "d");
            break;
    }
}



void
TideEditor::slotClosePrefs ()
{
    delete (prefsD);
}



void
TideEditor::slotCloseSearch ()
{
    delete (searchD);
}



void
TideEditor::slotZoomOut ()
{
    map->setZoomLevel (-1);
}


void
TideEditor::slotReset ()
{
    map->setZoomLevel (0);
}



void
TideEditor::slotDown ()
{
    map->moveMap (MAP_DOWN);
}



void
TideEditor::slotUp ()
{
    map->moveMap (MAP_UP);
}



void
TideEditor::slotRight ()
{
    map->moveMap (MAP_RIGHT);
}



void
TideEditor::slotLeft ()
{
    map->moveMap (MAP_LEFT);
}



void
TideEditor::about ()
{
    QMessageBox::about (this, VERSION,
        "Simple program to edit the OAML standard Tide Constituent Database."
        "\n\nAuthor : Jan C. Depner (depnerj@navo.navy.mil)\n"
        "Date : 16 August 2002");
}


void
TideEditor::aboutQt ()
{
    QMessageBox::aboutQt (this, VERSION);
}
