/* acpi application yacpi 1.2.1
 * Copyright (C) 2005 Nico Golde <nico@ngolde.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define _GNU_SOURCE

#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <getopt.h>
#include <sys/types.h>
#include "libacpi.h"
#include "get_cpu.h"

#define ACPI_VER "2.0"

global_t *globals;

/* options */
static struct option long_options[] = {
    {"help", 0, NULL, 'h'},
    {"version", 0, NULL, 'v'},
    {"samples", 1, NULL, 's'},
    {NULL, 0, NULL, 0}
};

/* next 2 functions taken from snownews and customized */
void helpbox (int x1, int y1, int x2, int y2)
{
    int i, j;

    attron (WA_REVERSE);
    for (i = y1; i <= y2; i++) {
        for (j = x1; j <= x2; j++) {
            mvaddch (i, j, ' ');
        }
    }
    attroff (WA_REVERSE);
}

/* get value for repeat the screen resfresh */
int monitor_help (void)
{
    int rval = 0;

    attrset (WA_REVERSE);
    curs_set (2);
    echo ();
    mvprintw (LINES - 2, 0, "Seconds before refresh: ");
    move (LINES - 2, 24);
    scanw ("%d", &rval);
    curs_set (0);
    noecho ();

    return rval;
}

int samples_help (void)
{
    int rval = 0;

    attrset (WA_REVERSE);
    curs_set (2);
    echo ();
    mvprintw (LINES - 2, 0, "Number of samples: ");
    move (LINES - 2, 19);
    scanw ("%d", &rval);
    curs_set (0);
    noecho ();

    return rval;
}

void help (void)
{
    int centerx, centery;       /* Screen center x/y coordinate. */
    int userinput;
    int offset = 13;
    int offsetstr = 11;
    char *helptext = "Press any[tm] key to continue.";

    centerx = COLS / 2;
    centery = LINES / 2;

    helpbox (centerx - 15, centery, centerx + 14, centery + 5);

    attron (WA_REVERSE);
    /* Keys */
    mvprintw (centery + 1, centerx - offset, "q");
    mvprintw (centery + 2, centerx - offset, "r");
    mvprintw (centery + 3, centerx - offset, "h");
    mvprintw (centery + 4, centerx - offset, "s");

    /* Descriptions */
    mvaddstr (centery + 1, centerx - offsetstr, "End Yacpi ...");
    mvaddstr (centery + 2, centerx - offsetstr, "Reload ACPI data ...");
    mvaddstr (centery + 3, centerx - offsetstr, "Show this help ...");
    mvaddstr (centery + 4, centerx - offsetstr, "Specify samples ...");

    mvprintw (LINES - 1, COLS - strlen (helptext), "%s", helptext);
    attroff (WA_REVERSE);

    userinput = getch ();

    /* Return input back into input queue so it gets automatically
       executed. */
    if ((userinput != '\n') && (userinput != 'h') && (userinput != 'q'))
        ungetch (userinput);
}

/* change colors in the ncurses interface */
void color_change (int pct, int tmp)
{
    if (tmp == 0) {
        if (pct >= 70)
            attron (COLOR_PAIR (1));
        else if (pct >= 50 && pct <= 74)
            attron (COLOR_PAIR (2));
        else if (pct >= 0 && pct <= 49)
            attron (COLOR_PAIR (3));
    }
    else {
        if (pct <= 59)
            attron (COLOR_PAIR (1));
        else if (pct <= 69 && pct >= 60)
            attron (COLOR_PAIR (2));
        else if (pct >= 70)
            attron (COLOR_PAIR (3));
    }
}

/* get acpi version number */
int get_acpi_version (void)
{
    FILE *acpi;
    int acpi_ver = 0;
    char buffer[512];

    if (!(acpi = fopen ("/proc/acpi/info", "r"))) {
        endwin ();
        printf ("This system does not support ACPI\n");
        exit (1);
    }
    fread (buffer, 512, 1, acpi);
    acpi_ver = strtol (buffer + 25, NULL, 10);
    fclose (acpi);
    if (acpi_ver < 20020214) {
        endwin ();
        printf ("This version requires ACPI subsystem version 20020214\n");
        return 1;
    }

    return acpi_ver;
}

/* make the window border etc. */
void print_window (void)
{
    int i;

    attron (WA_REVERSE);
    /* print white lines and version numbers */
    for (i = 0; i < COLS; i++) {
        mvaddch (0, i, ' ');
        mvaddch (LINES - 1, i, ' ');
    }
    mvaddstr (0, 0, "* YACPI 1.2.1 by Nico Golde");
    mvaddstr (LINES - 1, 0, "ACPI version detected: ");
    mvprintw (LINES - 1, 23, "%d", get_acpi_version ());

    attroff (WA_REVERSE);
    move (2, 0);
}

/* main acpi handling */
void acpi_handling (int samplef, int samplev)
{
    int i, j;
    int sleep_time = 0;
    int line = 2;
    int sampleflag = samplef, samples = samplev;
    attrset(WA_NORMAL);
    adapter_t *ap;
    battery_t *binfo;
    temperature_t *temperature;

    binfo = NULL;
    globals = (global_t *) malloc (sizeof (global_t)); /* checked */
    power_init (globals);
    get_temp (globals);
    ap = &globals->adapter;

    /* make the window */
    print_window ();

    if (sampleflag == 1)
        sleep_time = 1000000 / samples;

    /* we want to acquire samples over some period of time, so . . . */
    for (i = 0; i < samples + globals->battery_count; i++) {
        for (j = 0; j < globals->battery_count; j++) /* comment this line if you only use one of your possible batteries to increase speed of yacpi */ 
            acquire_batt_info (globals, j);
        acquire_global_info (globals);
        usleep (sleep_time);
    }                           /* this line too */
    temperature = &globals->temperature;


    /* if the terminal width isnt big enough */
    if (COLS <= 87) {
        for (i = 0; i < globals->battery_count; i++) {
            binfo = &batteries[i];
            if (binfo->present) {
                mvprintw (line, 0, "Battery %s on %2d%%", binfo->name,
                          binfo->percentage);
                line = line + 2;
            }
        }
        mvprintw(line,0, "AC:");
        if (ap->power == AC)
        {
            attron(COLOR_PAIR(1));
            mvprintw (line, 4, "on");
            attroff(COLOR_PAIR(1));
        }
        else
        {
            attron(COLOR_PAIR(3));
            mvprintw (line, 4, "off");
            attroff(COLOR_PAIR(3));
        }
        line=line+2;
        if (binfo->charge_state == CHARGE)
            mvprintw (line, 0, "Battery Status: charging   ");
        else if (binfo->charge_state == DISCHARGE)
            mvprintw (line, 0, "Battery Status: discharging");
        else if (binfo->charge_state == CHARGED)
            mvprintw (line, 0, "Battery Status: charged    ");
        else if (binfo->charge_state == NOINFO)
            mvprintw (line, 0, "Battery Status: not supported");
        line = line + 2;
        if (globals->rtime >= 0 && ap->power != AC) {
            mvprintw (line, 0, "Remaining time: %02d:%02d h        ",
                      globals->rtime / 60, globals->rtime % 60);
            line = line + 2;
        }

        if (ap->power == AC && binfo->charge_time >= 0) {
            mvprintw (line, 0, "Remaining charge time: %02d:%02d h",
                      binfo->charge_time / 60, binfo->charge_time % 60);
            line = line + 2;
        }
        mvprintw(line,0,"cpu frequency: %d/%d MHz",get_cpu_cur()/1000, get_cpu_max()/1000); /* get cpu frequency */
        line=line+2;
        {
            char *cpu_gov=get_cpu_gov();
            mvprintw(line,0,"cpu governor: %s",cpu_gov); /* get cpu governor */
            curs_set(0);
            free (cpu_gov);
        }
        line=line+2;
    }
    else {
        for (i = 0; i < globals->battery_count; i++) {
            binfo = &batteries[i];
            if (binfo->present) {
                mvprintw (line, 0, "%s Capacity [", binfo->name);
                color_change (binfo->percentage, 0);
                for (i = 0; i <= (binfo->percentage / 2); i++)
                    printw ("|");
                attron (COLOR_PAIR (4));
                for (i = 0; i < (50 - binfo->percentage / 2); i++)
                    printw (" ");
                printw ("]");
                printw (" %2d%% - ", binfo->percentage);

            }
        }
        if (ap->power == AC){
            printw ("AC status: ");
            attron(COLOR_PAIR(1));
            printw ("on\n");
            attroff(COLOR_PAIR(1));
        }
        else{
            printw ("AC status: ");
            attron(COLOR_PAIR(3));
            printw ("off\n");
            attroff(COLOR_PAIR(3));
        }
        line = line + 2;
        if (binfo->charge_state == CHARGE)
            mvprintw (line, 0, "Battery Status: charging   ");
        else if (binfo->charge_state == DISCHARGE)
            mvprintw (line, 0, "Battery Status: discharging");
        else if (binfo->charge_state == CHARGED)
            mvprintw (line, 0, "Battery Status: charged    ");
        else if (binfo->charge_state == NOINFO)
            mvprintw (line, 0, "Battery Status: not supported");

        line = line + 2;
        if (globals->rtime >= 0 && ap->power != AC) {
            mvprintw (line, 0, "Remaining time: %02d:%02d h       ",
                      globals->rtime / 60, globals->rtime % 60);
            line = line + 2;
        }
        if (ap->power == AC && binfo->charge_time >= 0) {
            mvprintw (line, 0, "Remaining charge time: %02d:%02d h",
                      binfo->charge_time / 60, binfo->charge_time % 60);
            line = line + 2;
        }
        mvprintw(line,0,"cpu frequency: %d/%d MHz",get_cpu_cur()/1000, get_cpu_max()/1000); /* get cpu frequency */
        line=line+2;
        {
            char *cpu_gov=get_cpu_gov();
            mvprintw(line,0,"cpu governor: %s",get_cpu_gov()); /* get cpu governor */
            free (cpu_gov);
            curs_set(0);
        }
        line=line+2;
    }
    mvaddstr (line, 0, "Temperature:");
    color_change (temperature->temp, 1);
    move (line, 13);
    printw ("%d", temperature->temp);
    attron (COLOR_PAIR (4));
    move (line, 16);
    printw ("degrees C");
    refresh ();

}

void show_error (void)
{
    char *help_string = "Unknown key, use 'h' for help.";

    attron (WA_REVERSE);
    mvprintw (LINES - 1, COLS - strlen (help_string), "%s", help_string);
    attroff (WA_REVERSE);
}

void usage (void)
{
    puts ("YACPI 1.2.1 by Nico Golde <nico@ngolde.de>");
    puts ("usage: yacpi [options]\n"
          "short options:\n"
          "-v    print version number + release date.\n"
          "-h    print this help.\n\n"
          "long options:\n" "--samples=value between 1 and 1000\n");
    exit (0);
}

int main (int argc, char *argv[])
{
    char ch;
    int sample_number = 0, sample_flag = 0;
    int option, akt_optind, option_index = 0;

    /* parse command line arguments */
    while (1) {
        akt_optind = optind;
        option = getopt_long (argc, argv, "s:hv", long_options, &option_index);

        if (option == EOF)
            break;
        switch (option) {
        case 'h':
            usage ();
            break;
        case 'v':
            puts ("YACPI 1.2.1 by Nico Golde - Released: 30.03.2005\n");
            exit (0);
            break;
        case 's':
            sample_number = atoi (optarg);
            if (sample_number <= 0 || sample_number > 1000) {
                printf ("\nUse a correct sample number (1-1000) ...\n");
                exit (1);
            }
            sample_flag = 1;
            break;
        default:
            usage ();
            break;
        }
    }

    initscr ();
    cbreak ();
    noecho ();
    curs_set (0);
    start_color ();             /* initialize colors */ 
    use_default_colors();
    init_pair (1, COLOR_GREEN, -1);
    init_pair (2, COLOR_YELLOW, -1);
    init_pair (3, COLOR_RED, -1);
    init_pair (4, COLOR_WHITE, -1);

    acpi_handling (sample_flag, sample_number);
    while (1) {
        // move (2, 0);
        ch = getch ();
        switch (ch) {
        case 'q':
            endwin ();
            return 0;
            break;
        case 'r':
            clear ();
            acpi_handling (sample_flag, sample_number);
            break;
        case 'h':
            help ();
            clear ();
            acpi_handling (sample_flag, sample_number);
            break;
        case 's':
            sample_number = samples_help ();
            sample_flag = 1;
            clear ();
            acpi_handling (sample_flag, sample_number);
            break;
/*            case 'm' :
                monitor_help();
                clear();
                acpi_handling(sample_flag,sample_number);
                break;*/
        default:
            show_error ();
            break;
        }
    }

    getch ();
    endwin ();
    return 0;
}
