#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include <glib.h>

#include <sigc++/object_slot.h>
#include <sigcx/util.h>
#include "yehia/io.h"

namespace Yehia
{

using namespace SigCX;
using namespace SigC;

DataSource::DataSource()
{
}

DataSource::~DataSource()
{
}

DataSink::DataSink()
{
}

DataSink::~DataSink()
{
}

CircularBuffer::CircularBuffer(size_type bufsz)
{
  buffer_ = new unsigned char[bufsz];
  bufget_ = bufput_ = buffer_;
  bufend_ = buffer_ + bufsz;
  empty_ = true;
}

CircularBuffer::~CircularBuffer()
{
  delete[] buffer_;
}

DataStream::DataStream(DataSource& src, DataSink& sink, int bufsz)
    : src_(src), sink_(sink), buffer_(bufsz), input_exhausted_(false)
{
  src_.reference();
  sink_.reference();

  src_.attach(slot(*this, &DataStream::input_ready),
              slot(*this, &DataStream::input_exhausted));
  sink_.attach(slot(*this, &DataStream::output_ready));
  
  src_.set_notification(true); // we need data
}

StackedDataSource::StackedDataSource(size_type bufsz)
    : buffer(bufsz)
{
  notify_ = true;
  exhausted_ = false;
  src_ = 0;
}

void StackedDataSource::attach(DataSource& src)
{
  detach();
  src_ = &src;
  src_->reference();
  src_->attach(slot(*this, &StackedDataSource::source_ready),
               slot(*this, &StackedDataSource::source_exhausted));
}

void StackedDataSource::detach()
{
  if (src_ == 0)
    return;

  src_->detach();
  src_->unreference();
  src_ = 0;
}

void StackedDataSource::set_notification(bool do_notify)
{
  notify_ = do_notify;
  if (src_)
  {
    if (!notify_ && buffer.full())
      src_->set_notification(false);
    else if (notify_)
      src_->set_notification(true);
  }
}

void StackedDataSource::source_ready()
{
  g_assert(src_ != 0);
  
  buffer.put_advance(src_->read(buffer.put_ptr(), buffer.put_size()));
  
  if (notify_ && !buffer.empty())
    ready();
  if (!notify_ && src_ && buffer.full())
    src_->set_notification(false);
}

void StackedDataSource::source_exhausted()
{
  g_assert(src_ != 0);
  
  exhausted_ = true;
  if (buffer.empty())
    exhausted();
}

DataStream::~DataStream()
{
  // Note: we disconnect the handlers automatically
  
  src_.unreference();
  sink_.unreference();
}

void DataStream::input_ready()
{
  if (buffer_.full())
    src_.set_notification(false);
  else
  {
    size_type nread = src_.read(buffer_.put_ptr(), buffer_.put_size());
    if (nread > 0)
    {
      buffer_.put_advance(nread);
      sink_.set_notification(true);
    }
  }
}

void DataStream::input_exhausted()
{
  input_exhausted_ = true;
  if (buffer_.empty())
    done.emit();
}

void DataStream::output_ready()
{
  if (buffer_.empty())
  {
    sink_.set_notification(false);
    if (input_exhausted_)
      done.emit();
  }
  else
  {
    size_type nwritten = sink_.write(buffer_.get_ptr(), buffer_.get_size());
    if (nwritten > 0)
    {
      buffer_.get_advance(nwritten);
      src_.set_notification(true);
    }
  }
}

DispatcherDataSource::DispatcherDataSource(SigCX::Dispatcher& disp, int fd)
    : disp_(disp), fd_(fd)
{
  int flags = fcntl(fd, F_GETFD, 0);
  if (flags < 0)
    throw std::runtime_error(errno_string());
  flags |= O_NONBLOCK;
  if (fcntl(fd, F_SETFD, flags) < 0)
    throw std::runtime_error(errno_string());

  disp_.reference();
  hid_ = disp_.add_input_handler(
          slot(*this, &DispatcherDataSource::input_handler), fd);
}

DispatcherDataSource::~DispatcherDataSource()
{
  set_notification(false); // remove handler
  disp_.unreference();
}

DataSource::size_type 
DispatcherDataSource::read(void *data, size_type len) throw (IOException)
{
  ssize_t readsz;
  bool retry;

  if (len == 0) 
    return 0;
  
  do
  {
    retry = false;
    if ((readsz = ::read(fd_, data, len)) == -1)
    {
      if (errno == EAGAIN)
        return 0;
      if (errno == EINTR)
        retry = true;
      else
        throw IOException(errno_string());
    }
    else if (readsz == 0)
      exhausted();
  } while (retry);

  return readsz;
}

void DispatcherDataSource::set_notification(bool do_notify)
{
  if (do_notify && hid_ == Dispatcher::InvalidHandlerID)
    hid_ = disp_.add_input_handler(
          slot(*this, &DispatcherDataSource::input_handler), fd_);
  else if (!do_notify && hid_ != Dispatcher::InvalidHandlerID)
  {
    disp_.remove(hid_);
    hid_ = Dispatcher::InvalidHandlerID;
  }
}

void DispatcherDataSource::input_handler()
{
  ready();
}

DispatcherDataSink::DispatcherDataSink(SigCX::Dispatcher& disp, int fd)
    : disp_(disp), fd_(fd)
{
  int flags = fcntl(fd, F_GETFD, 0);
  if (flags < 0)
    throw std::runtime_error(errno_string());
  flags |= O_NONBLOCK;
  if (fcntl(fd, F_SETFD, flags) < 0)
    throw std::runtime_error(errno_string());

  disp_.reference();
  hid_ = disp_.add_output_handler(
          slot(*this, &DispatcherDataSink::output_handler), fd);
}

DispatcherDataSink::~DispatcherDataSink()
{
  set_notification(false); // remove handler
  disp_.unreference();
}

DataSink::size_type 
DispatcherDataSink::write(const void *data, size_type len) throw (IOException)
{
  return ::write(fd_, data, len);
}

void DispatcherDataSink::set_notification(bool do_notify)
{
  if (do_notify && hid_ == Dispatcher::InvalidHandlerID)
    hid_ = disp_.add_output_handler(
          slot(*this, &DispatcherDataSink::output_handler), fd_);
  else if (!do_notify && hid_ != Dispatcher::InvalidHandlerID)
  {
    disp_.remove(hid_);
    hid_ = Dispatcher::InvalidHandlerID;
  }
}

void DispatcherDataSink::output_handler()
{
  ready();
}

}
