/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: zorp.c,v 1.15 2004/07/05 07:59:55 bazsi Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/zorp.h>
#include <zorp/log.h>


#include <zorp/zpython.h>
#include <zorp/policy.h>

/* libzorpll */
#include <zorp/process.h>


GMainLoop *main_loop;
gint exit_code = 0;

/* FIXME: the code in this module was straightly copied from main.c and as
 * such does not really fit into a library. Some generalization would be
 * appropriate. */

gboolean usr1_received = 0;
gboolean usr2_received = 0;
static gboolean term_received = 0;


void
z_main_loop_initiate_termination(gboolean called_from_sighandler)
{
  term_received = TRUE;
  if (!called_from_sighandler)
    g_main_context_wakeup(NULL);
}

gboolean
z_load_policy(const gchar *policy_file, const gchar *instance_name, const gchar **instance_policy_list)
{
  ZPolicy *policy;

  policy = z_policy_new(policy_file);
  if (!z_policy_boot(policy) || !z_policy_load(policy))
    {
      /*LOG
	This message indicates that Zorp was unable to load the policy.
	It is likely that the policy has any kind of syntactical problem.
	Check the traceback in the log to find out where the problem occurs.
       */
      z_log(NULL, CORE_ERROR, 0, "Error booting & parsing policy;");
      return FALSE;
    }
  current_policy = policy;
  z_log_set_fake_session_id(instance_name);
  if (!z_policy_init(policy, instance_policy_list))
    {
      /*LOG
	This message indicates that Zorp was unable to initialize the policy.
       */
      z_log(NULL, CORE_ERROR, 0, "Error initializing policy;");
      return FALSE;
    }
  return TRUE;
}

void
z_main_loop(const gchar *policy_file, const gchar *instance_name, const gchar **instance_policy_list)
{
  gint new_verbosity;
  
  main_loop = g_main_loop_new(NULL, TRUE);
  if (!z_load_policy(policy_file, instance_name, instance_policy_list))
    {
      /*LOG
	This message indicates that the loading of the initial policy failed, because of some policy problem.
	Check the log to find out where the problem occurs.
       */
      z_log(NULL, CORE_ERROR, 0, "Error loading initial policy, exiting;");
      /* hack to let our messages get out */
      sleep(1);
      exit_code = 2;
      return;
    }
  /* signal successful start */
  z_process_ok();
  if (term_received)
    z_main_loop_quit(0);
    
  z_blob_system_default_init();  /* FIXME: re-initialisation should happen on policy-reload also! */

  while (g_main_loop_is_running(main_loop))
    {
      g_main_context_iteration(NULL, TRUE);
      if (usr1_received)
        {
          usr1_received = 0;
          z_log_change_verbose_level(1, 1, &new_verbosity);
          z_mem_trace_stats();
        }
      if (usr2_received)
        {
          usr2_received = 0;
          z_log_change_verbose_level(-1, 1, &new_verbosity);
        }
      if (term_received)
        {
          z_main_loop_quit(0);
          break;
        }
    }
  z_mem_trace_dump();
  g_main_loop_unref(main_loop);
  main_loop = NULL;
}

/**
 * z_main_loop_quit:
 * @rc exit code
 *
 * Set the exit code to the specified value and tell glib to exit the main loop
 */
void
z_main_loop_quit(int rc)
{
  z_enter();
  exit_code = rc;
  g_main_quit(main_loop);
  z_leave();
}

/**
 * log_set_fake_session_id:
 * @instance_name instance name to use for generating session id
 *
 * Generate a fake session id in the form 'instance_name@firewall_name/nosession'
 * if 'firewall_name' is set, or 'noname/nosession' if it is missing, for
 * providing a sensible session id for logging in cases when there are no
 * sessions at all (eg. at initialisation).
 * 'firewall_name' is a Python variable defined in 'Zorp.py', 'instance_name'
 * comes from 'instances.conf', or from the '-a|--as or --also-as' commandline
 * argument of zorp.
 */
void
z_log_set_fake_session_id(const gchar *instance_name)
{
  PyObject *z, *name;
  
  z_policy_acquire_main(current_policy);
  z = PyImport_AddModule("Zorp.Zorp");
  name = PyObject_GetAttrString(z, "firewall_name");
  if (name && PyString_Check(name))
    {
      g_snprintf(fake_session_id, sizeof(fake_session_id), "%s@%s/nosession", instance_name, PyString_AsString(name));
    }
  else
    {
      strcpy(fake_session_id, "noname/nosession");
    }
  Py_XDECREF(name);
  z_policy_release_main(current_policy);
}
