#include <zorp/zorp.h>
#include <zorp/log.h>

#include <zorp/proxy/code.h>


static gint
z_code_dummy_transform(ZCode *self, const guchar *from, int fromlen)
{
  z_enter();
  z_code_grow(self, self->buf_used + fromlen);
  g_memmove(self->buf + self->buf_used, from, fromlen);
  self->buf_used += fromlen;
  z_leave();
  return fromlen;
}

static void
z_code_dummy_finish(ZCode *self G_GNUC_UNUSED)
{
  z_enter();
  z_leave();
}

static void
z_code_dummy_free(ZCode *self G_GNUC_UNUSED)
{
  z_enter();
  z_leave();
}

/**
 * z_code_grow:
 * @buf: pointer to current buffer
 * @len: pointer to current buffer length
 *
 * Resizes the buffer by a factor of 2.
 * FIXME: no error handling yet
 *
 * Returns:
 * TRUE on success
 */
gboolean
z_code_grow(ZCode *self, gint reqlen)
{
  gint    newlen;

  z_enter();
  newlen = self->buf_len;
  if (newlen <= 0)
    {
      newlen = 1;
    }
  
  while (newlen < reqlen)
    {
      newlen *= 2;
    }
  
  self->buf = g_realloc(self->buf, newlen);
  self->buf_len = newlen;
  z_leave();
  return TRUE;
}


/**
 * z_code_transform:
 * @self: this
 * @from: source buffer
 * @fromlen: number of bytes to transform
 *
 * Transforms data to the internal buffer. Can be used any number of
 * times the internal buffer grows automatically.
 *
 * Returns:
 * The number of bytes written
 */
gint
z_code_transform(ZCode *self, const guchar *from, int fromlen)
{
  gint    res;
  
  z_enter();
  res = self->transform(self, from, fromlen);
  z_leave();
  return res;
}

/**
 * z_code_finish:
 * @self: this
 *
 * Finalizes the output in the internal buffer. Should be called once
 * at the end of the encoding process.
 */
void
z_code_finish(ZCode *self)
{
  z_enter();
  self->finish(self);
  z_leave();
}

/**
 * z_code_unget_result:
 * @self: this
 * @from: source buffer
 * @fromlen: number of bytes to push back to the buffer
 *
 * Pushes back data to the buffer. (The data will be available for reading)
 *
 * Returns:
 * The number of bytes pushed back
 */
gint
z_code_unget_result(ZCode *self, const guchar *from, int fromlen)
{
  z_enter();
  if (fromlen > 0)
    {
      z_code_grow(self, self->buf_used + fromlen);
      g_memmove(self->buf + fromlen, self->buf, self->buf_used);
      g_memmove(self->buf, from, fromlen);
      self->buf_used += fromlen;
    }
  z_leave();
  return fromlen;
}

/**
 * z_code_get_result:
 * @self: this
 * @to: destination buffer
 * @tolen: length of the destination buffer
 *
 * Returns and removes a chunk of transformed bytes. The results are
 * returned in @to.
 *
 * Returns:
 * The number of bytes returned.
 */
gint
z_code_get_result(ZCode *self, guchar *to, int tolen)
{
  gint    res;
  
  z_enter();
  res = (tolen < self->buf_used) ? tolen : self->buf_used;
  if (res > 0)
    {
      z_log(NULL, CORE_DUMP, 8, "Reading ZCode data; len='%d', used='%d', partial='0x%02x'",
            res, self->buf_used, self->buf[self->buf_used]);
      z_log_data_dump(NULL, CORE_DEBUG, 8, self->buf, res);
      g_memmove(to, self->buf, res);
      self->buf_used -= res;
      g_memmove(self->buf, self->buf + res, self->buf_used + 1);
      z_log(NULL, CORE_DUMP, 8, "Remaining ZCode data; len='%d', used='%d', partial='0x%02x'",
            res, self->buf_used, self->buf[self->buf_used]);
    }
  z_leave();
  return res;
}

/**
 * z_code_peek_result:
 * @self: this
 * @dir: direction
 *
 * Returns a pointer to the transformed data (Non-destructive read
 * support). The length of the data returned by this function is
 * available by calling the z_code_get_result_length() function.
 *
 * Returns:
 * Pointer to the data
 */
const guchar *
z_code_peek_result(ZCode *self)
{
  guchar   *res;

  z_enter();
  res = self->buf;
  z_leave();
  return res;
}

/**
 * z_code_get_result_length:
 * @self: this
 *
 * How much data is available in the internal buffer, e.g. how much
 * output the coder has generated so far.
 *
 * Returns:
 * The amount of available data in bytes
 */
gint
z_code_get_result_length(ZCode *self)
{
  gint    res;

  z_enter();
  res = self->buf_used;
  z_leave();
  return res;
}

/**
 * z_code_init:
 * @self: ZCode instance
 * @bufsize: initial bufsize
 *
 * Initialize a caller-allocated ZCode structure with default values
 * for ZCode. Usually used by actual ZCode implementations.
 **/
void
z_code_init(ZCode *self, gint bufsize)
{
  z_enter();
  self->buf_len = (bufsize <= 0) ? ZCODE_BUFSIZE_DEFAULT : bufsize;
  self->buf = g_new0(guchar, self->buf_len);
  self->buf_used = 0;
  self->error_counter = 0;
  self->transform = z_code_dummy_transform;
  self->finish = z_code_dummy_finish;
  self->free_fn = z_code_dummy_free;
  z_leave();
}

/**
 * z_code_new:
 * @bufsize: Initial buffer size
 *
 * Constructor, creates a ZCode around the specified codec.  If
 * @bufsize is <= 0, then ZCODE_BUFSIZE_DEFAULT is used instead.
 *
 * Returns:
 * The new instance
 */
ZCode*
z_code_new(gint bufsize)
{
  ZCode *self;

  z_enter();
  self = g_new0(ZCode, 1);
  z_code_init(self, bufsize);
  z_leave();
  return self;
}

/**
 * z_code_free:
 * @self: this
 *
 * Free a ZCode instance by calling the virtual free_fn function.
 */
void
z_code_free(ZCode *self)
{
  z_enter();
  self->free_fn(self);
  g_free(self->buf);
  g_free(self);
  z_leave();
}
