;;;
;;; tm-ew-e.el --- RFC 1522 based multilingual MIME message header
;;;                encoder for GNU Emacs
;;;
;;; Copyright (C) 1995 Free Software Foundation, Inc.
;;; Copyright (C) 1993 .. 1996 MORIOKA Tomohiko
;;;
;;; Author: MORIOKA Tomohiko <morioka@jaist.ac.jp>
;;; Version: $Revision: 7.12 $
;;; Keywords: mail, news, MIME, RFC 1522, multilingual, encoded-word
;;;
;;; This file is part of tm (Tools for MIME).
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 2, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with This program.  If not, write to the Free Software
;;; Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; Code:

(require 'mel)
(require 'tl-822)
(require 'tm-def)


;;; @ version
;;;

(defconst tm-ew-e/RCS-ID
  "$Id: tm-ew-e.el,v 7.12 1996/01/11 18:31:43 morioka Exp $")
(defconst mime-eword/encoder-version (get-version-string tm-ew-e/RCS-ID))


;;; @ variables
;;;

(defvar mime/no-encoding-header-fields '("X-Nsubject" "Newsgroups"))

(defvar mime/use-X-Nsubject nil)

(defvar mime-eword/charset-encoding-alist
  '(("US-ASCII"       . nil)
    ("ISO-8859-1"     . "Q")
    ("ISO-8859-2"     . "Q")
    ("ISO-8859-3"     . "Q")
    ("ISO-8859-4"     . "Q")
;;; ("ISO-8859-5"     . "Q")
    ("KOI8-R"         . "Q")
    ("ISO-8859-7"     . "Q")
    ("ISO-8859-8"     . "Q")
    ("ISO-8859-9"     . "Q")
    ("ISO-2022-JP"    . "B")
    ("ISO-2022-KR"    . "B")
    ("EUC-KR"         . "B")
    ("ISO-2022-JP-2"  . "B")
    ("ISO-2022-INT-1" . "B")
    ))


;;; @ encoded-text encoder
;;;

(defun tm-eword::encode-encoded-text (charset encoding string &optional mode)
  (let ((text
	 (cond ((string= encoding "B")
		(base64-encode-string string))
	       ((string= encoding "Q")
		(q-encoding-encode-string string mode))
	       )
	 ))
    (if text
	(concat "=?" charset "?" encoding "?" text "?=")
      )))


;;; @ leading char
;;;

(defun tm-eword::char-type (chr)
  (if (or (= chr 32)(= chr ?\t))
      nil
    (char-leading-char chr)
    ))

(defun tm-eword::parse-lc-word (str)
  (let* ((rest (string-to-char-list str))
	 (chr (car rest))
	 (lc (tm-eword::char-type chr))
	 (p (char-bytes chr))
	 )
    (catch 'tag
      (while (setq rest (cdr rest))
	(setq chr (car rest))
	(if (not (eq lc (tm-eword::char-type chr)))
	    (throw 'tag nil)
	  )
	(setq p (+ p (char-bytes chr)))
	))
    (cons (cons lc (substring str 0 p)) (substring str p))
    ))

(defun tm-eword::split-to-lc-words (str)
  (let (ret dest)
    (while (and (not (string= str ""))
		(setq ret (tm-eword::parse-lc-word str))
		)
      (setq dest (cons (car ret) dest))
      (setq str (cdr ret))
      )
    (reverse dest)
    ))


;;; @ word
;;;

(defun tm-eword::parse-word (lcwl)
  (let* ((lcw (car lcwl))
	 (lc (car lcw))
	 )
    (if (null lc)
	lcwl
      (let ((lcl (list lc))
	    (str (cdr lcw))
	    )
	(catch 'tag
	  (while (setq lcwl (cdr lcwl))
	    (setq lcw (car lcwl))
	    (setq lc (car lcw))
	    (if (null lc)
		(throw 'tag nil)
	      )
	    (if (not (memq lc lcl))
		(setq lcl (cons lc lcl))
	      )
	    (setq str (concat str (cdr lcw)))
	    ))
	(cons (cons lcl str) lcwl)
	))))

(defun tm-eword::lc-words-to-words (lcwl)
  (let (ret dest)
    (while (setq ret (tm-eword::parse-word lcwl))
      (setq dest (cons (car ret) dest))
      (setq lcwl (cdr ret))
      )
    (reverse dest)
    ))


;;; @ rule
;;;

(defun tm-eword::find-charset-rule (lcl)
  (if lcl
      (let* ((charset (mime/find-charset lcl))
	     (encoding
	      (cdr (assoc charset mime-eword/charset-encoding-alist))
	      ))
	(list charset encoding)
	)))

(defun tm-eword::words-to-ruled-words (wl)
  (mapcar (function
	   (lambda (word)
	     (cons (cdr word) (tm-eword::find-charset-rule (car word)))
	     ))
	  wl))

(defun tm-eword::space-process (seq)
  (let (prev a ac b c cc)
    (while seq
      (setq b (car seq))
      (setq seq (cdr seq))
      (setq c (car seq))
      (setq cc (nth 1 c))
      (if (null (nth 1 b))
	  (progn
	    (setq a (car prev))
	    (setq ac (nth 1 a))
	    (if (and (nth 2 a)(nth 2 c))
		(cond ((equal ac cc)
		       (setq prev (cons
				   (cons (concat (car a)(car b)(car c))
					 (cdr a))
				   (cdr prev)
				   ))
		       (setq seq (cdr seq))
		       )
		      (t
		       (setq prev (cons
				   (cons (concat (car a)(car b))
					 (cdr a))
				   (cdr prev)
				   ))
		       ))
	      (setq prev (cons b prev))
	      ))
	(setq prev (cons b prev))
	))
    (reverse prev)
    ))

(defun tm-eword::split-string (str)
  (tm-eword::space-process
   (tm-eword::words-to-ruled-words
    (tm-eword::lc-words-to-words
     (tm-eword::split-to-lc-words str)
     ))))


;;; @ length
;;;

(defun base64-length (string)
  (let ((l (length string)))
    (* (+ (/ l 3)
	  (if (= (mod l 3) 0) 0 1)
	  ) 4)
    ))

(defun q-encoding-length (string)
  (let ((l 0)(i 0)(len (length string)) chr)
    (while (< i len)
      (setq chr (elt string i))
      (if (string-match "[A-Za-z0-9!*+/=_---]" (char-to-string chr))
	  (setq l (+ l 1))
	(setq l (+ l 3))
	)
      (setq i (+ i 1)) )
    l))

(defun tm-eword::encoded-word-length (rword)
  (let ((charset  (nth 1 rword))
	(encoding (nth 2 rword))
	(string   (car rword))
	ret)
    (setq ret
	  (cond ((equal encoding "B")
		 (setq string
		       (mime/convert-string-from-emacs string charset))
		 (base64-length string)
		 )
		((equal encoding "Q")
		 (setq string
		       (mime/convert-string-from-emacs string charset))
		 (q-encoding-length string)
		 )))
    (if ret
	(cons (+ 7 (length charset) ret) string)
      )))


;;; @ encode-string
;;;

(defun tm-eword::encode-string-1 (column rwl &optional mode)
  (let* ((rword (car rwl))
	 (ret (tm-eword::encoded-word-length rword))
	 string len)
    (if (null ret)
	(cond ((and (setq string (car rword))
		    (<= (setq len (+ (length string) column)) 76)
		    )
	       (setq rwl (cdr rwl))
	       )
	      (t
	       (setq string "\n ")
	       (setq len 1)
	       ))
      (cond ((and (setq len (car ret))
		  (<= (+ column len) 76)
		  )
	     (setq string
		   (tm-eword::encode-encoded-text
		    (nth 1 rword) (nth 2 rword) (cdr ret)
		    ))
	     (setq len (+ (length string) column))
	     (setq rwl (cdr rwl))
	     )
	    (t
	     (setq string (car rword))
	     (let* ((ls (reverse (string-to-char-list string)))
		    (sl (length string))
		    (p sl) str)
	       (while (and ls
			   (progn
			     (setq p (- p (char-bytes (car ls))))
			     (setq str (substring string 0 p))
			     (setq ret (tm-eword::encoded-word-length
					(cons str (cdr rword))
					))
			     (setq str (cdr ret))
			     (setq len (+ (car ret) column))
			     (> len 76)
			     ))
		 (setq ls (cdr ls))
		 )
	       (if (and ls (not (string= str "")))
		   (progn
		     (setq rwl (cons (cons (substring string p) (cdr rword))
				     (cdr rwl)))
		     (setq string
			   (tm-eword::encode-encoded-text
			    (nth 1 rword) (nth 2 rword) str))
		     (setq len (+ (length string) column))
		     )
		 (setq string "\n ")
		 (setq len 1)
		 )
	       )))
      )
    (list string len rwl)
    ))

(defun tm-eword::encode-rwl (column rwl &optional mode)
  (let (ret dest ps special str)
    (while rwl
      (setq ret (tm-eword::encode-string-1 column rwl mode))
      (setq str (car ret))
      (if (eq (elt str 0) ?\n)
	  (if (eq special ?\()
	      (progn
		(setq dest (concat dest "\n ("))
		(setq ret (tm-eword::encode-string-1 2 rwl mode))
		(setq str (car ret))
		))
	(cond ((eq special 32)
	       (if (string= str "(")
		   (setq ps t)
		 (setq dest (concat dest " "))
		 (setq ps nil)
		 ))
	      ((eq special ?\()
	       (if ps
		   (progn
		     (setq dest (concat dest " ("))
		     (setq ps nil)
		     )
		 (setq dest (concat dest "("))
		 )
	       )))
      (cond ((string= str " ")
	     (setq special 32)
	     )
	    ((string= str "(")
	     (setq special ?\()
	     )
	    (t
	     (setq special nil)
	     (setq dest (concat dest str))
	     ))
      (setq column (nth 1 ret)
	    rwl (nth 2 ret))
      )
    (list dest column)
    ))

(defun tm-eword::encode-string (column str &optional mode)
  (tm-eword::encode-rwl column (tm-eword::split-string str) mode)
  )


;;; @ converter
;;;

(defun tm-eword::phrase-to-rwl (phrase)
  (let (token type dest str)
    (while phrase
      (setq token (car phrase))
      (setq type (car token))
      (cond ((eq type 'quoted-string)
	     (setq str (concat "\"" (cdr token) "\""))
	     (setq dest
		   (append dest
			   (list
			    (cons str (tm-eword::find-charset-rule
				       (find-charset-string str)))
			    )))
	     )
	    ((eq type 'comment)
	     (setq dest
		   (append dest
			   '(("(" nil nil))
			   (tm-eword::words-to-ruled-words
			    (tm-eword::lc-words-to-words
			     (tm-eword::split-to-lc-words (cdr token))))
			   '((")" nil nil))
			   ))
	     )
	    (t
	     (setq dest (append dest
				(tm-eword::words-to-ruled-words
				 (tm-eword::lc-words-to-words
				  (tm-eword::split-to-lc-words (cdr token))
				  ))))
	     ))
      (setq phrase (cdr phrase))
      )
    (tm-eword::space-process dest)
    ))

(defun tm-eword::phrase-route-addr-to-rwl (phrase-route-addr)
  (if (eq (car phrase-route-addr) 'phrase-route-addr)
      (let ((phrase (nth 1 phrase-route-addr))
	    (route (nth 2 phrase-route-addr))
	    dest)
	(if (eq (car (car phrase)) 'spaces)
	    (setq phrase (cdr phrase))
	  )
	(setq dest (tm-eword::phrase-to-rwl phrase))
	(if dest
	    (setq dest (append dest '((" " nil nil))))
	  )
	(append
	 dest
	 (list (list (concat "<" (rfc822/addr-to-string route) ">") nil nil))
	 ))))

(defun tm-eword::addr-spec-to-rwl (addr-spec)
  (if (eq (car addr-spec) 'addr-spec)
      (list (list (rfc822/addr-to-string (cdr addr-spec)) nil nil))
    ))

(defun tm-eword::mailbox-to-rwl (mbox)
  (let ((addr (nth 1 mbox))
	(comment (nth 2 mbox))
	dest)
    (setq dest (or (tm-eword::phrase-route-addr-to-rwl addr)
		   (tm-eword::addr-spec-to-rwl addr)
		   ))
    (if comment
	(setq dest
	      (append dest
		      '((" " nil nil)
			("(" nil nil))
		      (tm-eword::split-string comment)
		      '((")" nil nil))
		      )))
    dest))

(defun tm-eword::addresses-to-rwl (addresses)
  (let ((dest (tm-eword::mailbox-to-rwl (car addresses))))
    (if dest
	(while (setq addresses (cdr addresses))
	  (setq dest (append dest
			     '(("," nil nil))
			     '((" " nil nil))
			     (tm-eword::mailbox-to-rwl (car addresses))
			     ))
	  ))
    dest))

(defun tm-eword::encode-address-list (column str)
  (tm-eword::encode-rwl
   column
   (tm-eword::addresses-to-rwl
    (rfc822/parse-addresses
     (rfc822/lexical-analyze str)))))


;;; @ application interfaces
;;;

(defun mime/encode-field (str)
  (setq str (rfc822/unfolding-string str))
  (let ((ret (string-match rfc822/field-top-regexp str)))
    (if ret
	(let ((field-name (substring str 0 (match-end 1)))
	      (field-body (eliminate-top-spaces
			   (substring str (match-end 0))))
	      fname)
	  (concat field-name ": "
		  (cond ((string= field-body "") "")
			((member (setq fname (downcase field-name))
				 '("reply-to" "from" "sender"
				   "resent-reply-to" "resent-from"
				   "resent-sender" "to" "resent-to"
				   "cc" "resent-cc"
				   "bcc" "resent-bcc" "dcc")
				 )
			 (car (tm-eword::encode-address-list
			       (+ (length field-name) 1) field-body))
			 )
			(t
			 (catch 'tag
			   (let ((r mime/no-encoding-header-fields) fn)
			     (while r
			       (setq fn (car r))
			       (if (string= (downcase fn) fname)
				   (throw 'tag field-body)
				 )
			       (setq r (cdr r))
			       ))
			   (car (tm-eword::encode-string
				 (+ (length field-name) 1) field-body))
			   ))
			))
	  )
      str)))

(defun mime/exist-encoded-word-in-subject ()
  (let ((str (rfc822/get-field-body "Subject")))
    (if (and str (string-match mime/encoded-word-regexp str))
	str)))

(defun mime/encode-message-header ()
  (interactive "*")
  (save-excursion
    (save-restriction
      (narrow-to-region (goto-char (point-min))
			(progn
			  (re-search-forward
			   (concat
			    "^" (regexp-quote mail-header-separator) "$")
			   nil t)
			  (match-beginning 0)
			  ))
      (goto-char (point-min))
      (let (beg end field)
	(while (re-search-forward rfc822/field-top-regexp nil t)
	  (setq beg (match-beginning 0))
	  (setq end (rfc822/field-end))
	  (if (and (find-charset-region beg end)
		   (setq field
			 (mime/encode-field
			  (buffer-substring-no-properties beg end)
			  ))
		   )
	      (progn
		(delete-region beg end)
		(insert field)
		))
	  ))
      (if mime/use-X-Nsubject
	  (let ((str (mime/exist-encoded-word-in-subject)))
	    (if str
		(insert
		 (concat
		  "\nX-Nsubject: "
		  (mime-eword/decode-string (rfc822/unfolding-string str))
		  )))))
      )))

(defun mime-eword/encode-string (str &optional column mode)
  (car (tm-eword::encode-rwl (or column 0)
			     (tm-eword::split-string str) mode))
  )


;;; @ end
;;;

(provide 'tm-ew-e)

;;; tm-ew-e.el ends here
