/*  Part of FAST - FMRIB's Automated Segmentation Tool

    Yongyue Zhang, FMRIB Image Analysis Group

    Copyright (C) 1999-2003 University of Oxford  */

/*  CCOPYRIGHT */

#ifdef _MSC_VER
#pragma warning (disable : 4305 4786)
#endif

#include "imagehelper.h"
#include <string>
#include <map>
using namespace std;

const char* PixelTypeDescrip(EPixFmt pixfmt)
{
	switch (pixfmt)
	{
		default:				return "Unknown";
		case epixfmtGrayChar:	return "sChar";
		case epixfmtGrayShort:	return "sShort";
		case epixfmtGrayInt:	return "sInt";
		case epixfmtGrayLong:	return "sLong";
		case epixfmtGrayFloat:	return "GrayFloat";
		case epixfmtGrayDouble:	return "GrayDouble";
		case epixfmtGrayByte:	return "uChar";
		case epixfmtGrayUShort:	return "uShort";
		case epixfmtGrayUInt:	return "uInt";

		case epixfmtRGBChar:	return "sRGBChar";
		case epixfmtRGBShort:	return "sRGBShort";
		case epixfmtRGBInt:		return "sRGBInt";
		case epixfmtRGBLong:	return "sRGBLong";
		case epixfmtRGBFloat:	return "RGBFloat";
		case epixfmtRGBDouble:	return "RGBDouble";
		case epixfmtRGBByte:	return "uRGBChar";
		case epixfmtRGBUShort:	return "uRGBShort";
		case epixfmtRGBUInt:	return "uRGBInt";
	}
}

ZImageBase*	TypeCopyFrom(const ZImageBase* image)
{
	switch(image->PixFmt())
	{
	case epixfmtGrayByte:		return new ZImage<BYTE>;
	case epixfmtGrayShort:		return new ZImage<short>;
	case epixfmtGrayUShort:		return new ZImage<WORD>;
	case epixfmtGrayFloat:		return new ZImage<float>;
	case epixfmtGrayDouble:		return new ZImage<double>;
	case epixfmtRGBByte:		return new ZImage< ZRGB<BYTE> >;
	case epixfmtRGBShort:		return new ZImage< ZRGB<short> >;
	case epixfmtRGBUShort:		return new ZImage< ZRGB<WORD> >;
	case epixfmtRGBFloat:		return new ZImage< ZRGB<float> >;
	default:
		ZError("TypeCopyFrom", "Unsupported Pixel Format %s!", PixelTypeDescrip(image->PixFmt()));
		return 0;
	}
}

ZImageBase*	FloatTypeCopyFrom(const ZImageBase* image) { if(image->isColor()) return new ZRGBFloatImage; return new ZGrayFloatImage; }
ZImageBase*	ShortTypeCopyFrom(const ZImageBase* image) { if(image->isColor()) return new ZRGBShortImage; return new ZGrayShortImage; }
ZImageBase*	ByteTypeCopyFrom(const ZImageBase* image) { if(image->isColor()) return new ZRGBByteImage; return new ZGrayByteImage; }

ZImageBase* GrayTypeCopyFrom(const ZImageBase* image) 
{ 
	switch(image->PixFmt())
	{
	case epixfmtGrayByte:	case epixfmtRGBByte:	return new ZImage<BYTE>;
	case epixfmtGrayShort:	case epixfmtRGBShort:	return new ZImage<short>;
	case epixfmtGrayUShort: case epixfmtRGBUShort:	return new ZImage<WORD>;
	case epixfmtGrayFloat:	case epixfmtRGBFloat:	return new ZImage<float>;
	case epixfmtGrayDouble:	return new ZImage<double>;
	default: 
		ZError("GrayTypeCopyFrom", "Unsupported Pixel Format %s!", PixelTypeDescrip(image->PixFmt()));
		return 0;
	}
}

ZImageBase* RGBTypeCopyFrom(const ZImageBase* image) 
{ 
	switch(image->PixFmt())
	{
	case epixfmtGrayByte:	case epixfmtRGBByte:	return new ZRGBByteImage;
	case epixfmtGrayShort:	case epixfmtRGBShort:	return new ZRGBShortImage;
	case epixfmtGrayUShort: case epixfmtRGBUShort:	return new ZRGBUShortImage;
	case epixfmtGrayFloat:	case epixfmtRGBFloat:	return new ZRGBFloatImage;
	default: 
		ZError("RGBTypeCopyFrom", "Unsupported Pixel Format %s!", PixelTypeDescrip(image->PixFmt()));
		return 0;
	}
}

ZImageBase* RGBToGray(const ZImageBase* image)
{
	ZImageBase* pimage = GrayTypeCopyFrom(image);
	image->CopyPixelsTo(*pimage);
	return pimage;
}

ZImageBase* ToByteImage(const ZImageBase* image)
{
	ZImageBase* pImage = ByteTypeCopyFrom(image);
	image->CopyPixelsTo(*pImage);
	return pImage;
}

ZImageBase* ToShortImage(const ZImageBase* image)
{
	ZImageBase* pImage = ShortTypeCopyFrom(image);
	image->CopyPixelsTo(*pImage);
	return pImage;
}

ZImageBase* ToFloatImage(const ZImageBase* image)
{
	ZImageBase* pImage = FloatTypeCopyFrom(image);
	image->CopyPixelsTo(*pImage);
	return pImage;
}

ZImageBase* exp(ZImageBase& image)
{
	ZImageBase* pImage = FloatTypeCopyFrom(&image);
	image.CopyPixelsTo(*pImage);
	if(image.isColor()) exp(*(ZRGBFloatImage*)pImage);
	else exp(*(ZGrayFloatImage*)pImage);
	return pImage;
}

ZImageBase* log(ZImageBase& image)
{
	ZImageBase* pImage = FloatTypeCopyFrom(&image);
	image.CopyPixelsTo(*pImage);
	if(image.isColor()) log(*(ZRGBFloatImage*)pImage);
	else log(*(ZGrayFloatImage*)pImage);
	return pImage;
}

ZImageBase* log10(ZImageBase& image)
{
	ZImageBase* pImage = FloatTypeCopyFrom(&image);
	image.CopyPixelsTo(*pImage);
	if(image.isColor()) log10(*(ZRGBFloatImage*)pImage);
	else log10(*(ZGrayFloatImage*)pImage);
	return pImage;
}

ZImageBase* sqrt(ZImageBase& image)
{
	ZImageBase* pImage = FloatTypeCopyFrom(&image);
	image.CopyPixelsTo(*pImage);
	if(image.isColor()) sqrt(*(ZRGBFloatImage*)pImage);
	else sqrt(*(ZGrayFloatImage*)pImage);
	return pImage;
}

void SplitRGBChannel(const ZImageBase& image, ZImageBase& r, ZImageBase& g, ZImageBase& b)
{
	if(!image.isColor()) { ZWarning("SplitRGBChannel", "image is not color!"); r = g = b = image; return; }
	if(r.isColor() || g.isColor() || b.isColor())
	{
		ZError("SplitRGBChannel", "Channel image is color!");
		return;
	}
	if(r.PixFmt() != g.PixFmt() || r.PixFmt() != b.PixFmt() || 
	  (image.PixFmt() & ~epixfmtRGB) != r.PixFmt())
	{
		ZError("SplitRGBChannel", "Pixel format is not the same!");
		return;
	}

	int psize = r.BytesPerPixel(), psize3 = psize*3;
	int width = image.Width(), height = image.Height(), depth = image.Depth();
	int xsize = image.BytesPerRow(), xysize = image.BytesPerSlice();
	r.Resize(width, height, depth);
	g.Resize(width, height, depth);
	b.Resize(width, height, depth);

	PBYTE ps = image.Origin(), pr = r.Origin(), pg = g.Origin(), pb = b.Origin();
	for(int k=0; k<depth; k++, ps+=xysize)
	{
		PBYTE pss = ps;
		for(int j=0; j<height; j++, pss+=xsize)
		for(int i=0; i<width; i++, pr+=psize, pg+=psize, pb+=psize)
		{
			for(int m=0; m<psize; m++) 
			{
				pr[m] = pss[i*psize3+m];
				pg[m] = pss[i*psize3+psize+m];
				pb[m] = pss[i*psize3+psize+psize+m];
			}
		}
	}
}

void CombineRGBChannel(ZImageBase& image, const ZImageBase& r, const ZImageBase& g, const ZImageBase& b)
{
	if(!image.isColor()) { ZWarning("CombineRGBChannel", "image is not color!"); return; }
	if(r.isColor() || g.isColor() || b.isColor())
	{
		ZError("CombineRGBChannel", "Channel image is color!");
		return;
	}
	if(r.PixFmt() != g.PixFmt() || r.PixFmt() != b.PixFmt() || 
	  (image.PixFmt() & ~epixfmtRGB) != r.PixFmt())
	{
		ZError("CombineRGBChannel", "Pixel format is not the same!");
		return;
	}
	if(r.Size() != g.Size() || r.Size() != b.Size())
	{
		ZError("CombineRGBChannel", "Channel images are not of the same size!");
		return;
	}

	int psize = r.BytesPerPixel(), psize3 = psize*3;
	int width = r.Width(), height = r.Height(), depth = r.Depth(), size=r.Size();
	image.Resize(width, height, depth);

	PBYTE ps = image.Origin(), pr = r.Origin(), pg = g.Origin(), pb = b.Origin();
	for(int i=0; i<size; i++, pr+=psize, pg+=psize, pb+=psize)
	{
		for(int m=0; m<psize; m++) 
		{
			ps[i*psize3+m] = pr[m];
			ps[i*psize3+psize+m] = pg[m];
			ps[i*psize3+psize+psize+m] = pb[m];
		}
	}
}

void exp(ZGrayFloatImage& x) { for(ZGrayFloatImage::iterator p(x); p.more(); p++) *p = exp(*p); }
void log(ZGrayFloatImage& x) { for(ZGrayFloatImage::iterator p(x); p.more(); p++) *p = log(*p); }
void log10(ZGrayFloatImage& x) { for(ZGrayFloatImage::iterator p(x); p.more(); p++) *p = log10(*p); }
void sqrt(ZGrayFloatImage& x) { for(ZGrayFloatImage::iterator p(x); p.more(); p++) *p = sqrt(*p); }
void exp(ZRGBFloatImage& x) { for(ZRGBFloatImage::iterator p(x); p.more(); p++) *p = exp(*p); }
void log(ZRGBFloatImage& x) { for(ZRGBFloatImage::iterator p(x); p.more(); p++) *p = log(*p); }
void log10(ZRGBFloatImage& x) { for(ZRGBFloatImage::iterator p(x); p.more(); p++) *p = log10(*p); }
void sqrt(ZRGBFloatImage& x) { for(ZRGBFloatImage::iterator p(x); p.more(); p++) *p = sqrt(*p); }

typedef float TRI[3];

static TRI LUTfmri1[] = {
{0.000000,0.000000,0.000000},
{0.010000,0.010000,0.010000},
{0.020000,0.020000,0.020000},
{0.030000,0.030000,0.030000},
{0.040000,0.040000,0.040000},
{0.050000,0.050000,0.050000},
{0.060000,0.060000,0.060000},
{0.070000,0.070000,0.070000},
{0.080000,0.080000,0.080000},
{0.090000,0.090000,0.090000},
{0.100000,0.100000,0.100000},
{0.110000,0.110000,0.110000},
{0.120000,0.120000,0.120000},
{0.130000,0.130000,0.130000},
{0.140000,0.140000,0.140000},
{0.150000,0.150000,0.150000},
{0.160000,0.160000,0.160000},
{0.170000,0.170000,0.170000},
{0.180000,0.180000,0.180000},
{0.190000,0.190000,0.190000},
{0.200000,0.200000,0.200000},
{0.210000,0.210000,0.210000},
{0.220000,0.220000,0.220000},
{0.230000,0.230000,0.230000},
{0.240000,0.240000,0.240000},
{0.250000,0.250000,0.250000},
{0.260000,0.260000,0.260000},
{0.270000,0.270000,0.270000},
{0.280000,0.280000,0.280000},
{0.290000,0.290000,0.290000},
{0.300000,0.300000,0.300000},
{0.310000,0.310000,0.310000},
{0.320000,0.320000,0.320000},
{0.330000,0.330000,0.330000},
{0.340000,0.340000,0.340000},
{0.350000,0.350000,0.350000},
{0.360000,0.360000,0.360000},
{0.370000,0.370000,0.370000},
{0.380000,0.380000,0.380000},
{0.390000,0.390000,0.390000},
{0.400000,0.400000,0.400000},
{0.410000,0.410000,0.410000},
{0.420000,0.420000,0.420000},
{0.430000,0.430000,0.430000},
{0.440000,0.440000,0.440000},
{0.450000,0.450000,0.450000},
{0.460000,0.460000,0.460000},
{0.470000,0.470000,0.470000},
{0.480000,0.480000,0.480000},
{0.490000,0.490000,0.490000},
{0.500000,0.500000,0.500000},
{0.510000,0.510000,0.510000},
{0.520000,0.520000,0.520000},
{0.530000,0.530000,0.530000},
{0.540000,0.540000,0.540000},
{0.550000,0.550000,0.550000},
{0.560000,0.560000,0.560000},
{0.570000,0.570000,0.570000},
{0.580000,0.580000,0.580000},
{0.590000,0.590000,0.590000},
{0.600000,0.600000,0.600000},
{0.610000,0.610000,0.610000},
{0.620000,0.620000,0.620000},
{0.630000,0.630000,0.630000},
{0.640000,0.640000,0.640000},
{0.650000,0.650000,0.650000},
{0.660000,0.660000,0.660000},
{0.670000,0.670000,0.670000},
{0.680000,0.680000,0.680000},
{0.690000,0.690000,0.690000},
{0.700000,0.700000,0.700000},
{0.710000,0.710000,0.710000},
{0.720000,0.720000,0.720000},
{0.730000,0.730000,0.730000},
{0.740000,0.740000,0.740000},
{0.750000,0.750000,0.750000},
{0.760000,0.760000,0.760000},
{0.770000,0.770000,0.770000},
{0.780000,0.780000,0.780000},
{0.790000,0.790000,0.790000},
{0.800000,0.800000,0.800000},
{0.810000,0.810000,0.810000},
{0.820000,0.820000,0.820000},
{0.830000,0.830000,0.830000},
{0.840000,0.840000,0.840000},
{0.850000,0.850000,0.850000},
{0.860000,0.860000,0.860000},
{0.870000,0.870000,0.870000},
{0.880000,0.880000,0.880000},
{0.890000,0.890000,0.890000},
{0.900000,0.900000,0.900000},
{0.910000,0.910000,0.910000},
{0.920000,0.920000,0.920000},
{0.930000,0.930000,0.930000},
{0.940000,0.940000,0.940000},
{0.950000,0.950000,0.950000},
{0.960000,0.960000,0.960000},
{0.970000,0.970000,0.970000},
{0.980000,0.980000,0.980000},
{0.990000,0.990000,0.990000},
{1.0,0.000000,0.0},
{1.0,0.010000,0.0},
{1.0,0.020000,0.0},
{1.0,0.030000,0.0},
{1.0,0.040000,0.0},
{1.0,0.050000,0.0},
{1.0,0.060000,0.0},
{1.0,0.070000,0.0},
{1.0,0.080000,0.0},
{1.0,0.090000,0.0},
{1.0,0.100000,0.0},
{1.0,0.110000,0.0},
{1.0,0.120000,0.0},
{1.0,0.130000,0.0},
{1.0,0.140000,0.0},
{1.0,0.150000,0.0},
{1.0,0.160000,0.0},
{1.0,0.170000,0.0},
{1.0,0.180000,0.0},
{1.0,0.190000,0.0},
{1.0,0.200000,0.0},
{1.0,0.210000,0.0},
{1.0,0.220000,0.0},
{1.0,0.230000,0.0},
{1.0,0.240000,0.0},
{1.0,0.250000,0.0},
{1.0,0.260000,0.0},
{1.0,0.270000,0.0},
{1.0,0.280000,0.0},
{1.0,0.290000,0.0},
{1.0,0.300000,0.0},
{1.0,0.310000,0.0},
{1.0,0.320000,0.0},
{1.0,0.330000,0.0},
{1.0,0.340000,0.0},
{1.0,0.350000,0.0},
{1.0,0.360000,0.0},
{1.0,0.370000,0.0},
{1.0,0.380000,0.0},
{1.0,0.390000,0.0},
{1.0,0.400000,0.0},
{1.0,0.410000,0.0},
{1.0,0.420000,0.0},
{1.0,0.430000,0.0},
{1.0,0.440000,0.0},
{1.0,0.450000,0.0},
{1.0,0.460000,0.0},
{1.0,0.470000,0.0},
{1.0,0.480000,0.0},
{1.0,0.490000,0.0},
{1.0,0.500000,0.0},
{1.0,0.510000,0.0},
{1.0,0.520000,0.0},
{1.0,0.530000,0.0},
{1.0,0.540000,0.0},
{1.0,0.550000,0.0},
{1.0,0.560000,0.0},
{1.0,0.570000,0.0},
{1.0,0.580000,0.0},
{1.0,0.590000,0.0},
{1.0,0.600000,0.0},
{1.0,0.610000,0.0},
{1.0,0.620000,0.0},
{1.0,0.630000,0.0},
{1.0,0.640000,0.0},
{1.0,0.650000,0.0},
{1.0,0.660000,0.0},
{1.0,0.670000,0.0},
{1.0,0.680000,0.0},
{1.0,0.690000,0.0},
{1.0,0.700000,0.0},
{1.0,0.710000,0.0},
{1.0,0.720000,0.0},
{1.0,0.730000,0.0},
{1.0,0.740000,0.0},
{1.0,0.750000,0.0},
{1.0,0.760000,0.0},
{1.0,0.770000,0.0},
{1.0,0.780000,0.0},
{1.0,0.790000,0.0},
{1.0,0.800000,0.0},
{1.0,0.810000,0.0},
{1.0,0.820000,0.0},
{1.0,0.830000,0.0},
{1.0,0.840000,0.0},
{1.0,0.850000,0.0},
{1.0,0.860000,0.0},
{1.0,0.870000,0.0},
{1.0,0.880000,0.0},
{1.0,0.890000,0.0},
{1.0,0.900000,0.0},
{1.0,0.910000,0.0},
{1.0,0.920000,0.0},
{1.0,0.930000,0.0},
{1.0,0.940000,0.0},
{1.0,0.950000,0.0},
{1.0,0.960000,0.0},
{1.0,0.970000,0.0},
{1.0,0.980000,0.0},
{1.0,0.990000,0.0}
};

static TRI LUTfmri1t[] = {
{0.000000,0.000000,0.000000},
{0.010000,0.010000,0.010000},
{0.020000,0.020000,0.020000},
{0.030000,0.030000,0.030000},
{0.040000,0.040000,0.040000},
{0.050000,0.050000,0.050000},
{0.060000,0.060000,0.060000},
{0.070000,0.070000,0.070000},
{0.080000,0.080000,0.080000},
{0.090000,0.090000,0.090000},
{0.100000,0.100000,0.100000},
{0.110000,0.110000,0.110000},
{0.120000,0.120000,0.120000},
{0.130000,0.130000,0.130000},
{0.140000,0.140000,0.140000},
{0.150000,0.150000,0.150000},
{0.160000,0.160000,0.160000},
{0.170000,0.170000,0.170000},
{0.180000,0.180000,0.180000},
{0.190000,0.190000,0.190000},
{0.200000,0.200000,0.200000},
{0.210000,0.210000,0.210000},
{0.220000,0.220000,0.220000},
{0.230000,0.230000,0.230000},
{0.240000,0.240000,0.240000},
{0.250000,0.250000,0.250000},
{0.260000,0.260000,0.260000},
{0.270000,0.270000,0.270000},
{0.280000,0.280000,0.280000},
{0.290000,0.290000,0.290000},
{0.300000,0.300000,0.300000},
{0.310000,0.310000,0.310000},
{0.320000,0.320000,0.320000},
{0.330000,0.330000,0.330000},
{0.340000,0.340000,0.340000},
{0.350000,0.350000,0.350000},
{0.360000,0.360000,0.360000},
{0.370000,0.370000,0.370000},
{0.380000,0.380000,0.380000},
{0.390000,0.390000,0.390000},
{0.400000,0.400000,0.400000},
{0.410000,0.410000,0.410000},
{0.420000,0.420000,0.420000},
{0.430000,0.430000,0.430000},
{0.440000,0.440000,0.440000},
{0.450000,0.450000,0.450000},
{0.460000,0.460000,0.460000},
{0.470000,0.470000,0.470000},
{0.480000,0.480000,0.480000},
{0.490000,0.490000,0.490000},
{0.500000,0.500000,0.500000},
{0.510000,0.510000,0.510000},
{0.520000,0.520000,0.520000},
{0.530000,0.530000,0.530000},
{0.540000,0.540000,0.540000},
{0.550000,0.550000,0.550000},
{0.560000,0.560000,0.560000},
{0.570000,0.570000,0.570000},
{0.580000,0.580000,0.580000},
{0.590000,0.590000,0.590000},
{0.600000,0.600000,0.600000},
{0.610000,0.610000,0.610000},
{0.620000,0.620000,0.620000},
{0.630000,0.630000,0.630000},
{0.640000,0.640000,0.640000},
{0.650000,0.650000,0.650000},
{0.660000,0.660000,0.660000},
{0.670000,0.670000,0.670000},
{0.680000,0.680000,0.680000},
{0.690000,0.690000,0.690000},
{0.700000,0.700000,0.700000},
{0.710000,0.710000,0.710000},
{0.720000,0.720000,0.720000},
{0.730000,0.730000,0.730000},
{0.740000,0.740000,0.740000},
{0.750000,0.750000,0.750000},
{0.760000,0.760000,0.760000},
{0.770000,0.770000,0.770000},
{0.780000,0.780000,0.780000},
{0.790000,0.790000,0.790000},
{0.800000,0.800000,0.800000},
{0.810000,0.810000,0.810000},
{0.820000,0.820000,0.820000},
{0.830000,0.830000,0.830000},
{0.840000,0.840000,0.840000},
{0.850000,0.850000,0.850000},
{0.860000,0.860000,0.860000},
{0.870000,0.870000,0.870000},
{0.880000,0.880000,0.880000},
{0.890000,0.890000,0.890000},
{0.900000,0.900000,0.900000},
{0.910000,0.910000,0.910000},
{0.920000,0.920000,0.920000},
{0.930000,0.930000,0.930000},
{0.940000,0.940000,0.940000},
{0.950000,0.950000,0.950000},
{0.960000,0.960000,0.960000},
{0.970000,0.970000,0.970000},
{0.980000,0.980000,0.980000},
{0.990000,0.990000,0.990000},
{0.250000,0.000000,0.0},
{0.250000,0.000000,0.0},
{0.250000,0.000000,0.0},
{0.250000,0.000000,0.0},
{0.250000,0.000000,0.0},
{0.250000,0.000000,0.0},
{0.437500,0.000000,0.0},
{0.625000,0.000000,0.0},
{0.812500,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{0.250000,0.062500,0.0},
{0.250000,0.062500,0.0},
{0.250000,0.062500,0.0},
{0.250000,0.062500,0.0},
{0.250000,0.062500,0.0},
{0.250000,0.062500,0.0},
{0.437500,0.109375,0.0},
{0.625000,0.156250,0.0},
{0.812500,0.203125,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{0.250000,0.125000,0.0},
{0.250000,0.125000,0.0},
{0.250000,0.125000,0.0},
{0.250000,0.125000,0.0},
{0.250000,0.125000,0.0},
{0.250000,0.125000,0.0},
{0.437500,0.218750,0.0},
{0.625000,0.312500,0.0},
{0.812500,0.406250,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{0.250000,0.187500,0.0},
{0.250000,0.187500,0.0},
{0.250000,0.187500,0.0},
{0.250000,0.187500,0.0},
{0.250000,0.187500,0.0},
{0.250000,0.187500,0.0},
{0.437500,0.328125,0.0},
{0.625000,0.468750,0.0},
{0.812500,0.609375,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{0.250000,0.250000,0.0},
{0.250000,0.250000,0.0},
{0.250000,0.250000,0.0},
{0.250000,0.250000,0.0},
{0.250000,0.250000,0.0},
{0.250000,0.250000,0.0},
{0.437500,0.437500,0.0},
{0.625000,0.625000,0.0},
{0.812500,0.812500,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0}
};

static TRI LUTfmri2[] = {
{0.000000,0.000000,0.000000},
{0.010000,0.010000,0.010000},
{0.020000,0.020000,0.020000},
{0.030000,0.030000,0.030000},
{0.040000,0.040000,0.040000},
{0.050000,0.050000,0.050000},
{0.060000,0.060000,0.060000},
{0.070000,0.070000,0.070000},
{0.080000,0.080000,0.080000},
{0.090000,0.090000,0.090000},
{0.100000,0.100000,0.100000},
{0.110000,0.110000,0.110000},
{0.120000,0.120000,0.120000},
{0.130000,0.130000,0.130000},
{0.140000,0.140000,0.140000},
{0.150000,0.150000,0.150000},
{0.160000,0.160000,0.160000},
{0.170000,0.170000,0.170000},
{0.180000,0.180000,0.180000},
{0.190000,0.190000,0.190000},
{0.200000,0.200000,0.200000},
{0.210000,0.210000,0.210000},
{0.220000,0.220000,0.220000},
{0.230000,0.230000,0.230000},
{0.240000,0.240000,0.240000},
{0.250000,0.250000,0.250000},
{0.260000,0.260000,0.260000},
{0.270000,0.270000,0.270000},
{0.280000,0.280000,0.280000},
{0.290000,0.290000,0.290000},
{0.300000,0.300000,0.300000},
{0.310000,0.310000,0.310000},
{0.320000,0.320000,0.320000},
{0.330000,0.330000,0.330000},
{0.340000,0.340000,0.340000},
{0.350000,0.350000,0.350000},
{0.360000,0.360000,0.360000},
{0.370000,0.370000,0.370000},
{0.380000,0.380000,0.380000},
{0.390000,0.390000,0.390000},
{0.400000,0.400000,0.400000},
{0.410000,0.410000,0.410000},
{0.420000,0.420000,0.420000},
{0.430000,0.430000,0.430000},
{0.440000,0.440000,0.440000},
{0.450000,0.450000,0.450000},
{0.460000,0.460000,0.460000},
{0.470000,0.470000,0.470000},
{0.480000,0.480000,0.480000},
{0.490000,0.490000,0.490000},
{0.500000,0.500000,0.500000},
{0.510000,0.510000,0.510000},
{0.520000,0.520000,0.520000},
{0.530000,0.530000,0.530000},
{0.540000,0.540000,0.540000},
{0.550000,0.550000,0.550000},
{0.560000,0.560000,0.560000},
{0.570000,0.570000,0.570000},
{0.580000,0.580000,0.580000},
{0.590000,0.590000,0.590000},
{0.600000,0.600000,0.600000},
{0.610000,0.610000,0.610000},
{0.620000,0.620000,0.620000},
{0.630000,0.630000,0.630000},
{0.640000,0.640000,0.640000},
{0.650000,0.650000,0.650000},
{0.660000,0.660000,0.660000},
{0.670000,0.670000,0.670000},
{0.680000,0.680000,0.680000},
{0.690000,0.690000,0.690000},
{0.700000,0.700000,0.700000},
{0.710000,0.710000,0.710000},
{0.720000,0.720000,0.720000},
{0.730000,0.730000,0.730000},
{0.740000,0.740000,0.740000},
{0.750000,0.750000,0.750000},
{0.760000,0.760000,0.760000},
{0.770000,0.770000,0.770000},
{0.780000,0.780000,0.780000},
{0.790000,0.790000,0.790000},
{0.800000,0.800000,0.800000},
{0.810000,0.810000,0.810000},
{0.820000,0.820000,0.820000},
{0.830000,0.830000,0.830000},
{0.840000,0.840000,0.840000},
{0.850000,0.850000,0.850000},
{0.860000,0.860000,0.860000},
{0.870000,0.870000,0.870000},
{0.880000,0.880000,0.880000},
{0.890000,0.890000,0.890000},
{0.900000,0.900000,0.900000},
{0.910000,0.910000,0.910000},
{0.920000,0.920000,0.920000},
{0.930000,0.930000,0.930000},
{0.940000,0.940000,0.940000},
{0.950000,0.950000,0.950000},
{0.960000,0.960000,0.960000},
{0.970000,0.970000,0.970000},
{0.980000,0.980000,0.980000},
{0.990000,0.990000,0.990000},
{1.0,0.000000,0.0},
{1.0,0.010000,0.0},
{1.0,0.020000,0.0},
{1.0,0.030000,0.0},
{1.0,0.040000,0.0},
{1.0,0.050000,0.0},
{1.0,0.060000,0.0},
{1.0,0.070000,0.0},
{1.0,0.080000,0.0},
{1.0,0.090000,0.0},
{1.0,0.100000,0.0},
{1.0,0.110000,0.0},
{1.0,0.120000,0.0},
{1.0,0.130000,0.0},
{1.0,0.140000,0.0},
{1.0,0.150000,0.0},
{1.0,0.160000,0.0},
{1.0,0.170000,0.0},
{1.0,0.180000,0.0},
{1.0,0.190000,0.0},
{1.0,0.200000,0.0},
{1.0,0.210000,0.0},
{1.0,0.220000,0.0},
{1.0,0.230000,0.0},
{1.0,0.240000,0.0},
{1.0,0.250000,0.0},
{1.0,0.260000,0.0},
{1.0,0.270000,0.0},
{1.0,0.280000,0.0},
{1.0,0.290000,0.0},
{1.0,0.300000,0.0},
{1.0,0.310000,0.0},
{1.0,0.320000,0.0},
{1.0,0.330000,0.0},
{1.0,0.340000,0.0},
{1.0,0.350000,0.0},
{1.0,0.360000,0.0},
{1.0,0.370000,0.0},
{1.0,0.380000,0.0},
{1.0,0.390000,0.0},
{1.0,0.400000,0.0},
{1.0,0.410000,0.0},
{1.0,0.420000,0.0},
{1.0,0.430000,0.0},
{1.0,0.440000,0.0},
{1.0,0.450000,0.0},
{1.0,0.460000,0.0},
{1.0,0.470000,0.0},
{1.0,0.480000,0.0},
{1.0,0.490000,0.0},
{1.0,0.500000,0.0},
{1.0,0.510000,0.0},
{1.0,0.520000,0.0},
{1.0,0.530000,0.0},
{1.0,0.540000,0.0},
{1.0,0.550000,0.0},
{1.0,0.560000,0.0},
{1.0,0.570000,0.0},
{1.0,0.580000,0.0},
{1.0,0.590000,0.0},
{1.0,0.600000,0.0},
{1.0,0.610000,0.0},
{1.0,0.620000,0.0},
{1.0,0.630000,0.0},
{1.0,0.640000,0.0},
{1.0,0.650000,0.0},
{1.0,0.660000,0.0},
{1.0,0.670000,0.0},
{1.0,0.680000,0.0},
{1.0,0.690000,0.0},
{1.0,0.700000,0.0},
{1.0,0.710000,0.0},
{1.0,0.720000,0.0},
{1.0,0.730000,0.0},
{1.0,0.740000,0.0},
{1.0,0.750000,0.0},
{1.0,0.760000,0.0},
{1.0,0.770000,0.0},
{1.0,0.780000,0.0},
{1.0,0.790000,0.0},
{1.0,0.800000,0.0},
{1.0,0.810000,0.0},
{1.0,0.820000,0.0},
{1.0,0.830000,0.0},
{1.0,0.840000,0.0},
{1.0,0.850000,0.0},
{1.0,0.860000,0.0},
{1.0,0.870000,0.0},
{1.0,0.880000,0.0},
{1.0,0.890000,0.0},
{1.0,0.900000,0.0},
{1.0,0.910000,0.0},
{1.0,0.920000,0.0},
{1.0,0.930000,0.0},
{1.0,0.940000,0.0},
{1.0,0.950000,0.0},
{1.0,0.960000,0.0},
{1.0,0.970000,0.0},
{1.0,0.980000,0.0},
{1.0,0.990000,0.0},
{0.0,0.000000,1.0},
{0.0,0.010000,1.0},
{0.0,0.020000,1.0},
{0.0,0.030000,1.0},
{0.0,0.040000,1.0},
{0.0,0.050000,1.0},
{0.0,0.060000,1.0},
{0.0,0.070000,1.0},
{0.0,0.080000,1.0},
{0.0,0.090000,1.0},
{0.0,0.100000,1.0},
{0.0,0.110000,1.0},
{0.0,0.120000,1.0},
{0.0,0.130000,1.0},
{0.0,0.140000,1.0},
{0.0,0.150000,1.0},
{0.0,0.160000,1.0},
{0.0,0.170000,1.0},
{0.0,0.180000,1.0},
{0.0,0.190000,1.0},
{0.0,0.200000,1.0},
{0.0,0.210000,1.0},
{0.0,0.220000,1.0},
{0.0,0.230000,1.0},
{0.0,0.240000,1.0},
{0.0,0.250000,1.0},
{0.0,0.260000,1.0},
{0.0,0.270000,1.0},
{0.0,0.280000,1.0},
{0.0,0.290000,1.0},
{0.0,0.300000,1.0},
{0.0,0.310000,1.0},
{0.0,0.320000,1.0},
{0.0,0.330000,1.0},
{0.0,0.340000,1.0},
{0.0,0.350000,1.0},
{0.0,0.360000,1.0},
{0.0,0.370000,1.0},
{0.0,0.380000,1.0},
{0.0,0.390000,1.0},
{0.0,0.400000,1.0},
{0.0,0.410000,1.0},
{0.0,0.420000,1.0},
{0.0,0.430000,1.0},
{0.0,0.440000,1.0},
{0.0,0.450000,1.0},
{0.0,0.460000,1.0},
{0.0,0.470000,1.0},
{0.0,0.480000,1.0},
{0.0,0.490000,1.0},
{0.0,0.500000,1.0},
{0.0,0.510000,1.0},
{0.0,0.520000,1.0},
{0.0,0.530000,1.0},
{0.0,0.540000,1.0},
{0.0,0.550000,1.0},
{0.0,0.560000,1.0},
{0.0,0.570000,1.0},
{0.0,0.580000,1.0},
{0.0,0.590000,1.0},
{0.0,0.600000,1.0},
{0.0,0.610000,1.0},
{0.0,0.620000,1.0},
{0.0,0.630000,1.0},
{0.0,0.640000,1.0},
{0.0,0.650000,1.0},
{0.0,0.660000,1.0},
{0.0,0.670000,1.0},
{0.0,0.680000,1.0},
{0.0,0.690000,1.0},
{0.0,0.700000,1.0},
{0.0,0.710000,1.0},
{0.0,0.720000,1.0},
{0.0,0.730000,1.0},
{0.0,0.740000,1.0},
{0.0,0.750000,1.0},
{0.0,0.760000,1.0},
{0.0,0.770000,1.0},
{0.0,0.780000,1.0},
{0.0,0.790000,1.0},
{0.0,0.800000,1.0},
{0.0,0.810000,1.0},
{0.0,0.820000,1.0},
{0.0,0.830000,1.0},
{0.0,0.840000,1.0},
{0.0,0.850000,1.0},
{0.0,0.860000,1.0},
{0.0,0.870000,1.0},
{0.0,0.880000,1.0},
{0.0,0.890000,1.0},
{0.0,0.900000,1.0},
{0.0,0.910000,1.0},
{0.0,0.920000,1.0},
{0.0,0.930000,1.0},
{0.0,0.940000,1.0},
{0.0,0.950000,1.0},
{0.0,0.960000,1.0},
{0.0,0.970000,1.0},
{0.0,0.980000,1.0},
{0.0,0.990000,1.0},
{0.0,0.500000,0.0},
{0.0,0.502500,0.0},
{0.0,0.505000,0.0},
{0.0,0.507500,0.0},
{0.0,0.510000,0.0},
{0.0,0.512500,0.0},
{0.0,0.515000,0.0},
{0.0,0.517500,0.0},
{0.0,0.520000,0.0},
{0.0,0.522500,0.0},
{0.0,0.525000,0.0},
{0.0,0.527500,0.0},
{0.0,0.530000,0.0},
{0.0,0.532500,0.0},
{0.0,0.535000,0.0},
{0.0,0.537500,0.0},
{0.0,0.540000,0.0},
{0.0,0.542500,0.0},
{0.0,0.545000,0.0},
{0.0,0.547500,0.0},
{0.0,0.550000,0.0},
{0.0,0.552500,0.0},
{0.0,0.555000,0.0},
{0.0,0.557500,0.0},
{0.0,0.560000,0.0},
{0.0,0.562500,0.0},
{0.0,0.565000,0.0},
{0.0,0.567500,0.0},
{0.0,0.570000,0.0},
{0.0,0.572500,0.0},
{0.0,0.575000,0.0},
{0.0,0.577500,0.0},
{0.0,0.580000,0.0},
{0.0,0.582500,0.0},
{0.0,0.585000,0.0},
{0.0,0.587500,0.0},
{0.0,0.590000,0.0},
{0.0,0.592500,0.0},
{0.0,0.595000,0.0},
{0.0,0.597500,0.0},
{0.0,0.600000,0.0},
{0.0,0.602500,0.0},
{0.0,0.605000,0.0},
{0.0,0.607500,0.0},
{0.0,0.610000,0.0},
{0.0,0.612500,0.0},
{0.0,0.615000,0.0},
{0.0,0.617500,0.0},
{0.0,0.620000,0.0},
{0.0,0.622500,0.0},
{0.0,0.625000,0.0},
{0.0,0.627500,0.0},
{0.0,0.630000,0.0},
{0.0,0.632500,0.0},
{0.0,0.635000,0.0},
{0.0,0.637500,0.0},
{0.0,0.640000,0.0},
{0.0,0.642500,0.0},
{0.0,0.645000,0.0},
{0.0,0.647500,0.0},
{0.0,0.650000,0.0},
{0.0,0.652500,0.0},
{0.0,0.655000,0.0},
{0.0,0.657500,0.0},
{0.0,0.660000,0.0},
{0.0,0.662500,0.0},
{0.0,0.665000,0.0},
{0.0,0.667500,0.0},
{0.0,0.670000,0.0},
{0.0,0.672500,0.0},
{0.0,0.675000,0.0},
{0.0,0.677500,0.0},
{0.0,0.680000,0.0},
{0.0,0.682500,0.0},
{0.0,0.685000,0.0},
{0.0,0.687500,0.0},
{0.0,0.690000,0.0},
{0.0,0.692500,0.0},
{0.0,0.695000,0.0},
{0.0,0.697500,0.0},
{0.0,0.700000,0.0},
{0.0,0.702500,0.0},
{0.0,0.705000,0.0},
{0.0,0.707500,0.0},
{0.0,0.710000,0.0},
{0.0,0.712500,0.0},
{0.0,0.715000,0.0},
{0.0,0.717500,0.0},
{0.0,0.720000,0.0},
{0.0,0.722500,0.0},
{0.0,0.725000,0.0},
{0.0,0.727500,0.0},
{0.0,0.730000,0.0},
{0.0,0.732500,0.0},
{0.0,0.735000,0.0},
{0.0,0.737500,0.0},
{0.0,0.740000,0.0},
{0.0,0.742500,0.0},
{0.0,0.745000,0.0},
{0.0,0.747500,0.0},
{0.0,0.750000,0.0},
{0.0,0.752500,0.0},
{0.0,0.755000,0.0},
{0.0,0.757500,0.0},
{0.0,0.760000,0.0},
{0.0,0.762500,0.0},
{0.0,0.765000,0.0},
{0.0,0.767500,0.0},
{0.0,0.770000,0.0},
{0.0,0.772500,0.0},
{0.0,0.775000,0.0},
{0.0,0.777500,0.0},
{0.0,0.780000,0.0},
{0.0,0.782500,0.0},
{0.0,0.785000,0.0},
{0.0,0.787500,0.0},
{0.0,0.790000,0.0},
{0.0,0.792500,0.0},
{0.0,0.795000,0.0},
{0.0,0.797500,0.0},
{0.0,0.800000,0.0},
{0.0,0.802500,0.0},
{0.0,0.805000,0.0},
{0.0,0.807500,0.0},
{0.0,0.810000,0.0},
{0.0,0.812500,0.0},
{0.0,0.815000,0.0},
{0.0,0.817500,0.0},
{0.0,0.820000,0.0},
{0.0,0.822500,0.0},
{0.0,0.825000,0.0},
{0.0,0.827500,0.0},
{0.0,0.830000,0.0},
{0.0,0.832500,0.0},
{0.0,0.835000,0.0},
{0.0,0.837500,0.0},
{0.0,0.840000,0.0},
{0.0,0.842500,0.0},
{0.0,0.845000,0.0},
{0.0,0.847500,0.0},
{0.0,0.850000,0.0},
{0.0,0.852500,0.0},
{0.0,0.855000,0.0},
{0.0,0.857500,0.0},
{0.0,0.860000,0.0},
{0.0,0.862500,0.0},
{0.0,0.865000,0.0},
{0.0,0.867500,0.0},
{0.0,0.870000,0.0},
{0.0,0.872500,0.0},
{0.0,0.875000,0.0},
{0.0,0.877500,0.0},
{0.0,0.880000,0.0},
{0.0,0.882500,0.0},
{0.0,0.885000,0.0},
{0.0,0.887500,0.0},
{0.0,0.890000,0.0},
{0.0,0.892500,0.0},
{0.0,0.895000,0.0},
{0.0,0.897500,0.0},
{0.0,0.900000,0.0},
{0.0,0.902500,0.0},
{0.0,0.905000,0.0},
{0.0,0.907500,0.0},
{0.0,0.910000,0.0},
{0.0,0.912500,0.0},
{0.0,0.915000,0.0},
{0.0,0.917500,0.0},
{0.0,0.920000,0.0},
{0.0,0.922500,0.0},
{0.0,0.925000,0.0},
{0.0,0.927500,0.0},
{0.0,0.930000,0.0},
{0.0,0.932500,0.0},
{0.0,0.935000,0.0},
{0.0,0.937500,0.0},
{0.0,0.940000,0.0},
{0.0,0.942500,0.0},
{0.0,0.945000,0.0},
{0.0,0.947500,0.0},
{0.0,0.950000,0.0},
{0.0,0.952500,0.0},
{0.0,0.955000,0.0},
{0.0,0.957500,0.0},
{0.0,0.960000,0.0},
{0.0,0.962500,0.0},
{0.0,0.965000,0.0},
{0.0,0.967500,0.0},
{0.0,0.970000,0.0},
{0.0,0.972500,0.0},
{0.0,0.975000,0.0},
{0.0,0.977500,0.0},
{0.0,0.980000,0.0},
{0.0,0.982500,0.0},
{0.0,0.985000,0.0},
{0.0,0.987500,0.0},
{0.0,0.990000,0.0},
{0.0,0.992500,0.0},
{0.0,0.995000,0.0},
{0.0,0.997500,0.0}
};

static TRI LUTfmri2t[] = {
{0.000000,0.000000,0.000000},
{0.010000,0.010000,0.010000},
{0.020000,0.020000,0.020000},
{0.030000,0.030000,0.030000},
{0.040000,0.040000,0.040000},
{0.050000,0.050000,0.050000},
{0.060000,0.060000,0.060000},
{0.070000,0.070000,0.070000},
{0.080000,0.080000,0.080000},
{0.090000,0.090000,0.090000},
{0.100000,0.100000,0.100000},
{0.110000,0.110000,0.110000},
{0.120000,0.120000,0.120000},
{0.130000,0.130000,0.130000},
{0.140000,0.140000,0.140000},
{0.150000,0.150000,0.150000},
{0.160000,0.160000,0.160000},
{0.170000,0.170000,0.170000},
{0.180000,0.180000,0.180000},
{0.190000,0.190000,0.190000},
{0.200000,0.200000,0.200000},
{0.210000,0.210000,0.210000},
{0.220000,0.220000,0.220000},
{0.230000,0.230000,0.230000},
{0.240000,0.240000,0.240000},
{0.250000,0.250000,0.250000},
{0.260000,0.260000,0.260000},
{0.270000,0.270000,0.270000},
{0.280000,0.280000,0.280000},
{0.290000,0.290000,0.290000},
{0.300000,0.300000,0.300000},
{0.310000,0.310000,0.310000},
{0.320000,0.320000,0.320000},
{0.330000,0.330000,0.330000},
{0.340000,0.340000,0.340000},
{0.350000,0.350000,0.350000},
{0.360000,0.360000,0.360000},
{0.370000,0.370000,0.370000},
{0.380000,0.380000,0.380000},
{0.390000,0.390000,0.390000},
{0.400000,0.400000,0.400000},
{0.410000,0.410000,0.410000},
{0.420000,0.420000,0.420000},
{0.430000,0.430000,0.430000},
{0.440000,0.440000,0.440000},
{0.450000,0.450000,0.450000},
{0.460000,0.460000,0.460000},
{0.470000,0.470000,0.470000},
{0.480000,0.480000,0.480000},
{0.490000,0.490000,0.490000},
{0.500000,0.500000,0.500000},
{0.510000,0.510000,0.510000},
{0.520000,0.520000,0.520000},
{0.530000,0.530000,0.530000},
{0.540000,0.540000,0.540000},
{0.550000,0.550000,0.550000},
{0.560000,0.560000,0.560000},
{0.570000,0.570000,0.570000},
{0.580000,0.580000,0.580000},
{0.590000,0.590000,0.590000},
{0.600000,0.600000,0.600000},
{0.610000,0.610000,0.610000},
{0.620000,0.620000,0.620000},
{0.630000,0.630000,0.630000},
{0.640000,0.640000,0.640000},
{0.650000,0.650000,0.650000},
{0.660000,0.660000,0.660000},
{0.670000,0.670000,0.670000},
{0.680000,0.680000,0.680000},
{0.690000,0.690000,0.690000},
{0.700000,0.700000,0.700000},
{0.710000,0.710000,0.710000},
{0.720000,0.720000,0.720000},
{0.730000,0.730000,0.730000},
{0.740000,0.740000,0.740000},
{0.750000,0.750000,0.750000},
{0.760000,0.760000,0.760000},
{0.770000,0.770000,0.770000},
{0.780000,0.780000,0.780000},
{0.790000,0.790000,0.790000},
{0.800000,0.800000,0.800000},
{0.810000,0.810000,0.810000},
{0.820000,0.820000,0.820000},
{0.830000,0.830000,0.830000},
{0.840000,0.840000,0.840000},
{0.850000,0.850000,0.850000},
{0.860000,0.860000,0.860000},
{0.870000,0.870000,0.870000},
{0.880000,0.880000,0.880000},
{0.890000,0.890000,0.890000},
{0.900000,0.900000,0.900000},
{0.910000,0.910000,0.910000},
{0.920000,0.920000,0.920000},
{0.930000,0.930000,0.930000},
{0.940000,0.940000,0.940000},
{0.950000,0.950000,0.950000},
{0.960000,0.960000,0.960000},
{0.970000,0.970000,0.970000},
{0.980000,0.980000,0.980000},
{0.990000,0.990000,0.990000},
{0.250000,0.000000,0.0},
{0.250000,0.000000,0.0},
{0.250000,0.000000,0.0},
{0.250000,0.000000,0.0},
{0.250000,0.000000,0.0},
{0.250000,0.000000,0.0},
{0.437500,0.000000,0.0},
{0.625000,0.000000,0.0},
{0.812500,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{1.000000,0.000000,0.0},
{0.250000,0.062500,0.0},
{0.250000,0.062500,0.0},
{0.250000,0.062500,0.0},
{0.250000,0.062500,0.0},
{0.250000,0.062500,0.0},
{0.250000,0.062500,0.0},
{0.437500,0.109375,0.0},
{0.625000,0.156250,0.0},
{0.812500,0.203125,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{1.000000,0.250000,0.0},
{0.250000,0.125000,0.0},
{0.250000,0.125000,0.0},
{0.250000,0.125000,0.0},
{0.250000,0.125000,0.0},
{0.250000,0.125000,0.0},
{0.250000,0.125000,0.0},
{0.437500,0.218750,0.0},
{0.625000,0.312500,0.0},
{0.812500,0.406250,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{1.000000,0.500000,0.0},
{0.250000,0.187500,0.0},
{0.250000,0.187500,0.0},
{0.250000,0.187500,0.0},
{0.250000,0.187500,0.0},
{0.250000,0.187500,0.0},
{0.250000,0.187500,0.0},
{0.437500,0.328125,0.0},
{0.625000,0.468750,0.0},
{0.812500,0.609375,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{1.000000,0.750000,0.0},
{0.250000,0.250000,0.0},
{0.250000,0.250000,0.0},
{0.250000,0.250000,0.0},
{0.250000,0.250000,0.0},
{0.250000,0.250000,0.0},
{0.250000,0.250000,0.0},
{0.437500,0.437500,0.0},
{0.625000,0.625000,0.0},
{0.812500,0.812500,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{1.000000,1.000000,0.0},
{0.0,0.000000,0.250000},
{0.0,0.000000,0.250000},
{0.0,0.000000,0.250000},
{0.0,0.000000,0.250000},
{0.0,0.000000,0.250000},
{0.0,0.000000,0.250000},
{0.0,0.000000,0.437500},
{0.0,0.000000,0.625000},
{0.0,0.000000,0.812500},
{0.0,0.000000,1.000000},
{0.0,0.000000,1.000000},
{0.0,0.000000,1.000000},
{0.0,0.000000,1.000000},
{0.0,0.000000,1.000000},
{0.0,0.000000,1.000000},
{0.0,0.000000,1.000000},
{0.0,0.000000,1.000000},
{0.0,0.000000,1.000000},
{0.0,0.000000,1.000000},
{0.0,0.000000,1.000000},
{0.0,0.062500,0.250000},
{0.0,0.062500,0.250000},
{0.0,0.062500,0.250000},
{0.0,0.062500,0.250000},
{0.0,0.062500,0.250000},
{0.0,0.062500,0.250000},
{0.0,0.109375,0.437500},
{0.0,0.156250,0.625000},
{0.0,0.203125,0.812500},
{0.0,0.250000,1.000000},
{0.0,0.250000,1.000000},
{0.0,0.250000,1.000000},
{0.0,0.250000,1.000000},
{0.0,0.250000,1.000000},
{0.0,0.250000,1.000000},
{0.0,0.250000,1.000000},
{0.0,0.250000,1.000000},
{0.0,0.250000,1.000000},
{0.0,0.250000,1.000000},
{0.0,0.250000,1.000000},
{0.0,0.125000,0.250000},
{0.0,0.125000,0.250000},
{0.0,0.125000,0.250000},
{0.0,0.125000,0.250000},
{0.0,0.125000,0.250000},
{0.0,0.125000,0.250000},
{0.0,0.218750,0.437500},
{0.0,0.312500,0.625000},
{0.0,0.406250,0.812500},
{0.0,0.500000,1.000000},
{0.0,0.500000,1.000000},
{0.0,0.500000,1.000000},
{0.0,0.500000,1.000000},
{0.0,0.500000,1.000000},
{0.0,0.500000,1.000000},
{0.0,0.500000,1.000000},
{0.0,0.500000,1.000000},
{0.0,0.500000,1.000000},
{0.0,0.500000,1.000000},
{0.0,0.500000,1.000000},
{0.0,0.187500,0.250000},
{0.0,0.187500,0.250000},
{0.0,0.187500,0.250000},
{0.0,0.187500,0.250000},
{0.0,0.187500,0.250000},
{0.0,0.187500,0.250000},
{0.0,0.328125,0.437500},
{0.0,0.468750,0.625000},
{0.0,0.609375,0.812500},
{0.0,0.750000,1.000000},
{0.0,0.750000,1.000000},
{0.0,0.750000,1.000000},
{0.0,0.750000,1.000000},
{0.0,0.750000,1.000000},
{0.0,0.750000,1.000000},
{0.0,0.750000,1.000000},
{0.0,0.750000,1.000000},
{0.0,0.750000,1.000000},
{0.0,0.750000,1.000000},
{0.0,0.750000,1.000000},
{0.0,0.250000,0.250000},
{0.0,0.250000,0.250000},
{0.0,0.250000,0.250000},
{0.0,0.250000,0.250000},
{0.0,0.250000,0.250000},
{0.0,0.250000,0.250000},
{0.0,0.437500,0.437500},
{0.0,0.625000,0.625000},
{0.0,0.812500,0.812500},
{0.0,1.000000,1.000000},
{0.0,1.000000,1.000000},
{0.0,1.000000,1.000000},
{0.0,1.000000,1.000000},
{0.0,1.000000,1.000000},
{0.0,1.000000,1.000000},
{0.0,1.000000,1.000000},
{0.0,1.000000,1.000000},
{0.0,1.000000,1.000000},
{0.0,1.000000,1.000000},
{0.0,1.000000,1.000000},
{0.0,0.125000,0.0},
{0.0,0.125000,0.0},
{0.0,0.125000,0.0},
{0.0,0.125000,0.0},
{0.0,0.125000,0.0},
{0.0,0.125000,0.0},
{0.0,0.125000,0.0},
{0.0,0.125000,0.0},
{0.0,0.125000,0.0},
{0.0,0.125000,0.0},
{0.0,0.125000,0.0},
{0.0,0.166667,0.0},
{0.0,0.208333,0.0},
{0.0,0.250000,0.0},
{0.0,0.291667,0.0},
{0.0,0.333333,0.0},
{0.0,0.375000,0.0},
{0.0,0.416667,0.0},
{0.0,0.458333,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.500000,0.0},
{0.0,0.156250,0.0},
{0.0,0.156250,0.0},
{0.0,0.156250,0.0},
{0.0,0.156250,0.0},
{0.0,0.156250,0.0},
{0.0,0.156250,0.0},
{0.0,0.156250,0.0},
{0.0,0.156250,0.0},
{0.0,0.156250,0.0},
{0.0,0.156250,0.0},
{0.0,0.156250,0.0},
{0.0,0.208333,0.0},
{0.0,0.260417,0.0},
{0.0,0.312500,0.0},
{0.0,0.364583,0.0},
{0.0,0.416667,0.0},
{0.0,0.468750,0.0},
{0.0,0.520833,0.0},
{0.0,0.572917,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.625000,0.0},
{0.0,0.187500,0.0},
{0.0,0.187500,0.0},
{0.0,0.187500,0.0},
{0.0,0.187500,0.0},
{0.0,0.187500,0.0},
{0.0,0.187500,0.0},
{0.0,0.187500,0.0},
{0.0,0.187500,0.0},
{0.0,0.187500,0.0},
{0.0,0.187500,0.0},
{0.0,0.187500,0.0},
{0.0,0.250000,0.0},
{0.0,0.312500,0.0},
{0.0,0.375000,0.0},
{0.0,0.437500,0.0},
{0.0,0.500000,0.0},
{0.0,0.562500,0.0},
{0.0,0.625000,0.0},
{0.0,0.687500,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.750000,0.0},
{0.0,0.218750,0.0},
{0.0,0.218750,0.0},
{0.0,0.218750,0.0},
{0.0,0.218750,0.0},
{0.0,0.218750,0.0},
{0.0,0.218750,0.0},
{0.0,0.218750,0.0},
{0.0,0.218750,0.0},
{0.0,0.218750,0.0},
{0.0,0.218750,0.0},
{0.0,0.218750,0.0},
{0.0,0.291667,0.0},
{0.0,0.364583,0.0},
{0.0,0.437500,0.0},
{0.0,0.510417,0.0},
{0.0,0.583333,0.0},
{0.0,0.656250,0.0},
{0.0,0.729167,0.0},
{0.0,0.802083,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.875000,0.0},
{0.0,0.250000,0.0},
{0.0,0.250000,0.0},
{0.0,0.250000,0.0},
{0.0,0.250000,0.0},
{0.0,0.250000,0.0},
{0.0,0.250000,0.0},
{0.0,0.250000,0.0},
{0.0,0.250000,0.0},
{0.0,0.250000,0.0},
{0.0,0.250000,0.0},
{0.0,0.250000,0.0},
{0.0,0.333333,0.0},
{0.0,0.416667,0.0},
{0.0,0.500000,0.0},
{0.0,0.583333,0.0},
{0.0,0.666667,0.0},
{0.0,0.750000,0.0},
{0.0,0.833333,0.0},
{0.0,0.916667,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0},
{0.0,1.000000,0.0}
};

static TRI LUTfmri3[] = {
{0.0,1.000000,1.0},
{0.0,0.990000,1.0},
{0.0,0.980000,1.0},
{0.0,0.970000,1.0},
{0.0,0.960000,1.0},
{0.0,0.950000,1.0},
{0.0,0.940000,1.0},
{0.0,0.930000,1.0},
{0.0,0.920000,1.0},
{0.0,0.910000,1.0},
{0.0,0.900000,1.0},
{0.0,0.890000,1.0},
{0.0,0.880000,1.0},
{0.0,0.870000,1.0},
{0.0,0.860000,1.0},
{0.0,0.850000,1.0},
{0.0,0.840000,1.0},
{0.0,0.830000,1.0},
{0.0,0.820000,1.0},
{0.0,0.810000,1.0},
{0.0,0.800000,1.0},
{0.0,0.790000,1.0},
{0.0,0.780000,1.0},
{0.0,0.770000,1.0},
{0.0,0.760000,1.0},
{0.0,0.750000,1.0},
{0.0,0.740000,1.0},
{0.0,0.730000,1.0},
{0.0,0.720000,1.0},
{0.0,0.710000,1.0},
{0.0,0.700000,1.0},
{0.0,0.690000,1.0},
{0.0,0.680000,1.0},
{0.0,0.670000,1.0},
{0.0,0.660000,1.0},
{0.0,0.650000,1.0},
{0.0,0.640000,1.0},
{0.0,0.630000,1.0},
{0.0,0.620000,1.0},
{0.0,0.610000,1.0},
{0.0,0.600000,1.0},
{0.0,0.590000,1.0},
{0.0,0.580000,1.0},
{0.0,0.570000,1.0},
{0.0,0.560000,1.0},
{0.0,0.550000,1.0},
{0.0,0.540000,1.0},
{0.0,0.530000,1.0},
{0.0,0.520000,1.0},
{0.0,0.510000,1.0},
{0.0,0.500000,1.0},
{0.0,0.490000,1.0},
{0.0,0.480000,1.0},
{0.0,0.470000,1.0},
{0.0,0.460000,1.0},
{0.0,0.450000,1.0},
{0.0,0.440000,1.0},
{0.0,0.430000,1.0},
{0.0,0.420000,1.0},
{0.0,0.410000,1.0},
{0.0,0.400000,1.0},
{0.0,0.390000,1.0},
{0.0,0.380000,1.0},
{0.0,0.370000,1.0},
{0.0,0.360000,1.0},
{0.0,0.350000,1.0},
{0.0,0.340000,1.0},
{0.0,0.330000,1.0},
{0.0,0.320000,1.0},
{0.0,0.310000,1.0},
{0.0,0.300000,1.0},
{0.0,0.290000,1.0},
{0.0,0.280000,1.0},
{0.0,0.270000,1.0},
{0.0,0.260000,1.0},
{0.0,0.250000,1.0},
{0.0,0.240000,1.0},
{0.0,0.230000,1.0},
{0.0,0.220000,1.0},
{0.0,0.210000,1.0},
{0.0,0.200000,1.0},
{0.0,0.190000,1.0},
{0.0,0.180000,1.0},
{0.0,0.170000,1.0},
{0.0,0.160000,1.0},
{0.0,0.150000,1.0},
{0.0,0.140000,1.0},
{0.0,0.130000,1.0},
{0.0,0.120000,1.0},
{0.0,0.110000,1.0},
{0.0,0.100000,1.0},
{0.0,0.090000,1.0},
{0.0,0.080000,1.0},
{0.0,0.070000,1.0},
{0.0,0.060000,1.0},
{0.0,0.050000,1.0},
{0.0,0.040000,1.0},
{0.0,0.030000,1.0},
{0.0,0.020000,1.0},
{0.0,0.010000,1.0},
{0.000000,0.000000,1.000000},
{0.010000,0.010000,0.990000},
{0.020000,0.020000,0.980000},
{0.030000,0.030000,0.970000},
{0.040000,0.040000,0.960000},
{0.050000,0.050000,0.950000},
{0.060000,0.060000,0.940000},
{0.070000,0.070000,0.930000},
{0.080000,0.080000,0.920000},
{0.090000,0.090000,0.910000},
{0.100000,0.100000,0.900000},
{0.110000,0.110000,0.890000},
{0.120000,0.120000,0.880000},
{0.130000,0.130000,0.870000},
{0.140000,0.140000,0.860000},
{0.150000,0.150000,0.850000},
{0.160000,0.160000,0.840000},
{0.170000,0.170000,0.830000},
{0.180000,0.180000,0.820000},
{0.190000,0.190000,0.810000},
{0.200000,0.200000,0.800000},
{0.210000,0.210000,0.790000},
{0.220000,0.220000,0.780000},
{0.230000,0.230000,0.770000},
{0.240000,0.240000,0.760000},
{0.250000,0.250000,0.750000},
{0.260000,0.260000,0.740000},
{0.270000,0.270000,0.730000},
{0.280000,0.280000,0.720000},
{0.290000,0.290000,0.710000},
{0.300000,0.300000,0.700000},
{0.310000,0.310000,0.690000},
{0.320000,0.320000,0.680000},
{0.330000,0.330000,0.670000},
{0.340000,0.340000,0.660000},
{0.350000,0.350000,0.650000},
{0.360000,0.360000,0.640000},
{0.370000,0.370000,0.630000},
{0.380000,0.380000,0.620000},
{0.390000,0.390000,0.610000},
{0.400000,0.400000,0.600000},
{0.410000,0.410000,0.590000},
{0.420000,0.420000,0.580000},
{0.430000,0.430000,0.570000},
{0.440000,0.440000,0.560000},
{0.450000,0.450000,0.550000},
{0.460000,0.460000,0.540000},
{0.470000,0.470000,0.530000},
{0.480000,0.480000,0.520000},
{0.490000,0.490000,0.510000},
{0.500000,0.500000,0.500000},
{0.510000,0.490000,0.490000},
{0.520000,0.480000,0.480000},
{0.530000,0.470000,0.470000},
{0.540000,0.460000,0.460000},
{0.550000,0.450000,0.450000},
{0.560000,0.440000,0.440000},
{0.570000,0.430000,0.430000},
{0.580000,0.420000,0.420000},
{0.590000,0.410000,0.410000},
{0.600000,0.400000,0.400000},
{0.610000,0.390000,0.390000},
{0.620000,0.380000,0.380000},
{0.630000,0.370000,0.370000},
{0.640000,0.360000,0.360000},
{0.650000,0.350000,0.350000},
{0.660000,0.340000,0.340000},
{0.670000,0.330000,0.330000},
{0.680000,0.320000,0.320000},
{0.690000,0.310000,0.310000},
{0.700000,0.300000,0.300000},
{0.710000,0.290000,0.290000},
{0.720000,0.280000,0.280000},
{0.730000,0.270000,0.270000},
{0.740000,0.260000,0.260000},
{0.750000,0.250000,0.250000},
{0.760000,0.240000,0.240000},
{0.770000,0.230000,0.230000},
{0.780000,0.220000,0.220000},
{0.790000,0.210000,0.210000},
{0.800000,0.200000,0.200000},
{0.810000,0.190000,0.190000},
{0.820000,0.180000,0.180000},
{0.830000,0.170000,0.170000},
{0.840000,0.160000,0.160000},
{0.850000,0.150000,0.150000},
{0.860000,0.140000,0.140000},
{0.870000,0.130000,0.130000},
{0.880000,0.120000,0.120000},
{0.890000,0.110000,0.110000},
{0.900000,0.100000,0.100000},
{0.910000,0.090000,0.090000},
{0.920000,0.080000,0.080000},
{0.930000,0.070000,0.070000},
{0.940000,0.060000,0.060000},
{0.950000,0.050000,0.050000},
{0.960000,0.040000,0.040000},
{0.970000,0.030000,0.030000},
{0.980000,0.020000,0.020000},
{0.990000,0.010000,0.010000},
{1.0,0.000000,0.0},
{1.0,0.010000,0.0},
{1.0,0.020000,0.0},
{1.0,0.030000,0.0},
{1.0,0.040000,0.0},
{1.0,0.050000,0.0},
{1.0,0.060000,0.0},
{1.0,0.070000,0.0},
{1.0,0.080000,0.0},
{1.0,0.090000,0.0},
{1.0,0.100000,0.0},
{1.0,0.110000,0.0},
{1.0,0.120000,0.0},
{1.0,0.130000,0.0},
{1.0,0.140000,0.0},
{1.0,0.150000,0.0},
{1.0,0.160000,0.0},
{1.0,0.170000,0.0},
{1.0,0.180000,0.0},
{1.0,0.190000,0.0},
{1.0,0.200000,0.0},
{1.0,0.210000,0.0},
{1.0,0.220000,0.0},
{1.0,0.230000,0.0},
{1.0,0.240000,0.0},
{1.0,0.250000,0.0},
{1.0,0.260000,0.0},
{1.0,0.270000,0.0},
{1.0,0.280000,0.0},
{1.0,0.290000,0.0},
{1.0,0.300000,0.0},
{1.0,0.310000,0.0},
{1.0,0.320000,0.0},
{1.0,0.330000,0.0},
{1.0,0.340000,0.0},
{1.0,0.350000,0.0},
{1.0,0.360000,0.0},
{1.0,0.370000,0.0},
{1.0,0.380000,0.0},
{1.0,0.390000,0.0},
{1.0,0.400000,0.0},
{1.0,0.410000,0.0},
{1.0,0.420000,0.0},
{1.0,0.430000,0.0},
{1.0,0.440000,0.0},
{1.0,0.450000,0.0},
{1.0,0.460000,0.0},
{1.0,0.470000,0.0},
{1.0,0.480000,0.0},
{1.0,0.490000,0.0},
{1.0,0.500000,0.0},
{1.0,0.510000,0.0},
{1.0,0.520000,0.0},
{1.0,0.530000,0.0},
{1.0,0.540000,0.0},
{1.0,0.550000,0.0},
{1.0,0.560000,0.0},
{1.0,0.570000,0.0},
{1.0,0.580000,0.0},
{1.0,0.590000,0.0},
{1.0,0.600000,0.0},
{1.0,0.610000,0.0},
{1.0,0.620000,0.0},
{1.0,0.630000,0.0},
{1.0,0.640000,0.0},
{1.0,0.650000,0.0},
{1.0,0.660000,0.0},
{1.0,0.670000,0.0},
{1.0,0.680000,0.0},
{1.0,0.690000,0.0},
{1.0,0.700000,0.0},
{1.0,0.710000,0.0},
{1.0,0.720000,0.0},
{1.0,0.730000,0.0},
{1.0,0.740000,0.0},
{1.0,0.750000,0.0},
{1.0,0.760000,0.0},
{1.0,0.770000,0.0},
{1.0,0.780000,0.0},
{1.0,0.790000,0.0},
{1.0,0.800000,0.0},
{1.0,0.810000,0.0},
{1.0,0.820000,0.0},
{1.0,0.830000,0.0},
{1.0,0.840000,0.0},
{1.0,0.850000,0.0},
{1.0,0.860000,0.0},
{1.0,0.870000,0.0},
{1.0,0.880000,0.0},
{1.0,0.890000,0.0},
{1.0,0.900000,0.0},
{1.0,0.910000,0.0},
{1.0,0.920000,0.0},
{1.0,0.930000,0.0},
{1.0,0.940000,0.0},
{1.0,0.950000,0.0},
{1.0,0.960000,0.0},
{1.0,0.970000,0.0},
{1.0,0.980000,0.0},
{1.0,0.990000,0.0}
};

static string fmriLUT[] = {"render1", "render1t", "render2", "render2t", "render3"};

static map<string, TRI*> LUTs;
static map<string, int> LUTsize;

static int init_lut()
{
	LUTs["render1"] = LUTfmri1;  LUTsize["render1"] = sizeof(LUTfmri1) / sizeof(LUTfmri1[0]);
	LUTs["render1t"] = LUTfmri1t;  LUTsize["render1t"] = sizeof(LUTfmri1t) / sizeof(LUTfmri1t[0]);
	LUTs["render2"] = LUTfmri2;  LUTsize["render2"] = sizeof(LUTfmri2) / sizeof(LUTfmri2[0]);
	LUTs["render2t"] = LUTfmri2t;  LUTsize["render2t"] = sizeof(LUTfmri2t) / sizeof(LUTfmri2t[0]);
	LUTs["render3"] = LUTfmri3;  LUTsize["render3"] = sizeof(LUTfmri3) / sizeof(LUTfmri3[0]);

	return 0;
}

static int dummy_lut=init_lut();

bool LUTexsited(string lut) { return LUTs.find(lut) != LUTs.end(); }

ZImageBase* GrayToRGB(const ZImageBase* image, string lut)
{
	if(image->isColor()) return (ZImageBase*)image;

	ZImageBase* pimage = 0;

	if(lut.empty() || LUTs.find(lut) == LUTs.end())
	{
		pimage = RGBTypeCopyFrom(image);
		image->CopyPixelsTo(*pimage);
	}
	else
	{
		pimage = new ZRGBByteImage(image->Width(), image->Height(), image->Depth());
		float min=0, max=255;
		image->GetDisplayRange(min, max);
		if(max < min) swap(min, max);
		else if(max == min) 
		{
			image->MinMax(min, max);
			if(max == min) 
			{
				image->CopyPixelsTo(*pimage);
				return pimage;
			}
		}

		TRI* luts = LUTs[lut]; int size = LUTsize[lut];

		float nor = float(size) / (max - min);

		float x, y, z;
		image->GetPixelDim(x, y, z);
		pimage->SetPixelDim(x, y, z);

		ZRGB<BYTE> t(luts[size-1][0]*255, luts[size-1][1]*255, luts[size-1][2]*255);
		ZRGB<BYTE> b(luts[0][0]*255, luts[0][1]*255, luts[0][2]*255);

		ZRGBByteImage::iterator p_image(*(ZRGBByteImage*)pimage);
		for(UINT k=0; k<image->Depth(); k++)
		for(UINT j=0; j<image->Height(); j++)
		for(UINT i=0; i<image->Width(); i++, p_image++)
		{
			float v = image->GetIntensity(i, j, k);
			if(v >= max) *p_image = t;
			else if(v <= min) *p_image = b;
			else 
			{
				int l = int(nor * (v-min));
				p_image->set(BYTE(luts[l][0]*255), BYTE(luts[l][1]*255), BYTE(luts[l][2]*255));
			}
		}
	}

	return pimage;
}
