/*
 *   mwmpool.c -- Mwave Modem AT Command Parser
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
/*****************************************************************************/
/* Routines to work with V.42bis / V.34 Memory Pool.                         */
/*****************************************************************************/

// Original Design document....
// 4/21/95    V.42bis/Datapump shared memory design.
// 4/25/95    Change BLKDPREQ <> 0 to BLKDPREQ>=0 for MCTL to IPC 35
//
//
// Terminology:
// Block      = 768 contiguous data words.
// Page       = 256 contiguous data words.
// CORECHUNK  = 1792 contiguous words including the first 2 Blocks.
//
//
// PROTSUPR BLKPTR_ITCB layout:
//
//
//     BLKTOT        Total Number of allocated blocks.set by driver
//     BLKV34        DP used blocks                   set by driver
//     BLKV42B                                        set by driver
//     BLKDPREQ   ** New. Number of blocks requested by datapump.  set by dp
//
//     Pointer Table:
//
//           |-----------------|
//           | Page 0          |   Special page used by V.32bis and V.34.
//  BLKPTRP->|-----------------|
//           | Block 1         |
//           |-----------------|   Page 0, Block 1, and Block 2 form a
//           | Block 1 + 256   |   1792 word buffer and will be allocated
//           |-----------------|   as one contiguous segment before the
//           | Block 1 + 512   |   datapumps are loaded.
//           |-----------------|
//           | Block 2         |
//           |-----------------|
//           | Block 2 + 256   |
//           |-----------------|
//           | Block 2 + 512   |
//           |-----------------|    All other blocks are 768 words
//                   .              with page pointers to each 256 words
//                   .              in BLKPTRP table.
//                   .
//           |-----------------|
//           | Block 15        |
//           |-----------------|
//           | Block 15 + 256  |
//           |-----------------|
//           | Block 15 + 512  |
//           |-----------------|
//
//
// Algorithm
//
//
//     Driver                     MCTL                      Datapump
// ==============================================================================
//
// ------------------------------IPC 15------------------------------------------
// Allocate CORECHUNK
// Set BLKV34 = 2
// Set BLKDPREQ = -1
// Load Pumps
// Set V.32bis TIASPTR
// ----End IPC 15----------------------------------------------------------------
//                                   .
//                                   .
//                                   .
// ------------------------------IPC 6 ------------------------------------------
// Free unused pump(s)
// Load V42/MNP ***
// Alloc INI specified blks
// Set BLKV34=BLKTOT
//                                                        Set BLKDPREQ
// ----End IPC 6 ----------------------------------
//                            Wait for BLKDPREQ >= 0
// ------------------------------IPC 35------------
// Set BLKV34=min(BLKTOT,BLKDPREQ)
// Determine V42bis desired size
// Guess V42b max avail size                              Start using buffer
//                                                        (use BLKV34 blocks)****
// ----End IPC 35----------------------------------------------------------------
//                            Tell MNP/V42 GO!
//                                  .
//                                  .
//                                  .
// ------------------------------IPC 13------------------------------------------
// Remove unused protocol
// Alloc more v42b blocks if needed
// Dealloc v42b blocks if too many
// Load v42bis
// ----End IPC 13----------------------------------------------------------------
//

/*#include <pbmplus.h>*/

#include <mwmspcfc.h>
#include <mwmcntnd.h>

static char szThisFile[] = "MWMPOOL.C";

#define MWM_POOL_MAXSEGMENTS 15 // Max Number of 768 Word Segments.


/*****************************************************************************/
/* If a control word is added before the block of pointers, this constant    */
/* needs to be increased.  This constant is used by                          */
/* mwmPoolTransferControlBlock when reading/writing only the array of        */
/* Segment pointers portion of the Control Block.                            */
/*****************************************************************************/
#define MWM_POOL_CONTROL_WORDS        6
#define MWM_POOL_OFFSET_TO_POINTERS   8
typedef struct
{
   USHORT       usTotalBlocks;   // Total Number of Blocks Allocated
   USHORT       usV34Blocks;     // Number of Blocks in use by V.34
   USHORT       usV42bisBlocks;  // Number of Blocks in use by V.42bis
   USHORT       usDPRequestedBlocks;// Blocks requested by Datapump
   USHORT       usBasePageAddress;// Address of first oddball page.
   USHORT       aSegAddress[MWM_POOL_MAXSEGMENTS*3];
} MWM_MEMPOOL, *PMWM_MEMPOOL;


#define MWM_ALLOC_ADD        1
#define MWM_ALLOC_SUBTRACT   2
#define MWM_ALLOC_CLEARALL   4


/*****************************************************************************/
/* Constants for Control flags of mwmPoolTransferControlBlock                */
/*****************************************************************************/
#define CONTROL_WORDS        1
#define SEGMENT_POINTERS     2
#define DP_REQUEST_WORD      4


#define NUM_V34_DYNASEGS     4


ULONG mwmPoolTransferControlBlock(PMWM_DSPINFO pmwmDspInfo,
                                  PMWM_MEMPOOL pmwmMemPool,
                                  PULONG pulAddress,
                                  USHORT usControlFlags,        // Work with Control words, pointer, or both.
                                  USHORT usMemTransferFlags);   // Read or Write flags for memTransfer
ULONG mwmPoolChangeAllocation(PMWM_DSPINFO pmwmDspInfo,
                              USHORT *pusNumSegments,      // Number of 768 Word Segments
                              USHORT usFlags);
ULONG mwmPoolGetMaxConfiguredDictSize(PMWM_DSPINFO pmwmDspInfo,
                                      USHORT *pusSegmentsNeeded,
                                      USHORT *pusCompressBothWays);



ULONG mwmPoolAllocateCoreChunk(PMWM_DSPINFO pmwmDspInfo, USHORT *pusCoreChunkAddress)
{
  ULONG ulRC;
  MWM_MEMPOOL mwmMemPool;
  ULONG       ulAddress = 0;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolAllocateCoreChunk entry\n");  }


  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;


  memset(&mwmMemPool,0,sizeof(mwmMemPool));

  /***************************************************************************/
  /* Allocate a 1792 word contiguous buffer.                                 */
  /***************************************************************************/
  ulRC =  mwmCntndAllocSegment(pmwmDspInfo,
                               pmwmDspInfo->hmtProtSupr,
                               "MPOOLCOR",
                               1792,
                               1,
                               ulAddress + MWM_POOL_OFFSET_TO_POINTERS,
                               &pmwmDspInfo->ahsegMemPool[0]);

  if (ulRC)
    return ulRC;

  /***************************************************************************/
  /* Read Pointer Array portion of Control Block from the DSP                */
  /* to get usBasePageAddress ....                                           */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;



  *pusCoreChunkAddress = mwmMemPool.usBasePageAddress;

  mwmMemPool.aSegAddress[0] = mwmMemPool.usBasePageAddress + 512;  // 256 Words
  mwmMemPool.aSegAddress[1] = mwmMemPool.usBasePageAddress + 1024; // 512 Words
  mwmMemPool.aSegAddress[2] = mwmMemPool.usBasePageAddress + 1536; // 768 Words
  mwmMemPool.aSegAddress[3] = mwmMemPool.usBasePageAddress + 2048; // 1024 Words
  mwmMemPool.aSegAddress[4] = mwmMemPool.usBasePageAddress + 2560; // 1280 Words
  mwmMemPool.aSegAddress[5] = mwmMemPool.usBasePageAddress + 3072; // 1536 Words

  mwmMemPool.usV34Blocks         = 2;
  mwmMemPool.usTotalBlocks       = 2;
  mwmMemPool.usV42bisBlocks      = 0;
  mwmMemPool.usDPRequestedBlocks = 0xffff;

  /***************************************************************************/
  /* Write Entire Control Block to the DSP                                   */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | DP_REQUEST_WORD | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_WRITE );
  if (ulRC)
    return ulRC;

  #ifdef DEBUG
  ulAddress = 0;
  /***************************************************************************/
  /* For debug...dump the Memory Pool pointers and control block to pmprintf */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | DP_REQUEST_WORD | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;


  DPF("Total Blocks: %d  DP Blocks: %d  V.42bis Blocks: %d  DP Requested: %d",
      mwmMemPool.usTotalBlocks,
      mwmMemPool.usV34Blocks,
      mwmMemPool.usV42bisBlocks,
      mwmMemPool.usDPRequestedBlocks);

  #endif


	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolAllocateCoreChunk exit\n");  }



  return 0;
}





ULONG mwmPoolFreeCoreChunk(PMWM_DSPINFO pmwmDspInfo)
{

  ULONG ulRC;
  MWM_MEMPOOL mwmMemPool;
  ULONG       ulAddress = 0;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolFreeCoreChunk entry\n");  }


  memset(&mwmMemPool,0,sizeof(mwmMemPool));

  /***********************************************************************/
  /* Free the segment.                                                   */
  /***********************************************************************/
  ulRC = dspFreeSegment(pmwmDspInfo->ahsegMemPool[0]);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }
  pmwmDspInfo->ahsegMemPool[0] = 0;


  mwmMemPool.usDPRequestedBlocks = 0xffff;
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | DP_REQUEST_WORD | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_WRITE );
  if (ulRC)
    return ulRC;


	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolFreeCoreChunk exit\n");  }


  return 0;
}







/*****************************************************************************/
/* Called AT IPC 6                                                           */
/*                                                                           */
/* - Checks the INI file.                                                    */
/* - Allocates min(INI#,# that fits)                                         */
/*****************************************************************************/
ULONG mwmPoolAllocInitialECBuffer(PMWM_DSPINFO pmwmDspInfo)
{
  ULONG ulRC = 0;
  ULONG  ulDesiredECBlocks = 0;
  MWM_MEMPOOL mwmMemPool;
  MWM_MODEM_STATUS   ModemStatus;
  ULONG   ulAddressSTATUS = 0;
  ULONG ulAddress = 0;


	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolAllocInitialECBuffer entry\n");  }


  /***************************************************************************/
  /* Read INI file to see how many total blocks should be allocated          */
  /* up front for Echo Cancellation...                                       */
  /* This number is the total number of blocks available for DP ECBuffer use */
  /* If the INI file specifies 2 or less, then we don't allocate anything    */
  /* here because CORECHUNK was allocated earlier.                           */
  /***************************************************************************/
  ulDesiredECBlocks = GetPrivateProfileInt("FEATURES","ECBLOCKS",
                                           0,INIFile );


  if (ulDesiredECBlocks)
  {


    /*************************************************************************/
    /* Don't request more than we have allocated pointers for.               */
    /*************************************************************************/
    ulDesiredECBlocks = min(ulDesiredECBlocks,MWM_POOL_MAXSEGMENTS-2);


    /*************************************************************************/
    /* We already have 2 blocks allocated (CORECHUNK), so don't try to make  */
    /* an allocation of 0, 1, or 2.                                          */
    /*************************************************************************/
    if (ulDesiredECBlocks > 2)
    {

      /************************************************************
      ** Before trying to allocate EC buffers, load up data store
      ** with dummy V.34 segments. These segments will be loaded
      ** if DynaSegs are enabled, and we don't want assumptions
      ** to be made about extra space being available.
      ** MTS 7795 - Use the internal DynaSegm value instead of
      ** reading the value from the INI file.   02/17/97 DR
      *************************************************************/
      ulRC = mwmSpcfcQueryModemStatus(pmwmDspInfo,&ModemStatus,&ulAddressSTATUS);
      if (ulRC != DSP_NOERROR) {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__, MWM_DSP_ERROR, ulRC );
        return ulRC;
      } /* endif */

      if ( ModemStatus.usProtocol & MWM_STATUS_V34_CONNECT ) {
        if ( pmwmDspInfo->usDynaSegmValue ) {
          ulRC = mwmPoolDummyV34Segments( pmwmDspInfo, 1 );
          if (ulRC)
            return ulRC;
        } /* endif */
      } /* endif */

      ulDesiredECBlocks -=2;
      /***********************************************************************/
      /* Try to allocate the desired number of ECBLOCKS....                  */
      /***********************************************************************/
      ulRC =  mwmPoolChangeAllocation(pmwmDspInfo,
                                      (USHORT *)&ulDesiredECBlocks,
                                      MWM_ALLOC_ADD);
    }
    if (ulRC)
      return ulRC;

  }


  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;


  mwmMemPool.usV34Blocks = mwmMemPool.usTotalBlocks;

  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS,
                                     DSP_MEMXFER_DATA_WRITE );
  if (ulRC)
    return ulRC;



  #ifdef DEBUG
  /***************************************************************************/
  /* For debug...dump the Memory Pool pointers and control block to pmprintf */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | DP_REQUEST_WORD | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;


  DPF("Total Blocks: %d  V.34 Blocks: %d  V.42bis Blocks: %d DP Requested: %d",
      mwmMemPool.usTotalBlocks,
      mwmMemPool.usV34Blocks,
      mwmMemPool.usV42bisBlocks,
      mwmMemPool.usDPRequestedBlocks);

  #endif

  /************************************************************
  ** Free the space reserved for V.34 Dummy Segments.
  *************************************************************/
  ulRC = mwmPoolDummyV34Segments( pmwmDspInfo, 2 );
  if (ulRC)
    return(ulRC);

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolAllocInitialECBuffer exit\n");  }


  return 0;
}






ULONG mwmPoolFreeV42bisDictionary(PMWM_DSPINFO pmwmDspInfo)
{
  ULONG ulRC;
  USHORT usV42bisV34Segments = 0;
  ULONG  ulAddress = 0;
  MWM_MEMPOOL mwmMemPool;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolFreeV42bisDictionary entry\n");  }


  /***************************************************************************/
  /* Now, see how many segments are allocated                                */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;

//  usV42bisV34Segments = mwmMemPool.usTotalBlocks - mwmMemPool.usV34Blocks;
  usV42bisV34Segments = mwmMemPool.usTotalBlocks - 2;


  /***************************************************************************/
  /* Free the V.42bis segments.                                              */
  /***************************************************************************/
  if (usV42bisV34Segments)
  {
    ulRC =  mwmPoolChangeAllocation(pmwmDspInfo,
                                    &usV42bisV34Segments,
                                    MWM_ALLOC_SUBTRACT);
    if (ulRC)
      return ulRC;
  }


  /***************************************************************************/
  /* Write clear the "Number of segments used by V.42bis"                    */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS ,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;

  mwmMemPool.usV42bisBlocks = 0;

  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS,
                                     DSP_MEMXFER_DATA_WRITE );
  if (ulRC)
    return ulRC;


	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolFreeV42bisDictionary exit\n");  }


  return 0;
}







ULONG mwmPoolCleanupAfterNegotiation(PMWM_DSPINFO pmwmDspInfo, MWM_MODEM_STATUS *pModemStatus)
{
  ULONG ulRC;
  MWM_MEMPOOL mwmMemPool;
  USHORT usNeededSegments = 0;
  ULONG  ulAddress = 0;
  USHORT usExtraSegments = 0;


	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolCleanupAfterNegotiation entry\n");  }



  DPF("Actual V.42bis Dictionary Size (NEGDICT): %d",pModemStatus->usV42bisDictionarySize);


  /***************************************************************************/
  /* At this point, assume a 1 direction compression.  We will check later,  */
  /* and double the needed segments if it is bidirectional.                  */
  /***************************************************************************/
  if (pModemStatus->usV42bisDictionarySize == 0)
    usNeededSegments = 0;
  else
  if (pModemStatus->usV42bisDictionarySize == 512)
    usNeededSegments = 1;
  else
  if (pModemStatus->usV42bisDictionarySize <= 768)
    usNeededSegments = 2;
  else
  if (pModemStatus->usV42bisDictionarySize <= 1024)
    usNeededSegments = 3;
  else
  if (pModemStatus->usV42bisDictionarySize <= 1280)
    usNeededSegments = 4;
  else
  if (pModemStatus->usV42bisDictionarySize <= 1536)
    usNeededSegments = 5;
  else
  if (pModemStatus->usV42bisDictionarySize <= 1792)
    usNeededSegments = 6;
  else
  if (pModemStatus->usV42bisDictionarySize <= 2048)
    usNeededSegments = 7;
  else
  if (pModemStatus->usV42bisDictionarySize <= 2304)
    usNeededSegments = 8;
  else
  if (pModemStatus->usV42bisDictionarySize <= 2560)
    usNeededSegments = 9;
  else
  if (pModemStatus->usV42bisDictionarySize <= 2816)
    usNeededSegments = 10;
  else
  if (pModemStatus->usV42bisDictionarySize <= 3072)
    usNeededSegments = 11;
  else
  if (pModemStatus->usV42bisDictionarySize <= 3328)
    usNeededSegments = 12;
  else
  if (pModemStatus->usV42bisDictionarySize <= 3584)
    usNeededSegments = 13;
  else
  if (pModemStatus->usV42bisDictionarySize <= 3840)
    usNeededSegments = 14;
  else
  if (pModemStatus->usV42bisDictionarySize <= 4096)
    usNeededSegments = 15;


  /***************************************************************************/
  /* is compression bidirectional?  is it on at all?                         */
  /***************************************************************************/
  switch (pModemStatus->usProtocol & 0x0038)
  {
    case 0:
      /***********************************************************************/
      /* No V.42bis at all.                                                  */
      /***********************************************************************/
      usNeededSegments = 0;
      break;
    case 8:
      /***********************************************************************/
      /* Bidirectional...uses twice as many segments.                        */
      /***********************************************************************/
      usNeededSegments *= 2;
      break;
    case 0x10:
    case 0x20:
      /***********************************************************************/
      /* One Direction compression...leave the number of segments as is.     */
      /***********************************************************************/
      break;
  }


  /***************************************************************************/
  /* Now, see how many segments are allocated                                */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;


  if (mwmMemPool.usTotalBlocks - mwmMemPool.usV34Blocks > usNeededSegments)
  {
    usExtraSegments = mwmMemPool.usTotalBlocks - mwmMemPool.usV34Blocks - usNeededSegments;

    /*************************************************************************/
    /* If we happened to get some segments that we didn't need.              */
    /*************************************************************************/
    ulRC =  mwmPoolChangeAllocation(pmwmDspInfo,
                                    &usExtraSegments,
                                    MWM_ALLOC_SUBTRACT);
    if (ulRC)
      return ulRC;
  }
  else
  {
    usExtraSegments = usNeededSegments - mwmMemPool.usTotalBlocks + mwmMemPool.usV34Blocks ;

    /*************************************************************************/
    /* Allocate more segments to fill in the gap.                            */
    /*************************************************************************/
    ulRC =  mwmPoolChangeAllocation(pmwmDspInfo,
                                    &usExtraSegments,
                                    MWM_ALLOC_ADD);
    if (ulRC)
      return ulRC;
  }


  /***************************************************************************/
  /* Write in the "Number of segments used by V.42bis"                       */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;

  /***************************************************************************/
  /* 04/05/99 Gracefully disconnect if could not allocatate dictionary.      */
  /***************************************************************************/
  if (mwmMemPool.usTotalBlocks - mwmMemPool.usV34Blocks < usNeededSegments)
  {
     return DSP_INSUFF_DATA_MEM;
  }

  mwmMemPool.usV42bisBlocks = usNeededSegments;

  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_WRITE );
  if (ulRC)
    return ulRC;





  #ifdef DEBUG
  /***************************************************************************/
  /* For debug...dump the Memory Pool pointers and control block to pmprintf */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | DP_REQUEST_WORD | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;


  DPF("Total Blocks: %d  V.34 Blocks: %d  V.42bis Blocks: %d  DP Requested: %d",
      mwmMemPool.usTotalBlocks,
      mwmMemPool.usV34Blocks,
      mwmMemPool.usV42bisBlocks,
      mwmMemPool.usDPRequestedBlocks);


  #endif


  if (pmwmDspInfo->hsegV42BTempGPC1)
  {
    ulRC = dspFreeSegment(pmwmDspInfo->hsegV42BTempGPC1);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
    pmwmDspInfo->hsegV42BTempGPC1 = 0;
  }

  if (pmwmDspInfo->hsegV42BTempGPC2)
  {
    ulRC = dspFreeSegment(pmwmDspInfo->hsegV42BTempGPC2);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
    pmwmDspInfo->hsegV42BTempGPC2 = 0;
  }

  if (pmwmDspInfo->hsegV42BTempDSEG)
  {
    ulRC = dspFreeSegment(pmwmDspInfo->hsegV42BTempDSEG);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
    pmwmDspInfo->hsegV42BTempDSEG = 0;
  }

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolCleanupAfterNegotiation exit\n");  }


  return 0;
}






ULONG mwmPoolGuessMaxV42bisEntries(PMWM_DSPINFO pmwmDspInfo)
{
  ULONG   ulRC = 0;
  USHORT  usSegmentsNeeded     = 0;
  USHORT  usSegmentsAvailable  = 0;
  USHORT  usSegmentsAcquired   = 0;
  USHORT  usCompressBothWays   = 0;
  USHORT  usBestDictionarySize = 0;
  USHORT  usExtraSegments      = 0;
  USHORT  usDPSegmentsToFree   = 0;
  USHORT  usSegmentsInUseByProtocol = 0;
  ULONG   ulBraveV42BGuess = 0;
  ULONG   ulAddress = 0;
  MWM_MEMPOOL mwmMemPool;
  MWM_MODEM_STATUS   ModemStatus;
  ULONG   ulAddressSTATUS = 0;


	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolGuessMaxV42bisEntries entry\n");  }


  /***************************************************************************/
  /*  Set BLKV34=min(BLKTOT,BLKDPREQ)                                        */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | DP_REQUEST_WORD,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;

  mwmMemPool.usV34Blocks = min(mwmMemPool.usTotalBlocks,mwmMemPool.usDPRequestedBlocks);

  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS,
                                     DSP_MEMXFER_DATA_WRITE );
  if (ulRC)
    return ulRC;


  /***************************************************************************/
  /* Free all of the blocks not used by the datapumps.                       */
  /* (excluding the CORECHUNK)                                               */
  /***************************************************************************/
  usDPSegmentsToFree = mwmMemPool.usTotalBlocks - mwmMemPool.usV34Blocks;
  if (usDPSegmentsToFree)
  {
    ulRC =  mwmPoolChangeAllocation(pmwmDspInfo,
                                    &usDPSegmentsToFree,
                                    MWM_ALLOC_SUBTRACT);
  }

  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;


  DPF("Total Blocks: %d  V.34 Blocks: %d  V.42bis Blocks: %d  DP Requested: %d",
      mwmMemPool.usTotalBlocks,
      mwmMemPool.usV34Blocks,
      mwmMemPool.usV42bisBlocks,
      mwmMemPool.usDPRequestedBlocks);



  /***************************************************************************/
  /* Find out the maximum number of dictionary entries from AT Commands      */
  /***************************************************************************/
  if ( (pmwmDspInfo->ulFeaturesToLoad & FEATURE_MNP) ||
       (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V42) )
  {
    ulRC = mwmPoolGetMaxConfiguredDictSize(pmwmDspInfo, &usSegmentsNeeded, &usCompressBothWays);
    if (ulRC)
      return ulRC;
  }
  else
  {
    /*************************************************************************/
    /* if neither V.42 nor MNP are loaded, then we for sure don't need any   */
    /* V.42bis Dictionary...                                                 */
    /*************************************************************************/
    usSegmentsNeeded = 0;
  }


  if (usSegmentsNeeded)
  {

    /************************************************************
    ** Before attempting to allocate any space for the V.42bis
    ** space estimation, we need to fill up the space that the
    ** V.34 Data Segments will use when they are loaded.
    ** MTS 7795 - Use the internal DynaSegm value instead of
    ** reading the value from the INI file.   02/17/97 DR
    *************************************************************/
    ulRC = mwmSpcfcQueryModemStatus(pmwmDspInfo,&ModemStatus,&ulAddressSTATUS);
    if (ulRC!=DSP_NOERROR) {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    } /* endif */

    if ( ModemStatus.usProtocol & MWM_STATUS_V34_CONNECT ) {
      if ( pmwmDspInfo->usDynaSegmValue ) {
        ulRC = mwmPoolDummyV34Segments( pmwmDspInfo, 1 );
        if (ulRC)
          return ulRC;
      } /* endif */
    } /* endif */


    ulBraveV42BGuess = GetPrivateProfileInt("DEBUG","V42BGUESS", 0,
                                                   INIFile);
    /*************************************************************************/
    /* if we have a possibility of V42bis, Allocate a block of data memory   */
    /* the same size as the data memory used by V.42bis.  We will free this  */
    /* after determining how much data store is available for dictionary.    */
    /*                                                                       */
    /* If we can't get enough memory for this block, then it is certain      */
    /* that we can't get enough to establish a V.42bis dictionary.           */
    /*************************************************************************/
    if (ulBraveV42BGuess <= 2)
    {
      ulRC =  mwmCntndAllocSegment(pmwmDspInfo,
                                   pmwmDspInfo->hmtProtSupr,
                                   "V42BTEMPGPC1",
                                   256,
                                   1024,
                                   0,
                                   &pmwmDspInfo->hsegV42BTempGPC1);
      if (ulRC)
      {
        if (ulRC != DSP_INSUFF_DATA_MEM)
        {
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR,
                         ulRC);
          return ulRC;
        }
        else
        {
          DPF("Could not allocate GPC1");
          usSegmentsNeeded = 0;
        }
      }
    }


    if (ulBraveV42BGuess == 0)
    {
      ulRC =  mwmCntndAllocSegment(pmwmDspInfo,
                                   pmwmDspInfo->hmtProtSupr,
                                   "V42BTEMPGPC2",
                                   256,
                                   1024,
                                   0,
                                   &pmwmDspInfo->hsegV42BTempGPC2);
      if (ulRC)
      {
        if (ulRC != DSP_INSUFF_DATA_MEM)
        {
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR,
                         ulRC);
          return ulRC;
        }
        else
        {
          DPF("Could not allocate GPC2");
          usSegmentsNeeded = 0;
        }
      }
    }

    if (ulBraveV42BGuess <= 1)
    {
      ulRC =  mwmCntndAllocSegment(pmwmDspInfo,
                                   pmwmDspInfo->hmtProtSupr,
                                   "V42BTEMPDSEG",
                                   909,
                                   1,
                                   0,
                                   &pmwmDspInfo->hsegV42BTempDSEG);
      if (ulRC)
      {
        if (ulRC != DSP_INSUFF_DATA_MEM)
        {
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR,
                         ulRC);
          return ulRC;
        }
        else
        {
          DPF("Could not allocate DSEG");
          usSegmentsNeeded = 0;
        }
      }
    }

  }


  /***************************************************************************/
  /* 4/27/95                                                                 */
  /***************************************************************************/
  usSegmentsAvailable = (mwmMemPool.usTotalBlocks - mwmMemPool.usV34Blocks);

  /***************************************************************************/
  /* At this point, how can we make a reasonable guess as to how much memory */
  /* is available for V.42bis?                                               */
  /*                                                                         */
  /* Some of the Data Memory for V.42bis Dictionaries may currently be in    */
  /* use by MNP or V.42.                                                     */
  /* This is only true if we have BOTH MNP and V.42 Configured.  Otherwise,  */
  /* when we finish negotiating, no protocol will be freed.                  */
  /*                                                                         */
  /* We need to make an assumption as to how many Segments we will be able   */
  /* to fit into the V.42 or MNP space (when they are freed).                */
  /*                                                                         */
  /* CMVC 1387 - For V.90 connections, MNP is not loaded at this time.       */
  /* However, segments will be freed from V.90 itself.  We'll do this only   */
  /* when more than 3 segments are requested.  MNP WILL NOT LOAD!!           */
  /***************************************************************************/

  if ( ((pmwmDspInfo->ulFeaturesToLoad & FEATURE_MNP)  &&
        (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V42)     ) ||
       (pmwmDspInfo->hmodV34                           &&
         !strcmp(pmwmDspInfo->achV34ModuleName, "V90") &&
         usSegmentsNeeded > 3                             ) )
  {
    usSegmentsInUseByProtocol = 4;

    /*************************************************************************/
    /* If we couldn't allocate room for the V.42bis data segment, then       */
    /* let's assume that we can get 1 less segment in the place of the       */
    /* protocol.  This appears possible because V.42bis data could use       */
    /* some of this memory currently in use by the protocol                  */
    /*************************************************************************/
    if (ulBraveV42BGuess <= 1)
    {
      if (!pmwmDspInfo->hsegV42BTempDSEG)
        usSegmentsInUseByProtocol -=1;
    }

    /*************************************************************************/
    /* if we couldn't get either GPC then decrease the available count by    */
    /* more.                                                                 */
    /*************************************************************************/
    if ((!pmwmDspInfo->hsegV42BTempGPC1 && (ulBraveV42BGuess <= 2)) ||
        (!pmwmDspInfo->hsegV42BTempGPC2 && !ulBraveV42BGuess) )
      usSegmentsInUseByProtocol -=1;
  }

  usSegmentsAvailable += usSegmentsInUseByProtocol;

  DPF("Segments Needed: %d   Segments Available %d",usSegmentsNeeded,usSegmentsAvailable);

  if (usSegmentsNeeded > usSegmentsAvailable)
  {
    /*************************************************************************/
    /* Allocate the delta between how many we need, and how many we assume   */
    /* we can get later.                                                     */
    /*************************************************************************/

    usSegmentsAcquired = usSegmentsNeeded - usSegmentsAvailable;

    ulRC =  mwmPoolChangeAllocation(pmwmDspInfo,
                                    &usSegmentsAcquired,
                                    MWM_ALLOC_ADD);
    if (ulRC)
      return ulRC;
  }


  /***************************************************************************/
  /* 4/28/95                                                                 */
  /* Go check how many are available for V.42bis.                            */
  /* Trying to base this on how many additional segments was getting         */
  /* confusing.                                                              */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;

  usSegmentsAcquired = mwmMemPool.usTotalBlocks -
                       mwmMemPool.usV34Blocks +
                       usSegmentsInUseByProtocol;

  /***************************************************************************/
  /* In any case, we don't want to select a dictionary larger than what      */
  /* the user configured, so we will limit the segments acquired to the      */
  /* number needed.                                                          */
  /***************************************************************************/
  usSegmentsAcquired = min(usSegmentsNeeded,usSegmentsAcquired);

  if (usCompressBothWays)
  {
    switch (usSegmentsAcquired)
    {
      case 0:
        usBestDictionarySize = 0;
        usExtraSegments      = 0;
        break;
      case 1:
        usBestDictionarySize = 0;
        usExtraSegments      = 1;
        break;
      case 2:
        usBestDictionarySize = 512;
        usExtraSegments      = 0;
        break;
      case 3:
        usBestDictionarySize = 512;
        usExtraSegments      = 1;
        break;
      case 4:
        usBestDictionarySize = 768;
        usExtraSegments      = 0;
        break;
      case 5:
        usBestDictionarySize = 768;
        usExtraSegments      = 1;
        break;
      case 6:
        usBestDictionarySize = 1024;
        usExtraSegments      = 0;
        break;
      case 7:
        usBestDictionarySize = 1024;
        usExtraSegments      = 1;
        break;
      case 8:
        usBestDictionarySize = 1280;
        usExtraSegments      = 0;
        break;
      case 9:
        usBestDictionarySize = 1280;
        usExtraSegments      = 1;
        break;
      case 10:
        usBestDictionarySize = 1536;
        usExtraSegments      = 0;
        break;
      case 11:
        usBestDictionarySize = 1536;
        usExtraSegments      = 1;
        break;
      case 12:
        usBestDictionarySize = 1792;
        usExtraSegments      = 0;
        break;
      case 13:
        usBestDictionarySize = 1792;
        usExtraSegments      = 1;
        break;
      case 14:
        usBestDictionarySize = 2048;
        usExtraSegments      = 0;
        break;
    }
  }
  else
  {
    switch (usSegmentsAcquired)
    {
      case 0:
        usBestDictionarySize = 0;
        usExtraSegments      = 0;
        break;
      case 1:
        usBestDictionarySize = 512;
        usExtraSegments      = 0;
        break;
      case 2:
        usBestDictionarySize = 768;
        usExtraSegments      = 0;
        break;
      case 3:
        usBestDictionarySize = 1024;
        usExtraSegments      = 0;
        break;
      case 4:
        usBestDictionarySize = 1280;
        usExtraSegments      = 0;
        break;
      case 5:
        usBestDictionarySize = 1536;
        usExtraSegments      = 0;
        break;
      case 6:
        usBestDictionarySize = 1792;
        usExtraSegments      = 0;
        break;
      case 7:
        usBestDictionarySize = 2048;
        usExtraSegments      = 0;
        break;
    }
  }




  /***************************************************************************/
  /* If we happened to get some segments that we didn't need.                */
  /***************************************************************************/
  if (usExtraSegments)
  {
    ulRC =  mwmPoolChangeAllocation(pmwmDspInfo,
                                    &usExtraSegments,
                                    MWM_ALLOC_SUBTRACT);
    if (ulRC)
      return ulRC;
  }


  /***************************************************************************/
  /* Tell MCTL (and Protocols) my best guess for Dictionary Size             */
  /***************************************************************************/
  ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "NEGDICT", &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                        &usBestDictionarySize, 1,
                        DSP_MEMXFER_DATA_WRITE);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }


  DPF("Drivers Guess for Max V.42bis Dictionary size: %d",usBestDictionarySize);

  #ifdef DEBUG
  ulAddress = 0;
  /***************************************************************************/
  /* For debug...dump the Memory Pool pointers and control block to pmprintf */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | DP_REQUEST_WORD,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;


  DPF("Total Blocks: %d  V.34 Blocks: %d  V.42bis Blocks: %d  DP Requested: %d",
      mwmMemPool.usTotalBlocks,
      mwmMemPool.usV34Blocks,
      mwmMemPool.usV42bisBlocks,
      mwmMemPool.usDPRequestedBlocks);

  #endif

  /************************************************************
  ** Free the space reserved for V.34 Dummy Segments.
  *************************************************************/
  ulRC = mwmPoolDummyV34Segments( pmwmDspInfo, 2 );
  if (ulRC)
    return(ulRC);

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolGuessMaxV42bisEntries exit\n");  }


  return 0;
}







ULONG mwmPoolChangeAllocation(PMWM_DSPINFO pmwmDspInfo,
                              USHORT *pusNumSegments,      // Number of 768 Word Segments
                              USHORT usFlags)
{
  ULONG  ulRC = 0;
  USHORT usCount = 0;
  USHORT usDesiredTotal = 0;
  USHORT usOutOfMemory = 0;
//  HSEG   hTempSeg = 0;
  char   achSegName[9];
  MWM_MEMPOOL mwmMemPool;
  ULONG  ulAddress = 0;
  USHORT usNumDeltaSegments = 0; // Number of Segments Allocated or Deallocated

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolChangeAllocation entry\n");  }


  memset(achSegName,0,9);

  /***************************************************************************/
  /* Get label to the Control Block                                          */
  /***************************************************************************/
  ulRC = dspLabelToAddress(pmwmDspInfo->hmtProtSupr, "BLKPTR", &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }



  /***************************************************************************/
  /* Read Entire Control Block                                               */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | DP_REQUEST_WORD | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;

//  DPF("Total Blocks: %d  DP Blocks: %d  V.42bis Blocks: %d  DP Requested: %d",
//             mwmMemPool.usTotalBlocks,
//             mwmMemPool.usV34Blocks,
//             mwmMemPool.usV42bisBlocks,
//             mwmMemPool.usDPRequestedBlocks);

  /***************************************************************************/
  /* if we need to allocate more.                                            */
  /***************************************************************************/
  if (usFlags & MWM_ALLOC_ADD)
  {
    usDesiredTotal = mwmMemPool.usTotalBlocks + *pusNumSegments;
    /*************************************************************************/
    /* Check to make sure we have enough pointers available to make this     */
    /* Allocation.                                                           */
    /*************************************************************************/
    if ( usDesiredTotal <= MWM_POOL_MAXSEGMENTS)
    {
      /***********************************************************************/
      /* We have the pointers available.                                     */
      /* while there is room in the DSP, proceed to allocate segments.       */
      /***********************************************************************/
      for (usCount = mwmMemPool.usTotalBlocks*3;
           ((usCount < usDesiredTotal*3) &&
            (!usOutOfMemory)            );
           usCount+=3)
      {
        /*********************************************************************/
        /* Generate the Segment Name                                         */
        /*********************************************************************/
        sprintf(achSegName,"MPOOL%02d",mwmMemPool.usTotalBlocks);

        /*********************************************************************/
        /* Allocate the segment                                              */
        /*********************************************************************/
        ulRC =  mwmCntndAllocSegment(pmwmDspInfo,
                                  pmwmDspInfo->hmtProtSupr,
                                  achSegName,
                                  768,
                                  1,
                                  ulAddress + MWM_POOL_OFFSET_TO_POINTERS + 2 + usCount*2,
                                  &pmwmDspInfo->ahsegMemPool[mwmMemPool.usTotalBlocks]);
        if (ulRC)
        {
          if (ulRC != DSP_INSUFF_DATA_MEM)
          {
            mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                           MWM_DSP_ERROR,
                           ulRC);
            return ulRC;
          }
          else
            usOutOfMemory = 1;
        }
        else
        {
          /*******************************************************************/
          /* Increment the Total Number of 768 Blocks by 1.                  */
          /*******************************************************************/
          mwmMemPool.usTotalBlocks++;
          usNumDeltaSegments++;
        }
      }
    }
    else
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_OS_ERROR,
                     1);
      return 1;
    }
  }
  else
  if ((usFlags & MWM_ALLOC_SUBTRACT) || (usFlags & MWM_ALLOC_CLEARALL) )
  {
    /*************************************************************************/
    /* Figure out how many segments we should delete.                        */
    /*************************************************************************/
    if (usFlags & MWM_ALLOC_CLEARALL)
      usDesiredTotal = 0;
    else
      usDesiredTotal = mwmMemPool.usTotalBlocks - *pusNumSegments;

    /*************************************************************************/
    /* Loop thru deleting these segments...                                  */
    /* Last 2 blocks cannot be deleted using this method...                  */
    /*************************************************************************/
    for (usCount = mwmMemPool.usTotalBlocks;
         ((usCount > usDesiredTotal) && (usCount > 2));
         usCount--)
    {
      /***********************************************************************/
      /* Free the segment.                                                   */
      /***********************************************************************/
      ulRC = dspFreeSegment(pmwmDspInfo->ahsegMemPool[usCount-1]);
      if (ulRC!=DSP_NOERROR)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC);
        return(ulRC);
      }

      /***********************************************************************/
      /* Clear the hSeg for this segment                                     */
      /***********************************************************************/
      pmwmDspInfo->ahsegMemPool[usCount-1] = 0;

      /***********************************************************************/
      /* Decrement the Total Blocks pointer...                               */
      /***********************************************************************/
      mwmMemPool.usTotalBlocks--;
      usNumDeltaSegments++;
    }
  }

  /***************************************************************************/
  /* Read Pointer Array portion of Control Block from the DSP                */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_READ );
  if (ulRC)
    return ulRC;



  /***************************************************************************/
  /* Fill in 256 word intermediate pointers...                               */
  /***************************************************************************/
  for (usCount = 0;
       usCount < mwmMemPool.usTotalBlocks*3;
       usCount +=3)
  {
    /*************************************************************************/
    /* Using Base segment pointer, create the artifial pointers which        */
    /* identify the 256 word boundaries...                                   */
    /*************************************************************************/
    mwmMemPool.aSegAddress[usCount+1] = mwmMemPool.aSegAddress[usCount] + 512; // 256 Words
    mwmMemPool.aSegAddress[usCount+2] = mwmMemPool.aSegAddress[usCount] + 1024;// 512 Words
  }


  /***************************************************************************/
  /* Clear Segment Addresses above the number actually allocated now.        */
  /***************************************************************************/
  for (usCount = mwmMemPool.usTotalBlocks*3;
       usCount < MWM_POOL_MAXSEGMENTS*3;
       usCount++)
  {
    mwmMemPool.aSegAddress[usCount] = 0;
  }


  /***************************************************************************/
  /* Write Entire Control Block to the DSP                                   */
  /***************************************************************************/
  ulRC = mwmPoolTransferControlBlock(pmwmDspInfo,
                                     &mwmMemPool,
                                     &ulAddress,
                                     CONTROL_WORDS | SEGMENT_POINTERS,
                                     DSP_MEMXFER_DATA_WRITE );
  if (ulRC)
    return ulRC;

  *pusNumSegments = usNumDeltaSegments;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolChangeAllocation exit\n");  }


  return 0;
}







ULONG mwmPoolGetMaxConfiguredDictSize(PMWM_DSPINFO pmwmDspInfo,
                                      USHORT *pusSegmentsNeeded,
                                      USHORT *pusCompressBothWays)

{
  ULONG     ulRC;
  REGISTERS Registers;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolGetMaxConfiguredDictSize entry\n");  }


  /***************************************************************************/
  /* Query Modem Register Settings so that we can see what Maximum           */
  /* allowed dictionary size is configured from AT Commands.                 */
  /***************************************************************************/
  ulRC = mwmParseQueryModemRegisters(&Registers);
  if (ulRC)
    return ulRC;

  /***************************************************************************/
  /* Check to see if compression is allowed, and if it is allowed in both    */
  /* directions...                                                           */
  /***************************************************************************/
  switch(Registers.QuoteH)
  {
    case 0:
      *pusSegmentsNeeded = 0;
      break;
    case 1:
    case 2:
      *pusCompressBothWays = 0;
      break;
    case 3:
      *pusCompressBothWays = 1;
      break;
  }

  if (Registers.QuoteH)
  {
    switch (Registers.QuoteN)
    {
      case 0:
        if (*pusCompressBothWays)
          *pusSegmentsNeeded = 2;
        else
          *pusSegmentsNeeded = 1;
        break;
      case 1:
        if (*pusCompressBothWays)
          *pusSegmentsNeeded = 6;
        else
          *pusSegmentsNeeded = 3;
        break;
      case 2:
        if (*pusCompressBothWays)
          *pusSegmentsNeeded = 14;
        else
          *pusSegmentsNeeded = 7;
        break;
    }
  }

  /***************************************************************************/
  /* Now, check to see if it is even possible to get a V.42bis connection    */
  /***************************************************************************/
  if (!Registers.SlashN)
  {
    *pusSegmentsNeeded = 0;
  }

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolGetMaxConfiguredDictSize exit\n");  }


  return 0;
}


/****************************************************************************
** Function: mwmPoolDummyV34Segments
**
** This function was created to handle the creating of dummy segments in
** the V.34 task space. These dummy segments are needed because the newest
** DynaSegs changes cause new segments to be unloaded initially, freeing
** up much more space in Data Store. Because so much more space is available,
** estimations of free space when the EC Blocks are allocated and when
** a guess is made for V.42bis dictionary size will be too generous, and
** there will not be enough space to load these DynaSegments back up when
** they are needed.
*****************************************************************************/
ULONG mwmPoolDummyV34Segments(PMWM_DSPINFO pmwmDspInfo, USHORT usActionFlag)
{

  char achV34Dynasegs[][25] =  { "DATAMODEDATA1",
                                 "DATAMODEDATA2",
                                 "DATAMODEDATA3",
                                 "V34CONSTEL" };

  static HSEG ahsegDummySegments[NUM_V34_DYNASEGS];
  static ULONG aulSegmentSizes[NUM_V34_DYNASEGS];
  USHORT i;
  ULONG  ulRC = 0;
  HSEG   hSegment;
  USHORT usSize;
  PVOID  pMiscInfoBuffer;
  char   achSegmentName[50];
  BOOL   bSegmentTempLoad;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolDummyV34Segments entry\n");  }


  switch (usActionFlag) {

    /* 0 means get the V.34 segment sizes */
    case 0:

      for (i=0; i < NUM_V34_DYNASEGS; i++) {
        bSegmentTempLoad = FALSE;
        ulRC = dspNameToSegmentHandle( pmwmDspInfo->hmtV34, achV34Dynasegs[i],
                                       &hSegment );


        // If the segment is not currently loaded, load it temporarily
        if (ulRC == DSP_NAME_NOT_FOUND) {
          ulRC = mwmCntndLoadSegment( pmwmDspInfo, pmwmDspInfo->achV34ModuleName,
                                      achV34Dynasegs[i], achV34Dynasegs[i],
                                      pmwmDspInfo->hmtV34, &hSegment, 0L,
                                      szThisFile, __LINE__, MWM_MUSTHAVE );
          if (ulRC)
            return ulRC;

          bSegmentTempLoad = TRUE;

        } else if (ulRC != DSP_NOERROR) {
          mwmHandleError( pInstanceInfo,szThisFile,__LINE__,
                          MWM_DSP_ERROR, ulRC);
          return ulRC;
        } /* endif */

        usSize = 0;
        ulRC = dspQueryMiscInfo(MISC_SEG, hSegment, &usSize, NULL);
        if (ulRC != DSP_INS_BUFFER) {
          mwmHandleError( pInstanceInfo,szThisFile,__LINE__,
                          MWM_DSP_ERROR, ulRC);
          return ulRC;
        } /* endif */

        usSize += 20;

        pMiscInfoBuffer = calloc( 1, usSize );
        if (!pMiscInfoBuffer) {
          mwmHandleError( pInstanceInfo,szThisFile,__LINE__,
                          MWM_OS_ERROR, 0);
          return 1;
        } /* endif */

        ulRC = dspQueryMiscInfo(MISC_SEG, hSegment, &usSize, pMiscInfoBuffer);
        if (ulRC != DSP_NOERROR) {
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC );
          free(pMiscInfoBuffer);
          return ulRC;
        } /* endif */

        aulSegmentSizes[i] = ((RSEGINFO *)pMiscInfoBuffer)->SEGINFO_ulSize;
        free(pMiscInfoBuffer);

        if (bSegmentTempLoad) {
          ulRC = dspFreeSegment(hSegment);
          if (ulRC != DSP_NOERROR) {
            mwmHandleError( pInstanceInfo,szThisFile,__LINE__,
                            MWM_DSP_ERROR, ulRC);
            return ulRC;
          } /* endif */
        } /* endif */

      } /* endfor */
    break;

    /* 1 means allocate the space */
    case 1:
      for (i=0; i < NUM_V34_DYNASEGS; i++) {
        ahsegDummySegments[i] = 0;
        strcpy(achSegmentName, achV34Dynasegs[i]);
        strcat(achSegmentName, "_DUMMY");
        ulRC =  mwmCntndAllocSegment(pmwmDspInfo,
                                     pmwmDspInfo->hmtV34,
                                     achSegmentName,
                                     aulSegmentSizes[i],
                                     1, 0,
                                     &ahsegDummySegments[i]);
        if (ulRC) {
          if (ulRC != DSP_INSUFF_DATA_MEM) {
            mwmHandleError( pInstanceInfo,szThisFile,__LINE__,
                            MWM_DSP_ERROR, ulRC);
          } else {
            DPF("Could not allocate dummy V.34 Segment");
          } /* endif */
        } /* endif */
      } /* endfor */
    break;

    /* 2 means free the dummy segments */
    case 2:
      for (i=0; i < NUM_V34_DYNASEGS; i++) {
        if (ahsegDummySegments[i]) {
          ulRC = dspFreeSegment(ahsegDummySegments[i]);
          if (ulRC != DSP_NOERROR) {
            mwmHandleError( pInstanceInfo,szThisFile,__LINE__,
                            MWM_DSP_ERROR, ulRC);
            return(ulRC);
          } /* endif */
          ahsegDummySegments[i] = 0;
        } /* endif */
      } /* endfor */
    break;
  } /* endswitch */

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolDummyV34Segments exit\n");  }


  return(ulRC);

}







ULONG mwmPoolTransferControlBlock(PMWM_DSPINFO pmwmDspInfo,
                                  PMWM_MEMPOOL pmwmMemPool,
                                  PULONG pulAddress,
                                  USHORT usControlFlags,        // Work with Control words, pointer, or both.
                                  USHORT usMemTransferFlags)    // Read or Write flags for memTransfer
{
  ULONG ulRC = 0;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolTransferControlBlock entry\n");  }


  if (!*pulAddress)
  {
    /*************************************************************************/
    /* Get label to the Control Block                                        */
    /*************************************************************************/
    ulRC = dspLabelToAddress(pmwmDspInfo->hmtProtSupr, "BLKPTR", pulAddress);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
  }

  if (usControlFlags & CONTROL_WORDS)
  {
    ulRC = dspMemTransfer(pmwmDspInfo->hDSP, *pulAddress,
                          pmwmMemPool, MWM_POOL_CONTROL_WORDS,
                          usMemTransferFlags);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
  }

  if (usControlFlags & DP_REQUEST_WORD)
  {
    ulRC = dspMemTransfer(pmwmDspInfo->hDSP, *pulAddress + MWM_POOL_CONTROL_WORDS,
                          &pmwmMemPool->usDPRequestedBlocks, 1,
                          usMemTransferFlags);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
  }

  /***************************************************************************/
  /* Get all of the Segment pointers, beginning with the oddball one just    */
  /* before the beginning of the block.                                      */
  /***************************************************************************/
  if (usControlFlags & SEGMENT_POINTERS)
  {
    ulRC = dspMemTransfer(pmwmDspInfo->hDSP, *pulAddress + MWM_POOL_OFFSET_TO_POINTERS,
                          &pmwmMemPool->usBasePageAddress, MWM_POOL_MAXSEGMENTS*3+1,
                          usMemTransferFlags);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
  }

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmpool::mwmPoolTransferControlBlock exit\n");  }

  return 0;
}

