/*
 *   mwwttbl.c -- World Trade Table Access 
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

/*---------------------------------------------------------------------------*/
static char copyright[]="(C) Copyright IBM Corp. 1994 to 1997 all rights reserved."
                        "  US Government Users Restricted Rights - "
                        "Use duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.";



/*-------- Defines --------------------------------------------------------------------*/

extern char *ini_file;
#define WORLDTRADE      "WORLDTRADE"
#define DATABASE        "DATABASE"
#define COUNTRYENTRY    "Country" 
#define CURRENTCOUNTRY  "CurrentCountry"
#define WTTVERSION      "WTTversion"
#define CHECK_DIALING_PROPERTIES "Check_Dialing_Properties"
#define NO_OC_HANGUP    "No_OC_Hangup"
#define MWCOUNTRYSELECT "MW_Country_Select"
#define DPACCESSENTRY   "Last_DP_Access_Code"
#define DISABLEWTT      "Disable_WTT"
#define NEWCOUNTRY      "Newcountry"

#define WTTVERSION_MIN_SUPPORTED  1040  /* This is the minimum level of WTTversion supported*/
                                        /* by this version of the mwwttbl */

#define WTBLOCK_MAP_DATA   0x504D  /*defined here instead of in WTT.H since*/
                                   /*it is only used internal to this program*/

/* Status to tell the current state of the WT Table loading to be assigned to G.usWTTloaded.*/
/* Note that the order of these defines is important to program operations.*/
#define NOT_LOADED  0 /*Means that there is no WT Table in memory.*/
#define WTT_LOADING 1 /*Means that a WT Table has been read in from the file but has not yet been fixed up.*/
#define WTT_LOADED  2 /*Means that the WT Table is in memory.*/

/* Server client list*/
#define WT_MAX_CLIENTS                  10
#define MW_USE_REG_FOR_INI_SETTINGS     1 

#ifndef NULLHANDLE 
#define NULLHANDLE 0
#endif

#include <stdio.h>
#include <malloc.h>
#include <memory.h>
#include <string.h>
#include <stdlib.h>

#include "mwave.h"       /* Main Mwave header file*/
#include "mwagent.h"     /* Mwave notification server*/
//#include "ini2reg.h"

#include <port_types.h>
#include <port_functions.h>

#include <mwwttbl.h>     /* Header file */
#include "wtt.h"         /* Field and parameter definitions for the WTT*/
#include "mwwttcus.h"    /* Mwave.ini customization override strings*/
//#include "mwwtdbg.h"     /* Debug routine header file*/

#ifndef FALSE
  #define FALSE 0
  #define TRUE (!FALSE)
#endif


/*------------------------------------------------------------------------------*/
extern USHORT XLarray[];  /* The XLarray provides a look up table for transmit levels*/
                          /* based on a log scale.  A look array avoids floating point.*/

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  Global Structure definition*/
/**/
/*      The global structure provides global static variables for use*/
/*      throughout the function set.*/
/**/
/*-----------------------------------------------------------------------------*/
struct TAGGLOBAL
{
  /* Make sure this parameter is the first one since it is intialized below when G is declared*/
  BOOL bCountrySelectLaunched;      /*This flag is used to limit how many times*/
                                    /*country select gets launched in the*/
  /* Status to tell the current state of the WT Table loading.*/
  /* NOT_LOADED means that there is no WT Table in memory.*/
  /* WTT_LOADING means that a WT Table has been read in form the file but has not yet been fixed up.*/
  /* WTT_LOADED means that the WT Table is in memory.  Since loading occurs during*/
  /* initalization, this status variable also tells if initialization occurred*/
  /* correctly.*/
  USHORT usWTTloaded;

  /* This variable is set with the failing error number if initialization fails.*/
  ULONG ulInitError;

  /* Status to tell if the mapping information has been initalized correctly.*/
  BOOL bMappingInitialized;

  /* Mwave.ini file entries*/
  USHORT usDatabaseSetNumber;   /* Used to select alternate databases*/

  char szDatabase[_MAX_PATH];   /* WT Table file path and name*/
  USHORT usCurrentCountry;      /* The COUNTRY= entry*/
  USHORT usWTTversion;          /* The WTTversion= entry*/
  USHORT usNo_OC_Hangup;        /* Flag to tell when not to hang up on an OverCurrent*/
                                /* condition.*/
  USHORT usCheckDialingProperties;  /*This flag is used to specify if the WTT_CheckCountryNumber*/
                                    /*function should operate.*/


  char szCountrySelect [_MAX_PATH]; /*Fully qualified country select string*/

  /* This array records what valid WT Tables are present in the multi-country*/
  /* file.*/
  BOOL bWTTpresent[WT_COUNTRY_MAX+1];


  /* This array keeps a record of the coupler-ID for each country marked as*/
  /* valid in the bWTTpresent array.  This can be used to filter the country*/
  /* list to only those countries matching a certain DAA.*/
  USHORT usWTTcouplerID[WT_COUNTRY_MAX+1];


  /* This array defines what WTT to use for each telephone network access code*/
  /* from the Win 95 "I am in" dialing properties of the modem.*/
  USHORT usAccessCodeMapArray[ACCESS_CODE_MAX];


  /* This array maps Mwave country numbers into WT Table numbers.*/
  USHORT usCountryNumMapArray[WT_COUNTRY_MAX+1];

  /* This array holds the country record when it is read in.*/
  USHORT pusRecord[MAX_RECORD_SIZE];

  struct
  {
    BOOL bEnabled;
    unsigned short ulCouplerID;
  } CouplerIDfilter;

  /**/
  /* World Trade Server Related*/
  /**/

  HMWAVE   hMwave;
  HMSERVER hServer;

  HMCONV ClientTable[WT_MAX_CLIENTS];     /* clients of World Trade Server*/

  USHORT us_Loaded;
} G = { 0 };  /*define G as the global variable structure*/


/* end of Global Structure definition*/



/* Internal function prototypes, documentation is with each function*/
ULONG Initialize_WTT (void);
ULONG QueryInfo (USHORT usWTTBlockTag, USHORT usOffset, ULONG ulNumShorts, short FAR* lpsBuffer);
ULONG GetBlock (USHORT usType, USHORT **ppusBlock);
ULONG ReadINIfiles(char FAR *szDatabase, USHORT usMaxStringSize,
                   USHORT FAR *pusCurrentCountry,
                   USHORT FAR *pusWTTversion,
                   USHORT FAR *pusCheck_Dialing_Properties, char FAR *szCountrySelect,
                   USHORT FAR *pusNo_OC_Hangup);
ULONG GetTAPIcountry(USHORT *pusCountryNum, USHORT FAR *pusCountryAccessCode);
ULONG GetConfigCountry(USHORT *pusCountryNum, USHORT FAR *pusCountryAccessCode);
ULONG GetINIcountry(USHORT *pusCountryNum, USHORT FAR *pusCountryAccessCode);
ULONG FindAvailableCountries (void);
ULONG VerifyCountryRecord (RECHEADER *pRecHeader, USHORT *pusRecord,
                           USHORT usExpectedWTTversion);

ULONG VerifyCheckSum (RECHEADER *pheader, USHORT *pRecord);
ULONG ReadMapTable(void);
ULONG MapAccessToCountryNum (ULONG lAccessCode, ULONG *ulCountryNum);
ULONG MapCountryNumToWTTnum (ULONG ulCountryNum, ULONG *pulWTTnum);
ULONG MapCountryNumToAccess (ULONG ulCountryNum, ULONG *ulAccessCode);
ULONG LoadWTTable (ULONG ulCountryNum, ULONG ulAccessCode);
ULONG FindCountry (ULONG ulCountryNum);
BOOL ReadAdjustment (ULONG ulCountryNum, char FAR * szINIentry, short *psUpdateVal);
ULONG SetParam (USHORT usWTTBlockTag, USHORT usOffset, short sUpdateVal);
ULONG UpdateSettings (ULONG ulCountryNum);


BOOL Init_Lib(ULONG ulReason)
{
  ULONG ulError;
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::Init_Lib entered ulReason %lx\n",ulReason);
  
  switch (ulReason)
    {
    case 1:
      
      if(G.us_Loaded == 0) {
	G.us_Loaded = 1;
	/* Initialize the server handles*/
	G.hMwave  = NULLHANDLE;
	G.hServer = NULLHANDLE;
      
	/* This flag specifies if a coupler id filter has been set to recognize only countries*/
	/* with a matching coupler id.  This has to be set outside of the Initialize_WTT routine*/
	/* since Initialize_WTT is recalled after the coupler id filter is set.*/
	G.CouplerIDfilter.bEnabled = FALSE;
        
        
	/* Perform initialization for table access.*/
	G.ulInitError = Initialize_WTT();
	if (G.ulInitError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,">>>Init_Lib - failed Initialize_WTT with ulError %lu\n", G.ulInitError);
	  
	  ulError = 0; /* This 'if' structure must have something in it if the*/
	}
	
      }  
      break;
      
    case 0:
      MW_SYSLOG_1(TRACE_MWWTT32,"Init_Lib - unload complete\n");
    
    break;
    
    } /*end of switch*/
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::Init_Lib exit TRUE\n");
  return TRUE;
}




/* ------------------World Trade Table New API Functions---------------------------*/
//////////////////////////////////////////////////////////////////////////////**/
//////////////////////////////////////////////////////////////////////////////**/
/**/
/* Start of New API functions, the old API is listed later in the file*/
/**/
//////////////////////////////////////////////////////////////////////////////**/
//////////////////////////////////////////////////////////////////////////////**/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_GetNumCountries*/
/**/
/*     Retrieves the maximum country number defined in the current revision*/
/*     This is used by the country selection utility to*/
/*     search the database to find all installed WT Tables.*/
/**/
/*     Params:   hDsp             - The DSP handle, one WT Table per DSP.*/
/*               lpulNumCountries - Pointer to ULONG location where the*/
/*                                  country number should be returned in.*/
/*               ulReserved       - A reserved location for future API*/
/*                                  expansion.*/
/**/
/*     Returns:  ULONG WTT_ERR_... from MWWTT.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_GetNumCountries (HDSP hDsp, ULONG FAR* lpulNumCountries, ULONG ulReserved )
{
  (void) hDsp;
  (void) ulReserved;

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_GetNumCountries  entry\n");

  /* Use the setting from the MWWTT.H file.*/
  *lpulNumCountries = (ULONG) WT_COUNTRY_MAX;

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_GetNumCountries  exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} 

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_GetCountryName*/
/**/
/*     Retrieves the name of the country associated with the given country*/
/*     number.  Due to WT Table mapping, a particular WT Table could be*/
/*     used in a different country than it's number.  If the country is*/
/*     not present in the WT Table, then this function returns an error.*/
/**/
/*     Params:   hDsp            - The DSP handle, one WT Table per DSP.*/
/*               ulCountryNum    - Country number to be translated into a*/
/*                                 country name string.*/
/*               lpszCountryName - Pointer to a string location to copy the*/
/*                                 country name into.*/
/*               ulBufSize       - Size in bytes of the lpszCountryName string*/
/*                                 location.*/
/*               ulReserved      - A reserved location for future API expansion.*/
/**/
/*               Note:  this function uses the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_GetCountryName (HDSP hDsp, ULONG ulCountryNum, ULONG ulAccessCode, char FAR* lpszCountryName, ULONG ulBufSize, ULONG ulReserved )
{
  ULONG ulError;
  ULONG ulWTTnum;
  ULONG ulAccess=(ULONG)ulAccessCode;
  char szTemp[1024];
  DWORD dwCharBufSize;
  
  (void) hDsp;
  (void) ulReserved;
  
  MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryName  entered ulCountryNum %lx ulAccessCode %lx\n",ulCountryNum,ulAccessCode);
  
  if (G.usWTTloaded != WTT_LOADED) {         /*return if the WTT has not been initialized*/
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryName - init failed with ulError = %lu\n", G.ulInitError);
    return G.ulInitError;
  }
  
  ulError = MapCountryNumToWTTnum (ulCountryNum, &ulWTTnum);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryName - failed MapCountryNumToWTTnum with ulError = %lu\n", ulError);
    return ulError;
  }
  
  if (!(G.bWTTpresent[(int)ulWTTnum])) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryName - failed - returning WTTPOSERR_COUNTRY_NOT_AVAIL for ulWTTnum = %lu\n", ulWTTnum);
    return WTTPOSERR_COUNTRY_NOT_AVAIL;
  }

  if (!ulAccess) {
    ulError=MapCountryNumToAccess(ulCountryNum, &ulAccess);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryName - failed MapCountryNumToAccess with ulError = %lu\n", ulError);
      return ulError;
    }
  }
    
  sprintf (szTemp, "Telephony\\Country List\\%ld", ulAccess);
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryName getting name from config for %s\n",szTemp);
  dwCharBufSize=GetPrivateProfileString(szTemp,"Name","Failed", lpszCountryName,256, NULL);
  MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryName, dwCharBufSize %lx, szName %s\n",dwCharBufSize,lpszCountryName);    
  
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryName - failed MWcountryName with ulError = %lu\n", ulError);
    return ulError;
  }

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryName exit WTT_SUCCESSFUL lpszCountryName %s\n",lpszCountryName);
  return WTT_SUCCESSFUL;
}

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_GetCountryCouplerID*/
/**/
/*     Retrieves the WTTCID setting from the WTITCB block of the specifed*/
/*     country number.  The coupler ID is only valid if a valid WT Table*/
/*     exists for the country number, otherwise an error is returned.*/
/**/
/**/
/*     Params:   hDsp               - The DSP handle, one WT Table per DSP.*/
/*               ulCountryNum       - Country number to look up the coupler ID.*/
/*               lpulCouplerID      - The CouplerID from the WT Table.*/
/*               ulReserved         - A reserved location for future API expansion.*/
/**/
/*               Note:  this function uses the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_GetCountryCouplerID (HDSP hDsp, ULONG ulCountryNum, ULONG *lpulCouplerID, ULONG ulReserved )
{
  ULONG ulError;
  ULONG ulWTTnum;

  (void) hDsp;
  (void) ulReserved;

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryCouplerID entry, ulCountryNum %lx\n",ulCountryNum);

  if (G.usWTTloaded != WTT_LOADED) {         /*return if the WTT has not been initialized*/
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryCouplerID - init failed with ulError = %lu\n", G.ulInitError);
    return G.ulInitError;
  }
  
  ulError = MapCountryNumToWTTnum (ulCountryNum, &ulWTTnum);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryCouplerID - failed MapCountryNumToWTTnum with ulError = %lu\n", ulError);
    return ulError;
  }
  
  if (!(G.bWTTpresent[(int)ulWTTnum])) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryCouplerID - failed - returning WTTPOSERR_COUNTRY_NOT_AVAIL for ulWTTnum = %lu\n", ulWTTnum);
    return WTTPOSERR_COUNTRY_NOT_AVAIL;
  }
  
  /* If the country record is present, then return the coupler id for the*/
  /* WT Table used.*/
  *lpulCouplerID = G.usWTTcouplerID[(int)ulWTTnum];

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryCouplerID exit, ulCouplerID %lx\n",*lpulCouplerID);
  return WTT_SUCCESSFUL;
} 

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_SetCouplerIDfilter*/
/**/
/*     Sets the Coupler ID filter.  Once this filter is set, only WT Tables*/
/*     with an identical WTCID setting will be valid.  This will limit the*/
/*     number of countries listed in the country selection application.*/
/**/
/**/
/*     Params:   hDsp         - The DSP handle, one WT Table per DSP.*/
/*               ulCouplerID  - Coupler ID that the WTTCID setting must match*/
/*                              for the WT Table to be valid.*/
/**/
/*               ulDisableRestriction = 0 - Enable restriction*/
/*                                      1 = Disable restriction*/
/**/
/*               ulReserved   - A reserved location for future API expansion.*/
/**/
/*     Returns:  ULONG WTPOS_ERR_... from list above*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_SetCouplerIDfilter (HDSP hDsp, ULONG ulCouplerID, ULONG ulDisableRestriction, ULONG ulReserved ) {
  ULONG ulError;
  
  (void) hDsp;
  (void) ulReserved;

  MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryCouplerID entry, ulCouplerID %lx ulDisableRestriction %lx\n",ulCouplerID,ulDisableRestriction);

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_SetCouplerIDfilter - setting filter to %lu\n", ulCouplerID);

  if (ulDisableRestriction) {
    G.CouplerIDfilter.bEnabled = FALSE;
  } else {
    G.CouplerIDfilter.bEnabled = TRUE;
    G.CouplerIDfilter.ulCouplerID = (unsigned short) ulCouplerID;
  }

  ulError = WTT_WTTableHasChanged (NULL, 0L);

#if defined(MWWTDBG)
  if (ulError)
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_SetCouplerIDfilter - failed WTT_WTTableHasChanged with ulError = %lu\n", ulError);
#endif

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCountryCouplerID exit, ulError %lx\n",ulError);
  return ulError;
} /*end WTT_SetCouplerIDfilter*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_GetCurrentCountry*/
/**/
/*     Retrieves the current country number in use for a particular DSP.*/
/**/
/*     Params:   hDsp                  - The DSP handle, one WT Table per DSP.*/
/*               lpulCurrentCountryNum - Pointer to location where the current*/
/*                                       country number should be returned in.*/
/*               ulReserved            - A reserved location for future API*/
/*                                       expansion.*/
/**/
/*               Note:  this function uses the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_GetCurrentCountry (HDSP hDsp, ULONG FAR* lpulCurrentCountryNum, ULONG ulReserved ) {
  (void) hDsp;
  (void) ulReserved;
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_GetCurrentCountry entry\n");

  if (G.usWTTloaded != WTT_LOADED) {         /*return if the WTT has not been initialized*/
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCurrentCountry - init failed with ulError = %lu\n", G.ulInitError);
    return G.ulInitError;
  }
  
  /* This is found by the ReadINIfiles function*/
  *lpulCurrentCountryNum = G.usCurrentCountry;

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCurrentCountry exit WTT_SUCCESSFUL, ulCurrentCountryNum %lx\n",*lpulCurrentCountryNum);
  return WTT_SUCCESSFUL;
} /*end WTT_GetCurrentCountry*/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_GetCurrentWTTversion*/
/**/
/*     Retrieves the WTTversion of the current WT Table.  This can be used*/
/*     to determine what level of Mwave communications code mix is on the*/
/*     system.*/
/**/
/*     Params:   hDsp                  - The DSP handle, one WT Table per DSP.*/
/*               lpulCurrentWTTversion - Pointer to location where the current*/
/*                                       WTTversion setting should be returned.*/
/*               ulReserved            - A reserved location for future API*/
/*                                       expansion.*/
/**/
/*               Note:  this function uses the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_GetCurrentWTTversion (HDSP hDsp, ULONG FAR* lpulCurrentWTTversion, ULONG ulReserved ) {
  (void) hDsp;
  (void) ulReserved;
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_GetCurrentWTTversion entry\n");

  if (G.usWTTloaded != WTT_LOADED) {          /*return if the WTT has not been initialized*/
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCurrentWTTversion - init failed with ulError = %lu\n", G.ulInitError);
    return G.ulInitError;
  }
  
  /* This is found by the ReadINIfiles function*/
  *lpulCurrentWTTversion = G.usWTTversion;
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_GetCurrentWTTversion exit WTT_SUCCESSFUL, ulCurrentWTTversion %lx\n",*lpulCurrentWTTversion);
  return WTT_SUCCESSFUL;
} /*end WTT_GetCurrentWTTversion*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_ChangeWTTable*/
/**/
/*     This function is called by the country change utility to change the*/
/*     current country.  This function then writes the new country setting*/
/*     to the Mwave.ini file and calls WTT_WTTableHasChanged to update*/
/*     the WT Table and notify users that the change occurred.*/
/**/
/*     Params:   hDsp            - The DSP handle, one WT Table per DSP.*/
/*               ulCountryNum    - Country number to be translated into a*/
/*                                 country name string.*/
/*               ulReserved      - A reserved location for future API expansion.*/
/**/
/*               Note:  this function changes the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*     Overall flow:*/
/*        update Mwave.ini*/
/*        call WTT_WTTableHasChanged*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_ChangeWTTable (HDSP hDsp, ULONG ulCountryNum, ULONG ulReserved ) {
  char szTempBuf[6];  /*large enough to hold a 5 digit USHORT plus \0*/
  USHORT usCountryNum;
  ULONG ulError;
  USHORT usIamInCountry, usIamInAccessCode;

  (void) hDsp;
  (void) ulReserved;

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_ChangeWTTable entry, ulCountryNum %lx\n",ulCountryNum);

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::Entering WTT_ChangeWTTable\n");
  usCountryNum = (USHORT) ulCountryNum;

  sprintf(szTempBuf,"%u", usCountryNum);
  WritePrivateProfileString(WORLDTRADE, COUNTRYENTRY, szTempBuf, ini_file);

  /* Now, because it is possible to set the Mwave country when there is no*/
  /* equivalent match in the Dialing Properties, we must record what the*/
  /* the Dialing Properties is set to at this point.  This is used by*/
  /* WTT_CheckCountryNumber to determine if the user has set a country.*/
  if (G.usCheckDialingProperties) {
    /* Now we need to get the Dialing Properties country setting.*/
    ulError = GetTAPIcountry (&usIamInCountry, &usIamInAccessCode);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::ReadINIfiles - failed GetTAPIcountry with ulError %lu\n", ulError);
      return ulError;
    }
    
    if (usIamInAccessCode) {  /*only write it out if it is non-zero*/
      sprintf(szTempBuf,"%u", usIamInAccessCode);
      WritePrivateProfileString(WORLDTRADE, DPACCESSENTRY, szTempBuf, ini_file);
    }
  }

  /* If there is a change, make it happen*/
  if (G.usCurrentCountry != usCountryNum) {
    /* write the country to the mwave.ini file*/
    G.usWTTloaded = NOT_LOADED;  /*invalidate the current WTT*/
    G.ulInitError = WTTPOSERR_COUNTRY_NOT_LOADED;
    
    ulError = WTT_WTTableHasChanged (NULL, 0);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwtbl::WTT_ChangeWTTable - failed WTT_WTTableHasChanged with ulError = %lu\n", ulError);
      return ulError;
    }
  }

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_ChangeWTTable exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /*end WTT_ChangeWTTable*/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_WTTableHasChanged*/
/**/
/*     This function is called when the WT Table has changed.  This is called*/
/*     by the WTT_ChangeWTTable() function to reload the WT Table and to*/
/*     notify users of the change.  This function can also be called by a*/
/*     routine that is monitoring dialing properties if that*/
/*     is being used for country selection.  This function calls all connected*/
/*     clients to notify them that the table has changed.*/
/**/
/*     Params:   hDsp            - The DSP handle, one WT Table per DSP.*/
/*               ulReserved      - A reserved location for future API expansion.*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*     Overall flow:*/
/*       call Initialize_WTT*/
/*       call user notification function*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_WTTableHasChanged (HDSP hDsp, ULONG ulReserved )
{
  (void) hDsp;
  (void) ulReserved;

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_WTTableHasChanged entry\n");
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_WTTableHasChanged. DYNAMIC UPDATE NOT SUPPORTED WARNING\n");
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_WTTableHasChanged exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;

} /*end WTT_WTTableHasChanged*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_CheckCountryNumber  */
/**/
/*     This is a dummy copy of the WTT_CheckCountryNumber function.*/
/*     This function always returns TRUE.  This version is included*/
/*     so that the modem driver can always make a call without ifdef's.*/
/**/
/*     Params:   hDsp                         - The DSP handle, one WT Table per DSP.*/
/*               lpbCountryNumberIsOK         - is set to TRUE*/
/*               BOOL bCallMwaveCountrySelect - N/A*/
/*               ulReserved                   - A reserved location for future API expansion.*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from list in mwwttbl.h*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_CheckCountryNumber (HDSP hDsp, BOOL FAR *lpbCountryNumberIsOK,
                                       BOOL bCallMwaveCountrySelect, ULONG ulReserved ) {
  USHORT usDisableWTT;
  (void) hDsp;
  (void) bCallMwaveCountrySelect;
  (void) ulReserved;

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumber entry, bCallMaveCountrySelect %x\n",bCallMwaveCountrySelect);

  usDisableWTT = (USHORT) GetPrivateProfileInt(WORLDTRADE, DISABLEWTT, 0, ini_file);
  
  if (usDisableWTT)
    *lpbCountryNumberIsOK = FALSE;
  else
    *lpbCountryNumberIsOK = TRUE;
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumber exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /*end WTT_CheckCountryNumber*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_CheckCountryNumberStatus*/
/**/
/*     This function verifies that the current country number matches the country*/
/*     selected in the Dialing Properties.  Returned is status for if*/
/*     the country matches or why it does not.*/
/**/
/*     Params:   hDsp                    - The DSP handle, one WT Table per DSP.*/
/*               lpulCountryNumberStatus - Pointer to return the status of the*/
/*                                         country number in.*/
/*               ulReserved              - A reserved location for future API expansion.*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from list in mwwttbl.h*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_CheckCountryNumberStatus (HDSP hDsp, ULONG FAR *lpulCountryNumberStatus,
                                             ULONG ulReserved )
{
  ULONG ulError;
  USHORT usIamInCountry, usIamInAccessCode, usLast_DP_Access_Code;
  ULONG  ulIamInWTTnum, usDisableWTT;
  BOOL   bCheckLastDialingProperties;
  char   szTempBuf[10];
  (void) hDsp;
  (void) ulReserved;

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus entry\n");

  bCheckLastDialingProperties = FALSE;
  
  /* First let's see if we are disabled*/
  usDisableWTT = (USHORT) GetPrivateProfileInt(WORLDTRADE, DISABLEWTT, 0, ini_file);

  if (usDisableWTT) {
    *lpulCountryNumberStatus = NOT_YET_AVAILABLE; /* assume OK*/
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus exit WTT_SUCCESSFUL\n");
    return WTT_SUCCESSFUL;
  } else
    *lpulCountryNumberStatus = CS_OK; /* assume OK*/

  /*This should only do something if the Check_Dialing_Properties flag was set*/
  /*in the Mwave.ini file.*/
  if (G.usCheckDialingProperties) {
    /*------ Get the country specified by TAPI Dialing Properties-----------------*/
    ulError = GetTAPIcountry (&usIamInCountry, &usIamInAccessCode); /* pass the current country pointer*/
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus - failed GetTAPIcountry with ulError %lu\n", ulError);
      return ulError;
    }
    
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus - dialing properties country number = %u\n", usIamInCountry);
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus -              Mwave country number = %u\n", G.usCurrentCountry);
    
    /* If the mapping returns WT_COUNTRY_INVALID then the access code in the dialing*/
    /* properties is not in the mapping file.*/
    if (usIamInCountry != WT_COUNTRY_INVALID) {
      /* At this point we have a valid mapping, we need to see if that WTT is currently present*/
      ulError = MapCountryNumToWTTnum ((ULONG)usIamInCountry, &ulIamInWTTnum);
      if (ulError) {
        MW_SYSLOG_2(TRACE_MWWTT32,"mwwtbl::WTT_CheckCountryNumberStatus - failed MapCountryNumToWTTnum with ulError = %lu\n", ulError);
        return ulError;
      }
      
      if (G.bWTTpresent[(int)ulIamInWTTnum]) {
        /* Now we have a mapped country that is present, is it what is currently selected?*/
        if (G.usCurrentCountry != usIamInCountry) {
          MW_SYSLOG_1(TRACE_MWWTT32,"mwwtbl::WTT_CheckCountryNumberStatus - Dialing Properties changed\n");
	  /*          WTT_WTTableHasChanged (hDsp, ulReserved);*/
          sprintf(szTempBuf, "%u", usIamInCountry);
          WritePrivateProfileString("WORLDTRADE", "Newcountry", szTempBuf, ini_file);
          *lpulCountryNumberStatus = DP_CHANGED;
        } else {
          MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus - Dialing Properties did not change\n");
          *lpulCountryNumberStatus = CS_OK;
	  
          /*If this is the very first system run, it is possible for the*/
          /*Last_DP_Access_Code value to not be set in the Mwave.ini file,*/
          /*so write it here.*/
          sprintf (szTempBuf, "%u", usIamInAccessCode);
          WritePrivateProfileString (WORLDTRADE, DPACCESSENTRY, szTempBuf, ini_file);
        }
      } else {
        /* Always say country is not OK*/
        MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus - Country is not yet available\n");
        *lpulCountryNumberStatus = NOT_YET_AVAILABLE;
      }
    } else {
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus - unsupported country - GENERIC SUPPORT\n");
/*      bCheckLastDialingProperties = TRUE;*/
      if (G.usCurrentCountry != WT_COUNTRY_GENERIC) {
        sprintf(szTempBuf, "%u", WT_COUNTRY_GENERIC);
        WritePrivateProfileString("WORLDTRADE", "Newcountry", szTempBuf, ini_file);
        *lpulCountryNumberStatus = DP_CHANGED;
      } else {
        *lpulCountryNumberStatus = CS_OK;
        sprintf (szTempBuf, "%u", usIamInAccessCode);
        WritePrivateProfileString (WORLDTRADE, DPACCESSENTRY, szTempBuf, ini_file);
      }
    }
    
    
    /* If the dialing properties setting does not match the Mwave setting*/
    /* then we need to determine if anything has changed or if the country*/
    /* has previously been set up.*/
    if (bCheckLastDialingProperties) {
      usLast_DP_Access_Code = (USHORT) GetPrivateProfileInt(WORLDTRADE,
								  DPACCESSENTRY, 0, ini_file);

      if (!usLast_DP_Access_Code) {
        MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus - Last_DP_Access_Code is blank\n");
	
	sprintf (szTempBuf, "%u", usIamInAccessCode);
        WritePrivateProfileString (WORLDTRADE, DPACCESSENTRY, szTempBuf, ini_file);
      } else if (usIamInAccessCode == usLast_DP_Access_Code) {
        MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus - unsupported access code matches Last_DP_Access_Code\n");
        *lpulCountryNumberStatus = UNSUPPORTED_BUT_EQUAL_TO_LAST_DP;
      }
    } /* endif Check_Last_Dialing_Properties*/
  } /*end if Check_Dialing_Properties*/
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckCountryNumberStatus exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /*end WTT_CheckCountryNumberStatus*/

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_CheckForCountryNumberChange*/
/**/
/*     This function verifies that the current country number matches the country*/
/*     selected in the Dialing Properties.  Returned is status for if*/
/*     the country matches or why it does not.*/
/**/
/*     Params:   hDsp                    - The DSP handle, one WT Table per DSP.*/
/*               lpulCountryNumberStatus - Pointer to return the status of the*/
/*                                         country number in.*/
/*               ulReserved              - A reserved location for future API expansion.*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from list in mwwttbl.h*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_CheckForCountryNumberChange (HDSP hDsp, BOOL FAR *lpbCountryNumberChange,
                                                ULONG ulReserved )
{
  ULONG ulError;
  USHORT usIamInCountry, usIamInAccessCode;
  ULONG  ulIamInWTTnum, usDisableWTT;

  (void) hDsp;
  (void) ulReserved;

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange entry\n");

  *lpbCountryNumberChange = FALSE;
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::Entering WTT_CheckForCountryNumberChange\n");
  /* First let's see if we are disabled*/
  usDisableWTT = (USHORT) GetPrivateProfileInt(WORLDTRADE, DISABLEWTT, 0, ini_file);

  if (usDisableWTT) {
    *lpbCountryNumberChange = FALSE; /* assume OK*/
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange exit WTT_SUCCESSFUL\n");
    return WTT_SUCCESSFUL;
  } else
    *lpbCountryNumberChange = TRUE; /* assume OK*/
  
  /*This should only do something if the Check_Dialing_Properties flag was set*/
  /*in the Mwave.ini file.*/
  if (G.usCheckDialingProperties) {
    /*------ Get the country specified by TAPI Dialing Properties-----------------*/
    ulError = GetTAPIcountry (&usIamInCountry, &usIamInAccessCode); /* pass the current country pointer*/
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange - failed GetTAPIcountry with ulError %lu\n", ulError);
      return ulError;
    }
    
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange - dialing properties country number = %u\n", usIamInCountry);
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange -              Mwave country number = %u\n", G.usCurrentCountry);
    
    /* If the mapping returns WT_COUNTRY_INVALID then the access code in the dialing*/
    /* properties is not in the mapping file.*/
    if (usIamInCountry != WT_COUNTRY_INVALID) {
      /* At this point we have a valid mapping, we need to see if that WTT is currently present*/
      ulError = MapCountryNumToWTTnum ((ULONG)usIamInCountry, &ulIamInWTTnum);
      if (ulError) {
        MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange - failed MapCountryNumToWTTnum with ulError = %lu\n", ulError);
        return ulError;
      }

      if (G.bWTTpresent[(int)ulIamInWTTnum]) {
        /* Now we have a mapped country that is present, is it what is currently selected?*/
        if (G.usCurrentCountry != usIamInCountry) {
          MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange - Dialing Properties changed\n");
          *lpbCountryNumberChange = FALSE;
        } else {
          MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange - Country is OK\n");
          *lpbCountryNumberChange = TRUE;
        }
      } else {
        /* Always say country is not OK*/
        MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange - Country is not yet available\n");
        *lpbCountryNumberChange = FALSE;
      }
    } else {
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange - unsupported country - GENERIC SUPPORT\n");
      if (G.usCurrentCountry != WT_COUNTRY_GENERIC) {
        MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange - unsupported country - DP_CHANGED\n");
        *lpbCountryNumberChange = FALSE;
      } else {
        MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange - unsupported country - CS_OK\n");
        *lpbCountryNumberChange = TRUE;
      }
    }
  }
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_CheckForCountryNumberChange exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
  
} /*end WTT_CheckForCountryNumberChange*/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_QueryInfo*/
/**/
/*     Provides a means to get information from the WT Table.  Specifications*/
/*     for what is in the WT Table come from the WTT.H file.*/
/**/
/*     Params:   hDsp          - The DSP handle, one WT Table per DSP.*/
/*               usWTTBlockTag - WTITCB, Blacklist, etc. tag from WTT.H file.*/
/*               usOffect      - Parameter or field starting location from*/
/*                               WTT.H file.*/
/*               ulNumShorts   - Number words to copy from the WTT starting*/
/*                               with the offset into the block.  Field lengths*/
/*                               are defined in WTT.H file.  For full block lengths,*/
/*                               use the WTT_QueryBlockLength function.*/
/*               lpsBuffer     - location to return the information in*/
/*               ulReserved    - A reserved location for future API expansion.*/
/**/
/*               Note:  this function changes the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*     Overall flow:*/
/*       call QueryInfo*/
/*         call GetBlock*/
/*         move data to caller's memory*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_QueryInfo (HDSP hDsp, USHORT usWTTBlockTag, USHORT usOffset, ULONG ulNumShorts, short FAR* lpsBuffer, ULONG ulReserved )
{
   ULONG ulError;
  (void) hDsp;
  (void) ulReserved;

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_QueryInfo - entering with usWTTBlockTag = %04X\n", usWTTBlockTag);
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_QueryInfo - entering with usOffset = %u\n", usOffset);
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_QueryInfo - entering with ulNumShorts = %lu\n", ulNumShorts);
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_QueryInfo - entering with lpsBuffer = %p hex\n", lpsBuffer);

  /* Validate that the WT Table is loaded and ready to go.  This validation is what*/
  /* distinquishes WTT_QueryInfo() from QueryInfo().*/
  if (G.usWTTloaded != WTT_LOADED) {         /*return if the WTT has not been initialized*/
    BOOL bRC=0;
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_QueryInfo - calling Init_Lib\n");
    bRC=Init_Lib(1);
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_QueryInfo - Return from Init_Lib bRC %x\n", bRC);
    if (!bRC)
      return G.ulInitError;
  }
  
  ulError = QueryInfo (usWTTBlockTag, usOffset, ulNumShorts, lpsBuffer);
  
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_QueryInfo - failed QueryInfo with ulError = %lu\n", ulError);
    return ulError;
  }

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_QueryInfo - first word in return buffer is %u\n", *lpsBuffer);
  return WTT_SUCCESSFUL;
} /*end WTT_QueryInfo*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WTT_QueryBlockLength*/
/**/
/*     Provides a means to get the length of a block in the current WT record.*/
/**/
/*     Params:   hDsp          - The DSP handle, one WT Table per DSP.*/
/*               usWTTBlockTag - WTITCB, Blacklist, etc. tag from WTT.H file.*/
/*               lpulNumShorts - A pointer to where the block length in words*/
/*                               (shorts) will be returned.*/
/*               ulReserved    - A reserved location for future API expansion.*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from list MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY WTT_QueryBlockLength (HDSP hDsp, USHORT usWTTBlockTag, ULONG FAR* lpulNumShorts, ULONG ulReserved )
{
  ULONG ulError;
  USHORT* pusBlock = NULL;     /* pointer to Block portion of Record cache*/

  (void) hDsp;
  (void) ulReserved;

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_QueryBlockLength entry, usWTTBlockTag %x\n",usWTTBlockTag);
  
  if (G.usWTTloaded == NOT_LOADED) {       /*return if the WTT is not in memory*/
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_QueryBlockLength - failed since G.usWTTloaded == NOT_LOADED\n");
    return WTTPOSERR_COUNTRY_NOT_LOADED;
  }
  
  /* Get a pointer to the block of interest*/
  ulError = GetBlock (usWTTBlockTag, &pusBlock);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_QueryBlockLength - failed GetBlock with ulError %lu\n", ulError);
    return ulError;
  }
  
  /* *pusBlock is the block length in 16bit words which includes the block header.*/
  if (*pusBlock <= (sizeof (BLOCKHEADER) / 2)) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_QueryBlockLength - failed - the block length found is invalid\n");
    return WTTPOSERR_BLOCK_LENGTH_INVALID;
  }

  *lpulNumShorts = (ULONG) (*pusBlock - (sizeof (BLOCKHEADER) / 2));

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WTT_QueryBlockLength - returning a block length of %lu\n", *lpulNumShorts);
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::mwwttbl::WTT_QueryBlockLength exit WTT_SUCCESSFUL\n");

  return WTT_SUCCESSFUL;
}


/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/*   MWWTT internal functions*/
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  Initialize_WTT*/
/**/
/*     This function needs to be called when the system starts up to initialize*/
/*     the Mwave WT Table Access functions.*/
/**/
/*     Params:   none*/
/**/
/*               Note:  this function changes the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTT.H*/
/**/
/*     Overall flow:*/
/*        call ReadINIfiles*/
/*        call FindAvailableCountries*/
/*        call ReadMapTable*/
/*        call LoadWTTable*/
/*             call MapCountryNumToWTTnum*/
/*             call FindCountry*/
/*             call UpdateSettings*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG Initialize_WTT () {
  ULONG ulError;
  USHORT usIamInCountry, usIamInAccessCode;
  ULONG ulIamInWTTnum;
  char szTempBuf[6];

  (void) copyright;  /*global variable*/

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::Initialize_WTT entered\n");

  /* This variable is used to tell if the WT Table is loaded into memory.*/
  G.usWTTloaded = NOT_LOADED;

  /* This variable stores the error code if this init routine fails.*/
  /* Assume failure and then assign the actual return value when this*/
  /* routine returns, at the calling location.*/
  ulError = WTTPOSERR_COUNTRY_NOT_LOADED;

  /* This variable used to tell that the mapping information has not been intialized*/
  G.bMappingInitialized = FALSE;

  /* Now get the Mwave.ini settings and possibly the Telephon.ini settings*/
  /* depending on where the country selection is coming from.*/
  ulError = ReadINIfiles(G.szDatabase, sizeof(G.szDatabase), &(G.usCurrentCountry),
                         &(G.usWTTversion), &(G.usCheckDialingProperties),
                         G.szCountrySelect, &(G.usNo_OC_Hangup));
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::Initialize_WTT - failed ReadINIfiles with ulError %lu\n", ulError);
    return ulError;
  }
  
  /* At this point the Mwave.ini and Telephon.ini settings are stored in the*/
  /* Global structure.  We will proceed by finding out which WT Tables are*/
  /* present in the database.*/
  
  /* Note that FindAvailableCountries changes settings in the Global Structure.*/
  ulError = FindAvailableCountries ();
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::Initialize_WTT - failed FindAvailableCountries with ulError %lu\n", ulError);
    return ulError;
  }

  /* Now that we know what is available, let's set up the mapping.  ReadMapTable*/
  /* makes changes in the Global Structure.*/
  ulError = ReadMapTable();
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::Initialize_WTT - failed ReadMapTable with ulError %lu\n", ulError);
    return ulError;
  }
  
  /* Now we need to determine if the Dialing Properties country will override the*/
  /* Mwave.ini setting.*/
  if (G.usCheckDialingProperties) {
    /* Now we need to get the Dialing Properties country setting.*/
    ulError = GetTAPIcountry (&usIamInCountry, &usIamInAccessCode);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::Initialize_WTT - failed GetTAPIcountry with ulError %lu\n", ulError);
      return ulError;
    }
    MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::Initialize_WTT, from GetTAPIcountry usIamInCountry %x usIamInAccessCode %x\n",usIamInCountry,usIamInAccessCode);
    
    ////////////////////////////////////////////////////////////////////////////////////**/
    /* OK, at this point we have both the Mwave country number and the Dialing Properties*/
    /* country number.  Are they the same?*/
    
    /* If the mapping returns WT_COUNTRY_INVALID then the access code in the dialing*/
    /* properties is not in the mapping file and is not a country we care about.*/
    if (usIamInCountry != WT_COUNTRY_INVALID) {
      /* At this point we have a valid mapping, we need to see if that WTT is currently present*/
      ulError = MapCountryNumToWTTnum ((ULONG)usIamInCountry, &ulIamInWTTnum);
      
      if (ulError) {
        MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::Initialize_WTT - failed MapCountryNumToWTTnum with ulError = %lu\n", ulError);
        return ulError;
      }
      
      if (G.bWTTpresent[(int)ulIamInWTTnum]) {
        /* Now we have a mapped country that is present, is it what is currently selected?*/
        if (G.usCurrentCountry != usIamInCountry) {
          /* Write out this new setting to the Mwave.ini file*/
          G.usCurrentCountry = usIamInCountry;
          sprintf(szTempBuf,"%u", G.usCurrentCountry);
          WritePrivateProfileString(WORLDTRADE, COUNTRYENTRY, szTempBuf, ini_file);
	  
        }
	
        /* Note this country as the Last_DP_Access_Code*/
        sprintf(szTempBuf,"%u", usIamInAccessCode);
        WritePrivateProfileString(WORLDTRADE, DPACCESSENTRY, szTempBuf, ini_file);
      }
    } else {
      if (G.usCurrentCountry != WT_COUNTRY_GENERIC) {
        /* Write out this new setting to the Mwave.ini file*/
        G.usCurrentCountry = WT_COUNTRY_GENERIC;
        sprintf(szTempBuf,"%u", G.usCurrentCountry);
        WritePrivateProfileString(WORLDTRADE, COUNTRYENTRY, szTempBuf, ini_file);
	
      }
      
      /* Note this country as the Last_DP_Access_Code*/
      sprintf(szTempBuf,"%u", usIamInAccessCode);
      WritePrivateProfileString(WORLDTRADE, DPACCESSENTRY, szTempBuf, ini_file);
    }
  }

  /* At this point it is time to load the current country specified in the*/
  /* ini files.  This country will be mapped to the correct WT Table.*/
  /* The LoadWTTable routine assumes that Mwave.ini settings in the Global*/
  /* Structure are initialized and this routine fills in the pusRecord in*/
  /* the Global Structure.*/
  ulError = LoadWTTable ((ULONG)G.usCurrentCountry, usIamInAccessCode);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::Initialize_WTT - failed LoadWTTable with ulError %lu\n", ulError);
    return ulError;
  }
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::Initialize_WTT exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /* end Initialize_WTT*/





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  QueryInfo*/
/**/
/*     Internal workings for the WTT_QueryInfo function.  Unlike WTT_QueryInfo,*/
/*     this function can be called after the WT Table has been read into memory*/
/*     but before the G.usWTTloaded is set to WTT_LOADED.  This is used by the routine*/
/*     which updates the WT Table power levels and other settings.*/
/**/
/*     Params:   usWTTBlockTag - WTITCB, Blacklist, etc. tag from WTT.H file.*/
/*               usOffect      - Parameter or field starting location from*/
/*                               WTT.H file.*/
/*               ulNumShorts   - Number words to copy from the WTT starting*/
/*                               with the offset into the block.  Field lengths*/
/*                               are defined in WTT.H file.*/
/*               lpsBuffer     - location to return the information in*/
/**/
/*               Note:  this function uses the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTT.H*/
/**/
/*     Overall flow:*/
/*       call GetBlock*/
/*       move data to caller's memory*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG QueryInfo (USHORT usWTTBlockTag, USHORT usOffset, ULONG ulNumShorts, short FAR* lpsBuffer) {
  ULONG ulError;
  ULONG ulBytes;
  USHORT usCopyLength;
  USHORT* pusBlock = NULL;     /* pointer to Block portion of Record cache*/

  //MW_SYSLOG_4(TRACE_MWWTT32,"mwwttbl::QueryInfo entered, usWTTBlockTag %x usOffset %x ulNumShorts %lx\n",usWTTBlockTag,usOffset,ulNumShorts);


  if (G.usWTTloaded == NOT_LOADED) {      /*return if the WTT is not in memory*/
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::QueryInfo - failed since G.usWTTloaded == NOT_LOADED\n");
    return WTTPOSERR_COUNTRY_NOT_LOADED;
  }

  /* Get a pointer to the block of interest*/
  ulError = GetBlock (usWTTBlockTag, &pusBlock);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::QueryInfo - failed GetBlock with ulError %lu\n", ulError);
    return ulError;
  }

  /* Check to make sure the usOffset has a good value in it.  *pusBlock is the*/
  /* block length in 16bit words.*/
  if (usOffset >= *pusBlock) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::QueryInfo - failed because the requested offset is larger than the block length\n");
    return WTTPOSERR_INVALID_OFFSET;
  }
  
  /* Determine the amount to copy*/
  usCopyLength = *pusBlock - usOffset;    /* *pusBlock == block length*/
                                          /* Assume usCopyLength = rest of block*/

  if (ulNumShorts < (ULONG) usCopyLength)         /* Adjust usCopyLength to be the amount requested*/
    usCopyLength = (USHORT) ulNumShorts;

  /* At this point if the usCopyLength is not the same as the ulNumShorts then the call*/
  /* requested more data than was in the block after the offset.*/
  if (ulNumShorts != (ULONG) usCopyLength) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::QueryInfo - failed because the requested number of words is invalid\n");
    return WTTPOSERR_BLOCK_LENGTH_INVALID;
  }
  
  /* Move the Block pointer by the offset and length of the header*/
  /* pusBlock points to words and usOffset is in words*/
  pusBlock = pusBlock + usOffset + (sizeof(BLOCKHEADER) / TWO_BYTES);
  
  ulBytes = (ULONG) usCopyLength * TWO_BYTES;  /* convert words to bytes*/
  
  /* move data from cache to callers storage*/
  memcpy((void FAR*) lpsBuffer, (void FAR*) pusBlock, (size_t) ulBytes);
  //MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::QueryInfo exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /*end QueryInfo*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  GetBlock*/
/**/
/*     This function is used internally to find a block within*/
/*     the record obtained by GetCountryRecord.*/
/**/
/*     Params:   usType    - specifies the Block Type to match up*/
/*               ppusBlock  - a pointer to the block pointer*/
/**/
/*               Note:  this function uses the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG GetBlock (USHORT usType, USHORT **ppusBlock) {
  USHORT *pusBlock = NULL;

  //MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetBlock entered, usType %x\n",usType);
  
  if (G.usWTTloaded == NOT_LOADED) {       /*return if the WTT is not in memory*/
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetBlock - failed since G.usWTTloaded == NOT_LOADED\n");
    return WTTPOSERR_COUNTRY_NOT_LOADED;
  }
  
  pusBlock = G.pusRecord;           /* start at the top of the record*/
  
  /* Process the blocks one at a time until a 0 is found for the block length*/
  while (*pusBlock) {
    if (pusBlock[1] == usType) {
      *ppusBlock = pusBlock;
      //MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetBlock exit WTT_SUCCESSFUL\n");
      return WTT_SUCCESSFUL;
    }
    pusBlock += *pusBlock;  /* pusBlock is pointing to the block length field,*/
    /* skip to next block*/
  }
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetBlock exit WTTPOSERR_BLOCK_TYPE_NOT_FOUND\n");
  return WTTPOSERR_BLOCK_TYPE_NOT_FOUND;
} /*end GetBlock*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  ReadINIfiles*/
/**/
/*     This routine reads the Mwave.ini file to get the worldtrade settings.*/
/**/
/*     Params:   pszDatabase - location to return the database string*/
/*               usMaxStringSize - the maximum string size to tell the GetPrivateProfileString function*/
/*               pusCurrentCountry - location to return the current country setting*/
/*               pusWTTversion - location to return the WTTversion setting*/
/*               pusCheckDialingProperties - location to return the Check_Dialing_Properties flag*/
/*               pszCountrySelect - location to return the country select path and name in*/
/*               pusNo_OC_Hangup - location to return the No_OC_Hangup flag*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG ReadINIfiles(char FAR *pszDatabase, USHORT usMaxStringSize,
                   USHORT FAR *pusCurrentCountry,
                   USHORT FAR *pusWTTversion,
                   USHORT FAR *pusCheckDialingProperties,
                   char FAR *pszCountrySelect,
                   USHORT FAR *pusNo_OC_Hangup)
{
  char szSetNumber[3];
  char szDatabaseEntry[sizeof(DATABASE) + 3]; /*allow for the database set number append*/

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadINIfiles entered\n");
  
  strcpy (szDatabaseEntry, DATABASE);

  if (G.usDatabaseSetNumber) {  /* zero is not appended to the database string*/
    /* paulsch */
    /* itoa (G.usDatabaseSetNumber, szSetNumber, 10); */
    sprintf(szSetNumber, "%hu", G.usDatabaseSetNumber);
    strcat (szDatabaseEntry, szSetNumber);
  }
  
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::ReadINIFiles - getting property for szDatabaseEntry %s\n", szDatabaseEntry);
  /*------ Get the database string -------------------------------------------*/ 
  if (!GetPrivateProfileString(WORLDTRADE, szDatabaseEntry, "",
                               pszDatabase, usMaxStringSize, ini_file)) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadINIfiles - failed GetPrivateProfileString for database string\n");
    return WTTPOSERR_GETTING_DB_NAME;
  }

   MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::ReadINIfiles - using %s=%s\n", szDatabaseEntry, pszDatabase);
  
  /*------ Get the CheckDialingProperties flag----------------------------------*/
  *pusCheckDialingProperties = (USHORT)GetPrivateProfileInt(WORLDTRADE,
							    CHECK_DIALING_PROPERTIES, 0, ini_file);

  if (*pusCheckDialingProperties) {
    /*------ Now, get the path to the country select utility ---------------------*/
    if (!GetPrivateProfileString(WORLDTRADE, MWCOUNTRYSELECT, "",
                                 pszCountrySelect, usMaxStringSize, ini_file)) {
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadINIfiles - failed GetPrivateProfileString for Mwave country select string\n");
      //  @TBD return WTTPOSERR_GETTING_COUNTRY_SELECT_PATH;
    }
     MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::ReadINIfiles - the Mwave Country Select Utility is %s\n", pszCountrySelect);
  }

  *pusCurrentCountry=(USHORT)GetPrivateProfileInt(WORLDTRADE,
						  COUNTRYENTRY, 0, ini_file);
  if (!(*pusCurrentCountry)) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadINIfiles - failed GetPrivateProfileInt for country number\n");
    return WTTPOSERR_INVALID_MWAVEINI_COUNTRY;
  }

  /* Validate the country number*/
  if ((*pusCurrentCountry == 0) || (*pusCurrentCountry > ACCESS_CODE_MAX)) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::ReadINIfiles - failed - country number = %u from Mwave.ini file is out of range\n", *pusCurrentCountry);
    return WTTPOSERR_INVALID_MWAVEINI_COUNTRY;
  }

  /*------ Get the WTTversion ------------------------------------------------*/
  *pusWTTversion=(USHORT)GetPrivateProfileInt(WORLDTRADE,
                                              WTTVERSION, 0, ini_file);
  if (!(*pusWTTversion) || (*pusWTTversion < WTTVERSION_MIN_SUPPORTED)) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadINIfiles - failed to get WTTversion from the Mwave.ini file\n");
    return WTTPOSERR_WTTVERSION_INVALID;
  }

  *pusNo_OC_Hangup=(USHORT)GetPrivateProfileInt(WORLDTRADE,
						NO_OC_HANGUP, 0, ini_file);
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadINIfiles exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /*end ReadINIfiles*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  GetTAPIcountry*/
/**/
/*     This routine gets the current country set for the Dialing Properties.*/
/**/
/**/
/*     Params:  pusCountryNum - the Mwave equivalent country number.  This may*/
/*                              be WT_COUNTRY_INVALID.*/
/*              pusCountryAccessCode - the Dialing Properties access code setting.*/
/*                                     This may be zero if no access code is available.*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG GetTAPIcountry (USHORT FAR *pusCountryNum, USHORT FAR *pusCountryAccessCode) {
  ULONG ulError;

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetTAPIcountry entry\n");

  ulError=GetConfigCountry(pusCountryNum, pusCountryAccessCode);
  
  MW_SYSLOG_4(TRACE_MWWTT32,"mwwttbl::GetTAPIcountry exit, ulError %lx usCountryNum %x usCountryAccessCode %x\n",ulError,*pusCountryNum,*pusCountryAccessCode);
  return ulError;
} /* end GetTAPIcountry*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  GetINIcountry*/
/**/
/*     This routine calls the mwdpw95.exe utility to obtain the current*/
/*     TAPI country setting from the telephon.ini file.*/
/**/
/*     Params:  pusCountryNum - the Mwave equivalent country number.  This may*/
/*                              be WT_COUNTRY_INVALID.*/
/*              pusCountryAccessCode - the Dialing Properties access code setting.*/
/*                                     This may be zero if no access code is available.*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG GetINIcountry(USHORT FAR *pusCountryNum, USHORT FAR *pusCountryAccessCode) {
  ULONG ulCountryNum, ulError, ulAccessCode;
  long lLocationIndex, lStrLen;
  int index, count;
  BOOL bDigitFound;
  BOOL bInString;
#define MAX_SIZE  255
  char szLocation[MAX_SIZE];
  char szLocationInfo[MAX_SIZE];
  char szFailed[] = "FAILED"; /*used to detect if the default string is used*/
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetINIcountry entry\n");

  /* First get the CurrentLocation code from the Locations section of the*/
  /* Telephon.ini file.  This string is of the format Location_ID,Location_index*/
  lStrLen = (long) GetPrivateProfileString ("Locations", "CurrentLocation",
					    szFailed, szLocation,
					    MAX_SIZE, "TELEPHON.INI");

  if (!strcmp (szLocation, szFailed)) {
    /* There no locations currently set in the Telephon.ini file return 0 as*/
    /* the Access Code and an invalid WTT.*/
    *pusCountryNum = WT_COUNTRY_INVALID;
    *pusCountryAccessCode = 0;
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetINIcountry exit WTT_SUCCESSFUL (usCountryNum WT_COUNTRY_INVALID)\n");
    return WTT_SUCCESSFUL;
  }

  /* Note that GetPrivateProfileString returns the number of characters*/
  /* in the return string but leaves the NULL terminator out of the count.*/
  /* "0,0" is the minumum size string*/
  if (lStrLen < 3) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetINIcountry - failed to get the CurrentLocation from the telephon.ini file\n");
    return WTTPOSERR_TELEPHON_INI_INVALID;
  }

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetINIcountry - CurrentLocation = %s\n", szLocation);

  /* determine the locations index of the format:  Location_ID,Location_index*/
  index = 0;
  
  /* find the comma*/
  while ((index < lStrLen) && (szLocation[index] != ','))
    index++;
  
  /* move to one location beyond the comma*/
  index++;
  
  /* move to the start of the number (skip spaces)*/
  while ((index < lStrLen) && (szLocation[index] == ' '))
    index++;
  
  lLocationIndex = 0;
  bDigitFound = FALSE;
  /* get the number out of the string*/
  while ((index < lStrLen) && (szLocation[index] >= '0')
	 && (szLocation[index] <= '9')) {
    bDigitFound = TRUE;
    lLocationIndex = (lLocationIndex * 10) + (szLocation[index] - '0');
    index++;
  }
  
  if (!bDigitFound) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetINIcountry - failed - could not extract the location index\n");
    return WTTPOSERR_TELEPHON_INI_INVALID;
  }
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetINIcountry - lLocationIndex = %ld\n", lLocationIndex);
  
  if (lLocationIndex > 32767) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetINIcountry - failed - the location index was out of range\n");
    return WTTPOSERR_TELEPHON_INI_INVALID;
  }

  /* now find the Locationx string*/
  sprintf (szLocation, "Location%d", (int) lLocationIndex);

  lStrLen = (long) GetPrivateProfileString ("Locations", szLocation,
					    "", szLocationInfo,
					    MAX_SIZE, "TELEPHON.INI");

  /* Note that lStrLen does not include the NULL terminator as part of the count.*/
  if (lStrLen < 2) {  /* There needs to at least be a "1," type entry*/
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetINIcountry - failed - could not get the locationx string from the telephon.ini file\n");
    return WTTPOSERR_TELEPHON_INI_INVALID;
  }
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetINIcountry - szLocationInfo = %s\n", szLocationInfo);
  
  /* In the Locationx string, the access code is stored as the sixth parameter.*/
  /* That is, the access code is after the 5th comma outside strings ("").*/
  index = 0;
  bInString = FALSE;

  for (count = 1; count <= 5; count++) {
    while (index < lStrLen) {
      if (szLocationInfo[index] == '"')
        bInString = !bInString;
      
      if (!bInString && (szLocationInfo[index] == ','))
        break;
      
      index++;
    }
    /* move to one location beyond the comma*/
    index++;
  }
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetINIcountry - After searching for the 5th comma index = %u\n", index);
  
  /* move to the start of the number (skip blanks)*/
  while ((index < lStrLen) && (szLocationInfo[index] == ' '))
    index++;
  
  ulAccessCode = 0;
  bDigitFound = FALSE;
  /* get the number*/
  while ((index < lStrLen) && (szLocationInfo[index] >= '0')
                           && (szLocationInfo[index] <= '9')) {
    bDigitFound = TRUE;
    ulAccessCode = (ulAccessCode * 10) + (szLocationInfo[index] - '0');
    index++;
  }
  
  if (!bDigitFound) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetINIcountry - failed - could not get the access code from the locationx string\n");
    return WTTPOSERR_TELEPHON_INI_INVALID;
  }
  
  if (ulAccessCode > ACCESS_CODE_MAX) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetINIcountry - failed - the access code was out of range\n");
    return WTTPOSERR_TELEPHON_INI_INVALID;
  }
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetINIcountry - ulAccessCode = %lu\n", ulAccessCode);
  
  /* Now lets map the access code into an Mwave country number*/
  ulError = MapAccessToCountryNum (ulAccessCode, &ulCountryNum);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetINIcountry - failed MapAccessToCountryNum with ulError %lu\n", ulError);
    return ulError;
  }
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetINIcountry - ulCountryNum = %lu\n", ulCountryNum);
  
  *pusCountryNum = (USHORT) ulCountryNum;
  *pusCountryAccessCode = (USHORT) ulAccessCode;
  
  MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::GetINIcountry exit WTT_SUCCESSFUL, usCountryNum %x usCountryAccessCode %x\n",*pusCountryNum,*pusCountryAccessCode);
  return WTT_SUCCESSFUL;
} /*end GetINIcountry*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  GetConfigCountry*/
/**/
/*     Params:  pusCountryNum - the Mwave equivalent country number.  This may*/
/*                              be WT_COUNTRY_INVALID.*/
/*              pusCountryAccessCode - the Dialing Properties access code setting.*/
/*                                     This may be zero if no access code is available.*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG GetConfigCountry(USHORT FAR *pusCountryNum, USHORT FAR *pusCountryAccessCode) {
#define CHARBUFSIZE 4  /* 4 bytes, enough for a DWORD*/
#define LOCATION_LIST_VERSION 2
  DWORD dwCountry=0;
  ULONG ulError, ulCountryNum;

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetConfigCountry entry\n");

  /* If there are no locations currently set in the config, return 0 as*/
  /* the Access Code and an invalid WTT.*/
  *pusCountryNum = WT_COUNTRY_INVALID;
  *pusCountryAccessCode = 0;

  dwCountry=(USHORT)GetPrivateProfileInt(WORLDTRADE,COUNTRYENTRY, 0, ini_file);
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetConfigCountry, dwCountry %lx\n",dwCountry);
  if (dwCountry == 0) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetConfigCountry - RegQueryValueEx for Country not found (or is zero)\n");
    return WTTPOSERR_CONFIG_INVALID;
  }
  
  /* OK, now let's turn the international access number (country number) into*/
  /* a Mwave country number.*/
  if (dwCountry > ACCESS_CODE_MAX) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::GetConfigCountry - failed - the access code was out of range\n");
    return WTTPOSERR_CONFIG_INVALID;
  }
   MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetConfigCountry - dwCountry = %lx\n", dwCountry);
  
  /* Now let's map the access code into an Mwave country number*/
  ulError = MapAccessToCountryNum ((ULONG)dwCountry, &ulCountryNum);
  
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetConfigCountry - failed MapAccessToCountryNum with ulError %lu\n", ulError);
    return ulError;
  }
   MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::GetConfigCountry - ulCountryNum = %lu\n", ulCountryNum);
  
  *pusCountryNum = (USHORT) ulCountryNum;
  *pusCountryAccessCode = (USHORT) dwCountry;
  MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::GetConfigCountry, exit WTT_SUCCESSFUL, usCountryNum %x usCountryAccessCode %x\n",*pusCountryNum,*pusCountryAccessCode);  
  return WTT_SUCCESSFUL;
} /*end GetConfigCountry*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  FindAvailableCountries*/
/**/
/*         This function searches the multi-country WTT to find out what*/
/*         countries are available.*/
/**/
/*         This function assumes that the Mwave.ini settings have been*/
/*         initialized in the Global structure.*/
/**/
/*         This function also changes the Global record storage*/
/*         so if this function is called after a WT Table is*/
/*         loaded, it will no longer be loaded.*/
/**/
/**/
/*     Params:   none, but changes the Global Structure*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG FindAvailableCountries (void) {
  FILE* wtFile;                      /* File handle*/
  RECHEADER RecHeader;               /* header of file*/
  ULONG ulError;                     /* Error handling*/
  USHORT* pusBlock = NULL;           /* pointer to Block portion of Record cache*/
  USHORT usCount;
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries entry\n");

  /* Discard the current WT Table*/
  G.usWTTloaded = NOT_LOADED;
  G.ulInitError = WTTPOSERR_COUNTRY_NOT_LOADED;
  
  /* First initialize the bWTTpresent and usWTTcouplerID arrays*/
  for (usCount = 0; usCount <= WT_COUNTRY_MAX; usCount++) {
    G.bWTTpresent[usCount]    = FALSE;
    G.usWTTcouplerID[usCount] = 30000; /* an arbitrary number*/
  }
  
  /* init file handle*/
  wtFile = NULL;
  
  /* open file for read access*/
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries - fopen of G.szDatabase %s\n", G.szDatabase);
  wtFile = fopen(G.szDatabase, "rb");
  
  if (wtFile == NULL) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries - failed to open database file\n");
    return WTTPOSERR_DATABASE_OPEN_FAILED;
  }
  
  /* Step through the WTT file until a zero length header is found*/
  /* which is the end of file marker.*/
  do {
    ulError = 0;
    
    /* Discard the current WT Table*/
    G.usWTTloaded = NOT_LOADED;
    G.ulInitError = WTTPOSERR_COUNTRY_NOT_LOADED;
    
    /* Let's read in the country record header.  Use a one time do loop for*/
    /* error handling so we can close the file before returning.*/
    do {
      /* get header from file*/
      if (fread(&RecHeader, 1, sizeof(RecHeader), wtFile) != sizeof(RecHeader)) {
	MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries - failed to read a country record header\n");
	ulError = WTTPOSERR_DATABASE_READ_FAILED;
	break;              /* exit and close file*/
      }
      
      if (RecHeader.usHeaderLength != 0) /* this marks the end of the file*/ {
	if ( (RecHeader.usHeaderLength * TWO_BYTES) > MAX_RECORD_SIZE) {
	  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries - failed record size too large\n");
	  ulError = WTTPOSERR_RECORD_TOO_LARGE;
	  break;              /* exit and close file*/
	}
	
	/* read record*/
         if (fread(G.pusRecord, TWO_BYTES, RecHeader.usRecLength, wtFile) < RecHeader.usRecLength) {
           MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries - failed to read a country record\n");
           ulError = WTTPOSERR_DATABASE_READ_FAILED;
           break;              /* exit and close file*/
         }
      }
      
    } while (0); /* one time for error processing*/
    
    
    if (RecHeader.usHeaderLength == 0) /*end of file marker found*/
      break; /*ulError has already been set*/
    
    /* if an error occurred while reading the header or record then fail the function*/
    if (ulError) {
      /* close file*/
      fclose(wtFile);
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries exit ulError %lx\n",ulError);       
      return ulError;
    }
    
    /* WTT's marked as Test Only have the most significant bit of the country number set*/
    /* on.  If this is the case then skip this country as it is not available.*/
    if ((RecHeader.usCountryNum) & 0x8000) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries - country number %u is marked as Test Only\n", RecHeader.usCountryNum & 0x7FFF);
      continue; /* go on to the next record*/
    }
    
    G.usWTTloaded = WTT_LOADING;
    
    /* Now let's do some verification on the contents of the record header*/
    /* and the WT Table checksum.*/
    ulError = VerifyCountryRecord (&RecHeader, G.pusRecord, G.usWTTversion);
    
    if (ulError) {
      MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries - failed VerifyCountryRecord with ulError %lu for country %u\n", ulError, RecHeader.usCountryNum);
      break;
    }
    
    /* If we got this far, then the record is OK.  Now let's mark*/
    /* the WT Table as available and record the coupler ID.*/
    
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries - marking country = 0x%x as present\n", RecHeader.usCountryNum);
    
    G.bWTTpresent[RecHeader.usCountryNum] = TRUE;
    
    if (RecHeader.usCountryNum != 0) {  /*no coupler id for map files*/
      ulError = GetBlock (WTBLOCK_WTITCB, &pusBlock);
      
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries - failed GetBlock with ulError %lu\n", ulError);
	break;
      }
      
      /* Move the Block pointer by the offset and length of the header*/
      /* pBlock points to words and usOffset is in words*/
      pusBlock = pusBlock + WTPARAM_WTTCID + (sizeof(BLOCKHEADER) / TWO_BYTES);
      
      G.usWTTcouplerID[RecHeader.usCountryNum] = *pusBlock;
      
      /* As one last step, if the coupler id filter is set, then only allow countries that*/
      /* have a matching coupler id.*/
      if (G.CouplerIDfilter.bEnabled) {
	if (*pusBlock != G.CouplerIDfilter.ulCouplerID) { /* *pusBlock is currently pointing to the first*/
	  /* word of the WTITCB block which is the coupler id*/
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries - due to Coupler ID mismatch country 0x%x is not available\n", RecHeader.usCountryNum);
	  G.bWTTpresent[RecHeader.usCountryNum] = FALSE;
	}
      }
    }
    
  } while (RecHeader.usHeaderLength != 0);  /*allow the fread error handling to*/
  /*handle end of file reach abnormally*/
  
  G.usWTTloaded = NOT_LOADED;  /* Mark the file as not loaded in case it is*/
                                /* marked as WTT_LOADING.*/
  G.ulInitError = WTTPOSERR_COUNTRY_NOT_LOADED;
  
  /* close file*/
  fclose(wtFile);
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindAvailableCountries exit, ulError %lx\n",ulError);
  return ulError;
} /*end FindAvailableCountries*/

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  VerifyCountryRecord*/
/**/
/*         This function checks the county record header and record*/
/*         checksum.  If no error is returned, then the country record is OK.*/
/**/
/**/
/*     Params:   pRecHeader - a pointer to the record header*/
/*               pusRecord  - a pointer to the record*/
/*               usExpectedWTTversion - this is the value expected in the*/
/*                                      WTTversion field if the record is good.*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG VerifyCountryRecord (RECHEADER *pRecHeader, USHORT *pusRecord,
                           USHORT usExpectedWTTversion) {
  ULONG ulError;

  // MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::VerifyCountryRecord entered\n");

  if (G.usWTTloaded == NOT_LOADED) {       /*return if the WTT is not in memory*/
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::VerifyCountryRecord - failed with G.usWTTloaded == NOT_LOADED\n");
    return WTTPOSERR_COUNTRY_NOT_LOADED;
  }

  /* verify the country number*/
  if (pRecHeader->usCountryNum > WT_COUNTRY_MAX) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::VerifyCountryRecord - failed with usCountryNum = %u greater than the maximum\n", pRecHeader->usCountryNum);
    return WTTPOSERR_INVALID_COUNTRY;
  }
  
  /* verify table format*/
  if ((pRecHeader->usCountryNum) == 0) {  /*map files have different format numbers*/
    if (pRecHeader->usFormat > WT_MAP_TABLE_FORMAT_1) {
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::VerifyCountryRecord - failed with the map file format greater than the maximum\n");
      return WTTPOSERR_INVALID_MAP_TBL_FORMAT;
    }
  } else {
    if (pRecHeader->usFormat != WT_TABLE_FORMAT_5) {
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::VerifyCountryRecord - failed with an unsupported table format\n");
      return WTTPOSERR_INVALID_TABLE_FORMAT;
    }
  }
  
  /* verify the WT Table Version*/
  if (pRecHeader->usWTTableVersion != usExpectedWTTversion)
    return WTTPOSERR_TABLE_VERSION_INVALID;
  
  ulError = VerifyCheckSum (pRecHeader, pusRecord);
  
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::VerifyCountryRecord - failed VerifyCheckSum with ulError %lu\n", ulError);
    return ulError;
  }
  //MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::VerifyCountryRecord exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /*end VerifyCountryRecord*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  VerifyCheckSum*/
/**/
/**/
/*           This procedure calculates the checksum and compares to the value*/
/*           listed at the end of the WT Table.*/
/**/
/*     Params:  pheader  - pointer to the WTT header*/
/*              pRecord - pointer to the WTT record*/
/**/
/*     Globals:  none*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG VerifyCheckSum (RECHEADER *pheader, USHORT *pRecord) {
  USHORT usIndex;
  long lCheckSum;
  USHORT *pFileEndBlock;
  signed char *pByte;
  
  //MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::VerifyCheckSum entry pheader %p pRecord %p\n",pheader,pRecord);

  if (G.usWTTloaded == NOT_LOADED) {       /*return if the WTT is not in memory*/
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::VerifyCheckSum - failed with G.usWTTloaded == NOT_LOADED\n");
    return WTTPOSERR_COUNTRY_NOT_LOADED;
  }
  
  lCheckSum = 0;
  pByte = (signed char *) &(pheader->usHeaderLength);
  
  /*Calculate the checksum of the header*/
  for (usIndex = 0; usIndex < sizeof (RECHEADER); usIndex++) {
    lCheckSum = lCheckSum + (long) pByte[usIndex];
  }
  
  pByte = (signed char *) pRecord;
  
  /*Calculate the checksum of the record*/
  for (usIndex = 0; usIndex < ((pheader->usRecLength * TWO_BYTES) - sizeof (lCheckSum)); usIndex++) {
    lCheckSum = lCheckSum + (long) pByte[usIndex];
  }
  
  pFileEndBlock = pRecord;
  
  while (*pFileEndBlock) {
    pFileEndBlock += *pFileEndBlock;  /*pFileEndBlock is pointing to the block length field, skip to next block*/
  }
  
  if ( ((USHORT)(lCheckSum & 0x0000FFFF)) != *(pFileEndBlock+1) ) {
    MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::VerifyCheckSum - lower checksum word failed %04X should have been %04lX\n", *(pFileEndBlock+1), lCheckSum & 0x0000FFFF);
    return WTTPOSERR_INVALID_CHECKSUM;
  }
  
  if ( ((USHORT)(lCheckSum >> 16)) != *(pFileEndBlock+2) ) {
    MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::VerifyCheckSum - upper checksum word failed %04X should have been %04lX\n", *(pFileEndBlock+2), lCheckSum >> 16);
    return WTTPOSERR_INVALID_CHECKSUM;
  }

  //MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::VerifyCheckSum exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /*end VerifyCheckSum*/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  ReadMapTable*/
/**/
/*         This function reads in the mapping information and sets up the*/
/*         mapping arrays.*/
/**/
/**/
/*     Params:   none, but changes the Global Structure*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG ReadMapTable(void) {
  USHORT usCount;
  ULONG ulError;
  USHORT * pusBlock = NULL;
  MAPENTRY * pItem;                 /* pointer to a mapping set*/
  USHORT usNumItems, usItemCount;
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadMapTable entry\n");

  /* First, let's initialize the AccessCodeMapArray which identifies which*/
  /* WT Table to use for each international phone access code.*/
  for (usCount = 0; usCount < ACCESS_CODE_MAX; usCount++) {
    G.usAccessCodeMapArray[usCount] = WT_COUNTRY_INVALID;  /* initialize to a non-table*/
  }
  
  /* Next we need to initialize what WT Table to use for each country number.*/
  for (usCount = 0; usCount <= WT_COUNTRY_MAX; usCount++) {
    G.usCountryNumMapArray[usCount] = usCount;  /* initialize to point to itself*/
  }
  
  /*---------------------------------------------------------------------------*/
  /* Now we are ready to read the map file and fill in the arrays.*/
  /* FindCountry changes the G.pusRecord array.*/
  ulError = FindCountry(0);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::ReadMapTable - failed FindCountry with ulError %lu\n", ulError);
    return ulError;
  }
  
  /* Get a pointer to the map block*/
  ulError = GetBlock (WTBLOCK_MAP_DATA, &pusBlock);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::ReadMapTable - failed GetBlock with ulError %lu\n", ulError);
    return ulError;
  }
  
  pItem = (struct TAGMAPENTRY *) (pusBlock + (sizeof(BLOCKHEADER) / TWO_BYTES)); /* set the pointer on the first map set*/

  /* *pusBlock points to the block length in 16 bit words.*/
  usNumItems = ((*pusBlock * TWO_BYTES) - sizeof(BLOCKHEADER)) / sizeof(MAPENTRY);
  if (usNumItems > ACCESS_CODE_MAX) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadMapTable - failed with map table specifying too many entries\n");
    return WTTPOSERR_MAP_FILE_CORRUPTED;
  }

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::ReadMapTable, usNumItems %x\n",usNumItems);
  for (usItemCount = 0; usItemCount < usNumItems; usItemCount++, pItem++) {
    /*validate entries*/
    if (pItem->usAccessCode > ACCESS_CODE_MAX) {
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadMapTable - failed with an access code too large\n");
      return WTTPOSERR_MAP_ACCESSCODE_TOO_LRG; 
    }
    
    if (pItem->usCountryNum == 0) {               /* the map file is not mapped itself*/
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadMapTable - failed with a country number of zero\n");
      return WTTPOSERR_INVALID_COUNTRY;
    }
    
    if (pItem->usCountryNum > WT_COUNTRY_MAX) {
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::d>ReadMapTable - failed with a country number too large\n");
      return WTTPOSERR_INVALID_COUNTRY;
    }

    if (pItem->usWTT == 0) {                     /*nothing can map to the map file*/
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadMapTable - failed with a WT Table number of zero\n");
      return WTTPOSERR_INVALID_COUNTRY;
    }
    
    if (pItem->usWTT > WT_COUNTRY_MAX) {
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadMapTable - failed with a WT Table number that is too large\n");
      return WTTPOSERR_INVALID_COUNTRY;
    }

    G.usAccessCodeMapArray[pItem->usAccessCode] = pItem->usCountryNum;
    G.usCountryNumMapArray[pItem->usCountryNum] = pItem->usWTT;
    MW_SYSLOG_4(TRACE_MWWTT32,"mwwttbl::ReadMapTable, usAccessCode %x usCountryNum %x usWTT %x\n",
		pItem->usAccessCode,pItem->usCountryNum,pItem->usWTT);
  } /*end for*/
  
  /* Mark the mapping information as valid*/
  G.bMappingInitialized = TRUE;
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadMapTable exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /*end ReadMapTable*/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  MapAccessToCountryNum*/
/**/
/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
ULONG MapAccessToCountryNum (ULONG ulAccessCode, ULONG *ulCountryNum) {

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::MapAccessToCountryNum entry ulAccessCode %lx\n",ulAccessCode);

  if (!G.bMappingInitialized) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::MapAccessToCountryNum - failed because the mapping has not been initialized\n");
    return WTTPOSERR_NO_MAPPING_INFO;
  }
  
  if (ulAccessCode > ACCESS_CODE_MAX) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::MapAccessToCountryNum - failed because the mapping access code is too large\n");
    return WTTPOSERR_ACCESS_CODE_TOO_LARGE;
  }

  *ulCountryNum = (ULONG) G.usAccessCodeMapArray[(int)ulAccessCode];
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::MapAccessToCountryNum exit WTT_SUCCESSFUL, ulCountryNum %lx\n",*ulCountryNum);
  return WTT_SUCCESSFUL;
} /*end MapAccessToCountryNum*/


ULONG MapCountryNumToAccess (ULONG ulCountryNum, ULONG *ulAccessCode) {
  int i;

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::MapCountryNumToAccess entry ulCountryNum %lx\n",ulCountryNum);

  if (!G.bMappingInitialized) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::MapCountryNumToAccess - failed because the mapping has not been initialized\n");
    return WTTPOSERR_NO_MAPPING_INFO;
  }
  
  if ((ulCountryNum < 1) || (ulCountryNum > WT_COUNTRY_MAX)) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::MapCountryNumToAccess - failed because country number is out of range\n");
    return WTTPOSERR_INVALID_COUNTRY;
  }

  for(i=0;i<ACCESS_CODE_MAX;i++) {
    if (  G.usAccessCodeMapArray[i] == ulCountryNum ) {
      *ulAccessCode=i;
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::MapCountryNumToAccess - found access code %lx\n",*ulAccessCode);
      return WTT_SUCCESSFUL;
    }
  }

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::MapCountryNumToAccess exit WTTPOSERR_NO_MAPPING_INFO\n");
  return WTTPOSERR_NO_MAPPING_INFO;
} /*end MapAccessToCountryNum*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  MapCountryNumToWTTnum*/
/**/
/*     This function maps a country number into a WT Table number.*/
/**/
/*     Params:   usType    - specifies the Block Type to match up*/
/*               pusRecord - a pointer to the record to search*/
/**/
/*               Note:  this function uses the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG MapCountryNumToWTTnum (ULONG ulCountryNum, ULONG *pulWTTnum) {

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::MapCountryNumToWTTnum entry, ulCountryNum %lx\n",ulCountryNum);

  if (!G.bMappingInitialized) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::MapCountryNumToWTTnum - failed because the mapping has not been initialized\n");
    return WTTPOSERR_NO_MAPPING_INFO;
  }
  
  if ((ulCountryNum < 1) || (ulCountryNum > WT_COUNTRY_MAX)) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::MapAccessToCountryNum - failed because the country number is out of range\n");
    return WTTPOSERR_INVALID_COUNTRY;
  }

  *pulWTTnum = (ULONG) G.usCountryNumMapArray[(int)ulCountryNum];

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::MapCountryNumToWTTnum exit WTT_SUCCESSFUL, ulWTTnum %lx\n",*pulWTTnum);
  return WTT_SUCCESSFUL;
} /*end MapCountryNumToWTTnum*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  LoadWTTable*/
/**/
/*         This function loads the WT Table for a given Mwave country number.*/
/**/
/**/
/*     Params:   ulCountryNum  - the number of the country to find*/
/**/
/*               Note:  this function changes the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*     Overall flow:*/
/*        call MapCountryNumToWTTnum*/
/*        call FindCountry*/
/*        call UpdateSettings*/
/*        mark table as loaded*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG LoadWTTable (ULONG ulCountryNum, ULONG ulAccessCode) {
#define MAX_COUNTRY_NAME_SIZE  100
  char szCountryName[MAX_COUNTRY_NAME_SIZE];
  char szTempBuf [255];
  ULONG ulError, ulWTTnum;
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::LoadWTTable entry ulCountryNum %lx\n",ulCountryNum);

  /* First, mark the table as not loaded.*/
  G.usWTTloaded = NOT_LOADED;
  G.ulInitError = WTTPOSERR_COUNTRY_NOT_LOADED;
  
  /* map the Mwave country number into an actual WT Table*/
  ulError = MapCountryNumToWTTnum (ulCountryNum, &ulWTTnum);
  
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::LoadWTTable - failed MapCountryNumToWTTnum with ulError = %lu\n", ulError);
    return ulError;
  }
  
  /* At this point it is time to load the requested country.  The countries*/
  /* have already been validated by the FindAvailableCountries routine.*/
  /* FindCountry changes the G.pusRecord array.*/
  ulError = FindCountry(ulCountryNum);
  MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::LoadWTTable, return %lx from FindCountry ulCountryNum %lx\n",ulError,ulCountryNum);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::LoadWTTable - failed FindCountry with ulError = %lu\n", ulError);
    return ulError;
  }
  
  /* Now let's update the transmit power level settings and others*/
  ulError = UpdateSettings (ulCountryNum);
  MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::LoadWTTable, return %lx from UpdateSettings ulCountryNum %lx\n",ulError,ulCountryNum);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::LoadWTTable - failed UpdateSettings with ulError = %lu\n", ulError);
    return ulError;
  }
  
  /* mark table as loaded*/
  G.usWTTloaded = WTT_LOADED;
  G.ulInitError = WTT_SUCCESSFUL;
  
  /*write out the country name to the Mwave.ini file*/
  ulError = WTT_GetCountryName (NULL, ulCountryNum, ulAccessCode, szCountryName, MAX_COUNTRY_NAME_SIZE, 0L);
  MW_SYSLOG_4(TRACE_MWWTT32,"mwwttbl::LoadWTTable, return %lx from WTT_GetCountryName ulCountryNum %lx is %s\n",ulError,ulCountryNum,szCountryName);
  if (ulError){
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::LoadWTTable - failed WTT_GetCountryName with ulError = %lu\n", ulError);
    return ulError;
  } else {
    MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::LoadWTTable - WTT_GetCountry Name returns szCountryName %s for ulCountryNum %lx\n",szCountryName,ulCountryNum);
  }
  
  sprintf(szTempBuf,"%s, WTT=%u, Database=%u", szCountryName, (USHORT) ulWTTnum, G.usDatabaseSetNumber);
  if (!WritePrivateProfileString(WORLDTRADE, CURRENTCOUNTRY, szTempBuf, ini_file)) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::LoadWTTable - failed to write the current WTT to the Mwave.ini file\n");
    // @TBD return WTTPOSERR_FAILED_MWAVE_INI_WRITE;
  }

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::LoadWTTable exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;

} /*end LoadWTTable*/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  FindCountry*/
/**/
/*         This function searches the multi-country WTT to find a particular*/
/*         country.*/
/**/
/**/
/*     Params:   ulCountryNum  - the number of the country to find*/
/**/
/*               Note:  this function changes the Global Structure G*/
/**/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG FindCountry (ULONG ulCountryNum) {
  FILE* wtFile;
  ULONG ulError, ulWTTnum;
  RECHEADER RecHeader;
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindCountry entry ulCountryNum %lx\n",ulCountryNum);

  /*First, map the country number into a WT Table number*/
  if (ulCountryNum != 0)  { /* don't map the map file*/
    ulError = MapCountryNumToWTTnum (ulCountryNum, &ulWTTnum);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindCountry - failed MapCountryNumToWTTnum with ulError %lu\n", ulError);
      return ulError;
    }
  } else
    ulWTTnum = 0;

   MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::FindCountry - ulCountryNum = %lu is mapped to ulWTTnum = %lu\n", ulCountryNum, ulWTTnum);
  
  /* Now let's see if the country was previously found to be present.*/
  /* Note that the FindAvailableCountries routine fully verifies the*/
  /* record before saying that it's present.*/
  if (!G.bWTTpresent[(int) ulWTTnum]) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindCountry - failed - returning WTTPOSERR_COUNTRY_NOT_AVAIL for ulWTTnum = %lu\n", ulWTTnum);
    return WTTPOSERR_COUNTRY_NOT_AVAIL;
  }
  
  /* Country is present so let's get the information.*/
  /* Open file for read access.*/
  wtFile = fopen(G.szDatabase, "rb");
  if (wtFile == NULL) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindCountry - failed database open for database = %s\n", G.szDatabase);
    return WTTPOSERR_DATABASE_OPEN_FAILED;
  }

  do { /*step through the WTT file until a zero length header is found*/
    ulError = 0;
    do  {  /* use DO block for error processing*/
      /* get header from file*/
      if (fread(&RecHeader, 1, sizeof(RECHEADER), wtFile) != sizeof(RECHEADER)) {
        MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::FindCountry - failed reading record header\n");
        ulError = WTTPOSERR_DATABASE_READ_FAILED;
        break;              /* exit and close file*/
      }
      
      if (RecHeader.usHeaderLength == 0) /*this marks the file end*/
        break;
      
      /*****************************************************************************/
      /* Erase contents of previous record.                                        */
      /*****************************************************************************/
      memset(G.pusRecord, 0, MAX_RECORD_SIZE);

      if ( (RecHeader.usRecLength * TWO_BYTES) > MAX_RECORD_SIZE) {
        MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::FindCountry - failed record too large\n");
        ulError = WTTPOSERR_RECORD_TOO_LARGE;
        break;              /* exit and close file*/
      }
      
      /* read record*/
      if (fread(G.pusRecord, TWO_BYTES, RecHeader.usRecLength, wtFile) < RecHeader.usRecLength) {
        MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::FindCountry - failed reading record\n");
        ulError = WTTPOSERR_DATABASE_READ_FAILED;
        break;              /* exit and close file*/
      }
    } while (0); /* only one time*/
    
    if (ulError) {     /*fail the function*/
      /* close file*/
      fclose(wtFile);
      return ulError;
    }
    
  } while ((RecHeader.usCountryNum != (USHORT) ulWTTnum) &&
           (RecHeader.usHeaderLength != 0)); /* a length of zero marks the file end*/
  
  /* close file*/
  fclose(wtFile);

  if (RecHeader.usCountryNum == (USHORT) ulWTTnum) {
    G.usWTTloaded = WTT_LOADING; /* mark the table as in memory*/
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::FindCountry exit WTT_SUCCESSFUL\n");
    return WTT_SUCCESSFUL;
  } else {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::FindCountry - failed - WTT = %lu was marked as available but could not be found\n", ulWTTnum);
    return WTTPOSERR_COUNTRY_NOT_AVAIL;
  }
} /*end FindCountry*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  ReadAdjustment*/
/**/
/*     This procedure attempts to read an adjustment from the mwave.ini file*/
/*     for a particular country.  If an adjustment can be found then the*/
/*     function returns true, otherwise it returns false.  No verification*/
/*     is done on the data.*/
/**/
/**/
/*  Params:   lCountryNum  - number of country currently updating*/
/*            szINIentry   - string naming the mwave.ini entry to check*/
/*            psUpdateVal  - pointer to location to return mwave.ini value*/
/**/
/*  Returns:  TRUE:   if an entry is found*/
/*            FALSE:  if no entry is found*/
/**/
/*-----------------------------------------------------------------------------*/
BOOL ReadAdjustment (ULONG ulCountryNum, char FAR * szINIentry, short *psUpdateVal) {
#define INVALID_ENTRY 0x7FFF
  char szSection[sizeof(MWCOUNTRY)+6];
  
  //MW_SYSLOG_3(TRACE_MWWTT32,"mwwttbl::ReadAdjustment entry ulCountryNum %lx szINIentry %s\n",ulCountryNum,szINIentry);

  if ((ulCountryNum < 1) || (ulCountryNum > WT_COUNTRY_MAX))
    return FALSE;
  
  sprintf (szSection, "%s%d", MWCOUNTRY, (short) ulCountryNum);  /* no leading zeros with country number*/
  
  *psUpdateVal = (short) GetPrivateProfileInt(szSection, szINIentry,
							      INVALID_ENTRY, ini_file);
  
  //MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::ReadAdjustment exit\n");
  
  if (*psUpdateVal != INVALID_ENTRY)
    return TRUE;                     /* GetPrivateProfileInt could return*/
  else                               /* zero if the entry is equated to a*/
    return FALSE;                    /* non-number.  This is not protected against.*/
  
} /*end ReadAdjustment*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  SetParam*/
/**/
/*     This function changes the WTT data held in memory.*/
/**/
/*     Params:   usBlockTag - tag of block to be updated*/
/*               usOffset   - offset into the block where the parameter can be found*/
/*               nUpdateVal - value to put in memory*/
/**/
/*               Note:  this function uses the Global Structure G*/
/**/
/*  Returns:  ULONG WTTPOSERR_... from MWWTTBL.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG SetParam (USHORT usWTTBlockTag, USHORT usOffset, short sUpdateVal) {
  USHORT *pusBlock = NULL;
  ULONG ulError;
  
  MW_SYSLOG_4(TRACE_MWWTT32,"mwwttbl::SetParam entry usWTTBlockTag %x usOffset %x sUpdateVal %x\n",usWTTBlockTag,usOffset,sUpdateVal);

  /* Get a pointer to the block of interest*/
  ulError = GetBlock (usWTTBlockTag, &pusBlock);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::SetParam - failed GetBlock with ulError = %lu\n", ulError);
    return ulError;
  }
  
  /* Set the value*/
  *(pusBlock + usOffset + (sizeof(BLOCKHEADER) / TWO_BYTES)) = sUpdateVal;
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::SetParam exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /*end SetParam*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  UpdateSettings*/
/**/
/*     This function changes the settings in the WTT from overrides in the*/
/*     Mwave.ini file if allowed by the PS block, USER_AF parameter.  It*/
/*     also maps power level settings from the Platform Specific block to the*/
/*     WTITCB.*/
/**/
/*     Params:   lCountryNum - number of country currently updating*/
/**/
/*               Note:  this function uses the Global Structure G*/
/**/
/*     Returns:  ULONG WTTPOSERR_... from MWWTT.H*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG UpdateSettings(ULONG ulCountryNum) {
  ULONG ulError;
  short user_af, sUpdateVal, sSetParam, wttpdt3, wttpdt4, wttpdidd, wttaf;
  short sModemErr, sV34Err, sFaxErr, sToneErr;
  short sFaxTx, sModemTx, sToneTx, sVoiceTx, sMaxTx, sTxLevel;
  BOOL bUpdatePower;
  short sUDAA_RingDetectMinVoltage;
  short sUDAA_RingDetectVoltageMultiplier;
  
/* The following is a look up array for WTTVXL and WTTMXL values.*/
  short vxl[32] = {0x2D80,  /*-15 dB*/
                   0x3310,  /*-14 dB*/
                   0x3940,  /*-13 dB*/
                   0x4040,  /*-12 dB*/
                   0x4820,  /*-11 dB*/
                   0x50F0,  /*-10 dB*/
                   0x5AD0,  /* -9 dB*/
                   0x65E0,  /* -8 dB*/
                   0x3921,  /* -7 dB*/
                   0x4021,  /* -6 dB*/
                   0x47F1,  /* -5 dB*/
                   0x50C1,  /* -4 dB*/
                   0x5A91,  /* -3 dB*/
                   0x65A1,  /* -2 dB*/
                   0x3902,  /* -1 dB*/
                   0x4002,  /*  0 dB*/
                   0x47D2,  /*  1 dB*/
                   0x5092,  /*  2 dB*/
                   0x5A62,  /*  3 dB*/
                   0x6562,  /*  4 dB*/
                   0x71C2,  /*  5 dB*/
                   0x3FD3,  /*  6 dB*/
                   0x47A3,  /*  7 dB*/
                   0x5063,  /*  8 dB*/
                   0x5A33,  /*  9 dB*/
                   0x6533,  /* 10 dB*/
                   0x7183,  /* 11 dB*/
                   0x3F64,  /* 12 dB*/
                   0x4774,  /* 13 dB*/
                   0x5034,  /* 14 dB*/
                   0x59F4}; /* 15 dB*/


  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings entry ulCountryNum %lx\n",ulCountryNum);

  /* First, check to see if the country record is loaded*/
  if (G.usWTTloaded == NOT_LOADED) {       /*return if the WTT is not in memory*/
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::UpdateSettings - failed with G.usWTTloaded == NOT_LOADED\n");
    return WTTPOSERR_COUNTRY_NOT_LOADED;
  }
  
  /* get the user allowed functions parameter*/
  ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_USER_AF,
                       WTLENGTH_ONEPARAM, &user_af);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
    return ulError;
  }

  /*--------- Pulse dial adjustments -------------------------------------------*/
  if (user_af & USER_PULSE_DIAL_ADJUST_ALLOWED) {
    if (ReadAdjustment (ulCountryNum, PDMAKE, &sUpdateVal)) {
      if (sUpdateVal > 0) {
	ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTPDMD, sUpdateVal);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
      }
    }
    
    if (ReadAdjustment (ulCountryNum, PDBREAK, &sUpdateVal)) {
      if (sUpdateVal > 0) {
	ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTPDBD, sUpdateVal);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
      }
    }
    
    if (ReadAdjustment (ulCountryNum, PDIDD, &sUpdateVal)) {
      if (sUpdateVal > 0) {
	ulError = QueryInfo (WTBLOCK_WTITCB, WTPARAM_WTTPDT3,
			     WTLENGTH_ONEPARAM, &wttpdt3);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
	
	ulError = QueryInfo (WTBLOCK_WTITCB, WTPARAM_WTTPDT4,
			     WTLENGTH_ONEPARAM, &wttpdt4);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
	
	wttpdidd = sUpdateVal - wttpdt3 - wttpdt4;
	if (wttpdidd < 0)
	  wttpdidd = 0;
	
	ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTPDIDD, wttpdidd);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
      }
    }
    
    if (ReadAdjustment (ulCountryNum, FH, &sUpdateVal)) {
      if (sUpdateVal > 0) {
	ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTFH, sUpdateVal);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
      }
      
      /* Since a FH setting was found, set WTTAF bit 10 to force the*/
      /* FH value to be used.*/
      ulError = QueryInfo (WTBLOCK_WTITCB, WTPARAM_WTTAF,
			   WTLENGTH_ONEPARAM, &wttaf);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	return ulError;
      }
      wttaf = wttaf | 0x0400;
      
      ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTAF, wttaf);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	return ulError;
      }
    }
  }
  /* no adjustments from the Platform Specific block*/
  
  
  /*--------- Max Speaker Volume Adjustments -----------------------------------*/
  if (user_af & SPEAKER_VOLUME_ADJUST_ALLOWED) {
    if (ReadAdjustment (ulCountryNum, SPKR1, &sUpdateVal)) {
      ulError = SetParam (WTBLOCK_VOICE_CTRL, WTPARAM_MAXVOL1, sUpdateVal);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	return ulError;
      }
    }
    
    if (ReadAdjustment (ulCountryNum, SPKR2, &sUpdateVal)) {
      ulError = SetParam (WTBLOCK_VOICE_CTRL, WTPARAM_MAXVOL2, sUpdateVal);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	return ulError;
      }
    }
  }
  /* no adjustments from the Platform Specific block*/
  
  
  /*--------- Microphone Gains Adjustments -------------------------------------*/
  if (user_af & MICROPHONE_GAINS_ADJUST_ALLOWED) {
    if (ReadAdjustment (ulCountryNum, MICMAX1, &sUpdateVal)) {
      if ((sUpdateVal >= 0) && (sUpdateVal <= 100)) {
	ulError = SetParam (WTBLOCK_VOICE_CTRL, WTPARAM_GAINMAX1, sUpdateVal);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
      }
    }
    
    if (ReadAdjustment (ulCountryNum, MICNOM1, &sUpdateVal)) {
      if ((sUpdateVal >= 0) && (sUpdateVal <= 100)) {
	ulError = SetParam (WTBLOCK_VOICE_CTRL, WTPARAM_GAINNOM1, sUpdateVal);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
      }
    }

    if (ReadAdjustment (ulCountryNum, MICMIN1, &sUpdateVal)) {
      if ((sUpdateVal >= 0) && (sUpdateVal <= 100)) {
	ulError = SetParam (WTBLOCK_VOICE_CTRL, WTPARAM_GAINMIN1, sUpdateVal);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
      }
    }
    
    if (ReadAdjustment (ulCountryNum, MICMAX2, &sUpdateVal)) {
      if ((sUpdateVal >= 0) && (sUpdateVal <= 100)) {
	ulError = SetParam (WTBLOCK_VOICE_CTRL, WTPARAM_GAINMAX2, sUpdateVal);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
      }
    }
    
    if (ReadAdjustment (ulCountryNum, MICNOM2, &sUpdateVal)) {
      if ((sUpdateVal >= 0) && (sUpdateVal <= 100)) {
	ulError = SetParam (WTBLOCK_VOICE_CTRL, WTPARAM_GAINNOM2, sUpdateVal);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
      }
    }
    
    if (ReadAdjustment (ulCountryNum, MICMIN2, &sUpdateVal)) {
      if ((sUpdateVal >= 0) && (sUpdateVal <= 100)) {
	ulError = SetParam (WTBLOCK_VOICE_CTRL, WTPARAM_GAINMIN2, sUpdateVal);
	if (ulError) {
	  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
	  return ulError;
	}
      }
    }

  }
  /* no adjustments from the Platform Specific block*/
  
  /*--------- Speakerphone Tx power Adjustments ----------------------------*/
  bUpdatePower = FALSE;
  
  if (user_af & GREETING_MESSAGES_ADJUST_ALLOWED) {
    if (ReadAdjustment (ulCountryNum, SPKRPHTX, &sVoiceTx)) {
      if ((sVoiceTx >= -15) && (sVoiceTx <= 15)) {
	/*update is available and allowed*/
	bUpdatePower = TRUE;
      }
    }
  }

  if (!bUpdatePower) { /* get the default power number*/
    ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_SPKR_TX,
			 WTLENGTH_ONEPARAM, &sVoiceTx);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
    
    if ((sVoiceTx >= -15) && (sVoiceTx <= 15))
      bUpdatePower = TRUE;
  }
  
  if (bUpdatePower) {
    /* rather than calculate, use a look up array for vxl, mxl values*/
    sUpdateVal = vxl[sVoiceTx+15];
    
    ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTVXL, sUpdateVal);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
  }

  /*--------- Messages/Greetings Tx Adjustments ----------------------------*/
  bUpdatePower = FALSE;
  
  if (user_af & GREETING_MESSAGES_ADJUST_ALLOWED) {
    if (ReadAdjustment (ulCountryNum, GREETTX, &sVoiceTx)) {
      if ((sVoiceTx >= -15) && (sVoiceTx <= 15)) {
	/*update is available and allowed*/
	bUpdatePower = TRUE;
      }
    }
  }
  
  if (!bUpdatePower) { /* get the default power number*/
    ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_GREET_TX,
			 WTLENGTH_ONEPARAM, &sVoiceTx);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
    
    if ((sVoiceTx >= -15) && (sVoiceTx <= 15))
      bUpdatePower = TRUE;
  }
  
  if (bUpdatePower) {
    /* rather than calculate, use a look up array for vxl, mxl values*/
    sUpdateVal = vxl[sVoiceTx+15];
    
    ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTMXL, sUpdateVal);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
  }

  /*--------- Determine the maximum power level ----------------------------*/
  ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_MAX_PWR,
		       WTLENGTH_ONEPARAM, &sMaxTx);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
    return ulError;
  }
  
  if (sMaxTx > 0)
    return WTTPOSERR_INVALID_MAX_TX_PWR_LVL;
  
  
  /*--------- Fax Tx Power Adjustments -----------------------------------------*/
  bUpdatePower = FALSE;
  
  if (user_af & FAX_TX_PWR_ADJUST_ALLOWED) {
    if (ReadAdjustment (ulCountryNum, FAXTX, &sFaxTx)) {
      if (sFaxTx <= 0) {
	/*update is available and allowed*/
	bUpdatePower = TRUE;
      }
    }
  }
  
  if (!bUpdatePower) { /* get the default power number*/
    ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_FAX_TX,
			 WTLENGTH_ONEPARAM, &sFaxTx);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
    
    if (sFaxTx <= 0)
      bUpdatePower = TRUE;
  }
  
  if (bUpdatePower) {
    ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_FAX_ERR,
			 WTLENGTH_ONEPARAM, &sFaxErr);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }

    /* calculate the HDXL setting needed*/
    if (sFaxTx > sMaxTx)
      sFaxTx = sMaxTx;
    
    sTxLevel = sFaxTx + sFaxErr;
    if (sTxLevel < MIN_POWER_ALLOWED)
      sTxLevel = MIN_POWER_ALLOWED;
    
    sSetParam = XLarray[-sTxLevel];
    
    ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTHDXL, sSetParam);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
  }

  /*--------- Modem Tx Power Adjustments ---------------------------------------*/
  bUpdatePower = FALSE;
  
  if (user_af & MODEM_TX_PWR_ADJUST_ALLOWED) {
    if (ReadAdjustment (ulCountryNum, MODEMTX, &sModemTx)) {
      if (sModemTx <= 0) {
	/*update is available and allowed*/
	bUpdatePower = TRUE;
      }
    }
  }
  
  if (!bUpdatePower) { /* get the default power number*/
    ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_MODEM_TX,
			 WTLENGTH_ONEPARAM, &sModemTx);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
    
    if (sModemTx <= 0)
      bUpdatePower = TRUE;
  }
  
  if (bUpdatePower) {
    /* update wttfdxl first*/
    ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_MODEMERR,
			 WTLENGTH_ONEPARAM, &sModemErr);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
    
    /* calculate the FDXL setting needed*/
    if (sModemTx > sMaxTx)
      sModemTx = sMaxTx;
    
    sTxLevel = sModemTx + sModemErr;
    if (sTxLevel < MIN_POWER_ALLOWED)
      sSetParam = 0;
    else
      sSetParam = XLarray[-sTxLevel];
    
    ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTFDXL, sSetParam);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
    
    /* update v34 next*/
    ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_V34_ERR,
			 WTLENGTH_ONEPARAM, &sV34Err);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
    
    /* calculate the V34XL setting needed*/
    if (sModemTx > sMaxTx)
      sModemTx = sMaxTx;
    
    sTxLevel = sModemTx + sV34Err;
    if (sTxLevel < MIN_POWER_ALLOWED)
      sSetParam = 0;
    else
      sSetParam = XLarray[-sTxLevel];
    
    ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTV34XL, sSetParam);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
    
    //    /* calculate the V34RL setting needed*/
    /*    sTxLevel = -180 + sV34Err;*/
    /*    sSetParam = XLarray[-sTxLevel];*/
    /**/
    /*    ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTV34RL, sSetParam);*/
    /*    if (ulError)*/
    /*      return ulError;*/
    
  }


  /*--------- Get the tone error to be used in several places -------------------*/
  ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_TONE_ERR,
		       WTLENGTH_ONEPARAM, &sToneErr);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
    return ulError;
  }
  
  /*--------- Set WTTTONM from the ToneErr*/
    if (sToneErr >= 0) {  /* max out the tone multiplier*/
      ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTTONM, 0x7FFF);
      if (ulError)
        return ulError;
    } else {
      if (sToneErr < MIN_POWER_ALLOWED)
        sSetParam = 0;
      else
        sSetParam = XLarray[-sToneErr];
      
      ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTTONM, sSetParam);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
    }
    
    /*--------- Calling Tones Tx Power Adjustments --------------------------------*/
    bUpdatePower = FALSE;
    
    if (user_af & TONES_TX_PWR_ADJUST_ALLOWED) {
      if (ReadAdjustment (ulCountryNum, TONETX, &sToneTx)) {
        if (sToneTx <= 0) {
          /*update is available and allowed*/
          bUpdatePower = TRUE;
        }
      }
    }
    
    if (!bUpdatePower) { /* get the default power number*/
      ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_TONES_TX,
                           WTLENGTH_ONEPARAM, &sToneTx);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
      
      if (sToneTx <= 0)
        bUpdatePower = TRUE;
    }
    
    if (bUpdatePower) {
      /* calculate the WTTTONX setting needed*/
      if (sToneTx > sMaxTx)
        sToneTx = sMaxTx;
      
      if (sToneTx < MIN_POWER_ALLOWED)
        sSetParam = 0;
      else
        sSetParam = XLarray[-sToneTx];
      
      ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTTONX, sSetParam);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
    }
    
    /*--------- TAM Beep Tx Power Adjustments -----------------------------------*/
    bUpdatePower = FALSE;
    if (user_af & TONES_TX_PWR_ADJUST_ALLOWED) {
      if (ReadAdjustment (ulCountryNum, TAMBEEPTX, &sToneTx)) {
        if (sToneTx <= 0) {
          /*update is available and allowed*/
          bUpdatePower = TRUE;
        }
      }
    }
    
    if (!bUpdatePower)  { /* get the default power number*/
      ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_TAM_BEEP,
                           WTLENGTH_ONEPARAM, &sToneTx);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
      
      if (sToneTx <= 0)
        bUpdatePower = TRUE;
    }
    
    if (bUpdatePower) {
      /* calculate the WTTBPPWR setting needed*/
      if (sToneTx > sMaxTx)
        sToneTx = sMaxTx;
      
      if (sToneTx < MIN_POWER_ALLOWED)
        sSetParam = 0;
      else
        sSetParam = XLarray[-sToneTx];
      
      ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTBPPWR, sSetParam);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
    }
    
    /*--------- Conversation record Tx Power Adjustments ----------------------------*/
    bUpdatePower = FALSE;
    
    if (user_af & TONES_TX_PWR_ADJUST_ALLOWED) {
      if (ReadAdjustment (ulCountryNum, CONVBEEPTX, &sToneTx)) {
        if (sToneTx <= 0) {
          /*update is available and allowed*/
          bUpdatePower = TRUE;
        }
      }
    }
    
    if (!bUpdatePower) { /* get the default power number*/
      ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_CONVBEEP,
                           WTLENGTH_ONEPARAM, &sToneTx);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
      
      if (sToneTx <= 0)
        bUpdatePower = TRUE;
    }
    
    if (bUpdatePower) {
      /* calculate the WTTBPPWR setting needed*/
      if (sToneTx > sMaxTx)
        sToneTx = sMaxTx;
      
      if (sToneTx < MIN_POWER_ALLOWED)
        sSetParam = 0;
      else
        sSetParam = XLarray[-sToneTx];

      ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTCBPPWR, sSetParam);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
    }
    
    
    /*--------- DTMF High Frequency Tx Power Adjustments -------------------------*/
    bUpdatePower = FALSE;
    
    if (user_af & DTMF_TX_PWR_ADJUST_ALLOWED) {
      if (ReadAdjustment (ulCountryNum, DTMFTXHIGH, &sToneTx)) {
        if (sToneTx <= 0) {
          /*update is available and allowed*/
          bUpdatePower = TRUE;
        }
      }
    }
    
    if (!bUpdatePower) { /* get the default power number*/
      ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_DTMFHIGH,
                           WTLENGTH_ONEPARAM, &sToneTx);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
      
      if (sToneTx <= 0)
        bUpdatePower = TRUE;
    }
    
    if (bUpdatePower) {
      /* calculate the WTTHDTM setting needed*/
      if (sToneTx > sMaxTx)
        sToneTx = sMaxTx;
      
      if (sToneTx < MIN_POWER_ALLOWED)
        sSetParam = 0;
      else
        sSetParam = XLarray[-sToneTx];
      
      ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTHDTM, sSetParam);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
    }
    
    /*--------- DTMF Low Frequency Tx Power Adjustments -------------------------*/
    bUpdatePower = FALSE;
    
    if (user_af & DTMF_TX_PWR_ADJUST_ALLOWED) {
      if (ReadAdjustment (ulCountryNum, DTMFTXLOW, &sToneTx)) {
        if (sToneTx <= 0) {
          /*update is available and allowed*/
          bUpdatePower = TRUE;
        }
      }
    }
    
    if (!bUpdatePower) { /* get the default power number*/
      ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_DTMFLOW,
                           WTLENGTH_ONEPARAM, &sToneTx);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
      
      if (sToneTx <= 0)
        bUpdatePower = TRUE;
    }
    
    if (bUpdatePower) {
      /* calculate the WTTLDTM setting needed*/
      if (sToneTx > sMaxTx)
        sToneTx = sMaxTx;
      
      if (sToneTx < MIN_POWER_ALLOWED)
        sSetParam = 0;
      else
        sSetParam = XLarray[-sToneTx];

      ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTLDTM, sSetParam);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
    }
    
    /*--- Update the WTTRDNV parameter from the platform specific block.*/
    ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_UDAARDNV,
                         WTLENGTH_ONEPARAM, &sUDAA_RingDetectMinVoltage);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
    
    ulError = QueryInfo (WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_UDAARDVM,
                         WTLENGTH_ONEPARAM, &sUDAA_RingDetectVoltageMultiplier);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }
    
    ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTRDNV,
                        (short) (sUDAA_RingDetectMinVoltage * sUDAA_RingDetectVoltageMultiplier));
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
      return ulError;
    }

    /*--- Zero out the WTTOCDLY parameter if the Mwave.ini file had the*/
    /*--- No_OC_Hangup flag set.*/
    if (G.usNo_OC_Hangup) {
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::UpdateSettings - Zeroed out WTTOCDLY due to Mwave.ini file setting\n");
      ulError = SetParam (WTBLOCK_WTITCB, WTPARAM_WTTOCDLY, 0);
      if (ulError) {
	MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit ulError %lx\n",ulError);
        return ulError;
      }
    }

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::UpdateSettings exit WTT_SUCCESSFUL\n");
  return WTT_SUCCESSFUL;
} /*end UpdateSettings*/


///////////////////////////////////////////////////////////////////////////////**/
//                                                                            /**/
//                                                                            /**/
// ------------------World Trade Table OLD API Functions----------------------/**/
//                                                                            /**/
//                                                                            /**/
///////////////////////////////////////////////////////////////////////////////**/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtGetNumCountries*/
/**/
/*     Retrieves the number of country records in the World Trade*/
/*     database.*/
/**/
/*     Params:   None*/
/**/
/*     Returns:  (short) - number of country records*/
/**/
/*-----------------------------------------------------------------------------*/
short APIENTRY WtGetNumCountries( void ) {

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtGetNumCountries entry\n");

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtGetNumCountries exit WT_COUNTRY_MAX %x\n",WT_COUNTRY_MAX);
  return (WT_COUNTRY_MAX);               /* last record*/
}


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtGetCountryName*/
/**/
/*     Retrieves the country name corresponding to a record number if there*/
/*     is a corresponding country table available on the system.*/
/**/
/*     Params:   lCountryNum  - country record number*/
/*               lpstrName    - pointer to buffer where name string will be*/
/*                              placed*/
/*               nBufSize     - size of name string buffer*/
/**/
/*     Returns:  Successful:  lCount - number of bytes copied into buffer*/
/*               Error:       (long) < 0 (-WTTPOSERR_...)*/
/**/
/*-----------------------------------------------------------------------------*/
long APIENTRY WtGetCountryName(long lCountryNum, char *lpstrName, int nBufSize) {
  ULONG ulError;
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtGetCountryName entry lCountryNum %lx\n",lCountryNum);

  //MW_SYSLOG_1(TRACE_MWWTT32,WtGetCountryName - passing call to WTT_GetCountryName);
  ulError = WTT_GetCountryName (NULL, (ULONG) lCountryNum, 0, lpstrName, (ULONG) nBufSize, 0L);

  if (ulError) {      /* return the negative of the error*/
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtGetCountryName exit ulError %lx\n",-ulError);
    return -((long)ulError);
  }

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtGetCountryName exit WTT_SUCCESSFUL, lpstrName %s\n",lpstrName);
  return strlen(lpstrName);
} /*end WtGetCountryName*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtQueryRecord*/
/**/
/*     Obsolete, this function is not implemented.*/
/**/
/*     Returns:  Failure:   -WTPOSERR_OBSOLETE_FUNCTION*/
/**/
/*-----------------------------------------------------------------------------*/
long APIENTRY WtQueryRecord(long lCountryNum, unsigned short FAR* FAR* lplpRecordBuf) {
  (void) lCountryNum;
  (void) lplpRecordBuf;
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryRecord entry lCountryNum %lx\n",lCountryNum);

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtQueryRecord exit WTTPOSERR_OBSOLETE_FUNCTION\n");
  return -WTTPOSERR_OBSOLETE_FUNCTION;
}



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtQueryField*/
/**/
/*     Allocates and fills a buffer with the data for the country field*/
/*     requested.  This function will be used primarily by the MME drivers*/
/*     which need to transfer 'fields' of information to the DSP.  The caller*/
/*     must call WtFree() to release the storage allocated by this function.*/
/**/
/*     Params:   lpszAccessCode - FAR pointer to null terminated string*/
/*                                containing international access code.  If*/
/*                                NULL then the record for the default country*/
/*                                will be retrieved.  (Note:  this is not*/
/*                                currently used and should be NULL)*/
/**/
/*               lFieldQuery    - one of the constants listed below specifying*/
/*                                the field*/
/**/
/*               lplpFieldBuf   - FAR pointer which will be filled with a*/
/*                                pointer to the field for the country*/
/*                                requested.*/
/**/
/*     Returns:  Successful:  lBufSize > 0 is the size of the record in words*/
/*               Failure:     (long) < 0 ( -WTTPOSERR_... from MWWTTBL.H)*/
/*                            The data in the buffer is not valid.*/
/**/
/*-----------------------------------------------------------------------------*/
long APIENTRY WtQueryField(char FAR* lpszAccessCode, long lFieldQuery, unsigned short FAR* FAR* lplpFieldBuf) {
  ULONG ulWords;
  long lBytes;
  ULONG ulError;
  
  (void) lpszAccessCode;  /* not used*/
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryField entry lFieldQuery %lx\n",lFieldQuery);
  
  
  /* First we need to determine how large the memory needed is*/
  ulError = 0;
  
  switch (lFieldQuery) {
  case WT_FIELD_BLACKLIST:
     MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtQueryField - WT_FIELD_BLACKLIST requested\n");
  ulError = WTT_QueryBlockLength (NULL, WTBLOCK_BLACKLIST, &ulWords, 0L);
  break;
  case WT_FIELD_PSTN_CALL_PROGRESS:
     MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtQueryField - WT_FIELD_PSTN_CALL_PROGRESS requested\n");
  ulError = WTT_QueryBlockLength (NULL, WTBLOCK_CALL_PROG_DATA, &ulWords, 0L);
  break;
  case WT_FIELD_TAIO:
     MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtQueryField - WT_FIELD_TAIO requested\n");
  ulError = WTT_QueryBlockLength (NULL, WTBLOCK_WTITCB, &ulWords, 0L);
  break;
  case WT_FIELD_PLATFORM_SPECIFIC:
     MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtQueryField - WT_FIELD_PLATFORM_SPECIFIC requested\n");
  ulError = WTT_QueryBlockLength (NULL, WTBLOCK_PLATFORM_SPECIFIC, &ulWords, 0L);
  break;
  case WT_VOICE_CONTROLS:
     MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtQueryField - WT_VOICE_CONTROLS requested\n");
  ulError = WTT_QueryBlockLength (NULL, WTBLOCK_VOICE_CTRL, &ulWords, 0L);
  break;
  
  default:
     MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtQueryField - test on lFieldQuery failed\n");
  return WT_ERR_INVALID_FIELD;
  }
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryField ulError %lx from QueryBlockLength\n",ulError);

  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryField - failed WTT_QueryBlockLength with ulError = %lu\n", ulError);
    return ulError;
  }
  
  /* allocate storage for field query     - owned by application*/
  lBytes = (LONG) (ulWords * WT_ENTRY_SIZE);

  *lplpFieldBuf = (unsigned short FAR*) malloc(lBytes);

  if (*lplpFieldBuf == NULL) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtQueryField - failed to allocate memory\n");
    return WT_ERR_MEM_ALLOCATION;
  } else {
     MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryField - %lu bytes of memory has been allocated\n", lBytes);
    ulError = 0;  /*clear compiler warning when debug is off and nothing is in this else statement*/
  }
  
  //MW_SYSLOG_1(TRACE_MWWTT32,WtQueryField - passing call to WTT_QueryInfo);
  //MW_SYSLOG_2(TRACE_MWWTT32,WtQueryField - *lplpFieldBuf = %p hex, *lplpFieldBuf);
  
  /* Now let's get the data*/
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryField switching on lFieldQuery %lx\n",lFieldQuery);
  switch (lFieldQuery) {
    case WT_FIELD_BLACKLIST:
      ulError = WTT_QueryInfo (NULL, WTBLOCK_BLACKLIST, WTFIELD_BLACKLIST_BLOCK,
                               ulWords, (short FAR*) *lplpFieldBuf, 0L);
      break;

    case WT_FIELD_PSTN_CALL_PROGRESS:
      ulError = WTT_QueryInfo (NULL, WTBLOCK_CALL_PROG_DATA, WTFIELD_CALLPROGDATA_BLOCK,
                               ulWords, (short FAR*) *lplpFieldBuf, 0L);
      break;

    case WT_FIELD_TAIO:
      ulError = WTT_QueryInfo (NULL, WTBLOCK_WTITCB, WTFIELD_WTITCB_BLOCK,
                               ulWords, (short FAR*) *lplpFieldBuf, 0L);
      break;

    case WT_FIELD_PLATFORM_SPECIFIC:
      ulError = WTT_QueryInfo (NULL, WTBLOCK_PLATFORM_SPECIFIC, WTFIELD_PLATFORM_SPECIFIC_BLOCK,
                               ulWords, (short FAR*) *lplpFieldBuf, 0L);
      break;

    case WT_VOICE_CONTROLS:
      ulError = WTT_QueryInfo (NULL, WTBLOCK_VOICE_CTRL, WTFIELD_VC_BLOCK,
                               ulWords, (short FAR*) *lplpFieldBuf, 0L);
      break;

    default:  
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtQueryField exit WT_ERR_INVALID_FIELD\n");
      return WT_ERR_INVALID_FIELD;
  }

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryField ulError from WTT_QueryInfo %lx\n",ulError);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryField exit ulError %lx\n",ulError);
    return -((long)ulError);
  }
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryField exit WTT_SUCCESSFUL ulWords %lx\n",ulWords);
  /* return size of field*/
  return (long) ulWords;
} /*end WtQueryField*/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtQueryItem*/
/**/
/*     Returns a single 'item' of information.  This function will be used*/
/*     primarily by the MME drivers to query information such as functions*/
/*     allowed.  ItemQuery values will be added as requested by developers.*/
/**/
/*     Params:   lpszAccessCode - FAR pointer to null terminated string*/
/*                                containing internationl access code.  If NULL*/
/*                                then the record for the default country will*/
/*                                be retrieved.*/
/**/
/*               lItemQuery     - one of the constants listed below specifying*/
/*                                the item to be retrieved*/
/**/
/*               lpultem        - FAR pointer to storage for retrieved item*/
/**/
/*     Returns:  Successful:    (long) 0*/
/*               Failure:       (long) < 0 (-WTTPOSERR_... from MWWTTBL.H)*/
/**/
/*-----------------------------------------------------------------------------*/
long APIENTRY WtQueryItem(char FAR* lpszAccessCode, long lItemQuery, unsigned long FAR* lpulItem) {
  ULONG ulError;
  short sItem;
  USHORT usWTTBlockTag, usOffset;
  
  (void) lpszAccessCode; /*not used*/
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryItem entry lItemQuery %lx\n",lItemQuery);

  //MW_SYSLOG_1(TRACE_MWWTT32,WtQueryItem - passing call to WTT_QueryInfo);
  switch (lItemQuery) {
    case WT_SMART_CABLE_ID:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTCID;
      break;
    case WT_ALLOWED_FUNC_1:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTAF;
      break;
    case WT_ALLOWED_FUNC_2:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTAF2;
      break;
    case WT_G3_FAX_CTRL:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTG3FAX;
      break;
    case WT_DTMF_ALLOWED_SET:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTDTADS;
      break;
    case WT_CALL_PROG_CTRL:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTCPM;
      break;
    case WT_MIN_RINGS:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTRNGS;
      break;
    case WT_PULSE_DIAL_MODE:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTDMODE;
      break;
    case WT_BLIND_DIAL_WAIT_TIME:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTBDWT;
      break;
    case WT_DIAL_TONE_WAIT_TIME:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTDTWT;
      break;
    case WT_CARRIER_WAIT_TIME:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTCW;
      break;
    case WT_AUTO_MODEM_DISCONNECT_TIME:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTAMD;
      break;
    case WT_LOSS_OF_CARRIER_WAVE_TIME:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTLCW;
      break;
    case WT_FULL_DUPLEX_TX_LVL:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTFDXL;
      break;
    case WT_HALF_DUPLEX_TX_LVL:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTHDXL;
      break;
    case WT_AVG_VOICE_TX_LVL:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTVXL;
      break;
    case WT_AUTO_CALL_DELAY:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTACD;
      break;
    case WT_FLASH_HOOK_TIME:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTFH;
      break;
    case WT_BLIND_DIAL_WAIT_TIME_MAX:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTBDWTX;
      break;
    case WT_CARRIER_WAIT_TIME_MAX:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTCWX;
      break;
    case WT_LOSS_OF_CARRIER_WAVE_MAX:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTLCWX;
      break;
    case WT_DTMF_SIGNAL_DURATION:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTDTSD;
      break;
    case WT_MIC1_GAIN_MAX:
      usWTTBlockTag = WTBLOCK_VOICE_CTRL;
      usOffset      = WTPARAM_GAINMAX1;
      break;
    case WT_MIC1_GAIN_MIN:
      usWTTBlockTag = WTBLOCK_VOICE_CTRL;
      usOffset      = WTPARAM_GAINMIN1;
      break;
    case WT_MIC1_GAIN_NOM:
      usWTTBlockTag = WTBLOCK_VOICE_CTRL;
      usOffset      = WTPARAM_GAINNOM1;
      break;
    case WT_MIC2_GAIN_MAX:
      usWTTBlockTag = WTBLOCK_VOICE_CTRL;
      usOffset      = WTPARAM_GAINMAX2;
      break;
    case WT_MIC2_GAIN_MIN:
      usWTTBlockTag = WTBLOCK_VOICE_CTRL;
      usOffset      = WTPARAM_GAINMIN2;
      break;
    case WT_MIC2_GAIN_NOM:
      usWTTBlockTag = WTBLOCK_VOICE_CTRL;
      usOffset      = WTPARAM_GAINNOM2;
      break;
    case WT_MAX_SPEAKER1_VOLUME:
      usWTTBlockTag = WTBLOCK_VOICE_CTRL;
      usOffset      = WTPARAM_MAXVOL1;
      break;
    case WT_MAX_SPEAKER2_VOLUME:
      usWTTBlockTag = WTBLOCK_VOICE_CTRL;
      usOffset      = WTPARAM_MAXVOL2;
      break;
    case WT_WTTCFSR:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTTCFSR;
      break;
    case WT_WTT34FBR:
      usWTTBlockTag = WTBLOCK_WTITCB;
      usOffset      = WTPARAM_WTT34FBR;
      break;

    default:
      MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtQueryItem exit WT_ERR_INVALID_ITEM\n");
      return WT_ERR_INVALID_ITEM;
  }

  ulError = WTT_QueryInfo (NULL, usWTTBlockTag, usOffset,
                           WTLENGTH_ONEPARAM, &sItem, 0L );
  
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryItem exit ulError %lx\n",ulError);
    return -((long)ulError);
  }
  
  *lpulItem = (ULONG) sItem; /* assign the item for return*/
  //MW_SYSLOG_2(TRACE_MWWTT32,WtQueryItem - returning value %lX hex, *lpulItem);
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtQueryItem exit WTT_SUCCESSFUL, ulItem %lx\n",*lpulItem);
  return 0;
} /*end WtQueryItem*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtFree*/
/**/
/*     Frees storage allocated by a previous query (see calls WtQueryRecord and*/
/*     WtQueryField).*/
/**/
/*     Params:   lpBuffer - FAR pointer to FAR pointer to buffer which was*/
/*                          allocated by a previous query.*/
/**/
/*     Returns:  Successful:  (long) 0, storage was freed*/
/*               Failure:     (long) < 0 (WT_ERR_...)*/
/**/
/*-----------------------------------------------------------------------------*/
long APIENTRY WtFree(unsigned short FAR* FAR* lplpBuffer) {

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtFree entry\n");
  
   MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtFree called for location %p\n", *lplpBuffer);
  
  /* free memory only if this is a valid pointer*/
  
  if (*lplpBuffer) {
      free(*lplpBuffer);
  } else {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtFree - failed - called with an invalid pointer - returning WT_ERR_WTFREE_FAILED\n");
    return WT_ERR_WTFREE_FAILED;
  }
  
  *lplpBuffer = NULL;
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtFree exit WTT_SUCCESSFUL\n");
  return 0;
} /*end WtFree*/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtCountryRecordChanged*/
/**/
/*     Called by the Table maintenance utility when it changes a record.  This*/
/*     routine will refresh the data if the record changed is currently the*/
/*     default.  In this case, WT_TABLECHANGED message will be sent to all*/
/*     registered clients.  It is their responsibility to refresh data obtained*/
/*     from the database.*/
/**/
/*     Params:   lCountryNum - country number of record changed*/
/**/
/*     Returns:  None*/
/**/
/*-----------------------------------------------------------------------------*/
void APIENTRY WtCountryRecordChanged(long lCountryNum) {
  (void) lCountryNum; /* no longer used*/
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtCountryRecordChanged entry lCountryNum %lx\n",lCountryNum);

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtCountryRecordChanged - passing call to WTT_WTTableHasChanged\n");
  (void) WTT_WTTableHasChanged (NULL, 0L);
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtCountryRecordChanged exit  WTT_SUCCESSFUL\n");
}


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  This function is obsoleted by WtChangeCountry.*/
/**/
/*-----------------------------------------------------------------------------*/
void APIENTRY WtDefaultCountryHasChanged(void) {

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtDefaultCountryHasChanged entry\n");
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtDefaultCountryHasChanged exit WTT_SUCCESSFUL\n");
}

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtGetCurrentCountry*/
/**/
/*    Params:   None*/
/**/
/*    Returns:  Current country number*/
/**/
/*-----------------------------------------------------------------------------*/
long APIENTRY WtGetCurrentCountry(void) {
  ULONG ulError, ulCurrentCountryNum;
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtGetCurrentCountry entry\n");
  
  ulError = WTT_GetCurrentCountry (NULL, &ulCurrentCountryNum, 0L );
  
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtGetCurrentCountry exit ulError %lx\n",ulError);
    return -((long)ulError);
  }
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtGetCurrentCountry exit WTT_SUCCESSFUL, ulCurrentCountryNum %lx\n",ulCurrentCountryNum);
  return (long) ulCurrentCountryNum;
}



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtChangeCountry*/
/**/
/*    Params:   None*/
/**/
/*    Returns:  Current country number*/
/**/
/*-----------------------------------------------------------------------------*/
long APIENTRY WtChangeCountry(long lCountryNum) {
  ULONG ulError;

  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtChangeCountry entry lCountryNum %lx\n",lCountryNum);

  ulError = WTT_ChangeWTTable (NULL, (ULONG) lCountryNum, 0L);
  
  if (ulError){
    MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtChangeCountry exit ulError %lx\n",ulError);
    return -((long)ulError);
  }
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtChangeCountry exit WTT_SUCCESSFUL, lCountryNum %lx\n",lCountryNum);
  return lCountryNum;
}


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtGetDatabaseName*/
/**/
/*     Obsolete*/
/**/
/*     Params:   lCountryNum  - country ID number as defined by WtQueryRecord*/
/*               lpszName     - FAR pointer to storage for database name*/
/**/
/*     Returns:*/
/*               Failure:     FALSE*/
/**/
/*-----------------------------------------------------------------------------*/
BOOL APIENTRY WtGetDatabaseName(long lCountryNum, char FAR* lpszName) {
  (void) lCountryNum;  /* clear compiler warnings*/
  (void) lpszName;
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtGetDatabaseName entry, lCountryNum %lx\n",lCountryNum);

  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtGetDatabaseName exit FALSE\n");
  return FALSE;
}



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtCPsourceFileName*/
/**/
/*     Returns:*/
/*               Failure:     FALSE*/
/**/
/*-----------------------------------------------------------------------------*/
BOOL APIENTRY WtCPsourceFileName(long lCountryNum, char FAR* lpszName) {
  (void) lCountryNum;  /* clear compiler warnings*/
  (void) lpszName;
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtCPsourceFileName entry lCountryNum %lx\n",lCountryNum);
  
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtCPsourceFileName exit FALSE\n");
  return FALSE;
}


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  WtUseAlternateDatabase*/
/**/
/*     This function switches which database is the current default.  In the*/
/*     MWAVE.INI file under the [Worldtrade] section is a Database= statement.*/
/*     This is the default database used when loaded.  Alternate*/
/*     databases add a number to the Database= statement.  Alternate set*/
/*     1 is Database1=, alternate set 2 is Database2=.  To get back to the*/
/*     original Database= statement, specify set 0.*/
/**/
/*     Params:   usDatabaseSetNumber - The number of the Database set desired*/
/**/
/*     Returns:  Successful:  TRUE*/
/*               Failure:     FALSE if there is no corresponding Database? statement*/
/**/
/*-----------------------------------------------------------------------------*/
BOOL APIENTRY WtUseAlternateDatabase(unsigned short usLocalDatabaseSetNumber) {
  ULONG ulError;
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtUseAlternateDatabase entry usLocalDatabaseSetNumber %x\n",usLocalDatabaseSetNumber);
  
  /*store the set number away*/
  G.usDatabaseSetNumber = usLocalDatabaseSetNumber;
  
  MW_SYSLOG_2(TRACE_MWWTT32,"mwwttbl::WtUseAlternateDatabase - changing to database set number %u\n", G.usDatabaseSetNumber);
  
  /*reload the WTTable*/
  ulError = WTT_WTTableHasChanged (NULL, 0L);
  if (ulError) {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtUseAlternateDatabase exit FALSE\n");
    return FALSE;
  } else {
    MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WtUseAlternateDatabase exit TRUE\n");
    return TRUE;
  }
} /*end WtUseAlternateDatabase;*/


ULONG APIENTRY WTT_UnregisterCallback()
{
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_UnregisterCallback entered");
  MW_SYSLOG_1(TRACE_MWWTT32,"mwwttbl::WTT_UnregisterCallback exit UNSUPPORTED function\n");
  return WTT_SUCCESSFUL;
}

