/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTCPFlowControlModule.cpp

	Contains:	Implements object defined in .h file
	
	$Log: RTCPFlowControlModule.cpp,v $
	
*/

#include "RTCPFlowControlModule.h"
#include "RTPSession.h"
#include "RTPStream.h"
#include "RTSPServerInterface.h"
#include "RTSPPrefs.h"
#include "RTSPRollingLog.h"
#include "SocketUtils.h"

RTCPFlowControlModule::RTCPFlowControlModule()
 : RTPModule(kAllRTCPProcessingModule, "RTCPFlowControlModule"), fLogFile(NULL)
{
	fNumLossesAboveTolerance = StreamDictionary::sStreamDictValueIDManager.GenerateIDForSignature('latl');
	fNumLossesBelowTolerance = StreamDictionary::sStreamDictValueIDManager.GenerateIDForSignature('lbtl');
	fNumGettingWorses = StreamDictionary::sStreamDictValueIDManager.GenerateIDForSignature('ngws');
	fNumGettingBetters = StreamDictionary::sStreamDictValueIDManager.GenerateIDForSignature('ngbs');
	fMoreBandwidthMultiple = StreamDictionary::sStreamDictValueIDManager.GenerateIDForSignature('mbmt');
}

void RTCPFlowControlModule::ProcessRTCPPacket(RTPStream* inStream, StrPtrLen* /*inPacket*/)
{
#if FLOW_CONTROL_DEBUGGING
	if (inStream->GetCodecType() == RTPStream::kVideoCodecType)
		printf("Video track reporting:\n");
	else if (inStream->GetCodecType() == RTPStream::kAudioCodecType)
		printf("Audio track reporting:\n");
	else
		printf("Unknown track reporting\n");
#endif

	if ((fLogFile == NULL) && (RTSPServerInterface::GetRTSPPrefs()->IsFlowControlLogEnabled()))
		fLogFile = ::fopen("StreamThinning.log", "a+");
	else if ((fLogFile != NULL) && (!RTSPServerInterface::GetRTSPPrefs()->IsFlowControlLogEnabled()))
	{
		::fclose(fLogFile);
		fLogFile = NULL;
	}
	char theDateBuffer[RTSPRollingLog::kMaxDateBufferSizeInBytes];
	char theIPAddrBuffer[20];
	StrPtrLen theIPAddr(&theIPAddrBuffer[0], 20);
	if (fLogFile != NULL)
	{
		bool result = RTSPRollingLog::FormatDate(theDateBuffer);
		//for now, just ignore the error.
		if (!result)
			theDateBuffer[0] = '\0';
			
		struct in_addr theAddr;
		theAddr.s_addr = inStream->GetRemoteAddr();
		SocketUtils::ConvertAddrToString(theAddr, &theIPAddr);
		
		::fflush(fLogFile);
	}
		
	//all the values this module cares about are already in the dictionary,
	//so just use the dictionary
	StreamDictionary* theDictionary = inStream->GetDictionary();

	//ALGORITHM FOR DETERMINING WHEN TO MAKE QUALITY ADJUSTMENTS IN THE STREAM:
	
	//This routine makes quality adjustment determinations for the server. It is designed
	//to be flexible: you may swap this algorithm out for another implemented in another module,
	//and this algorithm uses settings that are adjustable at runtime.
	
	//It uses the % loss statistic in the RTCP packets, as well as the "getting better" &
	//"getting worse" fields.

	//Less bandwidth will be served if the loss % of N number of RTCP packets is above M, where
	//N and M are runtime settings.
	
	//Less bandwidth will be served if "getting worse" is reported N number of times.
	
	//More bandwidth will be served if the loss % of N number of RTCPs is below M.
	//N will scale up over time.
	
	//More bandwidth will be served if the client reports "getting better"
	
	//If the initial values of our dictionary items aren't yet in, put them in.
	this->InitializeDictionaryItems(theDictionary);	

	bool ratchetMore = false;
	bool ratchetLess = false;
	UInt32 theValue = 0;
	UInt32 theValueLen = 0;
	
	bool wasFound = false;
	
	bool clearPercentLossThinCount = true;
	bool clearPercentLossThickCount = true;
	
	//First take any action necessitated by the loss percent
	UInt16 thePercentLoss = theDictionary->GetStandardUInt16Value(StreamDictionary::kPercentPacketsLost, &wasFound);
	thePercentLoss /= 256; //Hmmm... looks like the client reports loss percent in multiples of 256
	
#if FLOW_CONTROL_DEBUGGING
	printf("Percent loss: %d\n", thePercentLoss);
#endif
	if (wasFound)
	{
		//check for a thinning condition
		if (thePercentLoss > RTSPServerInterface::GetRTSPPrefs()->GetPercentLossThinTolerance())
		{
			theDictionary->GetValue(fNumLossesAboveTolerance, &theValue, sizeof(theValue), &theValueLen);
			Assert(theValueLen == sizeof(theValue));
			
			theValue++;//we must count this loss

			//We only adjust after a certain number of these in a row. Check to see if we've
			//satisfied the thinning condition, and adjust the count
			if (theValue >= RTSPServerInterface::GetRTSPPrefs()->GetNumLossesToThin())
			{
#if FLOW_CONTROL_DEBUGGING
				printf("Percent loss too high: ratcheting less\n");
#endif
				if ((fLogFile != NULL) && (inStream->GetCodecType() == RTPStream::kVideoCodecType))
					::fprintf(fLogFile, "%s %s Percent loss too high. Ratcheting less. Percent loss = %d\n", theDateBuffer, theIPAddr.Ptr, thePercentLoss);
				ratchetLess = true;
			}
			else
			{
#if FLOW_CONTROL_DEBUGGING
				printf("Percent loss too high: Incrementing percent loss count\n");
#endif
				if ((fLogFile != NULL) && (inStream->GetCodecType() == RTPStream::kVideoCodecType))
					::fprintf(fLogFile, "%s %s Percent loss too high. Incrementing percent loss count. Percent loss = %d\n", theDateBuffer, theIPAddr.Ptr, thePercentLoss);
				theDictionary->SetValue(fNumLossesAboveTolerance, &theValue, sizeof(theValue));
				clearPercentLossThinCount = false;
			}
		}
		//check for a thickening condition
		else if (thePercentLoss < RTSPServerInterface::GetRTSPPrefs()->GetPercentLossThickTolerance())
		{
			UInt32 theMoreBandwidthMultiple = 0;
			theDictionary->GetValue(fNumLossesBelowTolerance, &theValue, sizeof(theValue), &theValueLen);
			Assert(theValueLen == sizeof(theValue));
			theDictionary->GetValue(fMoreBandwidthMultiple, &theMoreBandwidthMultiple, sizeof(theMoreBandwidthMultiple), &theValueLen);
			Assert(theValueLen == sizeof(theMoreBandwidthMultiple));
			Assert(theMoreBandwidthMultiple > 0);

			theValue++;//we must count this loss
			if (theValue >= (RTSPServerInterface::GetRTSPPrefs()->GetNumLossesToThick() * theMoreBandwidthMultiple))
			{
#if FLOW_CONTROL_DEBUGGING
				printf("Percent is low: ratcheting more\n");
#endif
				if ((fLogFile != NULL) && (inStream->GetCodecType() == RTPStream::kVideoCodecType))
					::fprintf(fLogFile, "%s %s Percent loss is low. ratcheting more. Percent loss = %d\n", theDateBuffer, theIPAddr.Ptr, thePercentLoss);
				ratchetMore = true;
				
				//Whenever we ratchet more here, we should increase the moreBandwidthMultiple. We
				//basically want to avoid a situation where we've thinned to a good bandwidth, but
				//we keep on ruining that because we keep on increasing the bandwidth when we shouldn't
				
				//theMoreBandwidthMultiple++;
				//theDictionary->SetValue(fMoreBandwidthMultiple, &theMoreBandwidthMultiple, sizeof(theMoreBandwidthMultiple));
			}
			else
			{
#if FLOW_CONTROL_DEBUGGING
				printf("Percent is low: Incrementing percent loss count\n");
#endif
				if ((fLogFile != NULL) && (inStream->GetCodecType() == RTPStream::kVideoCodecType))
					::fprintf(fLogFile, "%s %s Percent loss is low. Incrementing percent loss count. Percent loss = %d\n", theDateBuffer, theIPAddr.Ptr, thePercentLoss);
				theDictionary->SetValue(fNumLossesBelowTolerance, &theValue, sizeof(theValue));
				clearPercentLossThickCount = false;
			}			
		}
	}
	
	//Now take a look at the getting worse heuristic
	UInt16 isGettingWorse = theDictionary->GetStandardUInt16Value(StreamDictionary::kIsGettingWorse, &wasFound);
	if (wasFound)
	{
		if (isGettingWorse)
		{
			theDictionary->GetValue(fNumGettingWorses, &theValue, sizeof(theValue), &theValueLen);
			Assert(theValueLen == sizeof(theValue));
			theValue++;//we must count this getting worse
			
			//If we've gotten N number of getting worses, then thin. Otherwise, just
			//increment our count of getting worses
			if (theValue >= RTSPServerInterface::GetRTSPPrefs()->GetNumWorsesToThin())
			{
#if FLOW_CONTROL_DEBUGGING
				printf("Client reporting getting worse. Ratcheting less\n");
#endif
				if ((fLogFile != NULL) && (inStream->GetCodecType() == RTPStream::kVideoCodecType))
					::fprintf(fLogFile, "%s %s Client reporting getting worse. Ratcheting less\n", theDateBuffer, theIPAddr.Ptr);

				ratchetLess = true;
			}
			else
			{
#if FLOW_CONTROL_DEBUGGING
				printf("Client reporting getting worse. Incrementing num worses count\n");
#endif
				if ((fLogFile != NULL) && (inStream->GetCodecType() == RTPStream::kVideoCodecType))
					::fprintf(fLogFile, "%s %s Client reporting getting worse. Incrementing num worses count\n", theDateBuffer, theIPAddr.Ptr);
				theDictionary->SetValue(fNumGettingWorses, &theValue, sizeof(theValue));
			}
		}
	}

	//Finally, if we get a getting better, automatically ratchet up
	UInt16 isGettingBetter = theDictionary->GetStandardUInt16Value(StreamDictionary::kIsGettingBetter, &wasFound);
	if ((wasFound) && (isGettingBetter > 0))
		ratchetMore = true;

	//For clearing out counts below
	theValue = 0;

	//Based on the ratchetMore / ratchetLess variables, adjust the stream
	if (ratchetMore || ratchetLess)
	{
		if (ratchetLess)
			inStream->DecreaseQuality();
		else if (ratchetMore)
			inStream->IncreaseQuality();//note that if both bool's are set, only ratchetLess will run.

		//When adjusting the quality, ALWAYS clear out ALL our counts of EVERYTHING. Note
		//that this is the ONLY way that the fNumGettingWorses count gets cleared
		theDictionary->SetValue(fNumGettingWorses, &theValue, sizeof(theValue));
#if FLOW_CONTROL_DEBUGGING
		printf("Clearing num worses count\n");
#endif
		clearPercentLossThinCount = true;
		clearPercentLossThickCount = true;
	}
	
	//clear thick / thin counts if we are supposed to.
	if (clearPercentLossThinCount)
	{
#if FLOW_CONTROL_DEBUGGING
		printf("Clearing num losses above tolerance count\n");
#endif
		theDictionary->SetValue(fNumLossesAboveTolerance, &theValue, sizeof(theValue));
	}
	if (clearPercentLossThickCount)
	{
#if FLOW_CONTROL_DEBUGGING
		printf("Clearing num losses below tolerance count\n");
#endif
	
		theDictionary->SetValue(fNumLossesBelowTolerance, &theValue, sizeof(theValue));	
	}
}

void	RTCPFlowControlModule::InitializeDictionaryItems(StreamDictionary* inDictionary)
{
	UInt32 theValue = 0;
	UInt32 theValueLen = 0;
	inDictionary->GetValue(fNumLossesAboveTolerance, &theValue, sizeof(theValue), &theValueLen);
	if (theValueLen == 0)
	{
		//If one is not yet in the dictionary, they all aren't in the dictionary
		inDictionary->SetValue(fNumLossesAboveTolerance, &theValue, sizeof(theValue));
		inDictionary->SetValue(fNumLossesBelowTolerance, &theValue, sizeof(theValue));
		inDictionary->SetValue(fNumGettingWorses, &theValue, sizeof(theValue));
		inDictionary->SetValue(fNumGettingBetters, &theValue, sizeof(theValue));
		
		//the initial multiple should be 1 (we don't want to multiply by 0!)
		theValue = 1;
		inDictionary->SetValue(fMoreBandwidthMultiple, &theValue, sizeof(theValue));
	}
}
